/**
***************************************************************************
* @file dlrNumeric/array1D.h
*
* Header file declaring Array1D class.
*
* Copyright (C) 2001-2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1182 $
* $Date: 2009-09-09 11:25:46 -0400 (Wed, 09 Sep 2009) $
***************************************************************************
**/

#ifndef _DLR_ARRAY1D_H_
#define _DLR_ARRAY1D_H_

#include <iostream>
#include <string>
#include <dlrCommon/exception.h>

namespace dlr {


  /**
   ** This namespace contains code for 1D, 2D, and 3D arrays,
   ** matrices, coordinate transformations, rotation conversions, and
   ** much more.
   **/
  namespace numeric {
    
    /**
     ** The Array1D class template represents a 1D array of arbitrary type.
     ** This class has internal reference counting.
     **
     ** IMPORTANT: This class does _shallow_ copies by default.  If you
     ** type:
     **
     **       array1 = array2;
     **
     ** then array1 and array2 point to the same data.  To do a deep copy,
     ** type
     **
     **       array1.copy(array2);
     **
     ** or
     **
     **       array1 = array2.copy();
     **
     ** The advantage of the first form is that it doesn't involve allocating
     ** memory.  The advantage of the second form is that there's no error if
     ** array1 and array2 have different shapes/sizes.
     **/
    template <class Type>
    class Array1D {
    public:
      /* ======== Public typedefs ======== */

      /**
       ** Typedef for value_type describes the contents of the array.
       **/
      typedef Type value_type;

      /**
       ** Typedef for iterator type helps with standard library interface.
       **/
      typedef Type* iterator;
    
      /**
       ** Typedef for const_iterator type helps with standard library
       ** interface.
       **/
      typedef const Type* const_iterator;

      /* ======== Public member functions ======== */

      /** 
       * Default constructor initializes to zero size.
       */
      Array1D();
    
      /** 
       * Constructs an "arraySize" element array.
       * 
       * @param arraySize Number of elements in the array after successful
       * construction.
       */
      explicit
      Array1D(size_t arraySize);

      /** 
       * Constructs an Array1D by parsing an input string.  For example,
       * the code line
       *
       *   Array1D<double> testArray("[1.0, 2.0, 3.0]");
       *
       * would construct a 3 element array.
       * 
       * @param inputString A formatted string which could be reasonably
       * expected to parse into the desired array values.
       */
      explicit
      Array1D(const std::string& inputString);
    
      /** 
       * The copy constructor does a shallow copy.  The newly created
       * array points to the same data as copied array.
       * 
       * @param source The Array1D<> instance to be copied.
       */
      Array1D(const Array1D<Type> &source);
    
      /**
       * Construct an array around external data.  Arrays constructed
       * in this way will not implement reference counting, and will
       * not delete dataPtr when done.
       * 
       * @param arraySize Number of elements in the array after construction.
       * 
       * @param dataPtr A C-style array of Type into which the newly
       * constructed Array1D should index.
       */
      Array1D(size_t arraySize, Type* const dataPtr);
    
      /**
       * Construct an array around external data which was allocated by
       * an Array?D instance.  Arrays constructed in this way _do_
       * implement reference counting, and will delete dataPtr when
       * done.  This constructor is provided primarily so that higher
       * dimensionality array classes can return Array1D instances which
       * reference their data without being friend classes.  Caveat
       * emptor.
       * 
       * @param arraySize Number of elements in the array after construction.
       * 
       * @param dataPtr A C-style array of Type into which the newly
       * constructed Array1D should index.
       * 
       * @param referenceCountPtr Pointer to size_t indicating the number of
       * Array classes currently using dataPtr.
       */
      Array1D(size_t arraySize, Type* const dataPtr,
              size_t* const referenceCountPtr);
    
      /**
       * Destroys the Array1D instance and deletes the internal data
       * store if no remaining arrays point to it.
       */
      ~Array1D();

      /** 
       * Return begin() iterator for Standard Library algorithms.
       * 
       * @return Iterator pointing to the first element of the array.
       */
      iterator
      begin() {return m_dataPtr;}

      /** 
       * Return begin() const_iterator for Standard Library algorithms.
       * 
       * @return Const iterator pointing to the first element of the
       * array.
       */
      const_iterator
      begin() const {return m_dataPtr;}


      /** 
       * Optionally throw an exception if the shape of *this is
       * different than specified.
       *
       * @param arraySize The required array size.
       */
      inline void
      checkDimension(size_t arraySize) const;

    
      /** 
       * Reset the array to zero size, abandoning all contents.  This is
       * equivalent to this->reinit(0);
       */
      void
      clear() {this->reinit(0);}
    
      /** 
       * Allocate a new array and deep copy the contents of *this.
       * 
       * @return A new array which is a (deep) copy of *this.
       */
      Array1D<Type>
      copy() const;

      /** 
       * Deep copies the contents of source.  It is an error if source
       * does not have the same size as *this.
       *
       * @param source The array to be copied.
       *
       * @exception ValueException on incompatible array sizes
       *
       * @see void Array1D<Type>::copy(const Type2* dataPtr)
       */
      template <class Type2> void
      copy(const Array1D<Type2>& source);

      /**
       * Copies elements from dataPtr.  There must be valid data at all
       * addresses from dataPtr to (dataPtr + this->size());
       * 
       * @param dataPtr Pointer to the data to be copied.
       */
      template <class Type2> void
      copy(const Type2* dataPtr);

      /**
       * Returns a pointer to the internal data store.  This is ugly
       * but often necessary for interfacing with external libraries.
       * Data is stored contiguously.
       * 
       * @return Pointer to the internal data store.
       */
      Type*
      data() {return m_dataPtr;}

      /**
       * This version of data(void) is appropriate for const Array1D,
       * and returns a pointer-to-const.
       * 
       * @return Const pointer to the internal data store.
       */
      const Type*
      data() const {return m_dataPtr;}

      /** 
       * Just like data(void), which is documented above, but returns a
       * pointer to the (index)th element instead of the first element.
       * Note that "data(index)" is synonymous with "data() + index" .
       * 
       * @param index Indicates to which element of the array the return
       * value should point.
       * 
       * @return Pointer to the (index)th element of the array.
       */
      Type*
      data(size_t index) {
        this->checkBounds(index);
        return m_dataPtr + index;
      }
    
      /**
       * This version of data(size_t) is appropriate for const Array1D,
       * and returns a pointer-to-const.
       * 
       * @param index Indicates to which element of the array the return
       * value should point.
       * 
       * @return Const pointer to the (index)th element of the array.
       */
      const Type*
      data(size_t index) const {
        this->checkBounds(index);
        return m_dataPtr + index;
      }


      /** 
       * This member function returns true if the array instance
       * contains no elements.  It has complexity O(1).
       * 
       * @return The return value indicates whether or not the array is
       * empty.
       */
      bool
      empty() const {return this->size() == 0;}

    
      /** 
       * Return end() iterator for Standard Library algorithms.
       * 
       * @return Iterator pointing just past the last element of
       * the array.
       */
      iterator
      end() {return m_dataPtr + m_size;}

      /** 
       * Return end() const_iterator for Standard Library algorithms.
       * 
       * @return Const iterator pointing just past the last element of
       * the array.
       */
      const_iterator
      end() const {return m_dataPtr + m_size;}

      
      /** 
       * This member function returns a specific element of the array
       * by value.
       * 
       * @param index0 This argument specifies which element value
       * should be returned.
       * 
       * @return The return value is a copy of the requested element.
       */
      inline Type
      getElement(size_t index0) const {return this->operator()(index0);}


      /** 
       * Indicates whether the internal data array is being managed (and
       * reference counted) by *this.  This member function is only
       * needed in very unusual circumstances.
       * 
       * @return The return value is a bool indicating whether the internal
       * data is being managed by *this.
       */
      bool
      isAllocated() const {return m_isAllocated;}

      /** 
       * Returns the number of elements in the array.  This is a synonym
       * for size().
       * 
       * @return The number of elements in the array.
       */
      size_t
      length() const {return this->size();}

      /** 
       * This member function sets the value of the array from an input
       * stream.  The array is modified only if the read was successful,
       * otherwise the array is not modified, and failbit is set in the
       * stream state.  Because of this nice behavior, readFromStream is
       * quite slow.
       * 
       * @param inputStream This is the stream from which to read the
       * array.
       * 
       * @return The return value is a reference to inputStream.
       */
      std::istream&
      readFromStream(std::istream& inputStream);

      /**
       * Temporary function to ease porting of code from early versions
       * of dlr_libs.  The use of this function is discouraged.
       * 
       * @return The return value is a pointer to the internal reference
       * count.
       */
      size_t*
      refCountPtr() const {return m_refCountPtr;}
    

      /** 
       * Changes the shape of the array and reallocates storage.
       * The current array contents are lost.
       * 
       * @param size Number of elements in the array after reallocation.
       */
      void
      reinit(size_t arraySize);
    

      /** 
       * Behaves just like member function reinit() unless
       * this->size() matches the requested size, in which case this
       * member function does nothing.
       * 
       * @param size Required number of elements in the array.
       */
      inline void
      reinitIfNecessary(size_t arraySize);
    

      /** 
       * This member function sets the value of a specific element of
       * the array.
       * 
       * @param index0 This argument specifies which element value
       * should be set.
       * 
       * @param value This argument will be copied into the selected
       * array element.
       * 
       * @return The return value is a reference to the newly set
       * array element.
       */
      Type&
      setElement(size_t index0, const Type& value) {
        return this->operator()(index0) = value;
      }


      /** 
       * Returns the number of elements in the array.  This is a synonym
       * for length().
       * 
       * @return The number of elements in the array.
       */
      size_t
      size() const {return m_size;}
  
      /** 
       * Assignment operator shallow copies the contents of source.
       * After the copy, both arrays reference the same data.
       * 
       * @param source The Array1D instance to be copied.
       * 
       * @return Reference to *this.
       */
      Array1D<Type>&
      operator=(const Array1D<Type>& source);

      /** 
       * Assign value to every element in the array.
       * 
       * @param value The value to be copied.
       * 
       * @return Reference to *this.
       */
      Array1D<Type>&
      operator=(Type value);

      /** 
       * Returns the (index)th element of the array by reference.
       * 
       * @param index Indicates the selected element.
       * 
       * @return Reference to the (index)th element of the array.
       */
      Type&
      operator()(size_t index) {
        this->checkBounds(index);
        return m_dataPtr[index];
      }
  
      /** 
       * Returns the (index)th element of the array by value.
       * 
       * @param index Indicates the selected element.
       * 
       * @return Value of the (index)th element of the array.
       */
      Type operator()(size_t index) const {
        this->checkBounds(index);
        return m_dataPtr[index];
      }

      /** 
       * Returns the (index)th element of the array by reference.
       * Synonymous with operator()().
       * 
       * @param index Indicates the selected element.
       * 
       * @return Reference to the (index)th element of the array.
       */
      Type& operator[](size_t index) {return this->operator()(index);}

      /** 
       * Returns the (index)th element of the array by value.
       * Synonymous with operator()() const.
       * 
       * @param index Indicates the selected element.
       * 
       * @return Value of the (index)th element of the array.
       */
      Type operator[](size_t index) const {return this->operator()(index);}

      /** 
       * Increments each element of *this by the value of the
       * corresponding element of arg.
       * 
       * @param arg Array1D of values to be added to the elements of
       * *this.     
       *
       * @exception ValueException on incompatible array sizes
       * 
       * @return Reference to *this.
       */
      template <class Type2>
      Array1D<Type>&
      operator+=(const Array1D<Type2>& arg);
    
      /** 
       * Increments each element of *this by a constant.
       * 
       * @param arg Value by which array elements will be incremented.
       * 
       * @return Reference to *this.
       */
      Array1D<Type>&
      operator+=(const Type arg);

      /** 
       * Decrements each element of *this by the value of the
       * corresponding element of arg.
       * 
       * @param arg Array1D of values to be subtracted from the elements
       * of *this.
       *
       * @exception ValueException on incompatible array sizes
       * 
       * @return Reference to *this.
       */
      template <class Type2>
      Array1D<Type>&
      operator-=(const Array1D<Type2>& arg);

      /** 
       * Decrements each element of *this by a constant.
       * 
       * @param arg Value by which array elements will be decremented.
       * 
       * @return Reference to *this.
       */
      Array1D<Type>&
      operator-=(const Type arg);

      /** 
       * Multiplies each element of *this by the value of the
       * corresponding element of arg.
       * 
       * @param arg Array1D of values by which the elements of *this
       * should be multiplied.
       *
       * @exception ValueException on incompatible array sizes
       * 
       * @return Reference to *this.
       */
      template <class Type2>
      Array1D<Type>&
      operator*=(const Array1D<Type2>& arg);

      /** 
       * Multiplies each element of *this by a constant.
       * 
       * @param arg Value by which array elements will be multiplied.
       * 
       * @return Reference to *this.
       */
      Array1D<Type>&
      operator*=(const Type arg);

      /** 
       * Divides each element of *this by the value of the corresponding
       * element of arg.
       * 
       * @param arg Array1D of values by which the elements of *this
       * should be divided.
       *
       * @exception ValueException on incompatible array sizes
       * 
       * @return Reference to *this.
       */
      template <class Type2>
      Array1D<Type>&
      operator/=(const Array1D<Type2>& arg);

      /** 
       * Divides each element of *this by a constant.
       * 
       * @param arg Value by which array elements will be divided.
       * 
       * @return Reference to *this.
       */
      Array1D<Type>&
      operator/=(const Type arg);

    private:
      /* ======== Private member functions ======== */

      /** 
       * Allocate memory for array data and for reference count.
       */
      void
      allocate();

      /** 
       * Optionally throw an exception if index is beyond the range of
       * this array.
       *
       * @param index The index to check.
       */
      inline void
      checkBounds(size_t index) const;


      /** 
       * Release the memory for array data by decrementing the reference
       * count, or by returning the memory to the heap if it is not
       * referenced by any remaining Array instances.
       */
      void
      deAllocate();

      // Constants to help with formatting.  We use the initialization
      // on first use paradigm for the string constants to avoid
      // headaches.

      /**
       * Static constant describing how the string representation of an
       * Array1D should start.
       */
      static const std::string& ioIntro(); 

      /**
       * Static constant describing how the string representation of an
       * Array1D should end.
       */
      static const std::string& ioOutro();

      /**
       * Static constant describing how the the data portion of the
       * string representation of an Array1D should start.
       */
      static const char ioOpening = '[';

      /**
       * Static constant describing how the the data portion of the
       * string representation of an Array1D should end.
       */
      static const char ioClosing = ']';

      /**
       * Static constant describing how individual elements should be
       * separated in the string representation of Array1D.
       */
      static const char ioSeparator = ',';
    
      /* ======== Private data members ======== */
      size_t m_size;
      Type* m_dataPtr;
      size_t* m_refCountPtr;
      bool m_isAllocated;

    };

    /* Non-member functions which should maybe wind up in a different file */

    /** 
     * Elementwise addition of Array1D instances.
     * 
     * @param array0 First argument for addition.
     *
     * @param array1 Second argument for addition.
     *
     * @exception ValueException on incompatible array sizes
     *
     * @return Array1D instance in which the value of each element is
     * the sum of the values of the corresponding elements of the two
     * Array1D arguments.
     */
    template <class Type>
    Array1D<Type>
    operator+(const Array1D<Type>& array0, const Array1D<Type>& array1);
  
    /** 
     * Elementwise subtraction of Array1D instances.
     * 
     * @param array0 First argument for subtraction.
     *
     * @param array1 Second argument for subtraction.
     *
     * @exception ValueException on incompatible array sizes
     *
     * @return Array1D instance in which the value of each element is
     * the difference of the values of the corresponding elements of the two
     * Array1D arguments.
     */
    template <class Type>
    Array1D<Type>
    operator-(const Array1D<Type>& array0, const Array1D<Type>& array1);

    /** 
     * Elementwise multiplication of Array1D instances.
     * 
     * @param array0 First argument for multiplication.
     *
     * @param array1 Second argument for multiplication.
     *
     * @exception ValueException on incompatible array sizes
     *
     * @return Array1D instance in which the value of each element is
     * the product of the values of the corresponding elements of the two
     * Array1D arguments.
     */
    template <class Type>
    Array1D<Type>
    operator*(const Array1D<Type>& array0, const Array1D<Type>& array1);
  
    /** 
     * Elementwise division of Array1D instances.
     * 
     * @param array0 First argument for division.
     *
     * @param array1 Second argument for division.
     *
     * @exception ValueException on incompatible array sizes
     *
     * @return Array1D instance in which the value of each element is
     * the dividend of the values of the corresponding elements of the two
     * Array1D arguments.
     */
    template <class Type>
    Array1D<Type>
    operator/(const Array1D<Type>& array0, const Array1D<Type>& array1);

    /** 
     * Addition of Array1D and scalar.
     * 
     * @param array Array1D argument of the addition.
     *
     * @param scalar Scalar argument of the addition.
     *
     * @return Array1D instance in which the value of each element is
     * the sum of the corresponding element of the Array1D argument and
     * the scalar argument.
     */
    template <class Type>
    Array1D<Type> operator+(const Array1D<Type>& array, Type scalar);

    /** 
     * Subtraction of Array1D and scalar.
     * 
     * @param array0 Array1D argument of the subtraction.
     *
     * @param scalar Scalar argument of the subtraction.
     *
     * @return Array1D instance in which the value of each element is
     * the difference of the corresponding element of the Array1D
     * argument and the scalar argument.
     */
    template <class Type>
    Array1D<Type> operator-(const Array1D<Type>& array0, Type scalar);

    /** 
     * Multiplication of Array1D and scalar.
     * 
     * @param array0 Array1D argument of the multiplication.
     *
     * @param scalar Scalar argument of the multiplication.
     *
     * @return Array1D instance in which the value of each element is
     * the product of the corresponding element of the Array1D argument
     * and the scalar argument.
     */
    template <class Type>
    Array1D<Type> operator*(const Array1D<Type>& array0, Type scalar);

    /** 
     * Division of Array1D and scalar.
     * 
     * @param array0 Array1D argument of the division.
     *
     * @param scalar Scalar argument of the division.
     *
     * @return Array1D instance in which the value of each element is
     * the difference of the corresponding element of the Array1D
     * argument and the scalar argument.
     */
    template <class Type>
    Array1D<Type> operator/(const Array1D<Type>& array0, Type scalar);

    /** 
     * Addition of scalar and Array1D.
     * 
     * @param scalar Scalar argument of the addition.
     *
     * @param array0 Array1D argument of the addition.
     *
     * @return Array1D instance in which the value of each element is
     * the sum of the scalar argument and the corresponding element of
     * the Array1D argument.
     */
    template <class Type>
    inline Array1D<Type> operator+(Type scalar, const Array1D<Type>& array0);


    /** 
     * Subtraction of scalar and Array1D.
     * 
     * @param scalar Scalar argument of the subtraction.
     *
     * @param array0 Array1D argument of the subtraction.
     *
     * @return Array1D instance in which the value of each element is
     * the difference of the scalar argument and the corresponding
     * element of the Array1D argument.
     */
    template <class Type>
    inline Array1D<Type> operator-(Type scalar, const Array1D<Type>& array0);

    /** 
     * Multiplication of scalar and Array1D.
     * 
     * @param scalar Scalar argument of the multiplication.
     *
     * @param array0 Array1D argument of the multiplication.
     *
     * @return Array1D instance in which the value of each element is
     * the product of the scalar argument and the corresponding element
     * of the Array1D argument.
     */
    template <class Type>
    inline Array1D<Type> operator*(Type scalar, const Array1D<Type>& array0);

    /** 
     * Division of scalar and Array1D.
     * 
     * @param scalar Scalar argument of the division.
     *
     * @param array0 Array1D argument of the division.
     *
     * @return Array1D instance in which the value of each element is
     * the dividend of the scalar argument and the corresponding element
     * of the Array1D argument.
     */
    template <class Type>
    inline Array1D<Type> operator/(Type scalar, const Array1D<Type>& array0);


    /** 
     * Elementwise comparison of an Array1D with a constant.
     *
     * @param array0 An Array1D instance.
     * 
     * @param arg Value to which the elements of array0 should be
     * compared.
     * 
     * @return An Array1D<bool> in which each element has value "true"
     * if the corresponding element of array0 is equal to arg.
     */
    template <class Type>
    Array1D<bool>
    operator==(const Array1D<Type>& array0, const Type arg);

    
    /** 
     * Elementwise comparison of an Array1D with another array.
     * 
     * @param array0 An Array1D instance.
     *
     * @param array1 A second Array1D instance with the same size as
     * array0.
     * 
     * @return An Array1D<bool> in which each element has value "true"
     * if the corresponding element of array0 is equal to the
     * corresponding element of array1.
     */
    template <class Type>
    Array1D<bool>
    operator==(const Array1D<Type>& array0, const Array1D<Type>& array1);
    

    /** 
     * Elementwise comparison of an Array1D with a constant.
     * 
     * @param array0 An Array1D instance.
     *
     * @param arg Value to which array elements should be compared.
     * 
     * @return An Array1D<bool> in which each element has value "true"
     * if the corresponding element of array0 is greater than arg.
     */
    template <class Type>
    Array1D<bool>
    operator>(const Array1D<Type>& array0, const Type arg);

  
    /** 
     * Elementwise comparison of an Array1D with a constant.
     * 
     * @param array0 An Array1D instance.
     * 
     * @param arg Value to which array elements should be compared.
     * 
     * @return An Array1D<bool> in which each element has value "true"
     * if the corresponding element of array0 is greater than or equal
     * to arg.
     */
    template <class Type>
    Array1D<bool>
    operator>=(const Array1D<Type>& array0, const Type arg);


    /** 
     * Elementwise comparison of an Array1D with a constant.
     * 
     * @param array0 An Array1D instance.
     * 
     * @param arg Value to which array elements should be compared.
     * 
     * @return An Array1D<bool> in which each element has value "true"
     * if the corresponding element of array0 is less than arg.
     */
    template <class Type>
    Array1D<bool>
    operator<(const Array1D<Type>& array0, const Type arg);


    /** 
     * Elementwise comparison of an Array1D with a constant.
     * 
     * @param array0 An Array1D instance.
     * 
     * @param arg Value to which array elements should be compared.
     * 
     * @return An Array1D<bool> in which each element has value "true"
     * if the corresponding element of array0 is less than or equal to
     * arg.
     */
    template <class Type>
    Array1D<bool>
    operator<=(const Array1D<Type>& array0, const Type arg);

  
    /** 
     * Outputs a text representation of an Array1D instance to a
     * std::ostream.  The output format looks like this:
     *
     * Array1D([1, 2, 4, 8, 16])
     *
     * Where the array elements are output using
     * operator<<(std::ostream&, const Type&)
     * and each element is separated from its neighbors by a comma and
     * whitespace.
     * 
     * @param stream Reference to the the output stream.
     *
     * @param array0 const Reference to the Array1D to be output.
     *
     * @exception IOException on invalid stream.
     *
     * @return Reference to output stream.
     */
    template <class Type>
    std::ostream& operator<<(std::ostream& stream, const Array1D<Type>& array0);

    /** 
     * Sets the value of an Array1D instance from a std::istream.
     * The input format is as described for
     * operator<<(std::ostream&, const Array1D<Type>&) above.
     * 
     * @param stream Reference to the the input stream.
     *
     * @param array0 Reference to the Array1D which will take the input.
     *
     * @return Reference to input stream.
     */
    template <class Type>
    std::istream&
    operator>>(std::istream& stream, Array1D<Type>& array0);

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::Array1D;

} // namespace dlr


/* *****************************************************************
 * Member function definitions follow.  This would be a .C file
 * if it weren't templated.
 * ***************************************************************** */

#include <algorithm>
#include <sstream>
#include <vector>
#include <dlrCommon/inputStream.h>
#include <dlrNumeric/numericTraits.h>

namespace dlr {

  namespace numeric {
    
    // Static constant describing how the string representation of an
    // Array1D should start.
    template <class Type>
    const std::string&
    Array1D<Type>::
    ioIntro()
    {
      static const std::string intro = "Array1D(";
      return intro;
    }


    // Static constant describing how the string representation of an
    // Array1D should end.
    template <class Type>
    const std::string&
    Array1D<Type>::
    ioOutro()
    {
      static const std::string outro = ")";
      return outro;
    }

    // Non-static member functions below.

    template <class Type>
    Array1D<Type>::
    Array1D()
      : m_size(0),
        m_dataPtr(0),
        m_refCountPtr(0),
        m_isAllocated(false)
    {
      // Empty.
    }

  
    template <class Type>
    Array1D<Type>::
    Array1D(size_t arraySize)
      : m_size(arraySize),
        m_dataPtr(0),           // This will be set in the call to allocate().
        m_refCountPtr(0), // This will be set in the call to allocate().
        m_isAllocated(false)
    {
      this->allocate();
    }


    // Construct from an initialization string.
    template <class Type>
    Array1D<Type>::
    Array1D(const std::string& inputString)
      : m_size(0),
        m_dataPtr(0),
        m_refCountPtr(0),
        m_isAllocated(false)
    {
      // We'll use the stream input operator to parse the string.
      std::istringstream inputStream(inputString);

      // Now read the string into an array.
      Array1D<Type> inputArray;
      inputStream >> inputArray;
      if(!inputStream) {
        std::ostringstream message;
        message << "Couldn't parse input string: \"" << inputString << "\".";
        DLR_THROW3(ValueException, "Array1D::Array1D(const std::string&)",
                   message.str().c_str());                 
      }

      // If all went well, copy into *this.
      *this = inputArray;
    }

  
    /* When copying from a Array1D do a shallow copy */
    /* Update reference count if the array we're copying has */
    /* valid data. */
    template <class Type>
    Array1D<Type>::
    Array1D(const Array1D<Type>& source)
      : m_size(source.m_size),
        m_dataPtr(source.m_dataPtr),
        m_refCountPtr(source.m_refCountPtr),
        m_isAllocated(source.m_isAllocated)
    {
      if(m_isAllocated) {
        ++(*m_refCountPtr);
      }
    }


    template <class Type>
    Array1D<Type>::
    Array1D(size_t arraySize, Type* const dataPtr)
      : m_size(arraySize),
        m_dataPtr(dataPtr),
        m_refCountPtr(NULL),
        m_isAllocated(false)
    {
      // empty
    }


    template <class Type>
    Array1D<Type>::
    Array1D(size_t arraySize, Type* const dataPtr,
            size_t* const referenceCountPtr)
      : m_size(arraySize),
        m_dataPtr(dataPtr),
        m_refCountPtr(referenceCountPtr),
        m_isAllocated(true)
    {
      ++(*m_refCountPtr);
    }


    template <class Type>
    Array1D<Type>::~Array1D()
    {
      deAllocate();
    }


    template <class Type>
    inline void Array1D<Type>::
    checkDimension(size_t arraySize) const
    {
#ifdef _DLRNUMERIC_CHECKBOUNDS_
      if(arraySize != this->size()) {
        std::ostringstream message;
        message << "Size mismatch: required size is " << arraySize
                << " while *this has dimension " << this->size() << ".";
        DLR_THROW(IndexException, "Array1D::checkDimension()",
                  message.str().c_str());
      }
#endif
    }


    template <class Type>
    Array1D<Type> Array1D<Type>::
    copy() const
    {
      Array1D<Type> newArray(m_size);
      newArray.copy(*this);
      return newArray;
    }

  
    template <class Type> template <class Type2>
    void Array1D<Type>::
    copy(const Array1D<Type2>& source)
    {
      if(source.size() != m_size) {
        std::ostringstream message;
        message << "Mismatched array sizes. Source array has "
                << source.size() << " elements, while destination array has "
                << m_size << " elements.";
        DLR_THROW3(ValueException, "Array1D::copy(const Array1D&)",
                   message.str().c_str());
      }
      if(m_size != 0) {
        this->copy(source.data());
      }
    }

  
    template <class Type> template <class Type2>
    void
    Array1D<Type>::
    copy(const Type2* dataPtr)
    {
      std::copy(dataPtr, dataPtr + m_size, m_dataPtr);
    }


    template <class Type>
    Array1D<Type>& Array1D<Type>::
    operator=(Type val)
    {
      std::fill(m_dataPtr, m_dataPtr + m_size, val);
      return *this;
    }


    // This member function sets the value of the array from an input
    // stream.
    template <class Type>
    std::istream&
    Array1D<Type>::
    readFromStream(std::istream& inputStream)
    {
      // Most of the time, InputType will be the same as Type.
      typedef typename NumericTraits<Type>::TextOutputType InputType;

      // If stream is in a bad state, we can't read from it.
      if (!inputStream){
        return inputStream;
      }
    
      // It's a lot easier to use a try block than to be constantly
      // testing whether the IO has succeeded, so we tell inputStream to
      // complain if anything goes wrong.
      std::ios_base::iostate oldExceptionState = inputStream.exceptions();
      inputStream.exceptions(
        std::ios_base::badbit | std::ios_base::failbit | std::ios_base::eofbit);

      // Now on with the show.
      try{
        // Construct an InputStream instance so we can use our
        // convenience functions.
        InputStream stream(inputStream);

        stream.skipWhiteSpace();
      
        // We won't require the input format to start with "Array1D(", but
        // if it does we read it here.
        bool foundIntro = false;
        if(stream.peek() == ioIntro()[0]) {
          foundIntro = true;
          stream.expect(ioIntro());
        }

        // OK.  We've dispensed with the intro.  What's left should be of
        // the format "[#, #, #, ...]".  We require the square brackets to
        // be there.
        stream.expect(ioOpening);

        // Read the data.
        InputType inputValue;
        std::vector<Type> inputBuffer;
        while(1) {
          // Read the next value.
          stream >> inputValue;
          inputBuffer.push_back(static_cast<Type>(inputValue));

          // Read the separator, or else the closing character.
          char inChar = 0;
          stream >> inChar;
          if(inChar == ioClosing) {
            // Found a closing.  Stop here.
            break;
          }
          if(inChar != ioSeparator) {
            // Missing separator.  Fail here.
            stream.clear(std::ios_base::failbit);
          }
        }

        // If we found an intro, we expect the corresponding outro.
        if(foundIntro) {
          stream.expect(ioOutro());
        }

        // Now we're done with all of the parsing.  Copy the data to *this.
        this->reinit(inputBuffer.size());
        std::copy(inputBuffer.begin(), inputBuffer.end(), this->begin());

      } catch(std::ios_base::failure) {
        // Empty
      }
      inputStream.exceptions(oldExceptionState);
      return inputStream;
    }
  

    template <class Type>
    void Array1D<Type>::
    reinit(size_t arraySize)
    {
      this->deAllocate();
      this->m_size = arraySize;
      this->allocate();
    }
  

    template <class Type>
    void Array1D<Type>::
    reinitIfNecessary(size_t arraySize)
    {
      if(this->size() != arraySize) {
        this->reinit(arraySize);
      }
    }
  

    template <class Type>
    Array1D<Type>& Array1D<Type>::
    operator=(const Array1D<Type>& source)
    {
      // Check for self-assignment
      if(&source != this) {
        this->deAllocate();
        m_size = source.m_size;
        m_dataPtr = source.m_dataPtr;
        m_refCountPtr = source.m_refCountPtr;
        m_isAllocated = source.m_isAllocated;
        // Update reference count, if appropriate.
        if(m_isAllocated) {
          ++(*m_refCountPtr);
        }
      }
      return *this;
    }


    template <class Type> template <class Type2>
    Array1D<Type>&
    Array1D<Type>::
    operator+=(const Array1D<Type2>& arg)
    {
      if(m_size != arg.size()) {
        std::ostringstream message;
        message << "Mismatched array sizes. Argument array has "
                << arg.size() << " elements, while destination array has "
                << m_size << " elements.";
        DLR_THROW3(ValueException, "Array1D::operator+=(const Array1D&)",
                   message.str().c_str());
      }
      std::transform(m_dataPtr, m_dataPtr + m_size, arg.data(), m_dataPtr,
                     std::plus<Type>());
      return *this;
    }


    template <class Type>
    Array1D<Type>&
    Array1D<Type>::
    operator+=(const Type arg)
    {
      std::transform(m_dataPtr, m_dataPtr + m_size, m_dataPtr,
                     std::bind2nd(std::plus<Type>(), arg));
      return *this;
    }


    template <class Type> template <class Type2>
    Array1D<Type>&
    Array1D<Type>::
    operator-=(const Array1D<Type2>& arg)
    {
      if(m_size != arg.size()) {
        std::ostringstream message;
        message << "Mismatched array sizes. Argument array has "
                << arg.size() << " elements, while destination array has "
                << m_size << " elements.";
        DLR_THROW3(ValueException, "Array1D::operator-=(const Array1D&)",
                   message.str().c_str());
      }
      std::transform(m_dataPtr, m_dataPtr + m_size, arg.data(), m_dataPtr,
                     std::minus<Type>());
      return *this;
    }


    template <class Type>
    Array1D<Type>&
    Array1D<Type>::
    operator-=(const Type arg)
    {
      std::transform(m_dataPtr, m_dataPtr + m_size, m_dataPtr,
                     std::bind2nd(std::minus<Type>(), arg));
      return *this;
    }


    template <class Type> template <class Type2>
    Array1D<Type>&
    Array1D<Type>::
    operator*=(const Array1D<Type2>& arg)
    {
      if(m_size != arg.size()) {
        std::ostringstream message;
        message << "Mismatched array sizes. Argument array has "
                << arg.size() << " elements, while destination array has "
                << m_size << " elements.";
        DLR_THROW3(ValueException, "Array1D::operator*=(const Array1D&)",
                   message.str().c_str());
      }
      std::transform(m_dataPtr, m_dataPtr + m_size, arg.data(), m_dataPtr,
                     std::multiplies<Type>());
      return *this;
    }


    template <class Type>
    Array1D<Type>&
    Array1D<Type>::
    operator*=(const Type arg)
    {
      std::transform(m_dataPtr, m_dataPtr + m_size, m_dataPtr,
                     std::bind2nd(std::multiplies<Type>(), arg));
      return *this;
    }


    template <class Type> template <class Type2>
    Array1D<Type>&
    Array1D<Type>::
    operator/=(const Array1D<Type2>& arg)
    {
      if(m_size != arg.size()) {
        std::ostringstream message;
        message << "Mismatched array sizes. Argument array has "
                << arg.size() << " elements, while destination array has "
                << m_size << " elements.";
        DLR_THROW3(ValueException, "Array1D::operator/=(const Array1D&)",
                   message.str().c_str());
      }
      std::transform(m_dataPtr, m_dataPtr + m_size, arg.data(), m_dataPtr,
                     std::divides<Type>());
      return *this;
    }


    template <class Type>
    Array1D<Type>&
    Array1D<Type>::
    operator/=(const Type arg)
    {
      std::transform(m_dataPtr, m_dataPtr + m_size, m_dataPtr,
                     std::bind2nd(std::divides<Type>(), arg));
      return *this;
    }


    template <class Type>
    void Array1D<Type>::
    allocate()
    {
      // First check array size.  It doesn't make sense to allocate memory
      // for a zero size array.
      if(m_size > 0) {
        // Allocate data storage, and a new reference count.  new() should
        // throw an exception if we run out of memory.
        m_dataPtr = new(Type[m_size]);
        m_refCountPtr = new size_t;
        // Set reference count to show that exactly one Array is pointing
        // to this data.
        *m_refCountPtr = 1;
        m_isAllocated = true;
      } else {
        // Array size is zero, so no need to allocate memory.
        m_dataPtr = 0;
        m_refCountPtr = 0;
        m_isAllocated = false;
      }
      return;
    }


    template <class Type>
    inline void Array1D<Type>::
    checkBounds(size_t index) const
    {
#ifdef _DLRNUMERIC_CHECKBOUNDS_
      if(index >= m_size) {
        std::ostringstream message;
        message << "Index " << index << " is invalid for a(n) " << m_size
                << " element array.";
        DLR_THROW(IndexException, "Array1D::checkBounds()",
                  message.str().c_str());
      }
#endif
    }


    template <class Type>
    void Array1D<Type>::
    deAllocate()
    {
      // Are we responsible for deallocating the contents of this array?
      if(m_isAllocated == true) {
        // If yes, are we currently the only array pointing to this data?
        if(--(*m_refCountPtr) == 0) {
          // If yes, then delete both data and reference count, and change
          // m_isAllocated to reflect this.
          delete[] m_dataPtr;
          delete m_refCountPtr;
          m_isAllocated = false;
        }
      }
      // Abandon our pointers to data and reference count.
      m_dataPtr = 0;
      m_refCountPtr = 0;
    }

    /* Non-member functions which should maybe wind up in a different file? */
  
    template <class Type>
    Array1D<Type> operator+(const Array1D<Type>& array0,
                            const Array1D<Type>& array1)
    {
      if(array0.size() != array1.size()) {
        std::ostringstream message;
        message << "Array sizes do not match.  Array0 has " << array0.size()
                << " elements, while array1 has " << array1.size()
                << " elements.";
        DLR_THROW(ValueException, "Array1D::operator+()", message.str().c_str());
      }
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), array1.begin(),
                     result.begin(), std::plus<Type>());
      return result;
    }


    template <class Type>
    Array1D<Type> operator-(const Array1D<Type>& array0,
                            const Array1D<Type>& array1)
    {
      if(array0.size() != array1.size()) {
        std::ostringstream message;
        message << "Array sizes do not match.  Array0 has " << array0.size()
                << " elements, while array1 has " << array1.size()
                << " elements.";
        DLR_THROW(ValueException, "Array1D::operator-()", message.str().c_str());
      }
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), array1.begin(),
                     result.begin(), std::minus<Type>());
      return result;
    }


    template <class Type>
    Array1D<Type> operator*(const Array1D<Type>& array0,
                            const Array1D<Type>& array1)
    {
      if(array0.size() != array1.size()) {
        std::ostringstream message;
        message << "Array sizes do not match.  Array0 has " << array0.size()
                << " elements, while array1 has " << array1.size()
                << " elements.";
        DLR_THROW(ValueException, "Array1D::operator*()", message.str().c_str());
      }
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), array1.begin(),
                     result.begin(), std::multiplies<Type>());
      return result;
    }


    template <class Type>
    Array1D<Type> operator/(const Array1D<Type>& array0,
                            const Array1D<Type>& array1)
    {
      if(array0.size() != array1.size()) {
        std::ostringstream message;
        message << "Array sizes do not match.  Array0 has " << array0.size()
                << " elements, while array1 has " << array1.size()
                << " elements.";
        DLR_THROW(ValueException, "Array1D::operator/()", message.str().c_str());
      }
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), array1.begin(),
                     result.begin(), std::divides<Type>());
      return result;
    }


    template <class Type>
    Array1D<Type> operator+(const Array1D<Type>& array0, Type scalar)
    {
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind2nd(std::plus<Type>(), scalar));
      return result;
    }


    template <class Type>
    Array1D<Type> operator-(const Array1D<Type>& array0, Type scalar)
    {
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind2nd(std::minus<Type>(), scalar));
      return result;
    }


    template <class Type>
    Array1D<Type> operator*(const Array1D<Type>& array0, Type scalar)
    {
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind2nd(std::multiplies<Type>(), scalar));
      return result;
    }


    template <class Type>
    Array1D<Type> operator/(const Array1D<Type>& array0, Type scalar)
    {
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind2nd(std::divides<Type>(), scalar));
      return result;
    }


    template <class Type>
    inline Array1D<Type> operator+(Type scalar, const Array1D<Type>& array0)
    {
      return array0 + scalar;
    }


    template <class Type>
    Array1D<Type> operator-(Type scalar, const Array1D<Type>& array0)
    {
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind1st(std::minus<Type>(), scalar));
      return result;
    }

  
    template <class Type>
    inline Array1D<Type> operator*(Type scalar, const Array1D<Type>& array0)
    {
      return array0 * scalar;
    }

  
    template <class Type>
    Array1D<Type> operator/(Type scalar, const Array1D<Type>& array0)
    {
      Array1D<Type> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind1st(std::divides<Type>(), scalar));
      return result;
    }
  

    // Elementwise comparison of an Array1D with a constant.
    template <class Type>
    Array1D<bool>
    operator==(const Array1D<Type>& array0, const Type arg)
    {
      Array1D<bool> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.data(),
                     std::bind2nd(std::equal_to<Type>(), arg));
      return result;
    }

    
    // Elementwise comparison of an Array1D with another array.
    template <class Type>
    Array1D<bool>
    operator==(const Array1D<Type>& array0, const Array1D<Type>& array1)
    {
      array0.checkDimension(array1.size());
      Array1D<bool> result(array0.size());
      std::transform(array0.begin(), array0.end(), array1.begin(),
                     result.begin(), std::equal_to<Type>());
      return result;
    }

  
    template <class Type>
    Array1D<bool>
    operator>(const Array1D<Type>& array0, const Type arg)
    {
      Array1D<bool> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind2nd(std::greater<Type>(), arg));
      return result;
    }

  
    template <class Type>
    Array1D<bool>
    operator>=(const Array1D<Type>& array0, const Type arg)
    {
      Array1D<bool> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind2nd(std::greater_equal<Type>(), arg));
      return result;
    }


    template <class Type>
    Array1D<bool>
    operator<(const Array1D<Type>& array0, const Type arg)
    {
      Array1D<bool> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind2nd(std::less<Type>(), arg));
      return result;
    }


    template <class Type>
    Array1D<bool>
    operator<=(const Array1D<Type>& array0, const Type arg)
    {
      Array1D<bool> result(array0.size());
      std::transform(array0.begin(), array0.end(), result.begin(),
                     std::bind2nd(std::less_equal<Type>(), arg));
      return result;
    }


    template <class Type>
    std::ostream& operator<<(std::ostream& stream, const Array1D<Type>& array0)
    {
      // Most of the time, OutputType will be the same as Type.
      typedef typename NumericTraits<Type>::TextOutputType OutputType;
    
      if (!stream){
        DLR_THROW(IOException, "operator<<(std::ostream&, const Array1D&)",
                  "Invalid stream\n");
      }

      size_t index;
      stream << "Array1D([";
      if (array0.size() > 0){
        for(index = 0; index < array0.size() - 1; ++index) {
          stream << static_cast<OutputType>(array0(index)) << ", ";
        }
        stream << static_cast<OutputType>(array0(index));
      }
      stream << "])";
      return stream;
    }

    // Sets the value of an Array1D instance from a std::istream.
    template <class Type>
    std::istream& operator>>(std::istream& inputStream, Array1D<Type>& array0)
    {
      return array0.readFromStream(inputStream);
    }

  } // namespace numeric

} // namespace dlr

#endif /* #ifdef _DLR_ARRAY1D_H_ */
