/**
***************************************************************************
* @file dlrNumeric/amanatidesWoo3D.h
*
* Header file declaring AmanatidesWoo3D class.
*
* Copyright (C) 2004-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 975 $
* $Date: 2007-12-30 01:57:17 -0500 (Sun, 30 Dec 2007) $
***************************************************************************
**/

#ifndef _DLR_NUMERIC_AMANATIDESWOO3D_H_
#define _DLR_NUMERIC_AMANATIDESWOO3D_H_

#include <iostream>
#include <limits>
#include <dlrCommon/exception.h>
#include <dlrNumeric/vector3D.h>
#include <dlrNumeric/transform3D.h>
#include <dlrNumeric/amanatidesWoo3DIterator.h>

namespace dlr {

  namespace numeric {
    
    /**
     ** This class implements the Fast Voxel Traversal Algorithm of
     ** Amanatides and Woo [Ref] for 3D arrays.  The algorithm is for
     ** efficient ray tracing through regular grids.  For convenience,
     ** there are two coordinate systems associated with this class:
     **
     ** - The voxel coordinate system is an integer-based coordinate
     ** system associated with the array.
     **
     ** - The world coordinate system, which can differ from the voxel
     ** coordinate system only by translation, rotation, and
     ** non-isotropic scaling, but is in other respects arbitrary.
     **
     ** The voxel coordinate system has its origin at the corner of the
     ** 3D array, has its first coordinate (which we'll call U) parallel
     ** to the rows of the array, has its second coordinate (which we'll
     ** call V) parallel to the columns of the array, and has its third
     ** coordinate (which we'll call W) parallel to the direction of
     ** increasing slice number.  As you move along any row of any slice
     ** in the 3D array, the transition from the first column to the
     ** second occurs at U == 1.0, the transition from the second column
     ** to the third occurs at U == 2.0, and so on.  Similarly as you
     ** move along any column of any slice of the array, the transition
     ** from the first row to the second occurs at V == 1.0, and the
     ** transition from the second row to the third occurs at V ==
     ** 2.0. At any row, column coordinate, as you move through
     ** subsequent slices of the array, the transition from the first
     ** slice to the second occurs at W == 1.0, and the transition from
     ** the second slice to the third occurs at W == 2.0, and so on.
     **
     ** Once constructed, you can use the AmanatidesWoo3D class to
     ** access voxels along a straight line through the 3D array using
     ** STL-style iteration.  That is, the member functions
     ** AmanatidesWoo3D::begin() returns an iterator which references
     ** subsequent voxels along the line, and will become equal (==) to
     ** the iterator returned by AmanatidesWoo3D::end() when the line
     ** passes out of the 3D array.  Here's an example usage, which will
     ** probably format terribly in the Doxygen-generated doc:
     **
     **   typedef AmanatidesWoo3D< Array3D<int> >::iterator awIterator;
     **
     **   AmanatidesWoo3D< Array3D<int> > rayTracer(
     **     myArray3D, rayOrigin, rayDirection, false);
     **
     **   awIterator iterator0 = rayTracer.begin(); 
     **
     **   awIterator iterator1 = rayTracer.end();
     **
     **   for(; iterator0 != iterator1; ++iterator0) {
     **
     **     *iterator0 = 0;
     **
     **   }
     **/
    template <class ARRAY3D>
    class AmanatidesWoo3D {
    public:
      /* ================== Public typedefs ================== */

      /**
       ** This typedef specifies the type which will be returned by
       ** member functions begin() and end().  AmanatidesWoo3DIterator
       ** is a forward iterator, and has unusual semantics for
       ** operator==().  Be sure to understand them before using this
       ** class.
       **/
      typedef AmanatidesWoo3DIterator<ARRAY3D> iterator;

      // Iteration over const arrays will be done using a different
      // class.
      // typedef AmanatidesWoo3DConstIterator<ARRAY3D> const_iterator;
    
      /* ================== Public methods ================== */

      /** 
       * This constructor specifies all of the internal state of the
       * AmanatidesWoo3D class.  After construction, the AmanatidesWoo3D
       * instance is ready to be used.
       * 
       * @param data This argument specifies the 3D array over which to
       * iterate.
       *
       * @param voxelTworld This parameter specifies a coordinate
       * transformation which takes world coordinates and converts them
       * into voxel coordinates.
       *     
       * @param rayOrigin This parameter specifies the starting point
       * of the ray to be traced, expressed in world coordinates.  If
       * you'd rather express rayOrigin and rayDirection in voxel
       * coordinates, simply set argument voxelTworld to the identity
       * transform.  Note that rayOrigin does not have to lie inside the
       * voxel array.
       *     
       * @param rayDirection This parameter specifies the direction of
       * the ray to be traced, expressed in world coordinates.  In other
       * words, any point on the line of interest can be written as
       * (rayOrigin + t * rayDirection) for some value of t.  If
       * you'd rather express rayOrigin and rayDirection in voxel
       * coordinates, simply set argument voxelTworld to the identity
       * transform.
       *     
       * @param downstreamOnly This boolean argument specifies whether
       * voxels "upstream" of rayOrigin are to be included in the voxel
       * traversal.  If downstreamOnly is set to false, voxels which
       * intersect the line (rayOrigin + t * rayDirection) will be
       * included in the iteration even if they intersect the line at
       * negative values of t.
       */
      AmanatidesWoo3D(ARRAY3D& data,
                      const Transform3D& voxelTworld,
                      const Vector3D& rayOrigin,
                      const Vector3D& rayDirection,
                      bool downstreamOnly=true);

    
      /** 
       * This is the copy constructor.  After copying, the new
       * AmanatidesWoo3D instance and the copied instance both reference
       * the same voxel array.
       * 
       * @param source The AmanatidesWoo3D instance to be copied.
       */
      AmanatidesWoo3D(const AmanatidesWoo3D& source);

      /** 
       * This is the destructor.  It destroys the AmanatidesWoo3D
       * instance and cleans up any allocated memory.
       */
      ~AmanatidesWoo3D();

      /** 
       * This member function returns an iterator which references the
       * first voxel along the traced ray.  If constructor argument
       * downstreamOnly was set to false, this iterator will either
       * point to a voxel on the very edge of the array, or (in the case
       * that the ray does not intersect the voxel array) be equal to
       * this->end().  If constructor argument downstreamOnly was set to
       * true, then this iterator will point to an edge voxel, or point
       * to an interior voxel (in the case that rayOrigin lies within
       * the boundaries of the voxel array), or be equal to this->end()
       * (in the case that the ray does not intersect the voxel array,
       * and in the case that rayOrigin lies outside the voxel array and
       * rayDirection points away from the voxel array).
       * 
       * @return The return value is an iterator pointing to the first
       * voxel in the array, or else an iterator for which (iter ==
       * this->end()) is true.
       */
      iterator
      begin();

      /** 
       * This member function returns an iterator which references an
       * invalid voxel, and which will be equal (==) to the iterator
       * returned by member function begin() when that iterator has
       * fully traversed the line of voxels.
       * 
       * @return The return value is an iterator pointing to an invalid
       * voxel, for use as the end iterator in the comparison clause of
       * a loop: "while(rayIterator != endIterator) {++rayIterator; ...;}".
       */
      iterator
      end();

      /** 
       * This member function returns a reference to the array object
       * over which iteration is performed.
       * 
       * @return The return value is a reference to the data array.
       */
      ARRAY3D&
      getData() {return m_data;}

      /** 
       * This member function returns true if the iterator returned by
       * member function begin() will point to a valid voxel.  In other
       * words, if constructor argument downstreamOnly was set to
       * false, the return value of validIntersection() indicates
       * whether the ray specified in the constructor actually
       * intersects the voxel array.  If constructor argument
       * downstreamOnly was set to true, the return value of
       * validIntersection whether the "downstream" half of the ray
       * actually intersects the voxel array.
       * 
       * @return A boolean indicating whether or not (this->begin() ==
       * this->end()) will be true.
       */
      inline bool
      validIntersection();
    
      /** 
       * The assignment operator copies its argument.  After copying,
       * *this and the copied instance both reference the same voxel
       * array.
       * 
       * @param source The AmanatidesWoo3D instance to be copied.
       * @return The return value is a reference to *this.
       */
      AmanatidesWoo3D&
      operator=(const AmanatidesWoo3D& source);
  
    private:

      /** 
       * This private member function computes the parameters tEntry and
       * tExit such that (rayOrigin + tEntry * rayDirection) is very
       * first point of intersection between the ray and the voxel
       * array, and (rayOrigin + tExit * rayDirection) is very last of
       * intersection between the ray and the voxel array.
       * 
       * @param rayOriginVoxel This parameter specifies the starting
       * point of the ray to be traced, expressed in voxel coordinates.
       * @param rayDirectionVoxel This parameters specifies the
       * direction of the ray to be traced, expressed in voxel
       * coordinates.
       * @param m_data This parameter is a reference the voxel array.
       * @return The return value is a std::pair<double, double> in
       * which the first value is tEntry and the second value is tExit.
       */
      std::pair<double, double>
      findEntryAndExitPoints(const Vector3D& rayOriginVoxel,
                             const Vector3D& rayDirectionVoxel,
                             const ARRAY3D& m_data);

      /** 
       * This function is not documented because it will change very
       * soon.
       * 
       * @param rayOrigin 
       * @param rayDirection 
       * @param bVector 
       * @param cConstant 
       * @param defaultValue 
       * @return 
       */
      double
      findIntersection(const Vector3D& rayOrigin,
                       const Vector3D& rayDirection,
                       const Vector3D& bVector,
                       double cConstant,
                       double defaultValue);

      /// This data member is a reference the array over which to
      /// iterate.
      ARRAY3D& m_data;

      /// This data member indicates the U coordinate of the first voxel
      /// along the path.
      int m_initialU;

      /// This data member indicates the V coordinate of the first voxel
      /// along the path.
      int m_initialV;

      /// This data member indicates the W coordinate of the first voxel
      /// along the path.
      int m_initialW;
    
      /// This data member indicates whether the U coordinate will be
      /// increasing (m_stepU == 1) or decreasing (m_stepU == -1) as we
      /// travel along the ray path.
      int m_stepU;

      /// This data member indicates whether the V coordinate will be
      /// increasing (m_stepV == 1) or decreasing (m_stepV == -1) as we
      /// travel along the ray path.
      int m_stepV;
    
      /// This data member indicates whether the W coordinate will be
      /// increasing (m_stepW == 1) or decreasing (m_stepW == -1) as we
      /// travel along the ray path.
      int m_stepW;
    
      /// This data member indicates what increment of ray parameter t
      /// moves us exactly 1 voxel in the U direction.
      double m_tDeltaU;

      /// This data member indicates what increment of ray parameter t
      /// moves us exactly 1 voxel in the V direction.
      double m_tDeltaV;
    
      /// This data member indicates what increment of ray parameter t
      /// moves us exactly 1 voxel in the W direction.
      double m_tDeltaW;
    
      /// This data member indicates the value of ray parameter t at
      /// which the ray first enters a voxel with U coordinate not equal
      /// to m_initialU.
      double m_tMaxU;

      /// This data member indicates the value of ray parameter t at
      /// which the ray first enters a voxel with V coordinate not equal
      /// to m_initialV.
      double m_tMaxV;

      /// This data member indicates the value of ray parameter t at
      /// which the ray first enters a voxel with W coordinate not equal
      /// to m_initialW.
      double m_tMaxW;

      /// This data member indicates the value of ray parameter t at
      /// which we start following the ray.  It frequently corresponds
      /// to the point at which the ray enters the voxelated space, but
      /// may also be set to zero if the ray origin is within the
      /// voxelated space and constructor argument downstreamOnly was
      /// set to true.
      double m_tStart;

      /// This data member indicates whether the portion of the ray we
      /// care about actually intersects the specified voxel array.
      bool m_validIntersection;
    };

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::AmanatidesWoo3D;

} // namespace dlr


/* ========================================================= */
/* Implementation follows                                    */
/* ========================================================= */

namespace dlr {

  namespace numeric {
    
    template <class ARRAY3D>
    AmanatidesWoo3D<ARRAY3D>::
    AmanatidesWoo3D(ARRAY3D& data,
                    const Transform3D& voxelTworld,
                    const Vector3D& rayOrigin,
                    const Vector3D& rayDirection,
                    bool downstreamOnly)
      : m_data(data),
        m_initialU(-1),  // Initialize to illegal values.  These will be 
        m_initialV(-1),  // replaced with legal ones if there's a valid
        m_initialW(-1),  // intersection between the ray and the voxel
        // array.
        m_stepU(),
        m_stepV(),
        m_stepW(),
        m_tDeltaU(),
        m_tDeltaV(),
        m_tDeltaW(),
        m_tMaxU(),
        m_tMaxV(),
        m_tMaxW(),
        m_tStart(),
        m_validIntersection(true)    
    {
      // First convert everything into voxel coordinates.
      Vector3D rayOriginVoxel = voxelTworld * rayOrigin;
      Vector3D rayDirectionVoxel =
        (voxelTworld * (rayOrigin + rayDirection)) - rayOriginVoxel;

      // Now find points entry and exit from the CT volume.  These are
      // expressed in as parameter values tEntry and tExit such that
      // (rayOriginVoxel + tEntry * rayDirectionVoxel) is the entry
      // point and (rayOriginVoxel + tExit * rayDirectionVoxel) is the
      // exit point.
      std::pair<double, double> tEntry_tExit =
        this->findEntryAndExitPoints(rayOriginVoxel, rayDirectionVoxel, m_data);

      // Sometimes we want to disallow any voxels which are "behind" the
      // ray origin.  That is, sometimes we want to only trace those
      // parts of the ray which correspond to positive values of the
      // parameter t.
      if(downstreamOnly && (tEntry_tExit.first < 0.0)) {
        tEntry_tExit.first = 0.0;
      }

      // Sometimes the there's no intersection with the voxel array.  In
      // this case, tExit will be less than tEntry, and we don't bother
      // doing any more calculation.
      if(tEntry_tExit.second <= tEntry_tExit.first) {
        m_validIntersection = false;
        return;
      }

      // Now make the point of entry explicit.
      Vector3D entryPoint =
        rayOriginVoxel + tEntry_tExit.first * rayDirectionVoxel;

      // Correct for rounding error which sometimes makes entryPoint
      // have values like -1.0e-14.
      if(entryPoint.x() < 0.0) {
        entryPoint.x() = 0.0;
      }
      if(entryPoint.y() < 0.0) {
        entryPoint.y() = 0.0;
      }
      if(entryPoint.z() < 0.0) {
        entryPoint.z() = 0.0;
      }
    
      // Finally, assign the variables described in the Amanatides' and
      // Woo's paper.

      // Since we've already converted to voxel coords, finding the
      // first voxel on the path is trivial.
      m_initialU = static_cast<int>(entryPoint.x());
      m_initialV = static_cast<int>(entryPoint.y());
      m_initialW = static_cast<int>(entryPoint.z());

      // There's an similar case to the one handled by the rounding
      // error test above.  Member function findEntryAndExitPoints()
      // uses m_data.shape to define the maximum U, V, and W
      // coordinates, respectively.  This means that it's very possible
      // for m_initialU to be equal to m_data.columns(), which will
      // cause an out-of-bounds index. Correct for this now.
      if(m_initialU == static_cast<int>(m_data.shape()[2])) {
        --m_initialU;
      }
      if(m_initialV == static_cast<int>(m_data.shape()[1])) {
        --m_initialV;
      }
      if(m_initialW == static_cast<int>(m_data.shape()[0])) {
        --m_initialW;
      }

      // Sanity check.
      if((m_initialU >= static_cast<int>(m_data.shape()[2]))
         || (m_initialV >= static_cast<int>(m_data.shape()[1]))
         || (m_initialW >= static_cast<int>(m_data.shape()[0]))) {
        DLR_THROW3(LogicException,
                   "AmanatidesWoo3D::AmanatidesWoo3D(ARRAY3D&, ...)",
                   "Illegal value for m_initialU, m_initialV, or m_initialW.");
      }
    
      // m_tStart is just the same as tEntry.
      m_tStart = tEntry_tExit.first;
    
      // The remaining variables depend on whether U & V will be
      // increasing or decreasing as we travel along the ray, so we need
      // if clauses.  Please see the declaration for documentation on
      // what each of these member variables means.
      if(rayDirectionVoxel.x() > 0.0) {
        m_stepU = 1;
        m_tDeltaU = 1.0 / rayDirectionVoxel.x();
        m_tMaxU = m_tStart + (((m_initialU + 1) - entryPoint.x())
                              / rayDirectionVoxel.x());
      } else if(rayDirectionVoxel.x() < 0.0) {
        m_stepU = -1;
        m_tDeltaU = -(1.0 / rayDirectionVoxel.x());
        m_tMaxU = m_tStart + ((m_initialU - entryPoint.x())
                              / rayDirectionVoxel.x());
      } else { // rayDirectionVoxel.x() == 0.0;
        m_stepU = 0;
        m_tDeltaU = std::numeric_limits<double>::max();
        m_tMaxU = std::numeric_limits<double>::max();
      }
      if(rayDirectionVoxel.y() > 0.0) {
        m_stepV = 1;
        m_tDeltaV = 1.0 / rayDirectionVoxel.y();
        m_tMaxV = m_tStart + (((m_initialV + 1) - entryPoint.y())
                              / rayDirectionVoxel.y());
      } else if(rayDirectionVoxel.y() < 0.0) {
        m_stepV = -1;
        m_tDeltaV = -(1.0 / rayDirectionVoxel.y());
        m_tMaxV = m_tStart + ((m_initialV - entryPoint.y())
                              / rayDirectionVoxel.y());
      } else { // rayDirectionVoxel.y() == 0.0;
        m_stepV = 0;
        m_tDeltaV = std::numeric_limits<double>::max();
        m_tMaxV = std::numeric_limits<double>::max();
      }
      if(rayDirectionVoxel.z() > 0.0) {
        m_stepW = 1;
        m_tDeltaW = 1.0 / rayDirectionVoxel.z();
        m_tMaxW = m_tStart + (((m_initialW + 1) - entryPoint.z())
                              / rayDirectionVoxel.z());
      } else if(rayDirectionVoxel.z() < 0.0) {
        m_stepW = -1;
        m_tDeltaW = -(1.0 / rayDirectionVoxel.z());
        m_tMaxW = m_tStart + ((m_initialW - entryPoint.z())
                              / rayDirectionVoxel.z());
      } else { // rayDirectionVoxel.z() == 0.0;
        m_stepW = 0;
        m_tDeltaW = std::numeric_limits<double>::max();
        m_tMaxW = std::numeric_limits<double>::max();
      }
    }
    
    template <class ARRAY3D>
    AmanatidesWoo3D<ARRAY3D>::
    AmanatidesWoo3D(const AmanatidesWoo3D& source)
      : m_data(source.m_data),
        m_initialU(source.m_initialU),
        m_initialV(source.m_initialV),
        m_initialW(source.m_initialW),
        m_stepU(source.m_stepU),
        m_stepV(source.m_stepV),
        m_stepW(source.m_stepW),
        m_tDeltaU(source.m_tDeltaU),
        m_tDeltaV(source.m_tDeltaV),
        m_tDeltaW(source.m_tDeltaW),
        m_tMaxU(source.m_tMaxU),
        m_tMaxV(source.m_tMaxV),
        m_tMaxW(source.m_tMaxW),
        m_tStart(source.m_tStart),
        m_validIntersection(source.m_validIntersection)
    {
      // Empty
    }

    template <class ARRAY3D>
    AmanatidesWoo3D<ARRAY3D>::
    ~AmanatidesWoo3D() {};

    template <class ARRAY3D>
    typename AmanatidesWoo3D<ARRAY3D>::iterator
    AmanatidesWoo3D<ARRAY3D>::
    begin()
    {
      return AmanatidesWoo3DIterator<ARRAY3D>(
        m_data, m_initialU, m_initialV, m_initialW, m_stepU, m_stepV, m_stepW,
        m_tMaxU, m_tMaxV, m_tMaxW, m_tDeltaU, m_tDeltaV, m_tDeltaW, m_tStart);
    }

    template <class ARRAY3D>
    typename AmanatidesWoo3D<ARRAY3D>::iterator
    AmanatidesWoo3D<ARRAY3D>::
    end()
    {
      // AmanatidesWoo3DIterator<ARRAY3D>::operator==(...) considers all
      // iterators with illegal voxel coordinates to be equal, so all we
      // have to do here is return an iterator which references an
      // illegal voxel.
      return AmanatidesWoo3DIterator<ARRAY3D>(
        m_data, -1, -1, -1, m_stepU, m_stepV, m_stepW,
        m_tMaxU, m_tMaxV, m_tMaxW, m_tDeltaU, m_tDeltaV, m_tDeltaW, m_tStart);
    }

    template <class ARRAY3D>
    inline bool
    AmanatidesWoo3D<ARRAY3D>::
    validIntersection()
    {
      return m_validIntersection;
    }

    template <class ARRAY3D>
    AmanatidesWoo3D<ARRAY3D>&
    AmanatidesWoo3D<ARRAY3D>::
    operator=(const AmanatidesWoo3D& source)
    {
      m_data = source.m_data;
      m_initialU = source.m_initialU;
      m_initialV = source.m_initialV;
      m_initialW = source.m_initialW;
      m_stepU = source.m_stepU;
      m_stepV = source.m_stepV;
      m_stepW = source.m_stepW;
      m_tDeltaU = source.m_tDeltaU;
      m_tDeltaV = source.m_tDeltaV;
      m_tDeltaW = source.m_tDeltaW;
      m_tMaxU = source.m_tMaxU;
      m_tMaxV = source.m_tMaxV;
      m_tMaxW = source.m_tMaxW;
      m_tStart = source.m_tStart;
      m_validIntersection = source.m_validIntersection;
      return *this;
    }
    
    template <class ARRAY3D>
    std::pair<double, double>
    AmanatidesWoo3D<ARRAY3D>::
    findEntryAndExitPoints(const Vector3D& rayOriginVoxel,
                           const Vector3D& rayDirectionVoxel,
                           const ARRAY3D& m_data)
    {
      // First find intersection with each boundary line.

      // ... Find the intersection with the plane U = 0, or else a
      // really small number if rayDirection is parallel to the U axis.
      double tIntersectU0 = findIntersection(
        rayOriginVoxel, rayDirectionVoxel, Vector3D(1.0, 0.0, 0.0),
        0.0, std::numeric_limits<double>::min());
    
      // ... Find the intersection with the plane U = m_data.shape()[2],
      // or else a really big number if rayDirection is parallel to the
      // U axis.
      double tIntersectU1 = findIntersection(
        rayOriginVoxel, rayDirectionVoxel, Vector3D(1.0, 0.0, 0.0),
        m_data.shape()[2], std::numeric_limits<double>::max());
    
      // ... Find the intersection with the plane V = 0, or else a
      // really small number if rayDirection is parallel to the V axis.
      double tIntersectV0 = findIntersection(
        rayOriginVoxel, rayDirectionVoxel, Vector3D(0.0, 1.0, 0.0),
        0.0, std::numeric_limits<double>::min());
    
      // ... Find the intersection with the plane V = m_data.shape()[1],
      // or else a really big number if rayDirection is parallel to the
      // V axis.
      double tIntersectV1 = findIntersection(
        rayOriginVoxel, rayDirectionVoxel, Vector3D(0.0, 1.0, 0.0),
        m_data.shape()[1], std::numeric_limits<double>::max());
    
      // ... Find the intersection with the plane W = 0, or else a
      // really small number if rayDirection is parallel to the W axis.
      double tIntersectW0 = findIntersection(
        rayOriginVoxel, rayDirectionVoxel, Vector3D(0.0, 0.0, 1.0),
        0.0, std::numeric_limits<double>::min());
    
      // ... Find the intersection with the plane W = m_data.shape()[0],
      // or else a really big number if rayDirection is parallel to the
      // W axis.
      double tIntersectW1 = findIntersection(
        rayOriginVoxel, rayDirectionVoxel, Vector3D(0.0, 0.0, 1.0),
        m_data.shape()[0], std::numeric_limits<double>::max());

      // Now find the closer and farther of each pair of intersections.
      double tMinU = std::min(tIntersectU0, tIntersectU1);
      double tMinV = std::min(tIntersectV0, tIntersectV1);
      double tMinW = std::min(tIntersectW0, tIntersectW1);
      double tMaxU = std::max(tIntersectU0, tIntersectU1);
      double tMaxV = std::max(tIntersectV0, tIntersectV1);
      double tMaxW = std::max(tIntersectW0, tIntersectW1);

      // Compute closest point which could possibly intersect with the volume.
      double tEntry = std::max(tMinU, std::max(tMinV, tMinW));
      // Compute farthest point which could possibly intersect with the volume.
      double tExit = std::min(tMaxU, std::min(tMaxV, tMaxW));

      // Return our findings.
      return std::make_pair(tEntry, tExit);
    }

    template <class ARRAY3D>
    double
    AmanatidesWoo3D<ARRAY3D>::
    findIntersection(const Vector3D& rayOrigin,
                     const Vector3D& rayDirection,
                     const Vector3D& bVector,
                     double cConstant,
                     double defaultValue)
    {
      // bVector and cConstant describe the desired plane:
      //   dot(x, bVector) = cConstant.
      // Now, x is constrained to lie on the specified line:
      //   x = rayOrigin + alpha * rayDirection.
      // Substituting, we have:
      //   dot(rayOrigin + alpha * rayDirection, bVector) = cConstant.
      // Which we rewrite:
      //   dot(rayOrigin, bVector) + alpha * dot(rayDirection, bVector)
      //     = cConstant.
      // Solving for alpha, we have:
      //   alpha = (cConstant - dot(rayOrigin, bVector))
      //            / dot(rayDirection, bVector).
      double denominator = dot(rayDirection, bVector);
      if(denominator == 0.0) {
        return defaultValue;
      }
      // else
      return (cConstant - dot(rayOrigin, bVector)) / denominator;
    }      

  } // namespace numeric

} // namespace dlr

#endif // #ifndef _DLR_NUMERIC_AMANATIDESWOO3D_H_
