/**
***************************************************************************
* @file dlrNumeric/transform2D.h
*
* Header file declaring Transform2D class.
*
* Copyright (C) 2001-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1138 $
* $Date: 2009-04-02 18:07:24 -0400 (Thu, 02 Apr 2009) $
***************************************************************************
**/

#ifndef _DLR_TRANSFORM2D_H_
#define _DLR_TRANSFORM2D_H_

#include <dlrNumeric/array2D.h>
#include <dlrNumeric/vector2D.h>

namespace dlr {

  namespace numeric {
    
    // Forward declaration.
    class Transform2DFunctor;

  
    /**
     ** The Transform2D class represents a homogeneous coordinate
     ** transform from one 2D coordinate system to another 2D
     ** coordinate system.
     **/
    class Transform2D {
    public:
      /** 
       * Default constructor.  Initializes to identity.
       */
      inline
      Transform2D()
        : m_00(1.0), m_01(0.0), m_02(0.0),
          m_10(0.0), m_11(1.0), m_12(0.0),
          m_20(0.0), m_21(0.0) {}

      /** 
       * Build a Transform2D instance by explicitly setting element values
       * as if setting the elements of a 3x3 transformation matrix:
       *    [[a00, a01, a02],
       *     [a10, a11, a12],
       *     [a20, a21, a22]]
       * 
       * @param a00 The value of one element of the transformation matrix.
       * @param a01 The value of one element of the transformation matrix.
       * @param a02 The value of one element of the transformation matrix.
       * @param a10 The value of one element of the transformation matrix.
       * @param a11 The value of one element of the transformation matrix.
       * @param a12 The value of one element of the transformation matrix.
       * @param a20 The value of one element of the transformation matrix.
       * @param a21 The value of one element of the transformation matrix.
       * @param a22 The value of one element of the transformation matrix.
       */
      inline
      Transform2D(double a00, double a01, double a02,
                  double a10, double a11, double a12,
                  double a20, double a21, double a22)
        : m_00(a00), m_01(a01), m_02(a02),
          m_10(a10), m_11(a11), m_12(a12),
          m_20(a20), m_21(a21) {
        this->normalize(a22);
      }

      /** 
       * Build a Transform2D from a homogeneous 3x3 matrix.
       *
       * @param source A 2D array containing the elements of the desired
       * homogeneous transformation.
       */
      Transform2D(const Array2D<double>& source);

      /** 
       * The copy constructor simply duplicates its argument.
       * 
       * @param src This is the Transform2D instance to be copied.
       */
      inline
      Transform2D(const Transform2D& src)
        : m_00(src.m_00), m_01(src.m_01), m_02(src.m_02),
          m_10(src.m_10), m_11(src.m_11), m_12(src.m_12),
          m_20(src.m_20), m_21(src.m_21) {}

      /** 
       * Destructor.
       */
      ~Transform2D() {}


      /** 
       * This member function returns a functor which makes it easier to
       * transform arrays of points using algorithms such as
       * std::transform().  For example:
       *
       * @code
       * std::transform(myPoints.begin(), myPoints.end(), myNewPoints.begin(),
       *                myTransform.getFunctor());
       * @endcode
       * 
       * @return The return value is a functor instance which will
       * transform points according to the current value of *this.  The
       * functor will contain a copy of *this, so that subsequent
       * changes to *this will not affect the functor.
       */
      Transform2DFunctor
      getFunctor() const;
    
    
      /** 
       * This member function returns the inverse of *this.  It is an
       * error if *this is not invertible.
       * 
       * @return The return value is the inverse of *this.
       */
      Transform2D
      invert() const;

    
      /** 
       * Change the Transform2D value by explicitly setting element values
       * as if setting the elements of a 3x3 transformation matrix:
       *    [[a00, a01, a02],
       *     [a10, a11, a12],
       *     [a20, a21, a22]]
       * 
       * @param a00 The value of one element of the transformation matrix.
       * @param a01 The value of one element of the transformation matrix.
       * @param a02 The value of one element of the transformation matrix.
       * @param a10 The value of one element of the transformation matrix.
       * @param a11 The value of one element of the transformation matrix.
       * @param a12 The value of one element of the transformation matrix.
       * @param a20 The value of one element of the transformation matrix.
       * @param a21 The value of one element of the transformation matrix.
       * @param a22 The value of one element of the transformation matrix.
       */
      void
      setTransform(double a00, double a01, double a02,
                   double a10, double a11, double a12,
                   double a20, double a21, double a22);

      /** 
       * This member function sets one element of the matrix
       * representation of the coordinate transform.  Note that indexing
       * is zero-based.  For example, calling myTransform2D.setValue(1,
       * 2, 5.0) will set the element from the 2nd row, 3rd column of
       * the matrix representation of the coordinate transformation to
       * 5.0.  If blindingly fast execution is important, consider using
       * the setValue<size_t, size_t>(double) member function instead.
       *
       * It is not permitted to set the value of the lower-right element
       * of the matrix: setValue(2, 2, ...) will throw an
       * IndexException.
       * 
       * @param row This argument specifies the row of the matrix
       * element to be modified.
       * 
       * @param column This argument specifies the row of the matrix
       * element to be modified.
       * 
       * @param value This argument specifies the value to be copied
       * into the matrix element at the specified row & column.
       */
      void
      setValue(size_t row, size_t column, double value);

      /** 
       * This member function sets one element from the matrix
       * representation of the coordinate transform.  Note that indexing
       * is zero-based.  For example, calling myTransform2D.setValue<1,
       * 2>(5.0) will set the element from the 2nd row, 3rd column of
       * the matrix representation of the coordinate transformation to
       * 5.0.
       *
       * It is not permitted to set the value of the lower-right element
       * of the matrix: setValue<2, 2>(double) will throw an
       * IndexException.
       * 
       * @param value This argument specifies the value to be copied
       * into the matrix element at the specified row & column.
       */
      template <size_t row, size_t column>
      void
      setValue(double value);

      /** 
       * This member function returns one element from the matrix
       * representation of the coordinate transform by value.
       * For example, calling myTransform2D.value<1, 2>() will return
       * the element from the 2nd row, 3rd column of the matrix
       * representation of the coordinate transformation.
       * 
       * @return The value of the requested element.
       */
      template <size_t row, size_t column>
      double
      value() const;

      /** 
       * This operator returns one element from the matrix
       * representation of the coordinate transform by value.
       * If blindingly fast execution is important, consider using
       * value<size_t, size_t>() member function.
       *
       * @param row The row of the requested element.
       * @param column The column of the requested element.
       * @return The value of the requested element.
       */
      double
      operator()(size_t row, size_t column) const;

      /** 
       * This operator takes a point and applies the coordinate
       * transform, returning the result.
       * 
       * @param vector0 The point to be transformed.
       * @return The result of the transformation.
       */
      Vector2D
      operator*(const Vector2D& vector0) const;

      /** 
       * The assignment operator simply duplicates its argument.
       * 
       * @param source This is the Transform2D instance to be copied.
       * @return A reference to *this.
       */
      Transform2D&
      operator=(const Transform2D& source);

    private:
      void normalize(double scaleFactor);
      
      double m_00, m_01, m_02;
      double m_10, m_11, m_12;
      double m_20, m_21;

    }; // class Transform2D


    /* ============== Helper classes ============== */


    /**
     ** This helper class works with Transform2D::getFunctor()
     **/
    class Transform2DFunctor
      : public std::unary_function<Vector2D, Vector2D> {

    public:

      /** 
       * The constructor deep-copies its argument.
       * 
       *  @param transform This is the transform instance to be copied.
       */
      Transform2DFunctor(const Transform2D& transform)
        : m_transform(transform) {}

    
      /** 
       * The application operator transforms its argument.
       * 
       *  @param vec This is the vector to be transformed.
       * 
       *  @return The transformed vector.
       */
      inline Vector2D
      operator()(const Vector2D& vec) const {return m_transform * vec;}
    
    private:
      Transform2D m_transform;
    };
  
  
    /* ============== Non-member functions ============== */
  
    /** 
     * This operator composes two Transform2D instances.  The resulting
     * transform satisfies the equation:
     *
     *  (transform0 * transform1) * v0 = transform0 * (transform1 * v0),
     *
     * where v0 is a Vector2D instance.
     * 
     * @param transform0 This is the first of the two Transform2D instances to
     * be composed.
     * @param transform1 This is the second of the two Transform2D instances to
     * be composed.
     * @return A Transform2D instance which is equivalent to the composition
     * of the two arguments.
     */
    Transform2D
    operator*(const Transform2D& transform0, const Transform2D& transform1);

  
    /** 
     * Outputs a text representation of a Transform2D instance to a
     * std::ostream.  The output format looks like this:
     *
     * Transform2D(1.0, 2.0, 3.0,
     *             5.0, 6.0, 7.0,
     *             9.0, 10.0, 1.0)
     *
     * @param stream Reference to the the output stream.
     *
     * @param transform0 Const reference to the Transform2D instance to be
     * output.
     *
     * @return Reference to the output stream.
     */
    std::ostream&
    operator<<(std::ostream& stream, const Transform2D& transform0);

  
    /** 
     * Sets the value of a Transform2D instance from a std::istream.
     * The input format is as described for
     * operator<<(std::ostream&, const Transform2D&), above.
     * 
     * @param stream Reference to the the input stream.
     *
     * @param transform0 Reference to the Transform2D that will take 
     * the input.
     *
     * @return Reference to the input stream.
     */
    std::istream&
    operator>>(std::istream& stream, Transform2D& transform0);

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::Transform2D;
  using numeric::Transform2DFunctor;

} // namespace dlr


/*******************************************************************
 * Member function definitions follow.  This would be a .C file
 * if it weren't templated.
 *******************************************************************/

namespace dlr {

  namespace numeric {
    
    // This operator sets one element of the matrix representation of
    // the coordinate transform.  The case statements should optimize
    // away, since row and column are both known at compile time.
    template <size_t ROW, size_t COLUMN>
    void
    Transform2D::
    setValue(double value)
    {
      switch(ROW) {
      case 0:
        switch(COLUMN) {
        case 0: m_00 = value; return; break;
        case 1: m_01 = value; return; break;
        case 2: m_02 = value; return; break;
        default: break;
        }
        break;
      case 1:
        switch(COLUMN) {
        case 0: m_10 = value; return; break;
        case 1: m_11 = value; return; break;
        case 2: m_12 = value; return; break;
        default: break;
        }
        break;
      case 2:
        switch(COLUMN) {
        case 0: m_20 = value; return; break;
        case 1: m_21 = value; return; break;
        default: break;
        }
        break;
      default:
        break;
      }
      std::ostringstream message;
      message << "Indices (" << ROW << ", " << COLUMN << ") are out of bounds.";
      DLR_THROW(IndexException, "Transform2D::setValue<size_t, size_t>(double)",
                message.str().c_str());
    }

  
    // This operator returns one element from the matrix
    // representation of the coordinate transform by value.
    // The case statements should optimize away, since row and column
    // are both known at compile time.
    template <size_t ROW, size_t COLUMN>
    double
    Transform2D::
    value() const
    {
      // // Avoid ugly duplication of code using ugly const_cast.
      // return const_cast<Transform2D*>(this)->value<ROW, COLUMN>();
      switch(ROW) {
      case 0:
        switch(COLUMN) {
        case 0: return m_00; break;
        case 1: return m_01; break;
        case 2: return m_02; break;
        default: break;
        }
        break;
      case 1:
        switch(COLUMN) {
        case 0: return m_10; break;
        case 1: return m_11; break;
        case 2: return m_12; break;
        default: break;
        }
        break;
      case 2:
        switch(COLUMN) {
        case 0: return m_20; break;
        case 1: return m_21; break;
        case 2: return 1.0; break;
        default: break;
        }
        break;
      default:
        break;
      }
      std::ostringstream message;
      message << "Index (" << ROW << ", " << COLUMN << ") out of bounds.";
      DLR_THROW3(IndexException, "Transform2D::value<>()",
                 message.str().c_str());
      return 0.0; // Dummy return to keep the compiler happy.
    }

  } // namespace numeric

} // namespace dlr
  
#endif // #ifndef _DLR_TRANSFORM2D_H_
