/**
***************************************************************************
* @file sampledFunctionsTest.cpp
* 
* Source file defining sampledFunctionsTest class.
*
* Copyright (C) 2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1001 $
* $Date: 2008-06-11 20:43:38 -0400 (Wed, 11 Jun 2008) $
***************************************************************************
**/

#include <dlrNumeric/sampledFunctions.h>
#include <dlrNumeric/utilities.h>

#include <dlrCommon/functional.h>
#include <dlrTest/testFixture.h>

namespace dlr {

  namespace numeric {

    class SampledFunctionsTest : public TestFixture<SampledFunctionsTest> {

    public:

      SampledFunctionsTest();
      ~SampledFunctionsTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      void testGetGaussian1D();

    private:

      double m_defaultTolerance;
    
    }; // class SampledFunctionsTest


    /* ============== Member Function Definititions ============== */

    SampledFunctionsTest::
    SampledFunctionsTest()
      : TestFixture<SampledFunctionsTest>("SampledFunctionsTest"),
        m_defaultTolerance(1.0E-10)
    {
      // Register all tests.
      DLR_TEST_REGISTER_MEMBER(testGetGaussian1D);
    }


    void
    SampledFunctionsTest::
    testGetGaussian1D()
    {
      double const sigma = 1.1;
      size_t kernelSize = 7;
      Array1D<double> kernel = getGaussian1D<double>(sigma);
      DLR_TEST_ASSERT(kernel.size() == kernelSize);
      for(size_t ii = 0; ii < kernelSize; ++ii) {
        double x = ii - ((kernelSize - 1) / 2.0);
        double exponent = -(x * x) / (2.0 * sigma * sigma);
        double referenceValue =
          (1.0 / (std::sqrt(2 * constants::pi) * sigma)) * std::exp(exponent);
        DLR_TEST_ASSERT(
          approximatelyEqual(kernel[ii], referenceValue, m_defaultTolerance));
      }

      kernelSize = 9;
      kernel = getGaussian1D<double>(sigma, kernelSize);
      DLR_TEST_ASSERT(kernel.size() == kernelSize);
      double sum = 0.0;
      for(size_t ii = 0; ii < kernelSize; ++ii) {
        double x = ii - ((kernelSize - 1) / 2.0);
        double exponent = -(x * x) / (2.0 * sigma * sigma);
        double referenceValue =
          (1.0 / (std::sqrt(2 * constants::pi) * sigma)) * std::exp(exponent);
        DLR_TEST_ASSERT(
          approximatelyEqual(kernel[ii], referenceValue, m_defaultTolerance));
        sum += kernel[ii];
      }

      kernel = getGaussian1D<double>(sigma, kernelSize, true);
      DLR_TEST_ASSERT(kernel.size() == kernelSize);
      for(size_t ii = 0; ii < kernelSize; ++ii) {
        double x = ii - ((kernelSize - 1) / 2.0);
        double exponent = -(x * x) / (2.0 * sigma * sigma);
        double referenceValue =
          ((1.0 / (std::sqrt(2 * constants::pi) * sigma)) * std::exp(exponent)
           / sum);
        DLR_TEST_ASSERT(
          approximatelyEqual(kernel[ii], referenceValue, m_defaultTolerance));
      }
    }

  } // namespace numeric
  
} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::numeric::SampledFunctionsTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::numeric::SampledFunctionsTest currentTest;

}

#endif
