/**
***************************************************************************
* @file dlrComputerVision/subpixelInterpolate.h
*
* Header file declaring routines for fitting quadratics to local
* patches of images or arrays, and using those quadratics to find
* local maxima or minima with subpixel precision.
*
* Copyright (C) 2009 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef DLR_NUMERIC_SUBPIXELINTERPOLATE_H
#define DLR_NUMERIC_SUBPIXELINTERPOLATE_H


namespace dlr {

  namespace numeric {

    /** 
     * This function solves for the coefficients of a 2D quadratic,
     * given the function values on a 3x3 grid around the origin.  It
     * is used internally by subpixelInterpolate(), but is exposed
     * here in case its useful in other contexts.  This function is
     * completely hardcoded, so it should be reasonably fast.
     *
     * Note that this function talks in (x, y) coordinates, while
     * subpixelInterpolate() talks in (row, column) coordinates.  Our
     * normal convention is that row number goes up with increasing y
     * coordinate and column number goes up with increasing x
     * coordinate, so this makes the valueYX arguments match nicely
     * with the valueRowColumn arguments of subpixelInterpolate(), but
     * it's something to be aware of if you don't follow this
     * convention.
     *
     * More detail about what this routine does: assuming you have
     * the function
     *
     * @code
     *   f(x, y) = [x, y] [k0, k1] [x] + [x, y] [k3] + k5
     *                    [k1, k2] [y]          [k4]
     * @endcode
     *
     * and are given the values of f(x, y) for the 3x3 grid x = {-1,
     * 0, 1}, y = {-1, 0, 1}, this function returns the values of k0
     * ... k5 that most nearly satisfy the equation (in the least
     * squares sense).
     * 
     * @param value00 This argument is the value of f(-1, -1).
     * 
     * @param value01 This argument is the value of f(0, -1).
     * 
     * @param value02 This argument is the value of f(1, -1).
     * 
     * @param value10 This argument is the value of f(-1, 0).
     * 
     * @param value11 This argument is the value of f(0, 0).
     * 
     * @param value12 This argument is the value of f(1, 0).
     * 
     * @param value20 This argument is the value of f(-1, 1).
     * 
     * @param value21 This argument is the value of f(0, 1).
     * 
     * @param value22 This argument is the value of f(1, 1).
     * 
     * @param k0 This argument returns a recovered parameter of the
     * quadratic function.
     * 
     * @param k1 This argument returns a recovered parameter of the
     * quadratic function.
     * 
     * @param k2 This argument returns a recovered parameter of the
     * quadratic function.
     * 
     * @param k3 This argument returns a recovered parameter of the
     * quadratic function.
     * 
     * @param k4 This argument returns a recovered parameter of the
     * quadratic function.
     * 
     * @param k5 This argument returns a recovered parameter of the
     * quadratic function.
     */
    template <class Type0>
    inline void
    getQuadraticCoefficients3x3(
      Type0 value00, Type0 value01, Type0 value02,
      Type0 value10, Type0 value11, Type0 value12,
      Type0 value20, Type0 value21, Type0 value22,
      double& k0, double& k1, double& k2, double& k3, double& k4, double& k5);


    /** 
     * Given pixel values in a 3x3 array around (centerRow,
     * centerColumn), this function fits a quadratic to the array
     * values and returns the location and interpolated value of the
     * extremum (max or min value) of the quadratic.  Note that when
     * choosing the values of parameters centerRowCoord and
     * centerColumnCoord, you have to think carefully about your pixel
     * coordinate system.  It is necessary to decide whether a row
     * coordinate of 6.0 means "the center of the sixth row," or "On
     * the border between the fifth and sixth rows".  We encourage the
     * convention that the pixel at integer coordinates [i, j] has
     * corners at (i, j), (i, j+1), (i+1, j), and (i+1, j+1).  Under
     * this convention, the center of pixel [i, j] is at row
     * coordinate (i + 0.5), column coordinate (j + 0.5).
     * 
     * @param centerRowCoord This argument is the row coordinate of the
     * center of the center pixel of the 3x3 neighborhood.  
     * 
     * @param centerColumnCoord This argument is the column coordinate of the
     * center of the center pixels of the 3x3 neighborhood.
     * 
     * @param value00 This argument is the pixel value at (centerRowCoord -
     * 1, centerColumnCoord - 1).
     * 
     * @param value01 This argument is the pixel value at (centerRowCoord -
     * 1, centerColumnCoord).
     * 
     * @param value02 This argument is the pixel value at (centerRowCoord -
     * 1, centerColumnCoord + 1).
     * 
     * @param value10 This argument is the pixel value at (centerRowCoord,
     * centerColumnCoord - 1).
     * 
     * @param value11 This argument is the pixel value at (centerRowCoord,
     * centerColumnCoord).
     * 
     * @param value12 This argument is the pixel value at (centerRowCoord,
     * centerColumnCoord + 1).
     * 
     * @param value20 This argument is the pixel value at (centerRowCoord +
     * 1, centerColumnCoord - 1).
     * 
     * @param value21 This argument is the pixel value at (centerRowCoord +
     * 1, centerColumnCoord).
     * 
     * @param value22 This argument is the pixel value at (centerRowCoord +
     * 1, centerColumnCoord + 1).
     * 
     * @param extremumRowCoord This argument returns the (subpixel) row
     * coordinate of the recovered extremum.
     * 
     * @param extremumColumnCoord This argument returns the (subpixel)
     * column coordinate of the recovered extremum.
     * 
     * @param extremeValue This argument returns the interpolated
     * pixel value at the recovered extremum.
     *
     * @return The return value is true if the extremum is well
     * defined, false otherwise.  Note that this function does not
     * check to see whether the location is extremum is reasonable.
     * You'll have to check for yourself to make sure that it doesn't
     * think the extremum is miles from (centerRowCoord,
     * centerColumnCoord).  A reasonable check is to make sure it's
     * within the original 3x3 neighborhood.
     */
    template <class Type0, class Type1>
    bool
    subpixelInterpolate(double centerRowCoord, double centerColumnCoord,
                        Type0 value00, Type0 value01, Type0 value02,
                        Type0 value10, Type0 value11, Type0 value12,
                        Type0 value20, Type0 value21, Type0 value22,
                        double& extremumRowCoord, double& extremumColumnCoord,
                        Type1& extremeValue);


    /** 
     * Given pixel values in a 3x1 arrangement centerPosition, this
     * function fits a quadratic to the values and returns the
     * location and interpolated value of the extremum (max or min
     * value) of the quadratic.  Note that when choosing the values of
     * parameter centerPosition, you have to think carefully about
     * your pixel coordinate system.  It is necessary to decide
     * whether a coordinate of 6.0 means "the center of the sixth
     * pixel," or "On the border between the fifth and sixth pixels".
     * We encourage the convention that the pixel at integer
     * coordinate i has boundaries at i and (i+1).  Under this
     * convention, the center of pixel i is at coordinate (i + 0.5).
     * 
     * @param centerPosition This argument is the coordinate of the
     * center of the center pixel of the 3x1 neighborhood.
     * 
     * @param value0 This argument is the pixel value at
     * (centerPosition - 1).
     * 
     * @param value1 This argument is the pixel value at
     * centerPosition.
     * 
     * @param value2 This argument is the pixel value at
     * (centerPosition + 1).
     * 
     * @param extremumPosition This argument returns the (subpixel)
     * position of the recovered extremum.
     * 
     * @param extremeValue This argument returns the interpolated
     * pixel value at the recovered extremum.
     *
     * @return The return value is true if the extremum is well
     * defined, false otherwise.  Note that this function does not
     * check to see whether the location is extremum is reasonable.
     * You'll have to check for yourself to make sure that it doesn't
     * think the extremum is miles from centerPosition.
     */
    template <class Type0, class Type1>
    bool
    subpixelInterpolate(double centerPosition,
                        Type0 value0, Type0 value1, Type0 value2,
                        double& extremumPosition,
                        Type1& extremeValue);
    
  } // namespace numeric
  
} // namespace dlr


/* ============ Definitions of inline & template functions ============ */


#include <cmath>

namespace dlr {

  namespace numeric {

    // This function solves for the coefficients of a 2D quadratic,
    // given the function values on a 3x3 grid around the origin.
    template <class Type0>
    void
    getQuadraticCoefficients3x3(
      Type0 value00, Type0 value01, Type0 value02,
      Type0 value10, Type0 value11, Type0 value12,
      Type0 value20, Type0 value21, Type0 value22,
      double& k0, double& k1, double& k2, double& k3, double& k4, double& k5)
    {
      // Given f(x0, y0) = value00, we can rearrange the quadratic
      // equation like this:
      //
      // @code
      //   [x0*x0, 2*x0*y0, y0*y0, x0, y0, 1] [k0, k1, k2, k3, k5, k5]^T = value00
      // @endcode
      // 
      // We're always going to look at a 3x3 neighborhood around (0,
      // 0), so this gives us 9 simultaneous equations that are
      // quadratic in x, y, but linear in the parameters for which
      // we're solving.
      //
      // @code
      //   [     ][k0]   [value00]
      //   [  A  ][k1] = [value01]
      //   [     ][k2]   [value02]
      //          [k3]   [value10]
      //          [k4]   [value11]
      //          [k5]   [value12]
      //                 [value20]
      //                 [value21]
      //                 [value22]
      // @endcode
      // 
      // Where A is a 9x6 matrix:
      //
      // @code
      //       [1,  2,  1, -1, -1,  1]
      //       [0,  0,  1,  0, -1,  1]
      //       [1, -2,  1,  1, -1,  1]
      //       [1,  0,  0, -1,  0,  1]
      //   A = [0,  0,  0,  0,  0,  1]
      //       [1,  0,  0,  1,  0,  1]
      //       [1, -2,  1, -1,  1,  1]
      //       [0,  0,  1,  0,  1,  1]
      //       [1,  2,  1,  1,  1,  1]
      // @endcode
      //
      // We simply hardcode the pseudoInverse here.
      k0 = (0.16666666666666663 * value00
            + -0.33333333333333331 * value01
            + 0.16666666666666663 * value02
            + 0.16666666666666663 * value10
            + -0.33333333333333337 * value11
            + 0.16666666666666663 * value12
            + 0.16666666666666663 * value20
            + -0.33333333333333331 * value21
            + 0.16666666666666663 * value22);
      k1 = (0.125 * value00
            + -0.125 * value02
            + -0.125 * value20
            + 0.125 * value22);
      k2 = (0.16666666666666663 * value00
            + 0.16666666666666663 * value01
            + 0.16666666666666663 * value02
            + -0.33333333333333331 * value10
            + -0.33333333333333331 * value11
            + -0.33333333333333331 * value12
            + 0.16666666666666663 * value20
            + 0.16666666666666663 * value21
            + 0.16666666666666663 * value22);
      k3 = (-0.16666666666666666 * value00
            + 0.16666666666666666 * value02
            + -0.16666666666666666 * value10
            + 0.16666666666666666 * value12
            + -0.16666666666666666 * value20
            + 0.16666666666666666 * value22);
      k4 = (-0.16666666666666666 * value00
            + -0.16666666666666666 * value01
            + -0.16666666666666666 * value02
            + 0.16666666666666666 * value20
            + 0.16666666666666666 * value21
            + 0.16666666666666666 * value22);
      k5 = (-0.11111111111111116 * value00
            + 0.22222222222222227 * value01
            + -0.11111111111111116 * value02
            + 0.22222222222222221 * value10
            + 0.55555555555555558 * value11
            + 0.22222222222222221 * value12
            + -0.11111111111111116 * value20
            + 0.22222222222222227 * value21
            + -0.11111111111111116 * value22);
    }

    
    template <class Type0, class Type1>
    bool
    subpixelInterpolate(double centerRowCoord, double centerColumnCoord,
                        Type0 value00, Type0 value01, Type0 value02,
                        Type0 value10, Type0 value11, Type0 value12,
                        Type0 value20, Type0 value21, Type0 value22,
                        double& extremumRowCoord, double& extremumColumnCoord,
                        Type1& extremeValue)
    {
      // First, we solve for the coefficients of a quadratic in the
      // neighborhood of (centerRowCoord, centerColumnCoord), with the
      // parameters of the quadratic being location (rowCoord,
      // columnCoord), and the scalar quadratic values being provided
      // in the argument list (value00, value01, etc.) above.
      //
      // @code
      //   valueRC = [R, C] [k0, k1] [R] + [R, C] [k3] + k5
      //                    [k1, k2] [C]          [k4]
      // @endcode
      //
      // This function call solves for k0 through k5 in the above
      // equation.
      double k0, k1, k2, k3, k4, k5;
      getQuadraticCoefficients3x3(
        value00, value01, value02,
        value10, value11, value12,
        value20, value21, value22,
        k0, k1, k2, k3, k4, k5);

      // Now we have the parameters of the quadratic, we need to find
      // its extremum.  That is, the place where it's derivitive goes
      // to zero.  We need to solve this equation for x and y:
      //
      // @code
      //   2[k0, k1][x] + [k3] = [0]
      //    [k1, k2][y]   [k4]   [0]
      // @endcode
      //
      // We simply hardcode a cofactor method inverse below, starting
      // with finding the determinant.
      double determinant = k0 * k2 - k1 * k1;

      // Before we continue with the matrix inversion, consider
      // whether we're going to find a min, a max, or a saddle point.
      // Essentially, we need to know if matrix [k0,k1; k1, k2] has
      // one positive eigenvalue and one negative eigenvalue (in which
      // case we'll find a saddle point).  If we were to solve for
      // those eigenvalues, we'd be trying to find values of lambda
      // such that
      //
      // @code
      //   determinant([k0 - lambda, k1         ]) == 0
      //               [         k1, k2 - lambda]
      // @endcode
      //
      // If you write this out, you get a quadratic equation:
      //
      //   lambda^2 + (k0 + k2) * lambda + (k0 * k2 - k1^2) = 0
      //
      // the signs of the two solutions to this equation are different
      // if and only if (k0 * k2 - k1^2) is negative (check out the
      // documentation of solveQuadratic() in utilities.h to see why
      // this is true), but this is just the determinant we just computed.
      //
      // So, if determinant is zero, we can't do the matrix inverse.
      // If determinant < 0, we'll only find a saddle point.  In both
      // cases, we simply return false.
      if(determinant <= 0) {
        return false;
      }

      // Now continue with the matrix inversion, and compute the
      // position of the extremum.
      double detTimes2 = determinant * 2.0;
      double inverseMx00 = k2 / detTimes2;
      double inverseMx01 = -k1 / detTimes2;
      double inverseMx11 = k0 / detTimes2;

      double maxX = -k3 * inverseMx00 - k4 * inverseMx01;
      double maxY = -k3 * inverseMx01 - k4 * inverseMx11;

      extremumRowCoord = centerRowCoord + maxY;
      extremumColumnCoord = centerColumnCoord + maxX;

      // Now that we have the location of the extremum, use the
      // quadratic equation to estimate its "real" value.
      extremeValue =
        (k0 * maxX * maxX + 2 * k1 * maxX * maxY + k2 * maxY * maxY
         + k3 * maxX + k4 * maxY + k5);

      return true;
    }


    
    // Given pixel values in a 3x1 arrangement centerPosition, this
    // function fits a quadratic to the values and returns the
    // location and interpolated value of the extremum (max or min
    // value) of the quadratic.
    template <class Type0, class Type1>
    bool
    subpixelInterpolate(double centerPosition,
                        Type0 value0, Type0 value1, Type0 value2,
                        double& extremumPosition,
                        Type1& extremeValue)
    {
      // First, we solve for the coefficients of a quadratic in the
      // neighborhood of centerPosition, with the parameters of the
      // quadratic being position, and the scalar quadratic values
      // being provided in the argument list (value0, value1, value2)
      // above.
      //
      // @code
      //   valueP = k0*p*p + k1*p + k2
      // @endcode
      //
      // To make this easier, we solve as if centerPosition were 0.
      // We'll correct this later.  Assuming centerPosition is zero,
      // we can write the above equation three times, one for each of
      // the input values, assuming coodinates of -1, 0, 1.  Writing
      // this in matrix form gives:
      // 
      // @code
      //   [1, -1, 1][k0]   [value0]
      //   [0,  0, 1][k1] = [value1]
      //   [1,  1, 1][k2]   [value2]
      // @endcode
      //
      // Solving this using the Moore-Penrose pseudoinverse gives:
      //
      // @code
      //   [k0]   [ 0.5, -1.0, 0.5][value0]
      //   [k1] = [-0.5,  0.0, 0.5][value1]
      //   [k2]   [ 0.0,  1.0, 0.0][value2]
      // @endcode
      double k0 =  0.5 * value0 - value1 + 0.5 * value2;
      double k1 = -0.5 * value0 + 0.5 * value2;
      double k2 = value1;

      // Now we have the parameters of the quadratic, we need to find
      // its extremum.  That is, the place where it's derivitive goes
      // to zero.  We need to solve this equation for p:
      //
      // @code
      //   2*k0*p + k1 = 0
      // @code
      //
      // If k0 is zero, we can't solve this.  Instead we return false.
      if(k0 == 0) {
        return false;
      }
      double maxP = -k1 / (2 * k0);

      // Now correct the "assume centerPosition is zero" dodge we
      // introduced above.
      extremumPosition = centerPosition + maxP;

      // Now that we have the location of the extremum, use the
      // quadratic equation to estimate its "real" value.
      extremeValue = k0 * maxP * maxP + k1 * maxP + k2;

      return true;
    }
    
  } // namespace numeric

} // namespace dlr

#endif /* #ifndef DLR_NUMERIC_SUBPIXELINTERPOLATE_H */
