/**
***************************************************************************
* @file dlrNumeric/sampledFunctions.h
*
* Header file declaring code which return sampled versions of common
* functions, such as gaussians, etc.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef _DLRNUMERIC_SAMPLEDFUNCTIONS_H_
#define _DLRNUMERIC_SAMPLEDFUNCTIONS_H_

#include <dlrNumeric/array1D.h>

namespace dlr {

  namespace numeric {

    /** 
     * This function returns an array in which the elements are
     * sampled from a 1D Gaussian.  The maximum of the Gaussian is at
     * the center of the returned array, and the distance between
     * adjacent samples (elements of the array) is assumed to be
     * exactly 1.0.
     * 
     * @param sigma This argument specifies the desired standard
     * deviation of the Gaussian.
     * 
     * @param size This argument specifies how many elements should be
     * in the returned array.  If size is set to 0, the returned value
     * will have the smallest odd number of elements that is greater
     * than or equal to 6.0 * sigma. xxx is this correct?
     * 
     * @param normalize This argument specifies whether, after the
     * elements of the array have been computed, the array should be
     * rescaled so that its elements sum to 1.0.
     * 
     * @return The return value is an array containing the sampled
     * values.
     */
    template <class TYPE>
    Array1D<TYPE>
    getGaussian1D(double sigma,
                  size_t size = 0,
                  bool normalize = false);

  } // namespace numeric
  
} // namespace dlr


/* =============================================================== */
/* Implementation follows.                                         */
/* =============================================================== */


#include <cmath>
#include <dlrNumeric/functional.h>
#include <dlrNumeric/utilities.h>

namespace dlr {

  namespace numeric {

    // This function returns an array in which the elements are
    // sampled from a 1D Gaussian.
    template <class TYPE>
    Array1D<TYPE>
    getGaussian1D(double sigma, size_t size, bool normalize)
    {
      if(size == 0) {
        size = static_cast<size_t>(6.0 * sigma + 0.5);
        if(size % 2 == 0) {
          ++size;
        }
      }
      Array1D<TYPE> result(size);
      Gaussian1DFunctor<double> functor(sigma);
      double x = (1.0 - static_cast<double>(size))/2.0;
      for(size_t index0 = 0; index0 < size; ++index0) {
        result[index0] = functor(x);
        x += 1.0;
      }
      if(normalize) {
        result /= sum(result);
      }
      return result;
    }
    
  } // namespace numeric
  
} // namespace dlr

#endif /* #ifndef _DLRNUMERIC_SAMPLEDFUNCTIONS_H_ */
