/**
***************************************************************************
* @file dlrNumeric/numericTraits.h
*
* Header file declaring numericTraits types.  The classes in this file
* need to be updated soon to use the new portable types from
* dlrCommon/types.h.
*
* Copyright (C) 2003-2007, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 975 $
* $Date: 2007-12-30 01:57:17 -0500 (Sun, 30 Dec 2007) $
***************************************************************************
**/

#ifndef _DLR_NUMERICTRAITS_H_
#define _DLR_NUMERICTRAITS_H_

#include <dlrCommon/exception.h>

namespace dlr {

  namespace numeric {
    
    /**
     ** This class is used to customize the behavior of dlr_libs
     ** classes & functions for specific numeric types.  Ultimately
     ** we'll use it to control the precision of computations, numeric
     ** limits, etc.
     **/
    template <class Type>
    class NumericTraits {
    public:
      /* ==================== Public typedefs ==================== */

      /**
       * This typedef is used to control the type of the result when
       * values of Type are multiplied.  productType should have enough
       * precision that several instances of Type can be multiplied
       * without overflow or loss of information.
       */
      typedef Type ProductType;

      /**
       * This typedef is used to control the type of the result when
       * values of Type are added.  sumType should have enough precision
       * that several instances of Type can be added without overflow or
       * loss of information.
       */
      typedef Type SumType;

      /**
       * This typedef is used to control the way arrays of Type are
       * printed by operator<<().  Each element of the array will be
       * formatted as TextOutputType when it is written.
       */
      typedef Type TextOutputType;

      /* ==================== Public methods ==================== */

      /** 
       * This member function indicates whether the specified type is an
       * integer type or not.  It is not implemented for the general case,
       * so NumericTraits must be specialized for every class which requires
       * this functionality.
       * 
       * @return The return value should true if this type is an integer
       * type, false other wise.  The general version of this function just
       * throws and exception.
       */
      inline bool
      isIntegral() {
        DLR_THROW3(NotImplementedException, "NumericTraits::isIntegral()",
                   "In order to use isIntegral, NumericTraits must be "
                   "specialized for each type.");
        return true; // Appease the compiler.
      }

    };

    /**
     ** This class is used to customize the behavior of dlr_libs
     ** classes & functions for the char type.
     **/
    template <>
    class NumericTraits<char> {
    public:
      /* ==================== Public typedefs ==================== */

      /**
       * This typedef specifies that when chars are multiplied, the
       * result should be an int.  This prevents overflow when the sum
       * of the components is more than 255.
       */
      typedef int ProductType;

      /**
       * This typedef specifies that when chars are added, the result
       * should be an int.  This prevents overflow when the sum of the
       * components is more than 255.
       */
      typedef int SumType;

      /**
       * This typedef specifies that when a char is serialized for
       * stream output as part of an array, each char is formatted as a
       * number, rather than as an ascii symbol.  For example, you might
       * see
       *
       *   Array1D([64, 97, 32])
       *
       * rather than
       *
       *   Array1D([@, a,  ])
       */
      typedef int TextOutputType;

      /* ==================== Public methods ==================== */

      /** 
       * This member function indicates whether the specified type is an
       * integer type or not.
       * 
       * @return The return is true if this type is an integer
       * type, false other wise.
       */
      inline bool
      isIntegral() {return true;}
    
    };

  
    /**
     ** This class is used to customize the behavior of dlr_libs
     ** classes & functions for the unsigned char type.
     **/
    template <>
    class NumericTraits<unsigned char> {
    public:
      /* ==================== Public typedefs ==================== */

      /**
       * This typedef specifies that when unsigned chars are multiplied,
       * the result should be an unsigned int.  This prevents overflow
       * when the sum of the components is more than 255.
       */
      typedef unsigned int ProductType;

      /**
       * This typedef specifies that when unsigned chars are added, the 
       * result should be an unsigned int.  This prevents overflow when
       * the sum of the components is more than 255.
       */
      typedef unsigned int SumType;

      /**
       * This typedef specifies that when an unsigned char is serialized for
       * stream output, it should be formatted as an int.
       */
      typedef int TextOutputType;

      /* ==================== Public methods ==================== */
    
      /** 
       * This member function indicates whether the specified type is an
       * integer type or not.
       * 
       * @return The return is true if this type is an integer
       * type, false other wise.
       */
      inline bool
      isIntegral() {return true;}
    
    };

  
    /**
     ** This class is used to customize the behavior of dlr_libs
     ** classes & functions for the short type.
     **/
    template <>
    class NumericTraits<short> {
    public:
      /* ==================== Public typedefs ==================== */

      /**
       * This typedef specifies that when shorts are multiplied, the
       * result should be an int.  This prevents overflow when the sum
       * of the components is more than 2^16.
       */
      typedef int ProductType;

      /**
       * This typedef specifies that when shorts are added, the result
       * should be an int.  This prevents overflow when the sum of the
       * components is more than sizeof(short).
       */
      typedef int SumType;

      /**
       * This typedef specifies that when an int is serialized for stream
       * output, it should be formatted as a short.
       */
      typedef short TextOutputType;

      /* ==================== Public methods ==================== */

      /** 
       * This member function indicates whether the specified type is an
       * integer type or not.
       * 
       * @return The return is true if this type is an integer
       * type, false other wise.
       */
      inline bool
      isIntegral() {return true;}
    
    };

    /**
     ** This class is used to customize the behavior of dlr_libs
     ** classes & functions for the int type.
     **/
    template <>
    class NumericTraits<int> {
    public:
      /* ==================== Public typedefs ==================== */

      /**
       * This typedef specifies that when ints are multiplied, the
       * result should be a long.
       */
      typedef long ProductType;

      /**
       * This typedef specifies that when ints are added, the result
       * should be a long.
       */
      typedef long SumType;

      /**
       * This typedef specifies that when an int is serialized for stream
       * output, it should be formatted as an int.
       */
      typedef int TextOutputType;

      /* ==================== Public methods ==================== */

      /** 
       * This member function indicates whether the specified type is an
       * integer type or not.
       * 
       * @return The return is true if this type is an integer
       * type, false other wise.
       */
      inline bool
      isIntegral() {return true;}
    
    };

  
    /**
     ** This class is used to customize the behavior of dlr_libs
     ** classes & functions for the long type.
     **/
    template <>
    class NumericTraits<long> {
    public:
      /* ==================== Public typedefs ==================== */

      /**
       * This typedef specifies that when longs are multiplied, the
       * result should be a long.  Note the assumption that long ints
       * won't overflow.
       */
      typedef long ProductType;

      /**
       * This typedef specifies that when longs are added, the result
       * should be an long.  Note the assumption that long ints won't
       * overflow.
       */
      typedef long SumType;

      /**
       * This typedef specifies that when an long is serialized for stream
       * output, it should be formatted as an long.
       */
      typedef long TextOutputType;

      /* ==================== Public methods ==================== */

      /** 
       * This member function indicates whether the specified type is an
       * integer type or not.
       * 
       * @return The return is true if this type is an integer
       * type, false other wise.
       */
      inline bool
      isIntegral() {return true;}
    
    };

    /**
     ** This class is used to customize the behavior of dlr_libs
     ** classes & functions for the float type.
     **/
    template <>
    class NumericTraits<float> {
    public:
      /* ==================== Public typedefs ==================== */

      /**
       * This typedef specifies that when floats are multiplied, the
       * result should be a float.
       */
      typedef float ProductType;

      /**
       * This typedef specifies that when floats are added, the result
       * should be an float.
       */
      typedef float SumType;

      /**
       * This typedef specifies that when an float is serialized for stream
       * output, it should be formatted as an float.
       */
      typedef float TextOutputType;

      /* ==================== Public methods ==================== */

      /** 
       * This member function indicates whether the specified type is an
       * integer type or not.
       * 
       * @return The return is true if this type is an integer
       * type, false other wise.
       */
      inline bool
      isIntegral() {return false;}
    
    };

    /**
     ** This class is used to customize the behavior of dlr_libs
     ** classes & functions for the double type.
     **/
    template <>
    class NumericTraits<double> {
    public:
      /* ==================== Public typedefs ==================== */

      /**
       * This typedef specifies that when doubles are multiplied, the
       * result should be a double.
       */
      typedef double ProductType;

      /**
       * This typedef specifies that when doubles are added, the result
       * should be an double.
       */
      typedef double SumType;

      /**
       * This typedef specifies that when an double is serialized for stream
       * output, it should be formatted as an double.
       */
      typedef double TextOutputType;

      /* ==================== Public methods ==================== */

      /** 
       * This member function indicates whether the specified type is an
       * integer type or not.
       * 
       * @return The return is true if this type is an integer
       * type, false other wise.
       */
      inline bool
      isIntegral() {return false;}
    
    };

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::NumericTraits;

} // namespace dlr

#endif // #ifndef _DLR_NUMERICTRAITS_H_
