/**
***************************************************************************
* @file dlrNumeric/normalizedCorrelator.h
*
* Header file declaring NormalizedCorrelator class.
*
* Copyright (C) 1999-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 880 $
* $Date: 2007-05-04 00:33:49 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef DLR_NUMERIC_NORMALIZEDCORRELATOR_H
#define DLR_NUMERIC_NORMALIZEDCORRELATOR_H

#include <cmath>
#include <deque>
#include <dlrNumeric/array2D.h>

namespace dlr {

  namespace numeric {


    /**
     ** This class implements 1D normalized correlation, which is
     ** sometimes also called the Correlation Coefficient.  That is,
     ** given two signals, f[x] and g[x], it computes C(f, g), where
     **
     **   C(f, g) = sum_x(p[x] * q[x]),
     **
     ** and
     **
     **   p[x] = (f[x] - mean(f)) / variance(f)
     **
     ** and
     **
     **   q[x] = (g[x] - mean(g)) / variance(g)
     **
     ** For efficiency reasons, the actual computation doesn't
     ** explicitly represent the mean and variance of the input
     ** signals.
     **
     ** NormalizedCorrelator is designed to support efficient
     ** incremental updates of the signals being correlated, and makes
     ** it easy to add and remove samples from the correlation
     ** on-the-fly.
     **
     ** Here are some examples of how you might use this class.
     **
     ** @code
     **
     **   std::vector<double> signalF;
     **   std::vector<double> signalG;
     ** 
     **   // Real code would set the contents of signalF and signalG
     **   // here.  They must have the same size.
     **
     **   // Here, we compute the normalized correlation of the two
     **   // signals.
     **   NormalizedCorrelator<double> correlator0(
     **     signalF.begin(), signalF.end(), signalG.begin());
     **   double correlation0 = correlator0.getNormalizedCorrelation();
     **
     **   // Here, we repeat the computation using a slightly
     **   // different sequence of calls.
     **   NormalizedCorrelator<double> correlator1;
     **   correlator1.addSamples(
     **     signalF.begin(), signalF.end(), signalG.begin());
     **   double correlation1 = correlator1.getNormalizedCorrelation();
     **
     **   // These lines compute the normalized correlation of the
     **   // first 100 elements of the two signals.
     **   NormalizedCorrelator<double> correlator2;
     **   correlator2.addSamples(
     **     signalF.begin(), signalF.begin() + 100, signalG.begin());
     **   double correlation2 = correlator2.getNormalizedCorrelation();
     **
     **   // These lines illustrate incremental update of the signals
     **   // being correlated.  They compute the normalized
     **   // correlation of the 2nd through 101st elements of the two
     **   // signals, but use the result of the previous computation
     **   // to save time.
     **   correlator2.removeSamples(
     **     signalF.begin(), signalF.begin() + 2, signalG.begin());
     **   correlator2.addSamples(
     **     signalF.begin() + 100, signalF.begin() + 102,
     **     signalG.begin() + 100);
     **   double correlation3 = correlator1.getNormalizedCorrelation();
     **
     **   // These lines repeat the computation of correlation2 and
     **   // correlation3, but request that the NormalizedCorrelator
     **   // instance remember the order in which samples were added
     **   // so that it can automatically remove the oldest samples.
     **   // The call to removeOldestSamples() below has the same
     **   // effect as the call to removeSamples() above, but in most
     **   // situations requires less bookkeeping in the calling
     **   // context.
     **   NormalizedCorrelator<double> correlator4(true);
     **   correlator4.addSamples(
     **     signalF.begin(), signalF.begin() + 100, signalG.begin());
     **   double correlation4 = correlator4.getNormalizedCorrelation();
     **
     **   correlator4.removeOldestSamples(2);
     **   correlator4.addSamples(
     **     signalF.begin() + 100, signalF.begin() + 102,
     **     signalG.begin() + 100);
     **   double correlation5 = correlator1.getNormalizedCorrelation();
     **
     ** @endcode
     **
     **/
    template <class Type>
    class NormalizedCorrelator
    {
    public:

      /** 
       * This constructor initializes the NormalizedCorrelator
       * instance, but doesn't add any samples.
       * 
       * @param trackInput This argument indicates whether the
       * NormalizedCorrelator instance should keep a record of samples
       * as they're added so that it can automatically remove them in
       * order using member function removeSamples(size_t).
       */
      NormalizedCorrelator(bool trackInput = false);


      /** 
       * This constructor initializes the NormalizedCorrelator
       * instance using sequences of samples from the two signals to
       * be correlated.  After calling this constructor, the
       * normalized correlation of the two input signals is available
       * via member function getNormalizedCorrelation().  If the
       * addSamples() method is called after calling this constructor,
       * the effect will be as if the input sequences from the
       * constructor and addSamples() calls were simply concatenated.
       * 
       * @param begin0 This argument is an iterator pointing to the
       * beginning of the sequence of samples from the first of the
       * two signals to be correlated.
       * 
       * @param end0 This argument is an iterator pointing to the
       * end of the sequence of samples from the first of the two
       * signals to be correlated.  Just as with standard library
       * algorithms, the final element of the input sequence is the
       * one _before_ *end0.
       * 
       * @param begin1 This argument is an iterator pointing to the
       * beginning of the sequence of samples from the second of the
       * two signals to be correlated.
       * 
       * @param trackInput This argument indicates whether the
       * NormalizedCorrelator instance should keep a record of samples
       * as they're added so that it can automatically remove them in
       * order using member function removeSamples(size_t).
       */
      template <class IterType0, class IterType1>
      NormalizedCorrelator(IterType0 begin0,
                           IterType0 end0,
                           IterType1 begin1,
                           bool trackInput = false);


      /** 
       * The destructor destroys the NormalizedCorrelator instance and
       * cleans up any associated resources.
       */
      ~NormalizedCorrelator();


      /** 
       * This member function adds a single pair of samples (one from
       * each of the two signals to be correlated) to the normalized
       * correlation calculation.  You might call this repeatedly,
       * once for each of many different samples.  If input tracking
       * is enabled (via constructor argument) then this function will
       * update the internal record keeping so that pairs of samples
       * can be automatically removed by a call to member function
       * removeOldestSample() and removeOldestSamples().
       * 
       * @param sample0 This argument is the sample value from the
       * first of the two input signals.
       * 
       * @param sample1 This argument  is the sample value from the
       * second of the two input signals.
       */
      inline void
      addSample(Type sample0, Type sample1);


      /** 
       * This member function works identically to addSample(), with
       * the exception that input tracking is never updated.  Use this
       * member function instead of addSample() if you know that you
       * will never have input tracking enabled, you can't pass your
       * samples in en masse using addSamples(), and you're in such a
       * hurry that the run-time cost of one conditional branch is
       * worth avoiding.
       * 
       * @param sample0 This argument is the sample value from the
       * first of the two input signals.
       * 
       * @param sample1 This argument  is the sample value from the
       * second of the two input signals.
       */
      inline void
      addSampleWithoutTracking(Type sample0, Type sample1);


      /** 
       * This member function adds a sequence of pairs of samples
       * (each pair containing one sample from each of the two signals
       * to be correlated) to the normalized correlation calculation.
       * If input tracking is enabled (via constructor argument) then
       * this function will update the internal record keeping so that
       * pairs of samples can be automatically removed by a call to
       * member function removeOldestSamples().  Note that for the
       * purposes of this automatic removal, *begin0 is considered to
       * be added before *(begin0 + 1).
       * 
       * @param begin0 This argument is an iterator pointing to the
       * beginning of the sequence of samples from the first of the
       * two signals to be correlated.
       * 
       * @param end0 This argument is an iterator pointing to the
       * end of the sequence of samples from the first of the two
       * signals to be correlated.  Just as with standard library
       * algorithms, the final element of the input sequence is the
       * one _before_ *end0.
       * 
       * @param begin1 This argument is an iterator pointing to the
       * beginning of the sequence of samples from the second of the
       * two signals to be correlated.
       */
      template <class IterType0, class IterType1>
      void
      addSamples(IterType0 begin0, IterType0 end0, IterType1 begin1);


      /** 
       * This member function removes all samples from the
       * NormalizedCorrelator instance.
       */
      void
      clear();


      /** 
       * This member function enables (or disables) internal
       * recordkeeping that allows samples to be automatically removed
       * from the normalized correlation calculation following the
       * order in which they were added.  When input tracking is
       * disabled, execution and storage requirements are reduced.
       * When input tracking is enabled, the NormalizedCorrelator
       * instance maintains a record of all added samples and the
       * order in which they were added.  Subsequent calls to
       * removeOldestSamples() will discard the oldest samples.  This
       * is useful if you need to maintain a running normalized
       * correlation of, for example, the last 100 sample pairs: each
       * time you get a new set of samples, you can add them using
       * addSample() or addSamples(), discard the corresponding number
       * of sample pairs using removeOldestSamples(), and recover the
       * updated normalized correlation by calling
       * getNormalizedCorrelation().
       *
       * Note that even if input tracking is disabled, you can still
       * remove samples explicitly by calling removeSample() or
       * removeSamples().
       *
       * Calling enableInputTracking(true) when tracking is already
       * enabled has undefined result.
       *
       * Currently, you cannot call removeSample() or removeSamples()
       * on a NormalizedCorrelator instance for which input tracking
       * is enabled (we expect this to change), and you cannot call
       * removeOldestSamples() on a NormalizedCorrelator instance for
       * which input tracking is not enabled.
       * 
       * @param trackInput Setting this argument to true enables input
       * tracking. Setting this argument false disabled input
       * tracking.
       */
      void
      enableInputTracking(bool trackInput = true);
      

      /** 
       * This member function returns the number of sample pairs
       * contributing to the normalized correlation.  It indicates the
       * total number of sample pairs added by calls to the
       * constructor, addSample(), and addSamples(), less the number
       * of sample pairs removed by removeSample(), removeSamples(),
       * and removeOldestSamples().
       * 
       * @return The return value is number of sample pairs.
       */
      inline size_t
      getCount() const;


      /** 
       * This member function returns the normalized correlation of
       * all the currently added sample pairs.  If no sample pairs
       * have been added, the return value is 1.0.
       * 
       * @return The return value is the computed normalized
       * correlation.
       */
      Type
      getNormalizedCorrelation() const;


      /** 
       * This member function returns a bool indicating whether or not
       * input tracking is enabled (see member function
       * enableInputTracking()).
       * 
       * @return The return value is true if input tracking is
       * enabled, false otherwise.
       */
      inline bool
      isInputTrackingEnabled() const {return m_inputTracker0Ptr != 0;}
      

      /** 
       * If input tracking is enabled, this member function removes
       * pairs of samples from the normalized correlation calculation,
       * following the order in which they were added.  It has the
       * same effect as the removeSamples() member function, but does
       * not require the calling context to explicitly specify the
       * sample values to be removed.  Calling this member function
       * when input tracking is disabled is an error.
       * 
       * @param count This argument specifies how many sample pairs to
       * remove.
       */
      void
      removeOldestSamples(size_t count);


      /** 
       * This member function removes a pair of sample values from the
       * normalized correlation calculation.  Note that this function
       * works regardless of whether input tracking is enabled.
       * 
       * @param sample0 This argument is the sample value from the
       * first of the two input signals.
       * 
       * @param sample1 This argument  is the sample value from the
       * second of the two input signals.
       */
      void
      removeSample(Type sample0, Type sample1);


      /** 
       * This member function works identically to removeSample(),
       * with the exception that input tracking is never updated.  Use
       * this member function instead of removeSample() if you know
       * that you will never have input tracking enabled, you can't
       * pass your samples in en masse using removeSamples(), and
       * you're in such a hurry that the run-time cost of one
       * conditional branch is worth avoiding.
       * 
       * @param sample0 This argument is the sample value from the
       * first of the two input signals.
       * 
       * @param sample1 This argument  is the sample value from the
       * second of the two input signals.
       */
      inline void
      removeSampleWithoutTracking(Type sample0, Type sample1);


      /** 
       * This member function removes a sequence of pairs of samples
       * (each pair containing one sample from each of the two signals
       * to be correlated) from the normalized correlation
       * calculation.  Note that this function works regardless of
       * whether input tracking is enabled (see member function
       * enableInputTracking()).
       * 
       * @param begin0 This argument is an iterator pointing to the
       * beginning of the sequence of samples from the first of the
       * two signals.
       * 
       * @param end0 This argument is an iterator pointing to the end
       * of the sequence of samples from the first of the two signals.
       * Just as with standard library algorithms, the final element
       * of the input sequence is the one _before_ *end0.
       * 
       * @param begin1 This argument is an iterator pointing to the
       * beginning of the sequence of samples from the second of the
       * two signals.
       */
      template <class IterType0, class IterType1>
      void
      removeSamples(IterType0 begin0, IterType0 end0, IterType1 begin1);
      
    private:

      size_t m_count;
      std::deque<Type>* m_inputTracker0Ptr;
      std::deque<Type>* m_inputTracker1Ptr;
      Type m_sum0;
      Type m_sum1;
      Type m_sum00;
      Type m_sum01;
      Type m_sum11;

    };

  } // namespace numeric

} // namespace dlr


/* ============ Definitions of inline & template functions ============ */

namespace dlr {

  namespace numeric {
    
    // This constructor initializes the NormalizedCorrelator
    // instance, but doesn't add any samples.
    template <class Type>
    NormalizedCorrelator<Type>::
    NormalizedCorrelator(bool trackInput)
      : m_count(0),
        m_inputTracker0Ptr(0),
        m_inputTracker1Ptr(0),
        m_sum0(static_cast<Type>(0)),
        m_sum1(static_cast<Type>(0)),
        m_sum00(static_cast<Type>(0)),
        m_sum01(static_cast<Type>(0)),
        m_sum11(static_cast<Type>(0))
    {
      if(trackInput) {
        this->enableInputTracking();
      }
    }
    

    // This constructor initializes the NormalizedCorrelator
    // instance using sequences of samples from the two signals to
    // be correlated.
    template <class Type>
    template <class IterType0, class IterType1>
    NormalizedCorrelator<Type>::
    NormalizedCorrelator(IterType0 begin0, IterType0 end0, IterType1 begin1,
                         bool trackInput)
      : m_count(0),
        m_inputTracker0Ptr(0),
        m_inputTracker1Ptr(0),
        m_sum0(static_cast<Type>(0)),
        m_sum1(static_cast<Type>(0)),
        m_sum00(static_cast<Type>(0)),
        m_sum01(static_cast<Type>(0)),
        m_sum11(static_cast<Type>(0))
    {
      if(trackInput) {
        this->enableInputTracking();
      }
      this->addSamples(begin0, end0, begin1);
    }
    

    // The destructor destroys the NormalizedCorrelator instance and
    // cleans up any associated resources.
    template <class Type>
    NormalizedCorrelator<Type>::
    ~NormalizedCorrelator()
    {
      this->enableInputTracking(false);
    }

    
    // This member function adds a single pair of samples (one from
    // each of the two signals to be correlated) to the normalized
    // correlation calculation.
    template <class Type>
    void
    NormalizedCorrelator<Type>::
    addSample(Type sample0, Type sample1)
    {
      // Copy input, if required to do so.
      if(this->isInputTrackingEnabled()) {
        m_inputTracker0Ptr->push_back(sample0);
        m_inputTracker1Ptr->push_back(sample1);
      }
      this->addSampleWithoutTracking(sample0, sample1);
    }

    
    // This member function works identically to addSample(), with
    // the exception that input tracking is never updated.
    template <class Type>
    inline void
    NormalizedCorrelator<Type>::
    addSampleWithoutTracking(Type sample0, Type sample1)
    {
      m_sum0 += sample0;
      m_sum1 += sample1;
      m_sum00 += sample0 * sample0;
      m_sum01 += sample0 * sample1;
      m_sum11 += sample1 * sample1;
      ++m_count;
    }


    // This member function adds a sequence of pairs of samples
    // (each pair containing one sample from each of the two signals
    // to be correlated) to the normalized correlation calculation.
    template <class Type>
    template <class IterType0, class IterType1>
    void
    NormalizedCorrelator<Type>::
    addSamples(IterType0 begin0, IterType0 end0, IterType1 begin1)
    {
      // Copy input, if required to do so.
      if(this->isInputTrackingEnabled()) {
        IterType0 begin0Copy = begin0;
        IterType1 begin1Copy = begin1;
        while(begin0Copy != end0) {
          m_inputTracker0Ptr->push_back(*begin0Copy);
          m_inputTracker1Ptr->push_back(*begin1Copy);
          ++begin0Copy;
          ++begin1Copy;
        }
      }

      // Update statistics.
      while(begin0 != end0) {
        this->addSampleWithoutTracking(*begin0, *begin1);
        ++begin0;
        ++begin1;
      }
    }
      

    // This member function removes all samples from the
    // NormalizedCorrelator instance.
    template <class Type>
    void
    NormalizedCorrelator<Type>::
    clear()
    {
      if(this->isInputTrackingEnabled) {
        // Clear input tracking cache by re-enabling.  The
        // documentation above says this has undefined result, but
        // because we control the implementation, it's ok for us to
        // abuse it.
        this->enableInputTracking(true);
      }
      m_count = 0;
      m_sum0 = static_cast<Type>(0);
      m_sum1 = static_cast<Type>(0);
      m_sum00 = static_cast<Type>(0);
      m_sum01 = static_cast<Type>(0);
      m_sum11 = static_cast<Type>(0);
    }


    // This member function enables (or disables) internal
    // recordkeeping that allows samples to be automatically removed
    // from the normalized correlation calculation following the
    // order in which they were added.
    template <class Type>
    void
    NormalizedCorrelator<Type>::
    enableInputTracking(bool trackInput)
    {
      if(trackInput) {
        if(m_inputTracker0Ptr == 0) {
          m_inputTracker0Ptr = new std::deque<Type>;
        } else {
          m_inputTracker0Ptr->clear();
        }
        if(m_inputTracker1Ptr == 0) {
          m_inputTracker1Ptr = new std::deque<Type>;
        } else {
          m_inputTracker1Ptr->clear();
        }
      } else {
        if(m_inputTracker0Ptr != 0) {
          delete m_inputTracker0Ptr;
          m_inputTracker0Ptr = 0;
        }
        if(m_inputTracker1Ptr != 0) {
          delete m_inputTracker1Ptr;
          m_inputTracker1Ptr = 0;
        }
      }
    }
      

    // This member function returns the number of sample pairs
    // contributing to the normalized correlation.
    template <class Type>
    inline size_t
    NormalizedCorrelator<Type>::
    getCount() const
    {
      return m_count;
    }

      
    // This member function returns the normalized correlation of
    // all the currently added sample pairs.
    template <class Type>
    Type
    NormalizedCorrelator<Type>::
    getNormalizedCorrelation() const
    {
      if(m_count == 0) {
        return static_cast<Type>(1);
      }
      Type oneOverN = static_cast<Type>(1) / static_cast<Type>(m_count);
      Type numerator = m_sum01 - oneOverN * m_sum0 * m_sum1;
      Type denominator = std::sqrt((m_sum00 - oneOverN * m_sum0 * m_sum0)
                                   * (m_sum11 - oneOverN * m_sum1 * m_sum1));
      return numerator / denominator;
    }

    
    // If input tracking is enabled, this member function removes
    // pairs of samples from the normalized correlation calculation,
    // following the order in which they were added.
    template <class Type>
    void
    NormalizedCorrelator<Type>::
    removeOldestSamples(size_t count)
    {
      if(!(this->isInputTrackingEnabled())) {
        DLR_THROW(LogicException,
                  "NormalizedCorrelator::removeInputSamples()",
                  "Attempt to call removeOldestSamples() method of a "
                  "NormalizedCorrelator instance that does not have "
                  "input tracking enabled.");
      }
      if(count > (m_inputTracker0Ptr->size())) {
        DLR_THROW(ValueException,
                  "NormalizedCorrelator::removeInputSamples()",
                  "Trying to remove more samples than have been added.");
      }
      while(count != 0) {
        // Warning(xxx): if this call is changed to removeSample(),
        // then no tests fail, but calls to removeOldestSamples() will
        // throw.
        this->removeSampleWithoutTracking(m_inputTracker0Ptr->front(),
                                          m_inputTracker1Ptr->front());
        m_inputTracker0Ptr->pop_front();
        m_inputTracker1Ptr->pop_front();
        --count;
      }
    }

      
    // This member function removes a pair of sample values from the
    // normalized correlation calculation.
    template <class Type>
    void
    NormalizedCorrelator<Type>::
    removeSample(Type sample0, Type sample1)
    {
      if(this->isInputTrackingEnabled()) {
        DLR_THROW(NotImplementedException,
                  "NormalizedCorrelator::removeSample()",
                  "Currently, removeSample() can only be called if "
                  "input tracking is not enabled.");
      }
      this->removeSampleWithoutTracking(sample0, sample1);
    }

    
    // This member function works identically to removeSample(),
    // with the exception that input tracking is never updated.
    template <class Type>
    inline void
    NormalizedCorrelator<Type>::
    removeSampleWithoutTracking(Type sample0, Type sample1)
    {
      m_sum0 -= sample0;
      m_sum1 -= sample1;
      m_sum00 -= sample0 * sample0;
      m_sum01 -= sample0 * sample1;
      m_sum11 -= sample1 * sample1;
      --m_count;
    }        


    // This member function removes a sequence of pairs of samples
    // (each pair containing one sample from each of the two signals
    // to be correlated) from the normalized correlation
    // calculation.
    template <class Type>
    template <class IterType0, class IterType1>
    void
    NormalizedCorrelator<Type>::
    removeSamples(IterType0 begin0, IterType0 end0, IterType1 begin1)
    {
      if(this->isInputTrackingEnabled()) {
        DLR_THROW(NotImplementedException,
                  "NormalizedCorrelator::removeSamples()",
                  "Currently, removeSamples() can only be called if "
                  "input tracking is not enabled.");
      }

      while(begin0 != end0) {
        this->removeSampleWithoutTracking(*begin0, *begin1);
        ++begin0;
        ++begin1;
      }
    }


  } // namespace numeric

} // namespace dlr

#endif /* #ifndef DLR_NUMERIC_NORMALIZEDCORRELATOR_H */
