/**
***************************************************************************
* @file dlrNumeric/maxRecorder.h
*
* Header file declaring MaxRecorder class.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 880 $
* $Date: 2007-05-04 00:33:49 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef DLR_NUMERIC_MINRECORDER_H
#define DLR_NUMERIC_MINRECORDER_H

#include <limits>

namespace dlr {

  namespace numeric {
    
    /**
     ** A simple class template to help you avoid writing "if(myVar <
     ** minVal) {minVal = myVar; bestIndex = ii;} all over your code.  
     ** WARNING: This interface may change slightly in subsequent releases.
     **/
    template<class Type, class Payload>
    class MinRecorder {
    public:

      /** 
       * Default constructor initializes min to a very large number
       * (std::numeric_limits<Type>::max()), and sets the payload to
       * it's default value.
       */
      MinRecorder() : m_min(), m_payload() {this->reset();}


      /** 
       * This constructor allows the user to explicitly set the starting
       * minValue, and the default payload.
       */
      MinRecorder(Type const& minValue, Payload const& payload)
        : m_min(minValue), m_payload(payload) {}


      /** 
       * This member function compares its first argument with the
       * saved minimum value, and updates the saved minimum (and
       * payload) if the argument is larger.
       * 
       * @param value This argument will be compared to the saved
       * minimum, and copied into the saved minimum if doing so will
       * decrease the saved minimum.
       * 
       * @param payload This argument is copied into the saved payload
       * if and only if the saved minimum was updated.
       * 
       * @return The return value is true if the saved minimum was
       * updated, false otherwise.
       */
      bool
      test(const Type& value, const Payload& payload) {
        if(value < m_min) {
          m_min = value;
          m_payload = payload;
          return true;
        }
        return false;
      }


      /** 
       * This member function returns the saved minimum.
       * 
       * @return The return value is the largest value passed to
       * member function test() since construction, or since the last
       * call to member function reset().
       */
      const Type&
      getMin() {return m_min;}


      /** 
       * This member function returns the saved "payload".
       * 
       * @return The return value is the payload associated with the
       * largest value passed to member function test() since
       * construction, or since the last call to member function
       * reset().
       */
      const Payload&
      getPayload() {return m_payload;}


      /** 
       * This member function resets the saved minimum just as if
       * *this were freshly constructed with the default constructor.
       */
      void
      reset() {
        m_min = std::numeric_limits<Type>::max();
      }

    private:

      Type m_min;
      Payload m_payload;

    };
    
  } // namespace numeric

} // namespace dlr

#endif // #ifdef DLR_NUMERIC_MINRECORDER_H
