/**
***************************************************************************
* @file dlrNumeric/maxRecorder.h
*
* Header file declaring MaxRecorder class.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 880 $
* $Date: 2007-05-04 00:33:49 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef DLR_NUMERIC_MAXRECORDER_H
#define DLR_NUMERIC_MAXRECORDER_H

#include <limits>

namespace dlr {

  namespace numeric {
    
    /**
     ** A simple class template to help you avoid writing "if(myVar >
     ** maxVal) {maxVal = myVar; bestIndex = ii;} all over your code.  
     ** WARNING: This interface may change slightly in subsequent releases.
     **/
    template<class Type, class Payload>
    class MaxRecorder {
    public:

      /** 
       * Default constructor initializes max to a very small number
       * (either std::numeric_limits<Type>::min(),
       * -std::numeric_limits<Type>::max(), depending on the
       * signedness of Type), and sets the payload to it's default
       * value.
       */
      MaxRecorder() : m_max(), m_payload() {this->reset();}


      /** 
       * This constructor allows the user to explicitly set the starting
       * maxValue, and the default payload.
       */
      MaxRecorder(Type const& maxValue, Payload const& payload)
        : m_max(maxValue), m_payload(payload) {}


      /** 
       * This member function compares its first argument with the
       * saved maximum value, and updates the saved maximum (and
       * payload) if the argument is larger.
       * 
       * @param value This argument will be compared to the saved
       * maximum, and copied into the saved maximum if doing so will
       * increase the saved maximum.
       * 
       * @param payload This argument is copied into the saved payload
       * if and only if the saved maximum was updated.
       * 
       * @return The return value is true if the saved maximum was
       * updated, false otherwise.
       */
      bool
      test(const Type& value, const Payload& payload) {
        if(value > m_max) {
          m_max = value;
          m_payload = payload;
          return true;
        }
        return false;
      }


      /** 
       * This member function returns the saved maximum.
       * 
       * @return The return value is the largest value passed to
       * member function test() since construction, or since the last
       * call to member function reset().
       */
      const Type&
      getMax() {return m_max;}


      /** 
       * This member function returns the saved "payload".
       * 
       * @return The return value is the payload associated with the
       * largest value passed to member function test() since
       * construction, or since the last call to member function
       * reset().
       */
      const Payload&
      getPayload() {return m_payload;}


      /** 
       * This member function resets the saved maximum just as if
       * *this were freshly constructed with the default constructor.
       */
      void
      reset() {
        if(std::numeric_limits<Type>::is_signed) {
          m_max = -std::numeric_limits<Type>::max();
        } else {
          m_max = std::numeric_limits<Type>::min();
        }
      }

    private:

      Type m_max;
      Payload m_payload;

    };
    
  } // namespace numeric

} // namespace dlr

#endif // #ifdef DLR_NUMERIC_MAXRECORDER_H
