/**
***************************************************************************
* @file dlrNumeric/bilinearInterpolator.h
*
* Header file declaring BilinearInterpolator class.
*
* Copyright (C) 1999-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 996 $
* $Date: 2008-01-12 01:10:03 -0500 (Sat, 12 Jan 2008) $
***************************************************************************
**/

#ifndef DLR_NUMERIC_BILINEARINTERPOLATOR_H
#define DLR_NUMERIC_BILINEARINTERPOLATOR_H

#include <cmath>
#include <dlrNumeric/array2D.h>

namespace dlr {

  namespace numeric {
    
    template <class TYPE>
    class BilinearInterpolator
    {
    public:

      BilinearInterpolator(const Array2D<TYPE>& image0)
        : m_array(image0) {}


      ~BilinearInterpolator() {}

      
      inline double
      operator()(double row, double column);

    private:

      inline void
      checkBounds(double row, double column) const;

    
      Array2D<TYPE> m_array;
    
    };

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::BilinearInterpolator;

} // namespace dlr


namespace dlr {

  namespace numeric {
    
    template <class TYPE>
    inline double
    BilinearInterpolator<TYPE>::
    operator()(double row, double column)
    {
      this->checkBounds(row, column);
    
      double tmpDbl;
      double x1 = modf(column, &tmpDbl);
      size_t i0 = static_cast<size_t>(tmpDbl);
      double y1 = modf(row, &tmpDbl);
      size_t j0 = static_cast<size_t>(tmpDbl);
      double x0 = 1.0 - x1;
      double y0 = 1.0 - y1;
      size_t index0 = m_array.columns() * j0 + i0;
      return (x0 * (y0 * this->m_array(index0)
                    + y1 * this->m_array(index0 + m_array.columns()))
              + x1 * (y0 * this->m_array(index0 + 1)
                      + y1 * this->m_array(index0 + m_array.columns() + 1)));
    }


    template <class TYPE>
    inline void
    BilinearInterpolator<TYPE>::
    checkBounds(double row, double column) const
    {
#ifdef _DLRNUMERIC_CHECKBOUNDS_
      int row0 = static_cast<int>(row);
      if(row0 < 0 || (row0 + 1) >= static_cast<int>(m_array.rows())) {
        std::ostringstream message;
        message << "Row index " << row << " is invalid for a(n) "
                << m_array.rows() << " x " << m_array.columns() << " array.";
        DLR_THROW(IndexException, "BilinearInterpolator::checkBounds()",
                  message.str().c_str());
      }
      int column0 = static_cast<int>(column);
      if(column0 < 0 || (column0 + 1) >= static_cast<int>(m_array.columns())) {
        std::ostringstream message;
        message << "Column index " << column << " is invalid for a(n) "
                << m_array.rows() << " x " << m_array.columns() << " array.";
        DLR_THROW(IndexException, "BilinearInterpolator::checkBounds()",
                  message.str().c_str());
      }
#endif
    }

  } // namespace numeric

} // namespace dlr

#endif /* #ifndef DLR_NUMERIC_BILINEARINTERPOLATOR_H */
