/**
***************************************************************************
* @file dlrNumeric/amanatidesWoo2D.h
*
* Header file declaring AmanatidesWoo2D class.
*
* Copyright (C) 2004-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 975 $
* $Date: 2007-12-30 01:57:17 -0500 (Sun, 30 Dec 2007) $
***************************************************************************
**/

#ifndef _DLR_NUMERIC_AMANATIDESWOO2D_H_
#define _DLR_NUMERIC_AMANATIDESWOO2D_H_

#include <iostream>
#include <limits>
#include <dlrCommon/exception.h>
#include <dlrNumeric/vector2D.h>
#include <dlrNumeric/transform2D.h>
#include <dlrNumeric/utilities.h>
#include <dlrNumeric/amanatidesWoo2DIterator.h>

namespace dlr {

  namespace numeric {
    
    /**
     ** This class implements the Fast Voxel Traversal Algorithm of
     ** Amanatides and Woo [Ref] for 2D arrays.  The algorithm is for
     ** efficient ray tracing through regular grids.  For convenience,
     ** there are two coordinate systems associated with this class:
     **
     ** - The pixel coordinate system is an integer-based coordinate
     ** system associated with the array.
     **
     ** - The world coordinate system, which can differ from the pixel
     ** coordinate system only by translation, rotation, and
     ** non-isotropic scaling, but is in other respects arbitrary.
     **
     ** The pixel coordinate system has its origin at the corner of the
     ** 2D array, has its first coordinate (which we'll call U) parallel
     ** to the rows of the array, and has its second coordinate (which
     ** we'll call V) parallel to the columns of the array.  As you move
     ** along any row of the 2D array, the transition from the first
     ** column to the second occurs at U == 1.0, the transition from the
     ** second column to the third occurs at U == 2.0, and so on.
     ** Similarly as you move along any column of the array, the
     ** transition from the first row to the second occurs at V == 1.0,
     ** and the transition from the second row to the third occurs at V
     ** == 2.0.
     **
     ** Once constructed, you can use the AmanatidesWoo2D class to
     ** access pixels along a straight line through the 2D array using
     ** STL-style iteration.  That is, the member functions
     ** AmanatidesWoo2D::begin() returns an iterator which references
     ** subsequent pixels along the line, and will become equal (==) to
     ** the iterator returned by AmanatidesWoo2D::end() when the line
     ** passes out of the 2D array.  Here's an example usage, which will
     ** probably format terribly in the Doxygen-generated doc:
     **
     **   typedef AmanatidesWoo2D< Array2D<int> >::iterator awIterator;
     **
     **   AmanatidesWoo2D< Array2D<int> > rayTracer(
     **     myArray2D, rayOrigin, rayDirection, false);
     **
     **   awIterator iterator0 = rayTracer.begin(); 
     **
     **   awIterator iterator1 = rayTracer.end();
     **
     **   for(; iterator0 != iterator1; ++iterator0) {
     **
     **     *iterator0 = 0;
     **
     **   }
     **/
    template <class ARRAY2D>
    class AmanatidesWoo2D {
    public:
      /* ================== Public typedefs ================== */

      /**
       ** This typedef specifies the type which will be returned by
       ** member functions begin() and end().  AmanatidesWoo2DIterator
       ** is a forward iterator, and has unusual semantics for
       ** operator==().  Be sure to understand them before using this
       ** class.
       **/
      typedef AmanatidesWoo2DIterator<ARRAY2D> iterator;

      // Iteration over const arrays will be done using a different
      // class.
      // typedef AmanatidesWoo2DConstIterator<ARRAY2D> const_iterator;
    
      /* ================== Public methods ================== */

      /** 
       * This constructor specifies all of the internal state of the
       * AmanatidesWoo2D class.  After construction, the AmanatidesWoo2D
       * instance is ready to be used.
       * 
       * @param data This argument specifies the 2D array over which to
       * iterate.
       *
       * @param pixelTworld This parameter specifies a coordinate
       * transformation which takes world coordinates and converts them
       * into pixel coordinates.
       *     
       * @param rayOrigin This parameter specifies the starting point
       * of the ray to be traced, expressed in world coordinates.  If
       * you'd rather express rayOrigin and rayDirection in pixel
       * coordinates, simply set argument pixelTworld to the identity
       * transform.  Note that rayOrigin does not have to lie inside the
       * pixel array.
       *     
       * @param rayDirection This parameter specifies the direction of
       * the ray to be traced, expressed in world coordinates.  In other
       * words, any point on the line of interest can be written as
       * (rayOrigin + t * rayDirection) for some value of t.  If
       * you'd rather express rayOrigin and rayDirection in pixel
       * coordinates, simply set argument pixelTworld to the identity
       * transform.
       *     
       * @param downstreamOnly This boolean argument specifies whether
       * pixels "upstream" of rayOrigin are to be included in the pixel
       * traversal.  If downstreamOnly is set to false, pixels which
       * intersect the line (rayOrigin + t * rayDirection) will be
       * included in the iteration even if they intersect the line at
       * negative values of t.
       */
      AmanatidesWoo2D(ARRAY2D& data,
                      const Transform2D& pixelTworld,
                      const Vector2D& rayOrigin,
                      const Vector2D& rayDirection,
                      bool downstreamOnly=true);

    
      /** 
       * This is the copy constructor.  After copying, the new
       * AmanatidesWoo2D instance and the copied instance both reference
       * the same pixel array.
       * 
       * @param source The AmanatidesWoo2D instance to be copied.
       */
      AmanatidesWoo2D(const AmanatidesWoo2D& source);

      /** 
       * This is the destructor.  It destroys the AmanatidesWoo2D
       * instance and cleans up any allocated memory.
       */
      ~AmanatidesWoo2D();

      /** 
       * This member function returns an iterator which references the
       * first pixel along the traced ray.  If constructor argument
       * downstreamOnly was set to false, this iterator will either
       * point to a pixel on the very edge of the array, or (in the case
       * that the ray does not intersect the pixel array) be equal to
       * this->end().  If constructor argument downstreamOnly was set to
       * true, then this iterator will point to an edge pixel, or point
       * to an interior pixel (in the case that rayOrigin lies within
       * the boundaries of the pixel array), or be equal to this->end()
       * (in the case that the ray does not intersect the pixel array,
       * and in the case that rayOrigin lies outside the pixel array and
       * rayDirection points away from the pixel array).
       * 
       * @return The return value is an iterator pointing to the first
       * pixel in the array, or else an iterator for which (iter ==
       * this->end()) is true.
       */
      iterator
      begin();

      /** 
       * This member function returns an iterator which references an
       * invalid pixel, and which will be equal (==) to the iterator
       * returned by member function begin() when that iterator has
       * fully traversed the line of pixels.
       * 
       * @return The return value is an iterator pointing to an invalid
       * pixel, for use as the end iterator in the comparison clause of
       * a loop: "while(rayIterator != endIterator) {++rayIterator; ...;}".
       */
      iterator
      end();

      /** 
       * This member function returns a reference to the array object
       * over which iteration is performed.
       * 
       * @return The return value is a reference to the data array.
       */
      ARRAY2D&
      getData() {return m_data;}
    
      /** 
       * This member function returns true if the iterator returned by
       * member function begin() will point to a valid pixel.  In other
       * words, if constructor argument downstreamOnly was set to
       * false, the return value of validIntersection() indicates
       * whether the ray specified in the constructor actually
       * intersects the pixel array.  If constructor argument
       * downstreamOnly was set to true, the return value of
       * validIntersection whether the "downstream" half of the ray
       * actually intersects the pixel array.
       * 
       * @return A boolean indicating whether or not (this->begin() ==
       * this->end()) will be true.
       */
      inline bool
      validIntersection();

      /** 
       * The assignment operator copies its argument.  After copying,
       * *this and the copied instance both reference the same pixel
       * array.
       * 
       * @param source The AmanatidesWoo2D instance to be copied.
       * @return The return value is a reference to *this.
       */
      AmanatidesWoo2D&
      operator=(const AmanatidesWoo2D& source);
  
    private:

      /** 
       * This private member function computes the parameters tEntry and
       * tExit such that (rayOrigin + tEntry * rayDirection) is very
       * first point of intersection between the ray and the pixel
       * array, and (rayOrigin + tExit * rayDirection) is very last of
       * intersection between the ray and the pixel array.
       * 
       * @param rayOriginPixel This parameter specifies the starting
       * point of the ray to be traced, expressed in pixel coordinates.
       * @param rayDirectionPixel This parameters specifies the
       * direction of the ray to be traced, expressed in pixel
       * coordinates.
       * @param m_data This parameter is a reference the pixel array.
       * @return The return value is a std::pair<double, double> in
       * which the first value is tEntry and the second value is tExit.
       */
      std::pair<double, double>
      findEntryAndExitPoints(const Vector2D& rayOriginPixel,
                             const Vector2D& rayDirectionPixel,
                             const ARRAY2D& data);

      /** 
       * This function is not documented because it will change very
       * soon.
       * 
       * @param rayOrigin 
       * @param rayDirection 
       * @param bVector 
       * @param cConstant 
       * @param defaultValue 
       * @return 
       */
      double
      findIntersection(const Vector2D& rayOrigin,
                       const Vector2D& rayDirection,
                       const Vector2D& bVector,
                       double cConstant,
                       double defaultValue);

      /// This data member is a reference the array over which to
      /// iterate.
      ARRAY2D& m_data;

      /// This data member indicates the U coordinate of the first pixel
      /// along the path.
      int m_initialU;

      /// This data member indicates the V coordinate of the first pixel
      /// along the path.
      int m_initialV;

      /// This data member indicates whether the U coordinate will be
      /// increasing (m_stepU == 1) or decreasing (m_stepU == -1) as we
      /// travel along the ray path.
      int m_stepU;

      /// This data member indicates whether the V coordinate will be
      /// increasing (m_stepV == 1) or decreasing (m_stepV == -1) as we
      /// travel along the ray path.
      int m_stepV;
    
      /// This data member indicates what increment of ray parameter t
      /// moves us exactly 1 pixel in the U direction.
      double m_tDeltaU;

      /// This data member indicates what increment of ray parameter t
      /// moves us exactly 1 pixel in the V direction.
      double m_tDeltaV;
    
      /// This data member indicates the value of ray parameter t at
      /// which the ray first enters a pixel with U coordinate not equal
      /// to m_initialU.
      double m_tMaxU;

      /// This data member indicates the value of ray parameter t at
      /// which the ray first enters a pixel with V coordinate not equal
      /// to m_initialV.
      double m_tMaxV;

      /// This data member indicates the value of ray parameter t at
      /// which we start following the ray.  It frequently corresponds
      /// to the point at which the ray enters the pixelated space, but
      /// may also be set to zero if the ray origin is within the
      /// pixelated space and constructor argument downstreamOnly was
      /// set to true.
      double m_tStart;

      /// This data member indicates whether the portion of the ray we
      /// care about actually intersects the specified pixel array.
      bool m_validIntersection;
    };

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::AmanatidesWoo2D;

} // namespace dlr


/* ========================================================= */
/* Implementation follows                                    */
/* ========================================================= */

namespace dlr {

  namespace numeric {
    
    template <class ARRAY2D>
    AmanatidesWoo2D<ARRAY2D>::
    AmanatidesWoo2D(ARRAY2D& data,
                    const Transform2D& pixelTworld,
                    const Vector2D& rayOrigin,
                    const Vector2D& rayDirection,
                    bool downstreamOnly)
      : m_data(data),
        m_initialU(-1),  // Initialize to illegal values.  These will be 
        m_initialV(-1),  // replaced with legal ones if there's a valid
        // intersection between the ray and the pixel
        // array.
        m_stepU(),
        m_stepV(),
        m_tDeltaU(),
        m_tDeltaV(),
        m_tMaxU(),
        m_tMaxV(),
        m_tStart(),
        m_validIntersection(true)    
    {
      // First convert everything into pixel coordinates.
      Vector2D rayOriginPixel = pixelTworld * rayOrigin;
      Vector2D rayDirectionPixel =
        (pixelTworld * (rayOrigin + rayDirection)) - rayOriginPixel;

      // Now find points entry and exit from the CT volume.  These are
      // expressed in as parameter values tEntry and tExit such that
      // (rayOriginPixel + tEntry * rayDirectionPixel) is the entry
      // point and (rayOriginPixel + tExit * rayDirectionPixel) is the
      // exit point.
      std::pair<double, double> tEntry_tExit =
        this->findEntryAndExitPoints(rayOriginPixel, rayDirectionPixel, m_data);

      // Sometimes we want to disallow any pixels which are "behind" the
      // ray origin.  That is, sometimes we want to only trace those
      // parts of the ray which correspond to positive values of the
      // parameter t.
      if(downstreamOnly && (tEntry_tExit.first < 0.0)) {
        tEntry_tExit.first = 0.0;
      }

      // Sometimes the there's no intersection with the pixel array.  In
      // this case, tExit will be less than tEntry, and we don't bother
      // doing any more calculation.
      if(tEntry_tExit.second <= tEntry_tExit.first) {
        m_validIntersection = false;
        return;
      }

      // Now make the point of entry explicit.
      Vector2D entryPoint =
        rayOriginPixel + tEntry_tExit.first * rayDirectionPixel;

      // Correct for rounding error which sometimes makes entryPoint
      // have values like -1.0e-14.
      if(entryPoint.x() < 0.0) {
        entryPoint.x() = 0.0;
      }
      if(entryPoint.y() < 0.0) {
        entryPoint.y() = 0.0;
      }
    
      // Finally, assign the variables described in the Amanatides' and
      // Woo's paper.

      // Since we've already converted to pixel coords, finding the
      // first pixel on the path is trivial.
      m_initialU = static_cast<int>(entryPoint.x());
      m_initialV = static_cast<int>(entryPoint.y());

      // There's an similar case to the one handled by the rounding
      // error test above.  Member function findEntryAndExitPoints()
      // uses m_data.columns() and m_data.rows() to define the maximum U
      // and V coordinates, respectively.  This means that it's very
      // possible for m_initialU to be equal to m_data.columns(), which
      // will cause an out-of-bounds index. Correct for this now.
      if(m_initialU == static_cast<int>(m_data.columns())) {
        --m_initialU;
      }
      if(m_initialV == static_cast<int>(m_data.rows())) {
        --m_initialV;
      }

      // Sanity check.
      if((m_initialU >= static_cast<int>(m_data.columns()))
         || (m_initialV >= static_cast<int>(m_data.rows()))) {
        DLR_THROW3(LogicException,
                   "AmanatidesWoo2D::AmanatidesWoo2D(ARRAY2D&, ...)",
                   "Illegal value for m_initialU or m_initialV.");
      }
    
      // m_tStart is just the same as tEntry.
      m_tStart = tEntry_tExit.first;
    
      // The remaining variables depend on whether U & V will be
      // increasing or decreasing as we travel along the ray, so we need
      // if clauses.  Please see the declaration for documentation on
      // what each of these member variables means.
      if(rayDirectionPixel.x() > 0.0) {
        m_stepU = 1;
        m_tDeltaU = 1.0 / rayDirectionPixel.x();
        m_tMaxU = m_tStart + (((m_initialU + 1) - entryPoint.x())
                              / rayDirectionPixel.x());
      } else if(rayDirectionPixel.x() < 0.0) {
        m_stepU = -1;
        m_tDeltaU = -(1.0 / rayDirectionPixel.x());
        m_tMaxU = m_tStart + ((m_initialU - entryPoint.x())
                              / rayDirectionPixel.x());
      } else { // rayDirectionPixel.x() == 0.0;
        m_stepU = 0;
        m_tDeltaU = std::numeric_limits<double>::max();
        m_tMaxU = std::numeric_limits<double>::max();
      }
      if(rayDirectionPixel.y() > 0.0) {
        m_stepV = 1;
        m_tDeltaV = 1.0 / rayDirectionPixel.y();
        m_tMaxV = m_tStart + (((m_initialV + 1) - entryPoint.y())
                              / rayDirectionPixel.y());
      } else if(rayDirectionPixel.y() < 0.0) {
        m_stepV = -1;
        m_tDeltaV = -(1.0 / rayDirectionPixel.y());
        m_tMaxV = m_tStart + ((m_initialV - entryPoint.y())
                              / rayDirectionPixel.y());
      } else { // rayDirectionPixel.y() == 0.0;
        m_stepV = 0;
        m_tDeltaV = std::numeric_limits<double>::max();
        m_tMaxV = std::numeric_limits<double>::max();
      }
    }


    // The copy constructor deep copies its argument.
    template <class ARRAY2D>
    AmanatidesWoo2D<ARRAY2D>::
    AmanatidesWoo2D(const AmanatidesWoo2D& source)
      : m_data(source.m_data),
        m_initialU(source.m_initialU),
        m_initialV(source.m_initialV),
        m_stepU(source.m_stepU),
        m_stepV(source.m_stepV),
        m_tDeltaU(source.m_tDeltaU),
        m_tDeltaV(source.m_tDeltaV),
        m_tMaxU(source.m_tMaxU),
        m_tMaxV(source.m_tMaxV),
        m_tStart(source.m_tStart),
        m_validIntersection(source.m_validIntersection)
    {
      // Empty
    }

    template <class ARRAY2D>
    AmanatidesWoo2D<ARRAY2D>::
    ~AmanatidesWoo2D() {};

    template <class ARRAY2D>
    typename AmanatidesWoo2D<ARRAY2D>::iterator
    AmanatidesWoo2D<ARRAY2D>::
    begin()
    {
      return AmanatidesWoo2DIterator<ARRAY2D>(
        m_data, m_initialU, m_initialV, m_stepU, m_stepV, m_tMaxU, m_tMaxV,
        m_tDeltaU, m_tDeltaV, m_tStart);
    }

    template <class ARRAY2D>
    typename AmanatidesWoo2D<ARRAY2D>::iterator
    AmanatidesWoo2D<ARRAY2D>::
    end()
    {
      // AmanatidesWoo2DIterator<ARRAY2D>::operator==(...) considers all
      // iterators with illegal pixel coordinates to be equal, so all we
      // have to do here is return an iterator which references an
      // illegal pixel.
      return AmanatidesWoo2DIterator<ARRAY2D>(
        m_data, -1, -1, m_stepU, m_stepV, m_tMaxU, m_tMaxV,
        m_tDeltaU, m_tDeltaV, m_tStart);
    }

    template <class ARRAY2D>
    inline bool
    AmanatidesWoo2D<ARRAY2D>::
    validIntersection()
    {
      return m_validIntersection;
    }

    template <class ARRAY2D>
    AmanatidesWoo2D<ARRAY2D>&
    AmanatidesWoo2D<ARRAY2D>::
    operator=(const AmanatidesWoo2D& source)
    {
      m_data = source.m_data;
      m_initialU = source.m_initialU;
      m_initialV = source.m_initialV;
      m_stepU = source.m_stepU;
      m_stepV = source.m_stepV;
      m_tDeltaU = source.m_tDeltaU;
      m_tDeltaV = source.m_tDeltaV;
      m_tMaxU = source.m_tMaxU;
      m_tMaxV = source.m_tMaxV;
      m_tStart = source.m_tStart;
      m_validIntersection = source.m_validIntersection;
      return *this;
    }
    
    template <class ARRAY2D>
    std::pair<double, double>
    AmanatidesWoo2D<ARRAY2D>::
    findEntryAndExitPoints(const Vector2D& rayOriginPixel,
                           const Vector2D& rayDirectionPixel,
                           const ARRAY2D& data)
    {
      // First find intersection with each boundary line.

      // ... Find the intersection with the line U = 0, or else a really
      // small number if rayDirection is parallel to the rows of the
      // array.
      double tIntersectU0 = findIntersection(
        rayOriginPixel, rayDirectionPixel, Vector2D(1.0, 0.0), 0.0,
        std::numeric_limits<double>::min());

      // ... Find the intersection with the line U = data.columns(),
      // or else a really big number if rayDirection is parallel to the
      // rows of the array.
      double tIntersectU1 = findIntersection(
        rayOriginPixel, rayDirectionPixel, Vector2D(1.0, 0.0), data.columns(),
        std::numeric_limits<double>::max());

      // ... Find the intersection with the line V = 0, or else a really
      // small number if rayDirection is parallel to the columns of the
      // array.
      double tIntersectV0 = findIntersection(
        rayOriginPixel, rayDirectionPixel, Vector2D(0.0, 1.0), 0.0,
        std::numeric_limits<double>::min());
    
      // ... Find the intersection with the line V = data.rows(), or
      // else a really big number if rayDirection is parallel to the
      // columns of the array.
      double tIntersectV1 = findIntersection(
        rayOriginPixel, rayDirectionPixel, Vector2D(0.0, 1.0), data.rows(),
        std::numeric_limits<double>::max());

      // Now find the closer and farther of each pair of intersections.
      double tMinU = std::min(tIntersectU0, tIntersectU1);
      double tMinV = std::min(tIntersectV0, tIntersectV1);
      double tMaxU = std::max(tIntersectU0, tIntersectU1);
      double tMaxV = std::max(tIntersectV0, tIntersectV1);

      // Compute closest point which could possibly intersect with the volume.
      double tEntry = std::max(tMinU, tMinV);
      // Compute farthest point which could possibly intersect with the volume.
      double tExit = std::min(tMaxU, tMaxV);

      // Return our findings.
      return std::make_pair(tEntry, tExit);
    }

    template <class ARRAY2D>
    double
    AmanatidesWoo2D<ARRAY2D>::
    findIntersection(const Vector2D& rayOrigin,
                     const Vector2D& rayDirection,
                     const Vector2D& bVector,
                     double cConstant,
                     double defaultValue)
    {
      // bVector and cConstant describe the desired plane:
      //   dot(x, bVector) = cConstant.
      // Now, x is constrained to lie on the specified line:
      //   x = rayOrigin + t * rayDirection.
      // Substituting, we have:
      //   dot(rayOrigin + t * rayDirection, bVector) = cConstant.
      // Which we rewrite:
      //   dot(rayOrigin, bVector) + t * dot(rayDirection, bVector)
      //     = cConstant.
      // Solving for t, we have:
      //   t = (cConstant - dot(rayOrigin, bVector))
      //        / dot(rayDirection, bVector).
      double denominator = dot(rayDirection, bVector);
      if(denominator == 0.0) {
        return defaultValue;
      }
      // else
      return (cConstant - dot(rayOrigin, bVector)) / denominator;
    }      

  } // namespace numeric

} // namespace dlr

#endif // #ifndef _DLR_NUMERIC_AMANATIDESWOO2D_H_
