/**
***************************************************************************
* @file dlrNumeric/transform3DTo2D.h
*
* Header file declaring Transform3DTo2D class.
*
* Copyright (C) 2001-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 975 $
* $Date: 2007-12-30 01:57:17 -0500 (Sun, 30 Dec 2007) $
***************************************************************************
**/

#ifndef _DLR_TRANSFORM3DTO2D_H_
#define _DLR_TRANSFORM3DTO2D_H_

#include <dlrNumeric/array2D.h>
#include <dlrNumeric/transform3D.h>
#include <dlrNumeric/vector2D.h>
#include <dlrNumeric/vector3D.h>

namespace dlr {

  namespace numeric {
    
    // Forward declaration.
    class Transform3DTo2DFunctor;


    /**
     ** The Transform3DTo2D class represents a homogeneous coordinate
     ** transformation from a 3D coordinate system to a 2D coordinate
     ** system.
     **/
    class Transform3DTo2D {
    public:
      /** 
       * Default constructor
       */
      Transform3DTo2D()
        : m_00(1.0), m_01(0.0), m_02(0.0), m_03(0.0),
          m_10(0.0), m_11(1.0), m_12(0.0), m_13(0.0),
          m_20(0.0), m_21(0.0), m_22(1.0), m_23(1.0)
        {}

      /** 
       * Build a Transform3DTo2D instance by explicitly setting element 
       * values as if setting the elements of a 3x4 transformation matrix:
       *    [[a00, a01, a02, a03],
       *     [a10, a11, a12, a13],
       *     [a20, a21, a22, a23]]
       * 
       * @param a00 The value of one element of the transformation matrix.
       * @param a01 The value of one element of the transformation matrix.
       * @param a02 The value of one element of the transformation matrix.
       * @param a03 The value of one element of the transformation matrix.
       * @param a10 The value of one element of the transformation matrix.
       * @param a11 The value of one element of the transformation matrix.
       * @param a12 The value of one element of the transformation matrix.
       * @param a13 The value of one element of the transformation matrix.
       * @param a20 The value of one element of the transformation matrix.
       * @param a21 The value of one element of the transformation matrix.
       * @param a22 The value of one element of the transformation matrix.
       * @param a23 The value of one element of the transformation matrix.
       */
      inline
      Transform3DTo2D(double a00, double a01, double a02, double a03,
                      double a10, double a11, double a12, double a13,
                      double a20, double a21, double a22, double a23)
        : m_00(a00), m_01(a01), m_02(a02), m_03(a03),
          m_10(a10), m_11(a11), m_12(a12), m_13(a13),
          m_20(a20), m_21(a21), m_22(a22), m_23(a23) {
      }

      /** 
       * Build a Transform3D from a homogeneous 3x4 matrix.
       * 
       * @param source A 2D array containing the elements of the desired
       * homogeneous transformation.
       */
      Transform3DTo2D(const Array2D<double>& source);

      /** 
       * The copy constructor simply duplicates its argument.
       * 
       * @param src This is the Transform3DTo2D instance to be copied.
       */
      inline
      Transform3DTo2D(const Transform3DTo2D& src)
        : m_00(src.m_00), m_01(src.m_01), m_02(src.m_02), m_03(src.m_03),
          m_10(src.m_10), m_11(src.m_11), m_12(src.m_12), m_13(src.m_13),
          m_20(src.m_20), m_21(src.m_21), m_22(src.m_22), m_23(src.m_23) {
      }

      /** 
       * Destructor.
       */
      ~Transform3DTo2D() {}


      /** 
       * This member function returns a functor which makes it easier to
       * transform arrays of points using algorithms such as
       * std::transform().  For example:
       *
       * @code
       * std::transform(myPoints.begin(), myPoints.end(), myNewPoints.begin(),
       *                myTransform.getFunctor());
       * @endcode
       * 
       * @return The return value is a functor instance which will
       * transform points according to the current value of *this.  The
       * functor will contain a copy of *this, so that subsequent
       * changes to *this will not affect the functor.
       */
      Transform3DTo2DFunctor
      getFunctor() const;
    
    
      /** 
       * Change the Transform3DTo2D value by explicitly setting element values
       * as if setting the elements of a 3x4 transformation matrix:
       *    [[a00, a01, a02, a03],
       *     [a10, a11, a12, a13],
       *     [a20, a21, a22, a23]]
       * 
       * @param a00 The value of one element of the transformation matrix.
       * @param a01 The value of one element of the transformation matrix.
       * @param a02 The value of one element of the transformation matrix.
       * @param a03 The value of one element of the transformation matrix.
       * @param a10 The value of one element of the transformation matrix.
       * @param a11 The value of one element of the transformation matrix.
       * @param a12 The value of one element of the transformation matrix.
       * @param a13 The value of one element of the transformation matrix.
       * @param a20 The value of one element of the transformation matrix.
       * @param a21 The value of one element of the transformation matrix.
       * @param a22 The value of one element of the transformation matrix.
       * @param a23 The value of one element of the transformation matrix.
       */
      void
      setTransform(double a00, double a01, double a02, double a03,
                   double a10, double a11, double a12, double a13,
                   double a20, double a21, double a22, double a23);

      /** 
       * This member function returns one element from the matrix
       * representation of the coordinate transform by reference.
       * For example, calling myTransform3DTo2D.value<2, 3>() will 
       * return the element from the 2nd row, 3rd column of the matrix
       * representation of the coordinate transformation.
       * 
       * @return A reference to the requested element.
       */
      // John Culbertson argues convincingly that this breaks the
      // abstraction, so we remove it for now.
      // 
      // template <size_t row, size_t column>
      // double&
      // value();

      /** 
       * This member function returns one element from the matrix
       * representation of the coordinate transform by value.  For
       * example, calling myTransform3DTo2D.value<2, 3>() will return
       * the element from the 2nd row, 3rd column of the matrix
       * representation of the coordinate transformation.
       * 
       * @return The value of the requested element.
       */
      template <size_t row, size_t column>
      double
      value() const;

      /** 
       * This operator returns one element from the matrix
       * representation of the coordinate transform by reference.
       * If blindingly fast execution is important, consider using
       * value<size_t, size_t>() member function.
       *
       * @param row The row of the requested element.
       * @param column The column of the requested element.
       * @return A reference to the requested element.
       */
      // John Culbertson argues convincingly that this breaks the
      // abstraction, so we remove it for now.
      // 
      // double&
      // operator()(size_t row, size_t column);

      /** 
       * This operator returns one element from the matrix
       * representation of the coordinate transform by value.
       * If blindingly fast execution is important, consider using
       * value<size_t, size_t>() member function.
       *
       * @param row The row of the requested element.
       * @param column The column of the requested element.
       * @return The value of the requested element.
       */
      double
      operator()(size_t row, size_t column) const;

      /** 
       * Applies the coordinate transformation to a Vector3D point and
       * returns the result. 
       * 
       * @param vector0 The point to be transformed.
       * @return The result of transforming vector0.
       */
      Vector2D
      operator*(const Vector3D& vector0) const;

      /** 
       * The assignment operator simply duplicates its argument.
       * 
       * @param source This is the Transform3D instance to be copied.
       * @return A reference to *this.
       */
      Transform3DTo2D&
      operator=(const Transform3DTo2D& source);

    private:
      void normalize(double scaleFactor);

      double m_00, m_01, m_02, m_03;
      double m_10, m_11, m_12, m_13;
      double m_20, m_21, m_22, m_23;

    }; // class Transform3DTo2D


    /* ============== Helper classes ============== */


    /**
     ** This helper class works with Transform3DTo2D::getFunctor()
     **/
    class Transform3DTo2DFunctor
      : public std::unary_function<Vector3D, Vector2D> {

    public:

      /** 
       * The constructor deep-copies its argument.
       * 
       *  @param transform This is the transform instance to be copied.
       */
      Transform3DTo2DFunctor(const Transform3DTo2D& transform)
        : m_transform(transform) {}

    
      /** 
       * The application operator transforms its argument.
       * 
       *  @param vec This is the vector to be transformed.
       * 
       *  @return The transformed vector.
       */
      inline Vector2D
      operator()(const Vector3D& vec) const {return m_transform * vec;}
    
    private:
      Transform3DTo2D m_transform;
    };
  

    /* ================ Non member functions below ================ */

    /** 
     * This operator composes a Transform3DTo2D instance with a Transform3D
     * instance.  The resulting Transform3DTo2D instance the equation:
     *
     *  (transform0 * transform1) * v0 = transform0 * (transform1 * v0),
     *
     * where v0 is a Vector3D instance.
     * 
     * @param transform0 This is the Transform3DTo2D instance to be
     * composed.
     *
     * @param transform1 This is the Transform3D instance to be composed.
     *
     * @return A Transform3DTo2D instance which is equivalent to the 
     * composition of the two arguments.
     */
    Transform3DTo2D
    operator*(const Transform3DTo2D& transform0, const Transform3D& transform1);

  
  } // namespace numeric

}; // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::Transform3DTo2D;
  using numeric::Transform3DTo2DFunctor;

} // namespace dlr


/*******************************************************************
 * Member function definitions follow.  This would be a .C file
 * if it weren't templated.
 *******************************************************************/

namespace dlr {

  namespace numeric {
    
    // This operator returns one element from the matrix
    // representation of the coordinate transform by reference.
    // The case statements should optimize away, since row and column
    // are both known at compile time.
//   template <size_t ROW, size_t COLUMN>
//   double&
//   Transform3D::
//   value()
//   {
//     switch(ROW) {
//     case 0:
//       switch(COLUMN) {
//       case 0: return m_00; break;
//       case 1: return m_01; break;
//       case 2: return m_02; break;
//       case 3: return m_03; break;
//       default: break;
//       }
//       break;
//     case 1:
//       switch(COLUMN) {
//       case 0: return m_10; break;
//       case 1: return m_11; break;
//       case 2: return m_12; break;
//       case 3: return m_13; break;
//       default: break;
//       }
//       break;
//     case 2:
//       switch(COLUMN) {
//       case 0: return m_20; break;
//       case 1: return m_21; break;
//       case 2: return m_22; break;
//       case 3: return m_23; break;
//       default: break;
//       }
//       break;
//     case 3:
//       switch(COLUMN) {
//       case 0: return m_30; break;
//       case 1: return m_31; break;
//       case 2: return m_32; break;
//       default: break;
//       }
//       break;
//     default:
//       break;
//     }
//     std::ostringstream message;
//     message << "Index (" << ROW << ", " << COLUMN << ") out of bounds.";
//     DLR_THROW(IndexException, message.str().c_str());
//     return m_00; // Dummy return to keep the compiler happy.
//   }

    // This operator returns one element from the matrix
    // representation of the coordinate transform by value.
    // The case statements should optimize away, since row and column
    // are both known at compile time.
    template <size_t ROW, size_t COLUMN>
    double
    Transform3DTo2D::
    value() const
    {
      // // Avoid ugly duplication of code using ugly const_cast.
      // return const_cast<Transform3D*>(this)->value<ROW, COLUMN>();
      switch(ROW) {
      case 0:
        switch(COLUMN) {
        case 0: return m_00; break;
        case 1: return m_01; break;
        case 2: return m_02; break;
        case 3: return m_03; break;
        default: break;
        }
        break;
      case 1:
        switch(COLUMN) {
        case 0: return m_10; break;
        case 1: return m_11; break;
        case 2: return m_12; break;
        case 3: return m_13; break;
        default: break;
        }
        break;
      case 2:
        switch(COLUMN) {
        case 0: return m_20; break;
        case 1: return m_21; break;
        case 2: return m_22; break;
        case 3: return m_23; break;
        default: break;
        }
        break;
      default:
        break;
      }
      std::ostringstream message;
      message << "Indices (" << ROW << ", " << COLUMN << ") are out of bounds.";
      DLR_THROW(IndexException, "Transform3DTo2D::value<size_t, size_t>()",
                message.str().c_str());
      return 0.0; // Dummy return to keep the compiler happy.
    }

  } // namespace numeric

} // namespace dlr

#endif // #ifndef _DLR_TRANSFORM3DTO2D_H_
