/**
***************************************************************************
* @file dlrNumeric/vector2D.h
*
* Header file declaring Vector2D class.
*
* Copyright (C) 2001-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 880 $
* $Date: 2007-05-04 00:33:49 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef _DLR_VECTOR2D_H_
#define _DLR_VECTOR2D_H_

#include <iostream>
#include <dlrCommon/exception.h>

namespace dlr {

  namespace numeric {
    
    /**
     ** The Vector2D class represents a real valued 2D vector.
     **/
    class Vector2D {
    public:

      /** 
       * The default constructor initializes to (0, 0).
       */
      Vector2D()
        : m_x(0), m_y(0) {}
    

      /** 
       * This constructor explicitly sets the 2D coordinates.
       * 
       * @param x The first component of the Vector2D.
       * 
       * @param y The second component of the Vector2D.
       */
      Vector2D(double x, double y)
        : m_x(x), m_y(y) {}


      /** 
       * This constructor explicitly sets 2D homogeneous coordinates.
       * The equivalent 2D coordinates are (x/alpha, y/alpha).
       * 
       * @param x The first component of the Vector2D.
       * 
       * @param y The second component of the Vector2D.
       * 
       * @param alpha The projective scale parameter.
       */
      Vector2D(double x, double y, double alpha)
        : m_x(x), m_y(y) {this->normalize(alpha);}


      /** 
       * The copy constructor deep copies its argument.
       * 
       * @param other This argument is the Vector2D instance to be
       * copied.
       */
      Vector2D(const Vector2D& vec)
        : m_x(vec.m_x), m_y(vec.m_y) {}


      /** 
       * The destructor destroys the Vector2D instance.
       */
      ~Vector2D() {}


      /** 
       * This member function explicitly sets the sets coordinates of
       * the Vector2D instance.
       * 
       * @param x The first component of the Vector2D.
       * 
       * @param y The second component of the Vector2D.
       * 
       */
      inline void setValue(double x, double y) {
        m_x = x; m_y = y;
      }


      /** 
       * This constructor explicitly sets 2D homogeneous coordinates.
       * The equivalent 2D coordinates are (x/alpha, y/alpha).
       * 
       * @param x The first component of the Vector2D.
       * 
       * @param y The second component of the Vector2D.
       * 
       * @param alpha The projective scale parameter.
       */
      inline void setValue(double x, double y, double alpha) {
        m_x = x; m_y = y; normalize(alpha);
      }


      /** 
       * This member function returns the X component of the Vector2D
       * by reference.
       * 
       * @return The return value is a reference to the X coordinate.
       */
      inline double& x() {return m_x;}


      /** 
       * This member function returns the X component of the Vector2D
       * by value.
       * 
       * @return The return value the X coordinate.
       */
      inline double x() const {return m_x;}

    
      /** 
       * This member function returns the Y component of the Vector2D
       * by value.
       * 
       * @return The return value the Y coordinate.
       */
      inline double& y() {return m_y;}

    
      /** 
       * This member function returns the Y component of the Vector2D
       * by value.
       * 
       * @return The return value the Y coordinate.
       */
      inline double y() const {return m_y;}
    

      /** 
       * The assignment operator deep copies its argument.
       * 
       * @param other This argument is the Vector2D instance to be
       * copied.
       *
       * @return The return value is a reference to *this after the
       * operation has been performed.
       */
      Vector2D& operator=(const Vector2D& vec) {
        setValue(vec.m_x, vec.m_y); return *this; // Self-assignment OK.
      }


      /** 
       * This operator multiplies each component of the Vector2D instance
       * by a scalar.
       * 
       * @param scalar This argument is the scalar by which to multiply.
       * 
       * @return The return value is a reference to *this after the
       * operation has been performed.
       */
      Vector2D& operator*=(double scalar) {
        m_x *= scalar; m_y *= scalar; return *this;
      }


      /** 
       * This operator divides each component of the Vector2D instance
       * by a scalar.
       * 
       * @param scalar This argument is the scalar by which to divide.
       * 
       * @return The return value is a reference to *this after the
       * operation has been performed.
       */
      Vector2D& operator/=(double scalar) {
        if (scalar == 0 ) {
          DLR_THROW(ValueException, "Vector2D::operator/=()",
                    "Bad scalar: Divide by Zero\n");
        }
        m_x /= scalar; m_y /= scalar; return *this;
      }


      /** 
       * This operator adds a scalar to each component of the Vector2D
       * instance.
       * 
       * @param scalar This argument is the scalar to be added.
       * 
       * @return The return value is a reference to *this after the
       * operation has been performed.
       */
      Vector2D& operator+=(const Vector2D& vec) {
        m_x += vec.m_x; m_y += vec.m_y; return *this;
      }


      /** 
       * This operator subtracts a scalar from each component of the
       * Vector2D instance.
       * 
       * @param scalar This argument is the scalar to be subtracted.
       * 
       * @return The return value is a reference to *this after the
       * operation has been performed.
       */
      Vector2D& operator-=(const Vector2D& vec) {
        m_x -= vec.m_x; m_y -= vec.m_y; return *this;
      }


      /** 
       * This operator returns a Vector2D equal to *this, but with each
       * element negated.
       * 
       * @return The return value is a negated copy of *this.
       */
      Vector2D operator-() {
        return Vector2D(-m_x, -m_y);
      }
    
    private:
      // Private member functions.
      inline void normalize(double alpha) {
        if(alpha == 1.0) {return;}
        if(alpha == 0.0) {
          DLR_THROW(ValueException, "Vector2D::normalize()",
                    "Bad alpha (0.0).");
        }
        m_x /= alpha; m_y /= alpha;
        return;
      }

      // Private data members.
      double m_x;
      double m_y;
    }; // class Vector2D


    /* ============== Non-member function declarations ============== */
  
    /** 
     * This operator returns the elementwise sum of two Vector2D instances.
     * 
     * @param vector0 This is the first of the two Vector2D instances to
     * be added.
     * @param vector1 This is the second of the two Vector2D instances to
     * be added.
     * @return A Vector2D instance in which the value of each element is
     * equal to the sum of the corresponding elements of the two arguments.
     */
    Vector2D
    operator+(const Vector2D& vector0, const Vector2D& vector1);

    /** 
     * This operator returns the elementwise difference of two Vector2D
     * instances.
     * 
     * @param vector0 This is the first of the two Vector2D instances to
     * be subtracted.
     * @param vector1 This is the second of the two Vector2D instances to
     * be subtracted.
     * @return A Vector2D instance in which the value of each element is
     * equal to the difference of the corresponding elements of the two
     * arguments.
     */
    Vector2D
    operator-(const Vector2D& vector0, const Vector2D& vector1);
  
    /** 
     * This operator returns the elementwise product of two Vector2D instances.
     * 
     * @param vector0 This is the first of the two Vector2D instances to
     * be multiplied.
     * @param vector1 This is the second of the two Vector2D instances to
     * be multiplied.
     * @return A Vector2D instance in which the value of each element is
     * equal to the product of the corresponding elements of the two arguments.
     */
    Vector2D
    operator*(const Vector2D& vector0, const Vector2D& vector1);

    /** 
     * This operator returns the elementwise dividend of two Vector2D instances.
     * 
     * @param vector0 This is the Vector2D instance whose element values
     * are to be divided.
     * @param vector1 This is the Vector2D instance by whose elements
     * the first argument is to be divided.
     * @return A Vector2D instance in which the value of each element is
     * equal to the corresponding value of the first argument divided by
     * the corresponding value of the second argument.
     */
    Vector2D
    operator/(const Vector2D& vector0, const Vector2D& vector1);

    /** 
     * This operator adds a scalar and a Vector2D.
     * 
     * @param vector0 This is the Vector2D instance to which the scalar
     * should be added.
     * @param scalar0 This is amount which should be added to each
     * element of argument vector0.
     * @return A Vector2D instance in which the value of each element is
     * equal to the corresponding value of the first argument plus the
     * value of the second argument.
     */
    Vector2D operator+(const Vector2D& vector0, double scalar0);

    /** 
     * This operator subtracts a scalar from a Vector2D.
     * 
     * @param vector0 This is the Vector2D instance from which the scalar
     * should be subtracted.
     * @param scalar0 This is amount which should be subtracted from each
     * element of argument vector0.
     * @return A Vector2D instance in which the value of each element is
     * equal to the corresponding value of the first argument minus the
     * value of the second argument.
     */
    Vector2D operator-(const Vector2D& vector0, double scalar0);

    /** 
     * This operator multiplies a Vector2D by scalar.
     * 
     * @param vector0 This is the Vector2D instance which is to be
     * multiplied by the scalar.
     * @param scalar0 This is amount by which should argument vector0 is
     * to be multiplied.
     * @return A Vector2D instance in which the value of each element is
     * equal to the corresponding value of the first argument multiplied by
     * the value of the second argument.
     */
    Vector2D operator*(const Vector2D& vector0, double scalar0);

    /** 
     * This operator divides a Vector2D by scalar.
     * 
     * @param vector0 This is the Vector2D instance which is to be
     * divided by the scalar.
     * @param scalar0 This is amount by which should argument vector0 is
     * to be divided.
     * @return A Vector2D instance in which the value of each element is
     * equal to the corresponding value of the first argument divided by
     * the value of the second argument.
     */
    Vector2D operator/(const Vector2D& vector0, double scalar0);

    /**
     * This operator checks the supplied vectors for equality.
     *
     * @param  vector0  First vector to compare.
     * @param  vector1  Second vector to compare.
     * @return  Result of comparing @p vector0 to @p vector1 for equality.
     */
    bool operator==(const Vector2D& vector0, const Vector2D& vector1);

    /**
     * This operator checks the supplied vectors for inequality.
     *
     * @param  vector0  First vector to compare.
     * @param  vector1  Second vector to compare.
     * @return  Result of comparing @p vector0 to @p vector1 for
     *          inequality.
     */
    bool operator!=(const Vector2D& vector0, const Vector2D& vector1);


    /** 
     * This operator adds a scalar value to each element of a Vector2D
     * instance.
     * 
     * @param scalar0 Scalar argument of the addition.
     *
     * @param vector0 Vector argument of the addition.
     *
     * @return Vector2D instance in which the value of each element is
     * the sum of the scalar argument and the corresponding element of
     * the Vector2D argument.
     */
    Vector2D operator+(double scalar0, const Vector2D& vector0);


    /** 
     * This operator multiplies a scalar value with each element of a
     * Vector2D instance.
     * 
     * @param scalar0 Scalar argument of the multiplication.
     *
     * @param vector0 Vector argument of the multiplication.
     *
     * @return Vector2D instance in which the value of each element is
     * the product of the scalar argument and the corresponding element
     * of the Vector2D argument.
     */
    Vector2D operator*(double scalar0, const Vector2D& vector0);

  
    /** 
     * This function outputs a text representation of a Vector2D
     * instance to a std::ostream.  The output format looks like this:
     *
     * Vector2D(1.0, 2.0)
     *
     * @param stream This argument is a reference to the the output
     * stream.
     *
     * @param vector0 This argument is a const reference to the
     * Vector2D instance to be output.
     *
     * @return The return value is a reference to the input stream after
     * the write has taken place.
     */
    std::ostream& operator<<(std::ostream& stream, const Vector2D& vector0);

    /** 
     * This function sets the value of a Vector2D instance from a
     * std::istream.  The input format is as described for
     * operator<<(std::ostream&, const Vector2D&) above.
     * 
     * @param stream This argument is a reference to the the input
     * stream from which to read.
     *
     * @param vector0 This argument is a reference to the Vector2D
     * which will take the input.
     *
     * @return The return value is a reference to the input stream after
     * the read has taken place.
     */
    std::istream& operator>>(std::istream& stream, Vector2D& vector0);

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::Vector2D;

} // namespace dlr


/* ============ Definitions of inline & template functions ============ */

namespace dlr {

  namespace numeric {
    
    inline Vector2D operator+(double scalar0, const Vector2D& vector0)
    {
      return vector0 + scalar0;
    }
  
    inline Vector2D operator*(double scalar0, const Vector2D& vector0)
    {
      return vector0 * scalar0;
    }

  } // namespace numeric

} // namespace dlr

#endif // #ifndef _DLR_VECTOR2D_H_
