/**
***************************************************************************
* @file utilitiesTemplateTest.cpp
*
* Source file defining UtilitiesTemplateTest class.
*
* Copyright (C) 2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 930 $
* $Date: 2007-05-30 13:48:23 -0400 (Wed, 30 May 2007) $
***************************************************************************
**/

#include <dlrCommon/functional.h>
#include <dlrNumeric/utilities.h>
#include <dlrTest/testFixture.h>

namespace dlr {

  template <class Type>
  class UtilitiesTemplateTest
    : public TestFixture< UtilitiesTemplateTest<Type> >
  {

  public:

    typedef UtilitiesTemplateTest<Type> TestFixtureType;
    

    UtilitiesTemplateTest(const std::string& typeName);
    ~UtilitiesTemplateTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    void testAbs0();
    void testAbs1();
    void testAllTrueEtc();
    void testArgmax0();
    void testArgmax1();
    void testArgmin0();
    void testArgmin1();
    void testArgsort0();
    void testAxisMax();
    void testAxisMaximum0();
    void testAxisMaximum1();
    void testAxisMin();
    void testAxisMinimum0();
    void testAxisMinimum1();
    void testAxisSum0();
    void testAxisSum1();
    void testAxisSum2();
    void testColumnIndices();
    void testCompress0();
    void testCompress1();
    void testCount();
    void testSum0();

  private:

    template <class Type2>
    bool equivalent(const Array1D<Type2>& arg0,
                    const Array1D<Type2>& arg1,
                    Type2 tolerance);

    template <class Type2>
    bool equivalent(const Array2D<Type2>& arg0, const Array2D<Type2>& arg1,
                    Type2 tolerance);

    double m_defaultTolerance;
    
  }; // class UtilitiesTemplateTest


  /* ============== Member Function Definititions ============== */

  template <class Type>
  UtilitiesTemplateTest<Type>::
  UtilitiesTemplateTest(const std::string& typeName)
    : TestFixture<UtilitiesTemplateTest>(
        std::string("UtilitiesTemplateTest<" + typeName + ">")),
      m_defaultTolerance(1.0E-6)
  {
    // Register all tests.
    DLR_TEST_REGISTER_MEMBER(testAbs0);
    DLR_TEST_REGISTER_MEMBER(testAbs1);
    DLR_TEST_REGISTER_MEMBER(testAllTrueEtc);
    DLR_TEST_REGISTER_MEMBER(testArgmax0);
    DLR_TEST_REGISTER_MEMBER(testArgmax1);
    DLR_TEST_REGISTER_MEMBER(testArgmin0);
    DLR_TEST_REGISTER_MEMBER(testArgmin1);
    DLR_TEST_REGISTER_MEMBER(testArgsort0);
    DLR_TEST_REGISTER_MEMBER(testAxisMax);
    DLR_TEST_REGISTER_MEMBER(testAxisMaximum0);
    DLR_TEST_REGISTER_MEMBER(testAxisMaximum1);
    DLR_TEST_REGISTER_MEMBER(testAxisMin);
    DLR_TEST_REGISTER_MEMBER(testAxisMinimum0);
    DLR_TEST_REGISTER_MEMBER(testAxisMinimum1);
    DLR_TEST_REGISTER_MEMBER(testAxisSum0);
    DLR_TEST_REGISTER_MEMBER(testAxisSum1);
    DLR_TEST_REGISTER_MEMBER(testAxisSum2);
    DLR_TEST_REGISTER_MEMBER(testColumnIndices);
    DLR_TEST_REGISTER_MEMBER(testCompress0);
    DLR_TEST_REGISTER_MEMBER(testCompress1);
    DLR_TEST_REGISTER_MEMBER(testCount);
    DLR_TEST_REGISTER_MEMBER(testSum0);
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAbs0()
  {
    Array1D<Type> testArray0("[-4, -3, -2, -1, 0, 1, 2, 3]");
    Array1D<Type> testArray1("[4, 3, 2, 1, 0, 1, 2, 3]");
    Array1D<Type> testArray2 = abs(testArray0);
    DLR_TEST_ASSERT(
      this->equivalent(
        testArray1, testArray2, static_cast<Type>(m_defaultTolerance)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAbs1()
  {
    Array2D<Type> testArray0("[[-4, -3, -2],"
                             " [-1, 0, 1]]");
    Array2D<Type> testArray1("[[4, 3, 2],"
                             " [1, 0, 1]]");
    Array2D<Type> testArray2 = abs(testArray0);
    DLR_TEST_ASSERT(
      this->equivalent(
        testArray1, testArray2, static_cast<Type>(m_defaultTolerance)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAllTrueEtc()
  {
    Array1D<Type> testArray0("[-2, -1, -10, 5, 0, 3, 1]");
    Array1D<Type> testArray1("[-2, -1, -10, 5, -1, 3, 1]");
    Array1D<Type> testArray2("[0, 0, 1, 0, 0, 0, 0, 0]");
    Array1D<Type> testArray3("[0, 0, 0, 0, 0, 0, 0, 0]");

    DLR_TEST_ASSERT(!allTrue(testArray0));
    DLR_TEST_ASSERT(!allFalse(testArray0));
    DLR_TEST_ASSERT(anyTrue(testArray0));
    DLR_TEST_ASSERT(anyFalse(testArray0));

    DLR_TEST_ASSERT(allTrue(testArray1));
    DLR_TEST_ASSERT(!allFalse(testArray1));
    DLR_TEST_ASSERT(anyTrue(testArray1));
    DLR_TEST_ASSERT(!anyFalse(testArray1));
    
    DLR_TEST_ASSERT(!allTrue(testArray2));
    DLR_TEST_ASSERT(!allFalse(testArray2));
    DLR_TEST_ASSERT(anyTrue(testArray2));
    DLR_TEST_ASSERT(anyFalse(testArray2));

    DLR_TEST_ASSERT(!allTrue(testArray3));
    DLR_TEST_ASSERT(allFalse(testArray3));
    DLR_TEST_ASSERT(!anyTrue(testArray3));
    DLR_TEST_ASSERT(anyFalse(testArray3));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testArgmax0()
  {
    Array1D<Type> testArray0("[-2, -1, -10, 5, 2, 3, 0]");
    DLR_TEST_ASSERT(argmax(testArray0) == 3);
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testArgmax1()
  {
    Array1D<Type> testArray0("[-2, -1, -10, 5, 2, 3, 0]");
    DLR_TEST_ASSERT(argmax(testArray0, std::greater<Type>()) == 2);
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testArgmin0()
  {
    Array1D<Type> testArray0("[-2, -1, -10, 5, 2, 3, 0]");
    DLR_TEST_ASSERT(argmin(testArray0, std::greater<Type>()) == 3);
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testArgmin1()
  {
    Array1D<Type> testArray0("[-2, -1, -10, 5, 2, 3, 0]");
    DLR_TEST_ASSERT(argmin(testArray0, std::greater<Type>()) == 3);
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testArgsort0()
  {
    Array1D<Type> testArray0("[-4, 3, -2, 10, 6, 5, 7, 1]");
    Array1D<size_t> referenceArray0("[0, 2, 7, 1, 5, 4, 6, 3]");
    Array1D<size_t> resultArray0 = argsort(testArray0);
    DLR_TEST_ASSERT(
      this->equivalent(resultArray0, referenceArray0, size_t(0)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisMax()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    Array1D<Type> maxOverRows("[10, 2, 3]");
    Array1D<Type> maxOverColumns("[-1, 5, 10, 2]");
    DLR_TEST_ASSERT(
      this->equivalent(axisMax(testArray0, 0), maxOverRows,
                       static_cast<Type>(m_defaultTolerance)));
    DLR_TEST_ASSERT(
      this->equivalent(axisMax(testArray0, 1), maxOverColumns,
                       static_cast<Type>(m_defaultTolerance)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisMaximum0()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    Array1D<Type> maxOverRows("[10, 2, 3]");
    Array1D<Type> maxOverColumns("[-1, 5, 10, 2]");
    DLR_TEST_ASSERT(
      this->equivalent(axisMaximum(testArray0, 0), maxOverRows,
                       static_cast<Type>(m_defaultTolerance)));
    DLR_TEST_ASSERT(
      this->equivalent(axisMaximum(testArray0, 1), maxOverColumns,
                       static_cast<Type>(m_defaultTolerance)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisMaximum1()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    Array1D<Type> minOverRows("[-2, -1, -10]");
    Array1D<Type> minOverColumns("[-10, 2, -2, 1]");
    DLR_TEST_ASSERT(
      this->equivalent(axisMaximum(testArray0, 0, std::greater<Type>()),
                       minOverRows, static_cast<Type>(m_defaultTolerance)));
    DLR_TEST_ASSERT(
      this->equivalent(axisMaximum(testArray0, 1, std::greater<Type>()),
                       minOverColumns, static_cast<Type>(m_defaultTolerance)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisMin()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    Array1D<Type> minOverRows("[-2, -1, -10]");
    Array1D<Type> minOverColumns("[-10, 2, -2, 1]");
    DLR_TEST_ASSERT(
      this->equivalent(axisMin(testArray0, 0), minOverRows,
                       static_cast<Type>(m_defaultTolerance)));
    DLR_TEST_ASSERT(
      this->equivalent(axisMin(testArray0, 1), minOverColumns,
                       static_cast<Type>(m_defaultTolerance)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisMinimum0()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    Array1D<Type> minOverRows("[-2, -1, -10]");
    Array1D<Type> minOverColumns("[-10, 2, -2, 1]");
    DLR_TEST_ASSERT(
      this->equivalent(axisMinimum(testArray0, 0), minOverRows,
                       static_cast<Type>(m_defaultTolerance)));
    DLR_TEST_ASSERT(
      this->equivalent(axisMinimum(testArray0, 1), minOverColumns,
                       static_cast<Type>(m_defaultTolerance)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisMinimum1()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    Array1D<Type> maxOverRows("[10, 2, 3]");
    Array1D<Type> maxOverColumns("[-1, 5, 10, 2]");
    DLR_TEST_ASSERT(
      this->equivalent(axisMinimum(testArray0, 0, std::greater<Type>()),
                       maxOverRows, static_cast<Type>(m_defaultTolerance)));
    DLR_TEST_ASSERT(
      this->equivalent(axisMinimum(testArray0, 1, std::greater<Type>()),
                       maxOverColumns, static_cast<Type>(m_defaultTolerance)));
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisSum0()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    Array1D<typename NumericTraits<Type>::SumType> sumOverRows(
      "[14, 3, -7]");
    Array1D<typename NumericTraits<Type>::SumType> sumOverColumns(
      "[-13, 10, 8, 5]");
    Array1D<typename NumericTraits<Type>::SumType> computedSum0 =
      axisSum(testArray0, 0);
    Array1D<typename NumericTraits<Type>::SumType> computedSum1 =
      axisSum(testArray0, 1);
    
    DLR_TEST_ASSERT(
      this->equivalent(computedSum0, sumOverRows,
                       static_cast<typename NumericTraits<Type>::SumType>(
                         m_defaultTolerance)));
    DLR_TEST_ASSERT(
      this->equivalent(computedSum1, sumOverColumns,
                       static_cast<typename NumericTraits<Type>::SumType>(
                         m_defaultTolerance)));
  }
    

  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisSum1()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    Array1D<double> sumOverRows("[14, 3, -7]");
    Array1D<double> sumOverColumns("[-13, 10, 8, 5]");
    Array1D<double> computedSum0 = axisSum(testArray0, 0, Double);
    Array1D<double> computedSum1 = axisSum(testArray0, 1, Double);
    
    DLR_TEST_ASSERT(
      this->equivalent(computedSum0, sumOverRows,
                       static_cast<double>(m_defaultTolerance)));
    DLR_TEST_ASSERT(
      this->equivalent(computedSum1, sumOverColumns,
                       static_cast<double>(m_defaultTolerance)));
  }
    

  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testAxisSum2()
  {
    // Warning(xxx): Can't make this compile.
//     Array2D<Type> testArray0("[[-2, -1, -10],"
//                              " [5, 2, 3],"
//                              " [10, 0, -2],"
//                              " [1, 2, 2]]");
//     Array1D<double> sumOverRows("[14, 3, -7]");
//     Array1D<double> sumOverColumns("[-13, 10, 8, 5]");
    
//     Array1D<double> computedSum0 =
//       axisSum(testArray0, 0, Double, std::plus<double>());
//     Array1D<double> computedSum1 =
//       axisSum(testArray0, 1, Double, std::plus<double>());
    
//     DLR_TEST_ASSERT(
//       this->equivalent(computedSum0, sumOverRows,
//                        static_cast<double>(m_defaultTolerance)));
//     DLR_TEST_ASSERT(
//       this->equivalent(computedSum1, sumOverColumns,
//                        static_cast<double>(m_defaultTolerance)));
  }
    

  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testColumnIndices()
  {
    Array2D<Type> indices("[[0, 1, 2, 3],"
                          " [0, 1, 2, 3],"
                          " [0, 1, 2, 3]]");
    Array2D<Type> computedIndices =
      columnIndices(indices.rows(), indices.columns(), type_tag<Type>());
    
    DLR_TEST_ASSERT(
      this->equivalent(computedIndices, indices,
                       static_cast<Type>(m_defaultTolerance)));
  }

  
  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testCompress0()
  {
    Array1D<Type> inputArray("[1, 2, 3, 4, 5]");
    Array1D<bool> mask("[1, 1, 0, 1, 0]");
    Array1D<Type> target("[1, 2, 4]");
    Array1D<Type> computedArray = compress(mask, inputArray);
    
    DLR_TEST_ASSERT(
      this->equivalent(computedArray, target,
                       static_cast<Type>(m_defaultTolerance)));
  }

  
  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testCompress1()
  {
    Array1D<Type> inputArray("[1, 2, 3, 4, 5]");
    Array1D<bool> mask("[1, 1, 0, 1, 0]");
    Array1D<Type> target("[1, 2, 4]");
    size_t numTrue = 3;
    Array1D<Type> computedArray = compress(mask, inputArray, numTrue);
    
    DLR_TEST_ASSERT(
      this->equivalent(computedArray, target,
                       static_cast<Type>(m_defaultTolerance)));
  }

  
  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testCount()
  {
    Array1D<Type> inputArray("[1, 2, 0, 4, 0, 5, -1, 0]");
    size_t numTrue = 5;
    DLR_TEST_ASSERT(count(inputArray) == numTrue);
  }


  template <class Type>
  void
  UtilitiesTemplateTest<Type>::
  testSum0()
  {
    Array2D<Type> testArray0("[[-2, -1, -10],"
                             " [5, 2, 3],"
                             " [10, 0, -2],"
                             " [1, 2, 2]]");
    typename NumericTraits<Type>::SumType sum0_2_0_3 = -3;
    typename NumericTraits<Type>::SumType sum0_3_0_2 = 14;
    typename NumericTraits<Type>::SumType sum2_4_1_3 = 2;    
    
    typename NumericTraits<Type>::SumType computedSum0_2_0_3 =
      sum(testArray0, Index2D(0, 0), Index2D(2, 3));
    typename NumericTraits<Type>::SumType computedSum0_3_0_2 =
      sum(testArray0, Index2D(0, 0), Index2D(3, 2));
    typename NumericTraits<Type>::SumType computedSum2_4_1_3 =
      sum(testArray0, Index2D(2, 1), Index2D(4, 3));

    DLR_TEST_ASSERT(
      approximatelyEqual(computedSum0_2_0_3, sum0_2_0_3,
                         static_cast<typename NumericTraits<Type>::SumType>(
                           m_defaultTolerance)));
    DLR_TEST_ASSERT(
      approximatelyEqual(computedSum0_3_0_2, sum0_3_0_2,
                         static_cast<typename NumericTraits<Type>::SumType>(
                           m_defaultTolerance)));
    DLR_TEST_ASSERT(
      approximatelyEqual(computedSum2_4_1_3, sum2_4_1_3,
                         static_cast<typename NumericTraits<Type>::SumType>(
                           m_defaultTolerance)));
  }
    

  template <class Type>
  template <class Type2>
  bool
  UtilitiesTemplateTest<Type>::
  equivalent(const Array1D<Type2>& array0,
             const Array1D<Type2>& array1,
             Type2 tolerance)
  {
    if(array0.size() != array1.size()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<Type2>(tolerance));
  }


  template <class Type>
  template <class Type2>
  bool
  UtilitiesTemplateTest<Type>::
  equivalent(const Array2D<Type2>& array0,
             const Array2D<Type2>& array1,
             Type2 tolerance)
  {
    if(array0.rows() != array1.rows()) {
      return false;
    }
    if(array0.columns() != array1.columns()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<Type2>(tolerance));
  }
  
} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::UtilitiesTemplateTest<double> currentTest0("double");
  dlr::UtilitiesTemplateTest<float> currentTest1("float");
  dlr::UtilitiesTemplateTest<int> currentTest2("int");
  bool result = (currentTest0.run()
                 && currentTest1.run()
                 && currentTest2.run());
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::UtilitiesTemplateTest<double> currentTest0("double");
  dlr::UtilitiesTemplateTest<float> currentTest1("float");
  dlr::UtilitiesTemplateTest<int> currentTest2("int");

}

#endif
