/**
***************************************************************************
* @file dlrNumeric/geometry2D.h
*
* Header file declaring some 2D geometric routines for dlrNumeric.
*
* Copyright (C) 2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 869 $
* $Date: 2007-04-17 03:56:20 -0400 (Tue, 17 Apr 2007) $
***************************************************************************
**/

#ifndef _DLR_NUMERIC_GEOMETRY2D_H_
#define _DLR_NUMERIC_GEOMETRY2D_H_

#include <dlrNumeric/vector2D.h>

namespace dlr {

  namespace numeric {
    
    /** 
     * This function finds a 2D point, given two other points and
     * distance between each of those points and the point to be
     * recovered.  That is, it answers the question "what point is at
     * range0 from point0 and also at range1 from point1?"  Note that
     * there may be two points which satisfy this constraint, or there
     * may be none if the sum of range0 and range1 is less than the
     * distance between point0 and point1.
     * 
     * @param point0 This argument specifies the first known point.
     * 
     * @param point1 This argument specifies the second known point.
     * 
     * @param range0 This argument specifies the distance from point0 to
     * the target point.
     * 
     * @param range1 This argument specifies the distance from point1 to
     * the target point.
     * 
     * @param intersection0 This argument returns the first of the two
     * points which is at range0 from point0 and at range1 from point1.
     * If no solution is found, then this argument will not be touched.
     * 
     * @param intersection1 This argument returns the second of the two
     * points which is at range0 from point0 and at range1 from point1.
     * In degenerate cases, intersection1 will be identical to
     * intersection0.  If no solution is found, then this argument will
     * not be touched.
     * 
     * @return The return value is true if a solution was found, false
     * otherwise.
     */
    bool
    bilaterate(const Vector2D& point0, const Vector2D& point1,
               double range0, double range1,
               Vector2D& intersection0, Vector2D& intersection1);

  } // namespace numeric

} // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::bilaterate;

} // namespace dlr


/* ============ Definitions of inline and template functions ============ */

namespace dlr {
  
  namespace numeric {

    // Empty.
  
  } // namespace numeric
  
} // namespace dlr

#endif /* #ifndef _DLR_NUMERIC_GEOMETRY2D_H_ */
