/**
***************************************************************************
* @file dlrNumeric/functional.h
*
* Header file declaring numeric functors.
*
* Copyright (C) 2003-2007, David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 880 $
* $Date: 2007-05-04 00:33:49 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#include <cmath>
#include <functional>
#include <dlrNumeric/numericTraits.h>

#ifndef _DLR_NUMERIC_FUNCTIONAL_H_
#define _DLR_NUMERIC_FUNCTIONAL_H_

namespace dlr {

  namespace numeric {
    
    /**
     ** Functor template which computes the value of a Gaussian
     ** evaluated at its argument.
     **/
    template <class Type>
    struct Gaussian1DFunctor
      : public std::unary_function<Type, Type>
    {
      /** 
       * The constructor initializes the Gaussian to zero mean and the
       * specified standard deviation.
       * 
       * @param sigma This argument specifies the standard deviation of
       * the Gaussian.
       */
      Gaussian1DFunctor(Type sigma = 1.0)
        : m_k0(1.0 / (std::sqrt(2.0 * M_PI) * sigma)),
          m_k1(-1.0 / (2.0 * sigma * sigma)) {}

    
      /** 
       * Compute the value of the Gaussian at the specified input value,
       * and return the result.
       * 
       * @param input The point at which to evaluate the Guassian.
       */
      inline Type
      operator()(const Type& input) {
        return m_k0 * std::exp(input * input * m_k1);
      }

    private:
      Type m_k0;
      Type m_k1;
    };

  
    /**
     ** Functor template which computes the natural logarithm of its
     ** argument (using std::log(), if appropriate).
     **/
    template <class Type>
    struct LogFunctor
      : public std::unary_function<Type, Type>
    {
      /** 
       * Compute the log of the input argument, and return the result.
       * 
       * @param input The natural log of this argument will be computed.
       * @return The natural log of the argument.
       */
      inline Type
      operator()(const Type& input) {
        return std::log(input);
      }
    };

  
    /**
     ** Functor template which uses static_cast to convert instances of
     ** one type into instances of another, but does the right thing
     ** with regard to rounding, so that the difference between the
     ** input and the returned value is minimized.
     **/
    template <class TypeIn, class TypeOut>
    struct NumericTypeConversionFunctor
      : public std::unary_function<TypeIn, TypeOut>
    {
      /** 
       * This operator returns the output value which minimizes the
       * difference between input and output.  In general, this takes
       * some smarts, so the default case is not implemented.
       * 
       * @param input Will be converted to type TypeOut.
       * @return The result of the conversion.
       */
      inline TypeOut
      operator()(const TypeIn& input) {
        // This "if" should optimize away, since traits are known at compile
        // time.
        if(!NumericTraits<TypeIn>().isIntegral()
           && NumericTraits<TypeOut>().isIntegral()) {
          return static_cast<TypeOut>(input + 0.5);
        }
        return static_cast<TypeOut>(input);
      }
    };

  
    /**
     ** Functor template which computes the square root of its argument
     ** (using std::sqrt(), if appropriate).
     **/
    template <class Type>
    struct SquareRootFunctor
      : public std::unary_function<Type, Type>
    {
      /** 
       * Take the square root of the input argument, and return the result.
       * 
       * @param input This argument will be passed to a square root function.
       * @return The result of the square root call.
       */
      inline Type
      operator()(const Type& input) {
        return std::sqrt(input);
      }
    };

  } // namespace numeric

}  // namespace dlr


/* ======= Declarations to maintain compatibility with legacy code. ======= */

namespace dlr {

  using numeric::Gaussian1DFunctor;
  using numeric::LogFunctor;
  using numeric::NumericTypeConversionFunctor;
  using numeric::SquareRootFunctor;

}  // namespace dlr


/* ================= Specializations =================== */

namespace dlr {

  namespace numeric {
    
  // None

  } // namespace numeric
  
}  // namespace dlr

#endif // #ifndef _DLR_NUMERIC_FUNCTIONAL_H_
