/**
***************************************************************************
* @file convolveTest.cpp
*
* Source file defining ConvolveTest class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 795 $
* $Date: 2006-10-20 19:01:27 -0400 (Fri, 20 Oct 2006) $
***************************************************************************
**/

#include <dlrCommon/functional.h>
#include <dlrNumeric/convolve.h>
#include <dlrTest/testFixture.h>

namespace dlr {

  template <class Type>
  class ConvolveTest
    : public TestFixture< ConvolveTest<Type> >
  {

  public:

    typedef ConvolveTest<Type> TestFixtureType;
    

    ConvolveTest(const std::string& typeName);
    ~ConvolveTest() {}

    void setUp(const std::string& testName) {}
    void tearDown(const std::string& testName) {}

    void testConvolve_truncateResult();
    void testConvolve_zeroPadResult();
    void testConvolve_zeroPadSignal();
    void testConvolve_reflectSignal();
    void testConvolve_wrapSignal();
    void testCorrelate_truncateResult();
    void testCorrelate_zeroPadResult();
    void testCorrelate_zeroPadSignal();
    void testCorrelate_reflectSignal();
    void testCorrelate_wrapSignal();

  private:

    template <class Type2>
    bool equivalent(const Array1D<Type2>& arg0,
                    const Array1D<Type2>& arg1,
                    Type2 tolerance);


    Array1D<Type> m_convolveKernel;
    Array1D<Type> m_correlateKernel;
    Type m_defaultTolerance;
    Array1D<Type> m_signal;
    Array1D<Type> m_result_truncateResult;
    Array1D<Type> m_result_zeroPadResult;
    Array1D<Type> m_result_zeroPadSignal;
    Array1D<Type> m_result_reflectSignal;
    Array1D<Type> m_result_wrapSignal;
    
  }; // class ConvolveTest


  /* ============== Member Function Definititions ============== */

  template <class Type>
  ConvolveTest<Type>::
  ConvolveTest(const std::string& typeName)
    : TestFixture<ConvolveTest>(
      std::string("ConvolveTest<" + typeName + ">")),
      m_convolveKernel("[6, 4, 2]"),
      m_correlateKernel("[2, 4, 6]"),
      m_defaultTolerance(static_cast<Type>(1.0E-6)),
      m_signal("[1, 2, 3, 4, 5, 6]"),
      m_result_truncateResult("[28, 40, 52, 64]"),
      m_result_zeroPadResult("[0, 28, 40, 52, 64, 0]"),
      m_result_zeroPadSignal("[6, 16, 28, 40, 52, 64, 34, 12]"),
      m_result_reflectSignal("[20, 20, 28, 40, 52, 64, 64, 56]"),
      m_result_wrapSignal("[40, 28, 28, 40, 52, 64, 40, 28]")
  {
    // Register all tests.
    DLR_TEST_REGISTER_MEMBER(testConvolve_truncateResult);
    DLR_TEST_REGISTER_MEMBER(testConvolve_zeroPadResult);
    DLR_TEST_REGISTER_MEMBER(testConvolve_zeroPadSignal);
    DLR_TEST_REGISTER_MEMBER(testConvolve_reflectSignal);
    DLR_TEST_REGISTER_MEMBER(testConvolve_wrapSignal);
    DLR_TEST_REGISTER_MEMBER(testCorrelate_truncateResult);
    DLR_TEST_REGISTER_MEMBER(testCorrelate_zeroPadResult);
    DLR_TEST_REGISTER_MEMBER(testCorrelate_zeroPadSignal);
    DLR_TEST_REGISTER_MEMBER(testCorrelate_reflectSignal);
    DLR_TEST_REGISTER_MEMBER(testCorrelate_wrapSignal);
  }


  template <class Type>
  void
  ConvolveTest<Type>::
  testConvolve_truncateResult()
  {
    Array1D<Type> result = convolve(m_convolveKernel, m_signal,
                                    DLR_CONVOLVE_TRUNCATE_RESULT);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_truncateResult, m_defaultTolerance));
  }

  
  template <class Type>
  void
  ConvolveTest<Type>::
  testConvolve_zeroPadResult()
  {
    Array1D<Type> result = convolve(m_convolveKernel, m_signal,
                                    DLR_CONVOLVE_ZERO_PAD_RESULT);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_zeroPadResult, m_defaultTolerance));
  }


  template <class Type>
  void
  ConvolveTest<Type>::
  testConvolve_zeroPadSignal()
  {
    Array1D<Type> result = convolve(m_convolveKernel, m_signal,
                                    DLR_CONVOLVE_ZERO_PAD_SIGNAL);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_zeroPadSignal, m_defaultTolerance));
  }


  template <class Type>
  void
  ConvolveTest<Type>::
  testConvolve_reflectSignal()
  {
    Array1D<Type> result = convolve(m_convolveKernel, m_signal,
                                    DLR_CONVOLVE_REFLECT_SIGNAL);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_reflectSignal, m_defaultTolerance));
  }


  template <class Type>
  void
  ConvolveTest<Type>::
  testConvolve_wrapSignal()
  {
    Array1D<Type> result = convolve(m_convolveKernel, m_signal,
                                    DLR_CONVOLVE_WRAP_SIGNAL);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_wrapSignal, m_defaultTolerance));
  }


  template <class Type>
  void
  ConvolveTest<Type>::
  testCorrelate_truncateResult()
  {
    Array1D<Type> result = correlate(m_correlateKernel, m_signal,
                                    DLR_CONVOLVE_TRUNCATE_RESULT);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_truncateResult, m_defaultTolerance));
  }

  
  template <class Type>
  void
  ConvolveTest<Type>::
  testCorrelate_zeroPadResult()
  {
    Array1D<Type> result = correlate(m_correlateKernel, m_signal,
                                    DLR_CONVOLVE_ZERO_PAD_RESULT);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_zeroPadResult, m_defaultTolerance));
  }


  template <class Type>
  void
  ConvolveTest<Type>::
  testCorrelate_zeroPadSignal()
  {
    Array1D<Type> result = correlate(m_correlateKernel, m_signal,
                                    DLR_CONVOLVE_ZERO_PAD_SIGNAL);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_zeroPadSignal, m_defaultTolerance));
  }


  template <class Type>
  void
  ConvolveTest<Type>::
  testCorrelate_reflectSignal()
  {
    Array1D<Type> result = correlate(m_correlateKernel, m_signal,
                                    DLR_CONVOLVE_REFLECT_SIGNAL);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_reflectSignal, m_defaultTolerance));
  }


  template <class Type>
  void
  ConvolveTest<Type>::
  testCorrelate_wrapSignal()
  {
    Array1D<Type> result = correlate(m_correlateKernel, m_signal,
                                    DLR_CONVOLVE_WRAP_SIGNAL);
    DLR_TEST_ASSERT(
      this->equivalent(result, m_result_wrapSignal, m_defaultTolerance));
  }


  template <class Type>
  template <class Type2>
  bool
  ConvolveTest<Type>::
  equivalent(const Array1D<Type2>& array0,
             const Array1D<Type2>& array1,
             Type2 tolerance)
  {
    if(array0.size() != array1.size()) {
      return false;
    }
    return std::equal(array0.begin(), array0.end(), array1.begin(),
                      ApproximatelyEqualFunctor<Type2>(tolerance));
  }

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::ConvolveTest<double> currentTest0("double");
  dlr::ConvolveTest<float> currentTest1("float");
  dlr::ConvolveTest<int> currentTest2("int");
  bool result = (currentTest0.run()
                 && currentTest1.run()
                 && currentTest2.run());
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::ConvolveTest<double> currentTest0("double");
  dlr::ConvolveTest<float> currentTest1("float");
  dlr::ConvolveTest<int> currentTest2("int");
  
}

#endif
