/**
***************************************************************************
* @file dlrGeometry/triangle3D.h
*
* Header file declaring the Triangle3D class.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#ifndef DLR_GEOMETRY_TRIANGLE3D_H
#define DLR_GEOMETRY_TRIANGLE3D_H

#include <iostream>
#include <dlrNumeric/vector3D.h>

namespace dlr {

  namespace geometry {
    
    /**
     ** The Triangle3D class represents a triangle in 3D space.
     **/
    class Triangle3D {
    public:
      
      /** 
       * The default constructor initializes to a triangle in the X-Y
       * plane.
       */
      Triangle3D()
	: m_vertex0(0.0, 0.0, 0.0), m_vertex1(1.0, 0.0, 0.0),
          m_vertex2(0.0, 1.0, 0.0) {}

    
      /** 
       * This constructor initializes the triangle using three points.
       * 
       * @param vertex0 This argument is one of the three points that
       * define the triangle.
       * 
       * @param vertex1 This argument is one of the three points that
       * define the triangle.
       * 
       * @param vertex2 This argument is one of the three points that
       * define the triangle.
       */
      Triangle3D(const Vector3D& vertex0,
                 const Vector3D& vertex1,
                 const Vector3D& vertex2);

    
      /** 
       * The copy constructor deep copies its argument.
       * 
       * @param source This argument is the class instance to be copied.
       */
      Triangle3D(const Triangle3D& source);


      /** 
       * Destructor.
       */
      ~Triangle3D() {}


      /** 
       * The assignment operator deep copies its argument.
       * 
       * @param source This argument is the class instance to be copied.
       * 
       * @return The return value is a reference to *this.
       */
      Triangle3D&
      operator=(const Triangle3D& source);


      /** 
       * This member function returns the one of the three vertices
       * that define the triangle.
       * 
       * @return The return value is a Vector3D representing the
       * first vertex of the triangle.
       */
      const Vector3D&
      getVertex0() const {return m_vertex0;}
      

      /** 
       * This member function returns the one of the three vertices
       * that define the triangle.
       * 
       * @return The return value is a Vector3D representing the
       * second vertex of the triangle.
       */
      const Vector3D&
      getVertex1() const {return m_vertex1;}
      

      /** 
       * This member function returns the one of the three vertices
       * that define the triangle.
       * 
       * @return The return value is a Vector3D representing the
       * first vertex of the triangle.
       */
      const Vector3D&
      getVertex2() const {return m_vertex2;}
      

      Triangle3D&
      setValue(Vector3D const& vertex0,
               Vector3D const& vertex1,
               Vector3D const& vertex2) {
        m_vertex0 = vertex0;
        m_vertex1 = vertex1;
        m_vertex2 = vertex2;
        return *this;
      }
      
    private:

      // Private data members.
      Vector3D m_vertex0;
      Vector3D m_vertex1;
      Vector3D m_vertex2;

    }; // class Triangle3D



    /* ======= Non-member functions. ======= */

    std::ostream&
    operator<<(std::ostream& stream, const Triangle3D& triangle);
    

    std::istream&
    operator>>(std::istream& stream, Triangle3D& triangle);
    
    
  } // namespace utilities
    
} // namespace dlr


/* ======= Inline and template function definitions. ======= */

#include <dlrLinearAlgebra/linearAlgebra.h>
#include <dlrNumeric/subArray1D.h>

namespace dlr {

  namespace geometry {


  } // namespace geometry
  
} // namespace dlr


#endif /* #ifndef DLR_GEOMETRY_TRIANGLE3D_H */
