/**
***************************************************************************
* @file dlrGeometry/ray3D.h
*
* Header file declaring Ray3D class.
*
* Copyright (C) 2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef DLR_GEOMETRY_RAY3D_H
#define DLR_GEOMETRY_RAY3D_H

#include <iostream>
#include <dlrNumeric/vector3D.h>

namespace dlr {

  namespace geometry {
    
    /**
     ** The Ray3D class represents a ray in 3D space.
     **/
    class Ray3D {
    public:
      
      /** 
       * The default constructor initializes to the ray that starts
       * at the origin and points along the X axis.
       */
      Ray3D()
	: m_origin(0.0, 0.0, 0.0), m_direction(1.0, 0.0, 0.0) {}

    
      /** 
       * This constructor initializes the ray using a point and a direction.
       * 
       * @param point This argument specifies the start point of the ray.
       * 
       * @param direction This argument specifies the direction of the ray.
       * 
       * @param normalized If the direction vector is already
       * normalized to unit length, then you can save some computation
       * by setting this argument to false.
       */
      Ray3D(const Vector3D& point,
            const Vector3D& direction,
            bool normalize = true);

    
      /** 
       * The copy constructor deep copies its argument.
       * 
       * @param source This argument is the class instance to be
       * copied.
       */
      Ray3D(const Ray3D& source);


      /** 
       * Destructor.
       */
      ~Ray3D() {}

      
      /** 
       * The assignment operator deep copies its argument.
       * 
       * @param source This argument is the class instance to be
       * copied.
       * 
       * @return The return value is a reference to *this.
       */
      Ray3D&
      operator=(const Ray3D& source);


      /** 
       * This member function returns the direction of the ray.
       * 
       * @return The return value is a const reference to Vector3D
       * representing the direction of the ray.
       */
      const Vector3D&
      getDirection() const {return m_direction;}
      

      /** 
       * This member function returns the direction of the ray.
       * 
       * @return The return value is a const reference to Vector3D
       * representing the direction of the ray.
       */
      const Vector3D&
      getDirectionVector() const {return this->getDirection();}
      

      /** 
       * This member function returns the start point of the ray.
       * 
       * @return The return value is a const reference to Vector3D
       * representing the start point of the ray.
       */
      const Vector3D&
      getOrigin() const {return m_origin;}


   private:
      // Private member functions.

      // Private data members.
      Vector3D m_origin;
      Vector3D m_direction;

    }; // class Ray3D


    
    /* ======= Non-member functions. ======= */

    std::ostream&
    operator<<(std::ostream& stream, const Ray3D& ray);
    
    
  } // namespace utilities
    
} // namespace dlr


/* ======= Inline and template function definitions. ======= */

namespace dlr {

  namespace geometry {

    
  } // namespace geometry
  
} // namespace dlr


#endif /* #ifndef DLR_GEOMETRY_RAY3D_H */
