/**
***************************************************************************
* @file dlrGeometry/ray2D.h
*
* Header file declaring Ray2D class.
*
* Copyright (C) 2007-2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef DLR_GEOMETRY_RAY2D_H
#define DLR_GEOMETRY_RAY2D_H

#include <iostream>
#include <dlrNumeric/vector2D.h>

namespace dlr {

  namespace geometry {
    
    /**
     ** The Ray2D class represents a ray in 2D space.
     **/
    class Ray2D {
    public:
      
      /** 
       * The default constructor initializes to the ray that starts
       * at the origin and points along the X axis.
       */
      Ray2D()
	: m_origin(0.0, 0.0), m_direction(1.0, 0.0) {}

    
      /** 
       * This constructor initializes the ray using a point and a direction.
       * 
       * @param point This argument specifies the start point of the ray.
       * 
       * @param direction This argument specifies the direction of the ray.
       * 
       * @param normalized If the direction vector is already
       * normalized to unit length, then you can save some computation
       * by setting this argument to false.
       */
      Ray2D(const Vector2D& point,
            const Vector2D& direction,
            bool normalize = true);


      /** 
       * This constructor initializes the ray according to the
       * equation Ax + By + C = 0.  After construction, the ray origin
       * is set to [-aa*cc / k, -bb*cc / k], where k = aa*aa + bb*bb,
       * and the ray direction is set parallel to [-bb, aa].
       * 
       * @param aa This argument is the "A" coefficient of the line equation.
       * 
       * @param bb This argument is the "B" coefficient of the line equation.
       * 
       * @param cc This argument is the "C" coefficient of the line equation.
       */
      Ray2D(double aa, double bb, double cc);
      
    
      /** 
       * The copy constructor deep copies its argument.
       * 
       * @param source This argument is the class instance to be
       * copied.
       */
      Ray2D(const Ray2D& source);


      /** 
       * Destructor.
       */
      ~Ray2D() {}

      
      /** 
       * The assignment operator deep copies its argument.
       * 
       * @param source This argument is the class instance to be
       * copied.
       * 
       * @return The return value is a reference to *this.
       */
      Ray2D&
      operator=(const Ray2D& source);


      /** 
       * This member function returns the direction of the ray.
       * 
       * @return The return value is a const reference to Vector2D
       * representing the direction of the ray.
       */
      const Vector2D&
      getDirection() const {return m_direction;}
      

      /** 
       * This member function returns the direction of the ray.
       * 
       * @return The return value is a const reference to Vector2D
       * representing the direction of the ray.
       */
      const Vector2D&
      getDirectionVector() const {return this->getDirection();}
      

      /** 
       * This member function returns the start point of the ray.
       * 
       * @return The return value is a const reference to Vector2D
       * representing the start point of the ray.
       */
      const Vector2D&
      getOrigin() const {return m_origin;}


   private:
      // Private member functions.

      // Private data members.
      Vector2D m_origin;
      Vector2D m_direction;

    }; // class Ray2D


    
    /* ======= Non-member functions. ======= */

    std::ostream&
    operator<<(std::ostream& stream, const Ray2D& ray);
    
    
  } // namespace utilities
    
} // namespace dlr


/* ======= Inline and template function definitions. ======= */

namespace dlr {

  namespace geometry {

    
  } // namespace geometry
  
} // namespace dlr


#endif /* #ifndef DLR_GEOMETRY_RAY2D_H */
