/**
***************************************************************************
* @file dlrGeometry/lineSegment2D.cpp
*
* Source file defining the LineSegment2D class.
*
* Copyright (C) 2007-2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#include <dlrGeometry/lineSegment2D.h>
#include <dlrNumeric/utilities.h>

namespace dnum = dlr::numeric;

namespace dlr {

  namespace geometry {
    
    // The assignment operator deep copies its argument.
    LineSegment2D&
    LineSegment2D::
    operator=(const LineSegment2D& source)
    {
      if(&source != this) {
        m_startPoint = source.m_startPoint;
        m_endPoint = source.m_endPoint;
      }
      return *this;
    }


    /* ======= Non-member functions. ======= */

    std::istream&
    operator>>(std::istream& stream, LineSegment2D& lineSegment)
    {
      // If stream is in a bad state, we can't read from it.
      if (!stream){
        return stream;
      }
    
      double startPointX;
      double startPointY;
      double endPointX;
      double endPointY;
        
      // Construct an InputStream instance so we can use our
      // convenience functions.
      dlr::common::InputStream inputStream(
        stream, dlr::common::InputStream::SKIP_WHITESPACE);

      inputStream.expect("LineSegment2D");
      inputStream.expect("{");
      inputStream >> startPointX;
      inputStream.expect(",");
      inputStream >> startPointY;
      inputStream.expect(",");
      inputStream >> endPointX;
      inputStream.expect(",");
      inputStream >> endPointY;
      inputStream.expect("}");

      // If reading failed, don't change lineSegment.
      if (!stream){
        return stream;
      }
      
      lineSegment.setValue(startPointX, startPointY,
                           endPointX, endPointY);
      return stream;
    }
    

    std::ostream&
    operator<<(std::ostream& stream, const LineSegment2D& lineSegment)
    {
      stream << "LineSegment2D { "
             << lineSegment.getStartPoint().x() << ", "
             << lineSegment.getStartPoint().y() << ", "
             << lineSegment.getEndPoint().x() << ", "
             << lineSegment.getEndPoint().y() << " }";
      return stream;
    }

    
  } // namespace geometry
    
} // namespace dlr
