/**
***************************************************************************
* @file dlrGeometry/ellipse2D.h
*
* Header file declaring the Ellipse2D class.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef DLR_GEOMETRY_ELLIPSE2D_H
#define DLR_GEOMETRY_ELLIPSE2D_H

#include <iostream>
#include <dlrNumeric/vector2D.h>

namespace dlr {

  namespace geometry {
    
    /**
     ** The Ellipse2D class represents a planar ellipse in 2D space.
     **/
    class Ellipse2D {
    public:
      
      /** 
       * The default constructor initializes to the unit circle.
       */
      inline
      Ellipse2D();

      
      /** 
       * This constructor initializes the ellipse using explicitly
       * specified values.
       * 
       * @param origin This argument specifies the position of the
       * geometric center of the ellipse.
       * 
       * @param majorAxis This argument represents a vector pointing
       * from the center of the ellipse to one of the two points on
       * the boundary of the ellipse that is farthest from the center.
       * 
       * @param ratio This argument specifies the length of the minor
       * axis as a proportion of the lenth of the major axis.  It must
       * be less than or equal to 1.0.
       */
      inline
      Ellipse2D(const dlr::numeric::Vector2D& origin,
                const dlr::numeric::Vector2D& semimajorAxis,
                double ratio);

    
      /** 
       * The copy constructor deep copies its argument.
       * 
       * @param source This argument is the class instance to be copied.
       */
      inline
      Ellipse2D(const Ellipse2D& source);


      /** 
       * Destructor.
       */
      ~Ellipse2D() {}


      /** 
       * The assignment operator deep copies its argument.
       * 
       * @param source This argument is the class instance to be copied.
       * 
       * @return The return value is a reference to *this.
       */
      inline Ellipse2D&
      operator=(const Ellipse2D& source);


      /** 
       * This member function returns the geometric center of the ellipse.
       * 
       * @return The return value is the point at the centroid of the
       * ellipse.
       */
      const dlr::numeric::Vector2D&
      getOrigin() const {return m_origin;}
      

      /** 
       * This member function returns a vector pointing from the
       * center of the ellipse to the point on the edge of the ellipse
       * that is farthest from the center.  Note that there are two
       * such farthest points on opposite sides of the ellipse.  The
       * vector returned by this member function will remain
       * consistent for the life of the ellipse, and will reflect the
       * semimajor axis specified as a constructor argument (if the
       * three-argument constructor was used).
       * 
       * @return The return value is a vector pointing along the
       * semimajor axis of the ellipse.
       */
      const dlr::numeric::Vector2D&
      getSemimajorAxis() const {return m_semimajorAxis;}
      

      /** 
       * This member function returns a vector pointing from the
       * center of the ellipse to the point on the edge of the ellipse
       * that is closest to the center.  Note that there are two such
       * closest points on opposite sides of the ellipse.  The vector
       * returned by this member function will remain consistent for
       * the life of the ellipse, and will be normally be rotated 90
       * degrees counterclockwise from the vector returned by
       * getSemiMajorAxis().
       * 
       * @return The return value is a vector pointing along the
       * semiminor axis of the ellipse.
       */
      const dlr::numeric::Vector2D&
      getSemiminorAxis() const {return m_semimajorAxis;}
      

    private:
      // Private member functions.

      // Private data members.
      dlr::numeric::Vector2D m_origin;
      dlr::numeric::Vector2D m_semimajorAxis;
      dlr::numeric::Vector2D m_semiminorAxis;

    }; // class Ellipse2D



    /* ======= Non-member functions. ======= */

    std::ostream&
    operator<<(std::ostream& stream, const Ellipse2D& ellipse);
    
  } // namespace geometry
    
} // namespace dlr


/* ======= Inline and template function definitions. ======= */

namespace dlr {

  namespace geometry {

    // The default constructor initializes to the unit circle.
    Ellipse2D::
    Ellipse2D()
      : m_origin(0.0, 0.0),
        m_semimajorAxis(1.0, 0.0),
        m_semiminorAxis(0.0, 1.0)
    {
      // Empty.
    }

    
    // This constructor initializes the ellipse using explicitly
    // specified values.
    Ellipse2D::
    Ellipse2D(const dlr::numeric::Vector2D& origin,
              const dlr::numeric::Vector2D& semimajorAxis,
              double ratio)
      : m_origin(origin),
        m_semimajorAxis(semimajorAxis),
        m_semiminorAxis(semimajorAxis.y() * ratio, semimajorAxis.x() * ratio)
    {
      if(ratio > 1.0) {
        std::swap(m_semimajorAxis, m_semiminorAxis);
      }
    }

    
    // The copy constructor deep copies its argument.
    Ellipse2D::
    Ellipse2D(const Ellipse2D& source)
      : m_origin(source.m_origin),
        m_semimajorAxis(source.m_semimajorAxis),
        m_semiminorAxis(source.m_semiminorAxis)
    {
      // Empty.
    }


    // The assignment operator deep copies its argument.
    Ellipse2D&
    Ellipse2D::
    operator=(const Ellipse2D& source)
    {
      if(&source != this) {
        m_origin = source.m_origin;
        m_semimajorAxis = source.m_semimajorAxis;
        m_semiminorAxis = source.m_semiminorAxis;
      }
      return *this;
    }

  } // namespace geometry
  
} // namespace dlr


#endif /* #ifndef DLR_GEOMETRY_ELLIPSE2D_H */
