/**
***************************************************************************
* @file dlrGeometry/circle2D.h
*
* Header file declaring the Circle2D class.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#ifndef DLR_GEOMETRY_CIRCLE2D_H
#define DLR_GEOMETRY_CIRCLE2D_H

#include <iostream>
#include <dlrNumeric/vector2D.h>

namespace dlr {

  namespace geometry {
    
    /**
     ** The Circle2D class represents a circle in 2D space.
     **/
    class Circle2D {
    public:
      
      /** 
       * The default constructor initializes to the unit circle.
       */
      inline
      Circle2D();

      
      /** 
       * This constructor initializes the circle using explicitly
       * specified values.
       * 
       * @param origin This argument specifies the position of the
       * geometric center of the circle.
       * 
       * @param radius This argument specifies the radius of the
       * circle.  If the value of radius is less than 0.0, it will be
       * multiplied by -1.
       */
      inline
      Circle2D(const dlr::numeric::Vector2D& origin,
               double radius);

    
      /** 
       * The copy constructor deep copies its argument.
       * 
       * @param source This argument is the class instance to be copied.
       */
      inline
      Circle2D(const Circle2D& source);


      /** 
       * Destructor.
       */
      ~Circle2D() {}


      /** 
       * The assignment operator deep copies its argument.
       * 
       * @param source This argument is the class instance to be copied.
       * 
       * @return The return value is a reference to *this.
       */
      inline Circle2D&
      operator=(const Circle2D& source);


      /** 
       * This member function returns the geometric center of the circle.
       * 
       * @return The return value is the point at the center of the
       * circle.
       */
      dlr::numeric::Vector2D const&
      getOrigin() const {return m_origin;}
      

      /** 
       * This member function returns the radius of the circle.
       * 
       * @return The return value is the radius of the circle.
       */
      double
      getRadius() const {return m_radius;}
      

    private:
      // Private member functions.

      // Private data members.
      dlr::numeric::Vector2D m_origin;
      double m_radius;

    }; // class Circle2D



    /* ======= Non-member functions. ======= */

    std::ostream&
    operator<<(std::ostream& stream, const Circle2D& circle);
    
  } // namespace geometry
    
} // namespace dlr


/* ======= Inline and template function definitions. ======= */

namespace dlr {

  namespace geometry {

    // The default constructor initializes to the unit circle.
    Circle2D::
    Circle2D()
      : m_origin(0.0, 0.0),
        m_radius(1.0)
    {
      // Empty.
    }

    
    // This constructor initializes the circle using explicitly
    // specified values.
    Circle2D::
    Circle2D(const dlr::numeric::Vector2D& origin,
             double radius)
      : m_origin(origin),
        m_radius(radius)
    {
      if(radius < 0.0) {
        m_radius = -m_radius;
      }
    }

    
    // The copy constructor deep copies its argument.
    Circle2D::
    Circle2D(const Circle2D& source)
      : m_origin(source.m_origin),
        m_radius(source.m_radius)
    {
      // Empty.
    }


    // The assignment operator deep copies its argument.
    Circle2D&
    Circle2D::
    operator=(const Circle2D& source)
    {
      if(&source != this) {
        m_origin = source.m_origin;
        m_radius = source.m_radius;
      }
      return *this;
    }

  } // namespace geometry
  
} // namespace dlr


#endif /* #ifndef DLR_GEOMETRY_CIRCLE2D_H */
