/**
***************************************************************************
* @file dlrGeometry/triangle3D.cpp
*
* Source file defining the Triangle3D class.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrGeometry/triangle3D.h>
#include <dlrNumeric/utilities.h>

namespace dnum = dlr::numeric;

namespace dlr {

  namespace geometry {
    
    // This constructor initializes the triangle using three points.
    Triangle3D::
    Triangle3D(const Vector3D& vertex0,
            const Vector3D& vertex1,
            const Vector3D& vertex2)
      : m_vertex0(vertex0),
        m_vertex1(vertex1),
        m_vertex2(vertex2)
    {
      // Empty.
    }

    
    // The copy constructor deep copies its argument.
    Triangle3D::
    Triangle3D(const Triangle3D& source)
      : m_vertex0(source.m_vertex0),
        m_vertex1(source.m_vertex1),
        m_vertex2(source.m_vertex2)
    {
      // Empty.
    }


    // The assignment operator deep copies its argument.
    Triangle3D&
    Triangle3D::
    operator=(const Triangle3D& source)
    {
      if(&source != this) {
        m_vertex0 = source.m_vertex0;
        m_vertex1 = source.m_vertex1;
        m_vertex2 = source.m_vertex2;
      }
      return *this;
    }


    /* ======= Non-member functions. ======= */

    std::ostream&
    operator<<(std::ostream& stream, const Triangle3D& triangle)
    {
      stream << "Triangle3D { "
             << triangle.getVertex0() << ", "
             << triangle.getVertex1() << ", "
             << triangle.getVertex2() << " }";
      return stream;
    }


    std::istream&
    operator>>(std::istream& stream, Triangle3D& triangle)
    {
      // If stream is in a bad state, we can't read from it.
      if (!stream){
        return stream;
      }
    
      // It's a lot easier to use a try block than to be constantly
      // testing whether the IO has succeeded, so we tell inputStream to
      // complain if anything goes wrong.
      std::ios_base::iostate oldExceptionState = stream.exceptions();
      stream.exceptions(
        std::ios_base::badbit | std::ios_base::failbit | std::ios_base::eofbit);

      // Now on with the show.
      try{
        dnum::Vector3D vertex0;
        dnum::Vector3D vertex1;
        dnum::Vector3D vertex2;
        
        // Construct an InputStream instance so we can use our
        // convenience functions.
        dlr::common::InputStream inputStream(
          stream, dlr::common::InputStream::SKIP_WHITESPACE);

        inputStream.expect("Triangle3D");
        inputStream.expect("{");
        inputStream >> vertex0;
        inputStream.expect(",");
        inputStream >> vertex1;
        inputStream.expect(",");
        inputStream >> vertex2;
        inputStream.expect("}");

        triangle.setValue(vertex0, vertex1, vertex2);
        
      } catch(std::ios_base::failure) {
        // Empty
      }
      stream.exceptions(oldExceptionState);
      return stream;
    }

    
  } // namespace utilities
    
} // namespace dlr
