/**
***************************************************************************
* @file dlrGeometry/ray2D.cpp
*
* Source file defining the Ray2D class.
*
* Copyright (C) 2007-2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#include <dlrGeometry/ray2D.h>
#include <dlrNumeric/utilities.h>

namespace dnum = dlr::numeric;

namespace dlr {

  namespace geometry {
    
    // This constructor initializes the ray using a point and a
    // direction vector.
    Ray2D::
    Ray2D(const Vector2D& point,
          const Vector2D& direction,
          bool normalize)
      : m_origin(point),
        m_direction(direction)
    {
      if(normalize) {
        m_direction /= dnum::magnitude(m_direction);
      }
    }

    
    // This constructor initializes the ray according to the
    // equation Ax + By + C = 0.
    Ray2D::
    Ray2D(double aa, double bb, double cc)
      : m_origin(-(aa * cc), -(bb * cc)),
        m_direction(-bb, aa)
    {
      double kk = aa * aa + bb * bb;
      m_origin /= kk;
      m_direction /= dnum::magnitude(m_direction);
    }

         
    // The copy constructor deep copies its argument.
    Ray2D::
    Ray2D(const Ray2D& source)
      : m_origin(source.m_origin),
        m_direction(source.m_direction)
    {
      // Empty.
    }


    // The assignment operator deep copies its argument.
    Ray2D&
    Ray2D::
    operator=(const Ray2D& source)
    {
      if(&source != this) {
        m_origin = source.m_origin;
        m_direction = source.m_direction;
      }
      return *this;
    }


    /* ======= Non-member functions. ======= */

    std::ostream&
    operator<<(std::ostream& stream, const Ray2D& ray)
    {
      stream << "Ray2D { "
             << ray.getOrigin() << ", "
             << ray.getDirectionVector() << " }";
      return stream;
    }
    
  } // namespace geometry
    
} // namespace dlr
