/**
***************************************************************************
* @file dlrGeometry/lineSegment2D.h
*
* Header file declaring LineSegment2D class.
*
* Copyright (C) 2007-2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 885 $
* $Date: 2007-05-04 01:01:15 -0400 (Fri, 04 May 2007) $
***************************************************************************
**/

#ifndef DLR_GEOMETRY_LINESEGMENT2D_H
#define DLR_GEOMETRY_LINESEGMENT2D_H

#include <iostream>
#include <dlrNumeric/vector2D.h>

namespace dlr {

  namespace geometry {
    
    /**
     ** The LineSegment2D class represents a line segment in 2D space.
     **/
    class LineSegment2D {
    public:
      
      /** 
       * The default constructor initializes to the line segment that starts
       * at the origin and ends at (1, 0).
       */
      LineSegment2D()
	: m_startPoint(0.0, 0.0), m_endPoint(1.0, 0.0) {}

    
      /** 
       * This constructor initializes the line segment using a pair of points.
       * 
       * @param point This argument specifies the start point of the
       * line segment.
       * 
       * @param endPoint This argument specifies the end point of the
       * line segment.
       */
      LineSegment2D(const Vector2D& startPoint,
                    const Vector2D& endPoint)
        : m_startPoint(startPoint), m_endPoint(endPoint) {}


      /** 
       * The copy constructor deep copies its argument.
       * 
       * @param source This argument is the class instance to be
       * copied.
       */
      LineSegment2D(const LineSegment2D& source)
        : m_startPoint(source.m_startPoint), m_endPoint(source.m_endPoint) {}


      /** 
       * Destructor.
       */
      ~LineSegment2D() {}

      
      /** 
       * The assignment operator deep copies its argument.
       * 
       * @param source This argument is the class instance to be
       * copied.
       * 
       * @return The return value is a reference to *this.
       */
      LineSegment2D&
      operator=(const LineSegment2D& source);


      /** 
       * This member function returns the end point of the line segment.
       * 
       * @return The return value is a const reference to Vector2D
       * representing the end point of the line segment.
       */
      const Vector2D&
      getEndPoint() const {return m_endPoint;}


      /** 
       * This member function returns the start point of the line segment.
       * 
       * @return The return value is a const reference to Vector2D
       * representing the start point of the line segment.
       */
      const Vector2D&
      getVertex0() const {return m_startPoint;}


      /** 
       * This member function returns the end point of the line segment.
       * 
       * @return The return value is a const reference to Vector2D
       * representing the end point of the line segment.
       */
      const Vector2D&
      getVertex1() const {return m_endPoint;}


      /** 
       * This member function returns the start point of the line segment.
       * 
       * @return The return value is a const reference to Vector2D
       * representing the start point of the line segment.
       */
      const Vector2D&
      getStartPoint() const {return m_startPoint;}


      /** 
       * This member function changes the start point and end point of
       * the line segment.
       * 
       * @param startPointX This argument is the new start point X coordinate.
       * 
       * @param startPointY This argument is the new start point Y coordinate.
       * 
       * @param endPointX This argument is the new end point X coordinate.
       * 
       * @param endPointY This argument is the new end point Y coordinate.
       * 
       * @return The return value is a reference to *this after the
       * change has been applied.
       */
      LineSegment2D&
      setValue(double startPointX, double startPointY,
               double endPointX, double endPointY) {
        m_startPoint.setValue(startPointX, startPointY);
        m_endPoint.setValue(endPointX, endPointY);
        return *this;
      }

      
      /** 
       * This member function changes the start point and end point of
       * the line segment.
       * 
       * @param startPoint This argument is the new start point.
       * 
       * @param endPoint This argument is the new end point.
       *
       * @return The return value is a reference to *this after the
       * change has been applied.
       */
      LineSegment2D&
      setValue(Vector2D const& startPoint, Vector2D const& endPoint) {
        m_startPoint = startPoint; m_endPoint = endPoint; return *this;
      }

      
   private:
      // Private member functions.

      // Private data members.
      Vector2D m_startPoint;
      Vector2D m_endPoint;

    }; // class LineSegment2D


    
    /* ======= Non-member functions. ======= */

    std::istream&
    operator>>(std::istream& stream, LineSegment2D& lineSegment);
    
    
    std::ostream&
    operator<<(std::ostream& stream, const LineSegment2D& lineSegment);
    
    
  } // namespace utilities
    
} // namespace dlr


/* ======= Inline and template function definitions. ======= */

namespace dlr {

  namespace geometry {

    
  } // namespace geometry
  
} // namespace dlr


#endif /* #ifndef DLR_GEOMETRY_LINESEGMENT2D_H */
