/**
***************************************************************************
* @file utilitiesTest.cpp
*
* Source file defining tests for dlrComputerVision library utilities.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1155 $
* $Date: 2009-05-24 00:07:38 -0400 (Sun, 24 May 2009) $
***************************************************************************
**/

#include <dlrComputerVision/test/testImages.h>
#include <dlrComputerVision/colorspaceConverter.h>
#include <dlrComputerVision/image.h>
#include <dlrComputerVision/imageIO.h>
#include <dlrComputerVision/utilities.h>
#include <dlrTest/testFixture.h>


namespace dlr {

  namespace computerVision {
    
    class UtilitiesTest : public TestFixture<UtilitiesTest> {

    public:

      UtilitiesTest();
      ~UtilitiesTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      // Tests.
      void testAssociateColorComponents();
      void testConvertColorspace();
      void testDissociateColorComponents();
      void testSubsample();
      void testSupersample();
      void testToArray();
    
    private:

    }; // class UtilitiesTest


    /* ============== Member Function Definititions ============== */

    UtilitiesTest::
    UtilitiesTest()
      : TestFixture<UtilitiesTest>("UtilitiesTest")
    {
      DLR_TEST_REGISTER_MEMBER(testAssociateColorComponents);
      DLR_TEST_REGISTER_MEMBER(testConvertColorspace);
      DLR_TEST_REGISTER_MEMBER(testDissociateColorComponents);
      DLR_TEST_REGISTER_MEMBER(testSupersample);
      DLR_TEST_REGISTER_MEMBER(testSubsample);
      DLR_TEST_REGISTER_MEMBER(testToArray);
    }


    void
    UtilitiesTest::
    testAssociateColorComponents()
    {
      Image<RGB8> inputImage = readPPM8(getTestImageFileNamePPM0());
      Array2D<UnsignedInt8> dataArray = dissociateColorComponents(inputImage);
      Image<RGB8> imageAlias = associateColorComponents<RGB8>(dataArray);
      DLR_TEST_ASSERT(imageAlias.rows() == inputImage.rows());
      DLR_TEST_ASSERT(imageAlias.columns() == inputImage.columns());
      DLR_TEST_ASSERT(imageAlias.data() == inputImage.data());

      Image<HSV_FLOAT64> hsvImage = convertColorspace<HSV_FLOAT64>(inputImage);
      Array2D<double> dataArray2 = dissociateColorComponents(hsvImage);
      Image<HSV_FLOAT64> imageAlias2 =
        associateColorComponents<HSV_FLOAT64>(dataArray2);
      DLR_TEST_ASSERT(imageAlias2.rows() == hsvImage.rows());
      DLR_TEST_ASSERT(imageAlias2.columns() == hsvImage.columns());
      DLR_TEST_ASSERT(imageAlias2.data() == hsvImage.data());
    }


    void
    UtilitiesTest::
    testConvertColorspace()
    {
      Image<RGB8> inputImage2 = readPPM8(getTestImageFileNamePPM0());
      {
        Image<GRAY8> grayImage = convertColorspace<GRAY8>(inputImage2);
        DLR_TEST_ASSERT(grayImage.rows() == inputImage2.rows());
        DLR_TEST_ASSERT(grayImage.columns() == inputImage2.columns());
        ColorspaceConverter<RGB8, GRAY8> converter;
        for(size_t pixelIndex = 0; pixelIndex < inputImage2.size();
            ++pixelIndex) {
          DLR_TEST_ASSERT(
            grayImage[pixelIndex] == converter(inputImage2[pixelIndex]));
        }
      }

      {
        Image<RGBA8> rgbaImage = convertColorspace<RGBA8>(inputImage2);
        DLR_TEST_ASSERT(rgbaImage.rows() == inputImage2.rows());
        DLR_TEST_ASSERT(rgbaImage.columns() == inputImage2.columns());
        ColorspaceConverter<RGB8, RGBA8> converter;
        for(size_t pixelIndex = 0; pixelIndex < inputImage2.size();
            ++pixelIndex) {
          DLR_TEST_ASSERT(
            rgbaImage[pixelIndex] == converter(inputImage2[pixelIndex]));
        }
      }
    }


    void
    UtilitiesTest::
    testDissociateColorComponents()
    {
      Image<RGB8> inputImage = readPPM8(getTestImageFileNamePPM0());
      Array2D<UnsignedInt8> dataArray = dissociateColorComponents(inputImage);
      DLR_TEST_ASSERT(dataArray.rows() == inputImage.rows());
      DLR_TEST_ASSERT(dataArray.columns() == 3 * inputImage.columns());
      size_t dataArrayIndex = 0;
      for(size_t pixelIndex = 0; pixelIndex < inputImage.size(); ++pixelIndex) {
        DLR_TEST_ASSERT(
          inputImage[pixelIndex].red == dataArray[dataArrayIndex++]);
        DLR_TEST_ASSERT(
          inputImage[pixelIndex].green == dataArray[dataArrayIndex++]);
        DLR_TEST_ASSERT(
          inputImage[pixelIndex].blue == dataArray[dataArrayIndex++]);
      }

      Image<HSV_FLOAT64> hsvImage = convertColorspace<HSV_FLOAT64>(inputImage);
      Array2D<double> dataArray2 = dissociateColorComponents(hsvImage);
      DLR_TEST_ASSERT(dataArray2.rows() == hsvImage.rows());
      DLR_TEST_ASSERT(dataArray2.columns() == 3 * hsvImage.columns());
      size_t dataArray2Index = 0;
      for(size_t pixelIndex = 0; pixelIndex < hsvImage.size(); ++pixelIndex) {
        DLR_TEST_ASSERT(
          hsvImage[pixelIndex].hue == dataArray2[dataArray2Index++]);
        DLR_TEST_ASSERT(
          hsvImage[pixelIndex].saturation == dataArray2[dataArray2Index++]);
        DLR_TEST_ASSERT(
          hsvImage[pixelIndex].value == dataArray2[dataArray2Index++]);
      }

    }
  

    void
    UtilitiesTest::
    testSubsample()
    {
      const size_t imageRows = 2048;
      const size_t imageColumns = 1962;

      Image<RGB8> inputImage(imageRows, imageColumns);
      for(size_t rowIndex = 0; rowIndex < imageRows; ++rowIndex) {
        for(size_t columnIndex = 0; columnIndex < imageColumns; ++columnIndex) {
          PixelRGB8 inputPixel(static_cast<UnsignedInt8>(rowIndex),
                               static_cast<UnsignedInt8>(columnIndex),
                               static_cast<UnsignedInt8>(0));
          inputImage(rowIndex, columnIndex) = inputPixel;
        }
      }


      for(size_t rowStep = 2; rowStep < 4; ++rowStep) {
        for(size_t columnStep = 2; columnStep < 4; ++columnStep) {
          Image<RGB8> outputImage0 = subsample(inputImage, rowStep, columnStep);

          DLR_TEST_ASSERT(
            (outputImage0.rows() - 1) * rowStep + 1
            <= inputImage.rows());
          DLR_TEST_ASSERT(
            (outputImage0.columns() - 1) * columnStep + 1
            <= inputImage.columns());
          DLR_TEST_ASSERT(
            (outputImage0.rows() - 1) * rowStep + 1
            > inputImage.rows() - rowStep);
          DLR_TEST_ASSERT(
            (outputImage0.columns() - 1) * columnStep + 1
            > inputImage.columns() - columnStep);
          
          for(size_t rowIndex = 0; rowIndex < outputImage0.rows();
              ++rowIndex) {
            for(size_t columnIndex = 0; columnIndex < outputImage0.columns();
                ++columnIndex) {
              DLR_TEST_ASSERT(
                outputImage0(rowIndex, columnIndex)
                == inputImage(rowIndex * rowStep, columnIndex * columnStep));
            }
          }
        }
      }
    }


    void
    UtilitiesTest::
    testSupersample()
    {
      const size_t imageRows = 100;
      const size_t imageColumns = 121;

      Image<GRAY8> inputImage0(imageRows, imageColumns);
      Image<RGB8> inputImage1(imageRows, imageColumns);
      for(size_t rowIndex = 0; rowIndex < imageRows; ++rowIndex) {
        for(size_t columnIndex = 0; columnIndex < imageColumns; ++columnIndex) {
          inputImage0(rowIndex, columnIndex) =
            static_cast<UnsignedInt8>(rowIndex + columnIndex);
            
          PixelRGB8 inputPixel1(static_cast<UnsignedInt8>(rowIndex),
                                static_cast<UnsignedInt8>(columnIndex),
                                static_cast<UnsignedInt8>(0));
          inputImage1(rowIndex, columnIndex) = inputPixel1;
        }
      }


      Image<GRAY8> outputImage0 =
        supersample<GRAY8, GRAY8, GRAY16>(inputImage0);
      Image<RGB8> outputImage1 =
        supersample<RGB8, RGB8, RGB16>(inputImage1);

      DLR_TEST_ASSERT(outputImage0.rows() == inputImage0.rows() * 2 - 1);
      DLR_TEST_ASSERT(outputImage0.columns() == inputImage0.columns() * 2 - 1);
      for(size_t rowIndex = 0; rowIndex < outputImage0.rows();
          ++rowIndex) {
        for(size_t columnIndex = 0; columnIndex < outputImage0.columns();
            ++columnIndex) {
          UnsignedInt8 outputValue =
            static_cast<UnsignedInt8>((rowIndex + columnIndex) / 2);
          DLR_TEST_ASSERT(outputImage0(rowIndex, columnIndex) == outputValue);
        }
      }

      DLR_TEST_ASSERT(outputImage1.rows() == inputImage1.rows() * 2 - 1);
      DLR_TEST_ASSERT(outputImage1.columns() == inputImage1.columns() * 2 - 1);
      for(size_t rowIndex = 0; rowIndex < outputImage0.rows();
          ++rowIndex) {
        for(size_t columnIndex = 0; columnIndex < outputImage0.columns();
            ++columnIndex) {
          PixelRGB8 outputPixel = outputImage1(rowIndex, columnIndex);
          DLR_TEST_ASSERT(outputPixel.red == rowIndex / 2);
          DLR_TEST_ASSERT(outputPixel.green == columnIndex / 2);
          DLR_TEST_ASSERT(outputPixel.blue == 0);
        }
      }
    }


    void
    UtilitiesTest::
    testToArray()
    {
      Image<RGB8> inputImage = readPPM8(getTestImageFileNamePPM0());
      Array2D<double> dataArray = toArray<double>(inputImage);

      DLR_TEST_ASSERT(dataArray.rows() == inputImage.rows());
      DLR_TEST_ASSERT(dataArray.columns() == 3 * inputImage.columns());
      size_t dataArrayIndex = 0;
      for(size_t pixelIndex = 0; pixelIndex < inputImage.size(); ++pixelIndex) {
        DLR_TEST_ASSERT(
          inputImage[pixelIndex].red == dataArray[dataArrayIndex++]);
        DLR_TEST_ASSERT(
          inputImage[pixelIndex].green == dataArray[dataArrayIndex++]);
        DLR_TEST_ASSERT(
          inputImage[pixelIndex].blue == dataArray[dataArrayIndex++]);
      }
    }
  
  } // namespace computerVision

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::UtilitiesTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::UtilitiesTest currentTest;

}

#endif
