/**
***************************************************************************
* @file dlrComputerVision/test/quadMapTest.cpp
*
* Source file defining tests for the QuadMap data structure.
*
* Copyright (C) 2009 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrComputerVision/quadMap.h>
#include <dlrNumeric/index2D.h>
#include <dlrNumeric/vector2D.h>
#include <dlrNumeric/utilities.h>
#include <dlrTest/testFixture.h>

namespace num = dlr::numeric;


namespace dlr {

  namespace computerVision {

    class QuadMapTest
      : public TestFixture<QuadMapTest> {

    public:

      QuadMapTest();
      ~QuadMapTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      // Tests.
      void testAddElements();
      void testFind();
      void testFindNearest();
    
    private:

      num::Vector2D const&
      findNearest(num::Vector2D const& point,
                  std::vector<num::Vector2D> const& candidateVector,
                  double& distance);


      double m_defaultTolerance;
    
    }; // class QuadMapTest


    /* ========= Specialization of QuadMapComparator to work with ========= */
    /* ===========         dlr::numeric::Array1D<int>          ============ */

    template <>
    bool
    QuadMapComparator< dlr::numeric::Array1D<int> >::
    isEqual(dlr::numeric::Array1D<int> const& arg0,
            dlr::numeric::Array1D<int> const& arg1) const
    {
      if(arg0.size() != arg1.size()) {
        return false;
      }
      for(size_t ii = 0; ii < arg0.size(); ++ii) {
        if(arg0[ii] != arg1[ii]) {
          return false;
        }
      }
      return true;
    }

    
    /* ============ QuadMapTest Member Function Definititions ============ */

    QuadMapTest::
    QuadMapTest()
      : TestFixture<QuadMapTest>("QuadMapTest"),
        m_defaultTolerance(1.0E-10)
    {
      DLR_TEST_REGISTER_MEMBER(testAddElements);
      DLR_TEST_REGISTER_MEMBER(testFind);
      DLR_TEST_REGISTER_MEMBER(testFindNearest);
    }


    void
    QuadMapTest::
    testAddElements()
    {
      std::vector<num::Index2D> inPoints;
      std::vector<num::Index2D> outPoints;

      inPoints.push_back(num::Index2D(2, 3));
      inPoints.push_back(num::Index2D(-4, 2));
      inPoints.push_back(num::Index2D(1, -6));
      inPoints.push_back(num::Index2D(-1, -1));
      inPoints.push_back(num::Index2D(2, 3));
      inPoints.push_back(num::Index2D(2, 1));
      inPoints.push_back(num::Index2D(7, 4));
      inPoints.push_back(num::Index2D(-2, 6));
      inPoints.push_back(num::Index2D(7, -6));
      inPoints.push_back(num::Index2D(0, 0));
      inPoints.push_back(num::Index2D(3, 4));
      inPoints.push_back(num::Index2D(2, 1));
      inPoints.push_back(num::Index2D(3, 6));
      inPoints.push_back(num::Index2D(-2, 6));

      outPoints.push_back(num::Index2D(1, 3));
      outPoints.push_back(num::Index2D(5, 4));
      outPoints.push_back(num::Index2D(7, 2));
      outPoints.push_back(num::Index2D(2, 2));
      outPoints.push_back(num::Index2D(2, 7));

      QuadMap<num::Index2D> quadMap(-8, -8, 8, 8, 5);
      quadMap.addElements(inPoints.begin(), inPoints.end());

      for(size_t ii = 0; ii < inPoints.size(); ++ii) {
        DLR_TEST_ASSERT(quadMap.find(inPoints[ii]));
      }
      for(size_t ii = 0; ii < outPoints.size(); ++ii) {
        DLR_TEST_ASSERT(!(quadMap.find(outPoints[ii])));
      }
    }


    void
    QuadMapTest::
    testFind()
    {
      std::vector< num::Array1D<int> > inPoints;
      std::vector< num::Array1D<int> > outPoints;

      inPoints.push_back(num::Array1D<int>("[2, 3]"));
      inPoints.push_back(num::Array1D<int>("[5, 3]"));
      inPoints.push_back(num::Array1D<int>("[2, 1]"));
      inPoints.push_back(num::Array1D<int>("[7, 4]"));
      inPoints.push_back(num::Array1D<int>("[-2, 6]"));
      inPoints.push_back(num::Array1D<int>("[7, -6]"));
      inPoints.push_back(num::Array1D<int>("[0, 0]"));
      inPoints.push_back(num::Array1D<int>("[3, 4]"));
      inPoints.push_back(num::Array1D<int>("[2, 1]"));
      inPoints.push_back(num::Array1D<int>("[3, 6]"));
      inPoints.push_back(num::Array1D<int>("[-2, 6]"));

      outPoints.push_back(num::Array1D<int>("[1, 3]"));
      outPoints.push_back(num::Array1D<int>("[5, 4]"));
      outPoints.push_back(num::Array1D<int>("[7, 2]"));
      outPoints.push_back(num::Array1D<int>("[2, 2]"));
      outPoints.push_back(num::Array1D<int>("[2, 7]"));

      QuadMap< num::Array1D<int> > quadMap(-8, -8, 8, 8, 5);
      quadMap.addElements(inPoints.begin(), inPoints.end());

      for(size_t ii = 0; ii < inPoints.size(); ++ii) {
        DLR_TEST_ASSERT(quadMap.find(inPoints[ii]));
      }
      for(size_t ii = 0; ii < outPoints.size(); ++ii) {
        DLR_TEST_ASSERT(!(quadMap.find(outPoints[ii])));
      }
    }


    void
    QuadMapTest::
    testFindNearest()
    {
      std::vector<num::Vector2D> inPoints;
      std::vector<num::Vector2D> outPoints;

      inPoints.push_back(num::Vector2D(2.1, 3.5));
      inPoints.push_back(num::Vector2D(5.0, 3.2));
      inPoints.push_back(num::Vector2D(2.4, 1.6));
      inPoints.push_back(num::Vector2D(7.7, 4.7));
      inPoints.push_back(num::Vector2D(-2.0, 6.3));
      inPoints.push_back(num::Vector2D(0.0, 0.0));
      inPoints.push_back(num::Vector2D(3.1, 4.7));
      inPoints.push_back(num::Vector2D(2.2, 1.6));
      inPoints.push_back(num::Vector2D(3.5, 6.9));
      inPoints.push_back(num::Vector2D(-2.2, 6.8));

      outPoints.push_back(num::Vector2D(1.0, 3.4));
      outPoints.push_back(num::Vector2D(5.2, 4.5));
      outPoints.push_back(num::Vector2D(7.3, 2.5));
      outPoints.push_back(num::Vector2D(2.6, 2.8));
      outPoints.push_back(num::Vector2D(2.0, 7.9));

      QuadMap<num::Vector2D> quadMap(-8.0, -8.0, 8.0, 8.0, 5);
      quadMap.addElements(inPoints.begin(), inPoints.end());

      for(size_t ii = 0; ii < outPoints.size(); ++ii) {
        double distance;      
        num::Vector2D nearest = this->findNearest(
          outPoints[ii], inPoints, distance);
        double maybeDistance;
        num::Vector2D maybeNearest = quadMap.findNearest(
          outPoints[ii], maybeDistance);
        DLR_TEST_ASSERT(
          num::magnitude(nearest - maybeNearest) < m_defaultTolerance);
        DLR_TEST_ASSERT(std::fabs(distance - maybeDistance) < m_defaultTolerance);
      }
    }


    num::Vector2D const&
    QuadMapTest::
    findNearest(num::Vector2D const& point,
                std::vector<num::Vector2D> const& candidateVector,
                double& distance)
    {
      num::Vector2D const* nearestPointPtr = &(candidateVector[0]);
      distance = num::magnitudeSquared(point - candidateVector[0]);
      for(size_t ii = 1; ii < candidateVector.size(); ++ii) {
        double newDistance = num::magnitudeSquared(point - candidateVector[ii]);
        if(newDistance < distance) {
          distance = newDistance;
          nearestPointPtr = &candidateVector[ii];
        }
      }
      return *nearestPointPtr;
    }
  
  } // namespace computerVision
  
} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::QuadMapTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::QuadMapTest currentTest;
  
}

#endif
