/**
***************************************************************************
* @file imageWarperTest.cpp
*
* Source file defining tests for the ImageWarper class template.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrComputerVision/imageWarper.h>
#include <dlrTest/testFixture.h>

namespace num = dlr::numeric;

namespace dlr {

  namespace computerVision {
    
    class ImageWarperTest
      : public TestFixture<ImageWarperTest> {

    public:

      ImageWarperTest();
      ~ImageWarperTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      // Tests.
      void testImageWarper();
      void testImageWarperRGB();

    private:

      struct ShiftWarpFunctor {
        ShiftWarpFunctor(double xShift, double yShift)
          : m_shift(xShift, yShift) {}
        
        num::Vector2D
        operator()(num::Vector2D const& arg) const {return arg + m_shift;}

        num::Vector2D m_shift;
      };

      struct StretchXWarpFunctor {
        StretchXWarpFunctor(double stretchFactor)
          : m_factor(stretchFactor) {}
        
        num::Vector2D
        operator()(num::Vector2D const& arg) const {
          return Vector2D(arg.x() / m_factor, arg.y());
        }

        double m_factor;
      };

      struct StretchYWarpFunctor {
        StretchYWarpFunctor(double stretchFactor)
          : m_factor(stretchFactor) {}

        num::Vector2D
        operator()(num::Vector2D const& arg) const {
          return Vector2D(arg.x(), arg.y() / m_factor);
        }

        double m_factor;
      };


      template <ImageFormat Format>
      bool
      isInBounds(num::Vector2D const& coordinate,
                 Image<Format> const& image);
      
        
      double m_defaultTolerance;
      float m_defaultFloatTolerance;
      
    }; // class ImageWarperTest


    /* ============== Member Function Definititions ============== */

    ImageWarperTest::
    ImageWarperTest()
      : TestFixture<ImageWarperTest>("ImageWarperTest"),
        m_defaultTolerance(1.0E-10),
        m_defaultFloatTolerance(1.0E-6)
    {
      DLR_TEST_REGISTER_MEMBER(testImageWarper);
      DLR_TEST_REGISTER_MEMBER(testImageWarperRGB);
    }


    void
    ImageWarperTest::
    testImageWarper()
    {
      Image<GRAY_FLOAT64> xImage(4, 3);
      Image<GRAY_FLOAT64> yImage(4, 3);

      for(size_t row = 0; row < xImage.rows(); ++row) {
        for(size_t column = 0; column < xImage.columns(); ++column) {
          xImage(row, column) = column;
          yImage(row, column) = row;
        }
      }

      size_t outputRows = 5 * xImage.rows();
      size_t outputColumns = 6 * xImage.rows();
      
      double defaultValue = -1.0;
      double xShift = -1.2;
      double yShift = -2.5;
      double scaleFactor = 4.1;
      ShiftWarpFunctor shiftWarpFunctor(xShift, yShift);
      StretchXWarpFunctor stretchXWarpFunctor(scaleFactor);
      StretchYWarpFunctor stretchYWarpFunctor(scaleFactor);
      
      ImageWarper<double, ShiftWarpFunctor> shiftWarper(
        xImage.rows(), xImage.columns(), outputRows, outputColumns,
        shiftWarpFunctor);
      ImageWarper<double, StretchXWarpFunctor> xStretchWarper(
        xImage.rows(), xImage.columns(), outputRows, outputColumns,
        stretchXWarpFunctor);
      ImageWarper<double, StretchYWarpFunctor> yStretchWarper(
        xImage.rows(), xImage.columns(), outputRows, outputColumns,
        stretchYWarpFunctor);

      Image<GRAY_FLOAT64> shiftedXImage =
        shiftWarper.warpImage<GRAY_FLOAT64, GRAY_FLOAT64>(
          xImage, defaultValue);
      Image<GRAY_FLOAT64> shiftedYImage =
        shiftWarper.warpImage<GRAY_FLOAT64, GRAY_FLOAT64>(
          yImage, defaultValue);
      Image<GRAY_FLOAT64> xStretchedXImage =
        xStretchWarper.warpImage<GRAY_FLOAT64, GRAY_FLOAT64>(
          xImage, defaultValue);
      Image<GRAY_FLOAT64> xStretchedYImage =
        xStretchWarper.warpImage<GRAY_FLOAT64, GRAY_FLOAT64>(
          yImage, defaultValue);
      Image<GRAY_FLOAT64> yStretchedXImage =
        yStretchWarper.warpImage<GRAY_FLOAT64, GRAY_FLOAT64>(
          xImage, defaultValue);
      Image<GRAY_FLOAT64> yStretchedYImage =
        yStretchWarper.warpImage<GRAY_FLOAT64, GRAY_FLOAT64>(
          yImage, defaultValue);

      for(size_t row = 0; row < outputRows; ++row) {
        for(size_t column = 0; column < outputColumns; ++column) {
          if(this->isInBounds(
               shiftWarpFunctor(Vector2D(column, row)), xImage)) {
            DLR_TEST_ASSERT(
              approximatelyEqual(shiftedXImage(row, column), column + xShift,
                                 m_defaultTolerance));
            DLR_TEST_ASSERT(
              approximatelyEqual(shiftedYImage(row, column), row + yShift,
                                 m_defaultTolerance));
          } else {
            DLR_TEST_ASSERT(shiftedXImage(row, column) == defaultValue);
            DLR_TEST_ASSERT(shiftedYImage(row, column) == defaultValue);
          }
          if(this->isInBounds(
               stretchXWarpFunctor(Vector2D(column, row)), xImage)) {
            DLR_TEST_ASSERT(
              approximatelyEqual(xStretchedXImage(row, column),
                                 column / scaleFactor,
                                 m_defaultTolerance));
            DLR_TEST_ASSERT(
              approximatelyEqual(xStretchedYImage(row, column),
                                 static_cast<double>(row),
                                 m_defaultTolerance));
          } else {
            DLR_TEST_ASSERT(xStretchedXImage(row, column) == defaultValue);
            DLR_TEST_ASSERT(xStretchedYImage(row, column) == defaultValue);
          }
          if(this->isInBounds(
               stretchYWarpFunctor(Vector2D(column, row)), xImage)) {
            DLR_TEST_ASSERT(
              approximatelyEqual(yStretchedXImage(row, column),
                                 static_cast<double>(column),
                                 m_defaultTolerance));
            DLR_TEST_ASSERT(
              approximatelyEqual(yStretchedYImage(row, column),
                                 row / scaleFactor,
                                 m_defaultTolerance));
          } else {
            DLR_TEST_ASSERT(yStretchedXImage(row, column) == defaultValue);
            DLR_TEST_ASSERT(yStretchedYImage(row, column) == defaultValue);
          }
        }
      }
    }


    void
    ImageWarperTest::
    testImageWarperRGB()
    {
      Image<RGB_FLOAT32> xImage(4, 3);
      Image<RGB_FLOAT32> yImage(4, 3);

      for(size_t row = 0; row < xImage.rows(); ++row) {
        for(size_t column = 0; column < xImage.columns(); ++column) {
          xImage(row, column).red = column;
          xImage(row, column).green = column + 5;
          xImage(row, column).blue = 2 * column;
          yImage(row, column).red = row;
          yImage(row, column).green = row + 10;
          yImage(row, column).blue = 3 * row;
        }
      }

      size_t outputRows = 5 * xImage.rows();
      size_t outputColumns = 6 * xImage.rows();
      
      PixelRGBFloat32 defaultValue(-1.0, -1.0, -1.0);
      double xShift = -1.2;
      double yShift = -2.5;
      double scaleFactor = 4.1;
      ShiftWarpFunctor shiftWarpFunctor(xShift, yShift);
      StretchXWarpFunctor stretchXWarpFunctor(scaleFactor);
      StretchYWarpFunctor stretchYWarpFunctor(scaleFactor);
      
      ImageWarper<Float32, ShiftWarpFunctor> shiftWarper(
        xImage.rows(), xImage.columns(), outputRows, outputColumns,
        shiftWarpFunctor);
      ImageWarper<Float32, StretchXWarpFunctor> xStretchWarper(
        xImage.rows(), xImage.columns(), outputRows, outputColumns,
        stretchXWarpFunctor);
      ImageWarper<Float32, StretchYWarpFunctor> yStretchWarper(
        xImage.rows(), xImage.columns(), outputRows, outputColumns,
        stretchYWarpFunctor);

      Image<RGB_FLOAT32> shiftedXImage =
        shiftWarper.warpImage<RGB_FLOAT32, RGB_FLOAT32>(
          xImage, defaultValue);
      Image<RGB_FLOAT32> shiftedYImage =
        shiftWarper.warpImage<RGB_FLOAT32, RGB_FLOAT32>(
          yImage, defaultValue);
      Image<RGB_FLOAT32> xStretchedXImage =
        xStretchWarper.warpImage<RGB_FLOAT32, RGB_FLOAT32>(
          xImage, defaultValue);
      Image<RGB_FLOAT32> xStretchedYImage =
        xStretchWarper.warpImage<RGB_FLOAT32, RGB_FLOAT32>(
          yImage, defaultValue);
      Image<RGB_FLOAT32> yStretchedXImage =
        yStretchWarper.warpImage<RGB_FLOAT32, RGB_FLOAT32>(
          xImage, defaultValue);
      Image<RGB_FLOAT32> yStretchedYImage =
        yStretchWarper.warpImage<RGB_FLOAT32, RGB_FLOAT32>(
          yImage, defaultValue);

      for(size_t row = 0; row < outputRows; ++row) {
        for(size_t column = 0; column < outputColumns; ++column) {
          if(this->isInBounds(
               shiftWarpFunctor(Vector2D(column, row)), xImage)) {
            DLR_TEST_ASSERT(
              approximatelyEqual(
                shiftedXImage(row, column).red, Float32(column + xShift),
                m_defaultFloatTolerance));
            DLR_TEST_ASSERT(
              approximatelyEqual(
                shiftedYImage(row, column).red, Float32(row + yShift),
                m_defaultFloatTolerance));
          } else {
            DLR_TEST_ASSERT(shiftedXImage(row, column) == defaultValue);
            DLR_TEST_ASSERT(shiftedYImage(row, column) == defaultValue);
          }
          if(this->isInBounds(
               stretchXWarpFunctor(Vector2D(column, row)), xImage)) {
            DLR_TEST_ASSERT(
              approximatelyEqual(
                xStretchedXImage(row, column).red,
                Float32(column / scaleFactor),
                m_defaultFloatTolerance));
            DLR_TEST_ASSERT(
              approximatelyEqual(
                xStretchedYImage(row, column).red,
                static_cast<Float32>(row),
                m_defaultFloatTolerance));
          } else {
            DLR_TEST_ASSERT(xStretchedXImage(row, column) == defaultValue);
            DLR_TEST_ASSERT(xStretchedYImage(row, column) == defaultValue);
          }
          if(this->isInBounds(
               stretchYWarpFunctor(Vector2D(column, row)), xImage)) {
            DLR_TEST_ASSERT(
              approximatelyEqual(
                yStretchedXImage(row, column).red, static_cast<Float32>(column),
                m_defaultFloatTolerance));
            DLR_TEST_ASSERT(
              approximatelyEqual(
                yStretchedYImage(row, column).red, Float32(row / scaleFactor),
                m_defaultFloatTolerance));
          } else {
            DLR_TEST_ASSERT(yStretchedXImage(row, column) == defaultValue);
            DLR_TEST_ASSERT(yStretchedYImage(row, column) == defaultValue);
          }
        }
      }
    }


    template <ImageFormat Format>
    bool
    ImageWarperTest::
    isInBounds(num::Vector2D const& coordinate,
               Image<Format> const& image)
    {
      return ((coordinate.x() >= 0.0)
              && (coordinate.y() >= 0.0)
              && (coordinate.x() < (image.columns() - 1))
              && (coordinate.y() < (image.rows() - 1)));
    }
    
    
  } // namespace computerVision

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::ImageWarperTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::ImageWarperTest currentTest;

}

#endif

