/**
***************************************************************************
* @file calibrationToolsTest.cpp
*
* Source file defining tests for the calibration utility routines.
*
* Copyright (C) 2009 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrCommon/functional.h>
#include <dlrComputerVision/calibrationTools.h>
#include <dlrTest/testFixture.h>

using namespace dlr::common;
using namespace dlr::computerVision;
using namespace dlr::geometry;
using namespace dlr::numeric;
using namespace dlr::test;


class CalibrationToolsTest
  : public TestFixture<CalibrationToolsTest> {

public:

  CalibrationToolsTest();
  ~CalibrationToolsTest() {}

  void setUp(const std::string& testName) {}
  void tearDown(const std::string& testName) {}

  // Tests.
  void testEstimateCameraIntrinsicsPinhole();
  
private:

  double m_defaultTolerance;
  double m_relaxedTolerance;
  
}; // class CalibrationToolsTest


/* ============== Member Function Definititions ============== */

CalibrationToolsTest::
CalibrationToolsTest()
  : TestFixture<CalibrationToolsTest>("CalibrationToolsTest"),
    m_defaultTolerance(1.0E-10),
    m_relaxedTolerance(1.0E-7)
{
  DLR_TEST_REGISTER_MEMBER(testEstimateCameraIntrinsicsPinhole);
}


void
CalibrationToolsTest::
testEstimateCameraIntrinsicsPinhole()
{
  // Arbitrary camera params.
  size_t numPixelsX = 320;
  size_t numPixelsY = 240;
  double focalLength = 0.03;
  double pixelSizeX = 0.001;
  double pixelSizeY = 0.002;
  double centerU = 100.0;
  double centerV = 125.0;
  CameraIntrinsicsPinhole intrinsics(
    numPixelsX, numPixelsY, focalLength, pixelSizeX, pixelSizeY,
    centerU, centerV);

  std::vector<Vector3D> points3D;
  std::vector<Vector2D> points2D;
  for(double zCoord = 1.0; zCoord < 10.0; zCoord += 0.7) {
    for(double yCoord = -1.0; yCoord < 1.0; yCoord += 0.1) {
      for(double xCoord = -1.0; xCoord < 1.0; xCoord += 0.1) {
        Vector3D cameraCoord(xCoord, yCoord, zCoord);
        Vector2D pixelCoord = intrinsics.project(cameraCoord);
        points3D.push_back(cameraCoord);
        points2D.push_back(pixelCoord);
      }
    }
  }

  CameraIntrinsicsPinhole recoveredIntrinsics = estimateCameraIntrinsicsPinhole(
    numPixelsX, numPixelsY, points3D.begin(), points3D.end(), points2D.begin());

  DLR_TEST_ASSERT(
    recoveredIntrinsics.getNumPixelsX() == intrinsics.getNumPixelsX());
  DLR_TEST_ASSERT(
    recoveredIntrinsics.getNumPixelsY() == intrinsics.getNumPixelsY());
  DLR_TEST_ASSERT(approximatelyEqual(
                    recoveredIntrinsics.getKx(), intrinsics.getKx(),
                    m_relaxedTolerance));
  DLR_TEST_ASSERT(approximatelyEqual(
                    recoveredIntrinsics.getKy(), intrinsics.getKy(),
                    m_relaxedTolerance));
  DLR_TEST_ASSERT(approximatelyEqual(
                    recoveredIntrinsics.getCenterU(), intrinsics.getCenterU(),
                    m_relaxedTolerance));
  DLR_TEST_ASSERT(approximatelyEqual(
                    recoveredIntrinsics.getCenterV(), intrinsics.getCenterV(),
                    m_relaxedTolerance));
}


#if 0

int main(int argc, char** argv)
{
  CalibrationToolsTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  CalibrationToolsTest currentTest;

}

#endif
