/**
***************************************************************************
* @file dlrComputerVision/fivePointAlgorithm.cpp
*
* Source file declaring the fivePointAlgorithm() function template.
*
* Copyright (C) 2009 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#include <dlrComputerVision/fivePointAlgorithm.h>
#include <dlrGeometry/utilities3D.h>

namespace geom = dlr::geometry;
namespace num = dlr::numeric;

namespace dlr {

  namespace computerVision {


    double
    checkEpipolarConstraint(dlr::numeric::Array2D<double> const& fundamentalMx,
                            dlr::numeric::Vector2D& point0,
                            dlr::numeric::Vector2D& point1)
    {
      // Compute matrix-vector product F * q.
      double lineCoefficient0 = 
        fundamentalMx[0] * point0.x() + fundamentalMx[1] * point0.y()
        + fundamentalMx[2];
      double lineCoefficient1 = 
        fundamentalMx[3] * point0.x() + fundamentalMx[4] * point0.y()
        + fundamentalMx[5];
      double lineCoefficient2 = 
        fundamentalMx[6] * point0.x() + fundamentalMx[7] * point0.y()
        + fundamentalMx[8];

      // Get the epipolar line corresponding to point0.
      geom::Ray2D ray(lineCoefficient0, lineCoefficient1, lineCoefficient2);

      // Find the squared distance between q' and this ray.
      dlr::Vector2D closestPoint = geom::findClosestPoint(point1, ray);
      return num::magnitudeSquared(closestPoint - point1);
    }
    
    
    dlr::numeric::Transform3D
    getCameraMotionFromEssentialMatrix(
      dlr::numeric::Array2D<double> const& EE,
      dlr::numeric::Vector2D const& testPointCamera0,
      dlr::numeric::Vector2D const& testPointCamera1)
    {
      // Note(xxx): Need to put a comment here explaining why this
      // works.  Ref R. Hartley and A. Zisserman, Multiple View
      // Geometry in Computer Vision, Cambridge University Press, ISBN
      // 0-521-62304-9, 2000.  Or ref R. Tsai and T. Huang, Uniqueness
      // and Estimation of Three-Dimensional Motion Parameters of
      // Rigid Objects with Curved Surfaces, IEEE Transactions on
      // Pattern Analysis and Machine Intelligence, 6(1):13-27, 1984.
      num::Array2D<Float64> uArray;
      num::Array1D<Float64> sigmaArray;
      num::Array2D<Float64> vTransposeArray;
      dlr::linearAlgebra::singularValueDecomposition(
        EE, uArray, sigmaArray, vTransposeArray);

#if 0
      // Nister's paper outlines the following steps, which are faster
      // than what we're actually using below, and exactly equivalent,
      // but I haven't had time to test and make sure I've implemented
      // them correctly.  Note that the right-handedness check is
      // missing from this code.
      
      // Multiply by D == [[0, 1, 0], [-1, 0, 0], [0, 0, 1]].
      vTransposeArray.getRow(0) *= -1.0;
      std::swap(vTransposeArray[0], vTransposeArray[3]);
      std::swap(vTransposeArray[1], vTransposeArray[4]);
      std::swap(vTransposeArray[2], vTransposeArray[5]);
      num::Array2D<Float64> rotation0 = matrixMultiply(uArray, vTransposeArray);

      num::Transform3D c1Tc0(
        rotation0[0], rotation0[1], rotation0[2], uArray(0, 2),
        rotation0[3], rotation0[4], rotation0[5], uArray(1, 2),
        rotation0[6], rotation0[7], rotation0[8], uArray(2, 2),
        0.0,          0.0,          0.0,          1.0);

      // Note: this call might throw...
      num::Vector3D testPoint3D0 = triangulateCalibratedImagePoint(
        c1Tc0, testPointCamera1, testPointCamera0);
      num::Vector3D testPoint3D1 = c1Tc0 * testPoint3D0;
      if((testPoint3D0.z() < 0.0) && (testPoint3D1.z() < 0.0)) {
        // 3D position is behind both cameras.
        c1Tc0.setTransform(
          -rotation0[0], -rotation0[1], -rotation0[2], -uArray(0, 2),
          -rotation0[3], -rotation0[4], -rotation0[5], -uArray(1, 2),
          -rotation0[6], -rotation0[7], -rotation0[8], -uArray(2, 2),
          0.0,          0.0,          0.0,          1.0);
      } else if((testPoint3D0.z() * testPoint3D1.z()) < 0.0) {
        // 3D position is behind one camera, but not both.
        Transform3D Ht(-1.0, 0.0, 0.0, 0.0,
                       0.0, -1.0, 0.0, 0.0,
                       0.0, 0.0, -1.0, 0.0,
                       2*vTransposeArray(2, 0), 2*vTransposeArray(2, 1),
                       2*vTransposeArray(2, 1), 1.0);
        c1Tc0 = Ht * c1Tc0;
        testPoint3D0 = Ht * testPoint3D0;
        if((testPoint3D0.z() < 0.0) || (testPoint3D1.z() < 0.0)) {
          c1Tc0.setTransform(
            -c1Tc0.value<0, 0>(), -c1Tc0.value<0, 1>(),
            -c1Tc0.value<0, 2>(), -c1Tc0.value<0, 3>(), 
            -c1Tc0.value<1, 0>(), -c1Tc0.value<1, 1>(),
            -c1Tc0.value<1, 2>(), -c1Tc0.value<1, 3>(), 
            -c1Tc0.value<2, 0>(), -c1Tc0.value<2, 1>(),
            -c1Tc0.value<2, 2>(), -c1Tc0.value<2, 3>(),
            0.0, 0.0, 0.0, 1.0);            
        }
      }
      return c1Tc0;
#else
      // Here's a bonehead version of the above.
      
      // Multiply by D == [[0, 1, 0], [-1, 0, 0], [0, 0, 1]].
      vTransposeArray.getRow(0) *= -1.0;
      std::swap(vTransposeArray[0], vTransposeArray[3]);
      std::swap(vTransposeArray[1], vTransposeArray[4]);
      std::swap(vTransposeArray[2], vTransposeArray[5]);
      num::Array2D<Float64> rotation0 = matrixMultiply(uArray, vTransposeArray);

      // Undo previous contortion, and multiply by D == [[0, -1, 0],
      // [1, 0, 0], [0, 0, 1]].
      vTransposeArray.getRow(0) *= -1.0;
      vTransposeArray.getRow(1) *= -1.0;
      num::Array2D<Float64> rotation1 = matrixMultiply(uArray, vTransposeArray);

      // SVD could easily return -1 * the rotatin we want.  Of course, this
      // would make the resulting coordinate system left handed.  Check for
      // this here.
      num::Vector3D e0(rotation0(0, 0), rotation0(1, 0), rotation0(2, 0));
      num::Vector3D e1(rotation0(0, 1), rotation0(1, 1), rotation0(2, 1));
      num::Vector3D e2(rotation0(0, 2), rotation0(1, 2), rotation0(2, 2));
      bool isRightHanded = num::dot(num::cross(e0, e1), e2) > 0.0;
      if(!isRightHanded) {
        rotation0 *= -1.0;
        rotation1 *= -1.0;
      }

      // Now we have some candidates for rotation.  We'll test them
      // using the test point that was passed in.
      num::Transform3D c1Tc0_0(
        rotation0[0], rotation0[1], rotation0[2], uArray(0, 2),
        rotation0[3], rotation0[4], rotation0[5], uArray(1, 2),
        rotation0[6], rotation0[7], rotation0[8], uArray(2, 2),
        0.0,          0.0,          0.0,          1.0);
      num::Transform3D c1Tc0_1(
        rotation0[0], rotation0[1], rotation0[2], -uArray(0, 2),
        rotation0[3], rotation0[4], rotation0[5], -uArray(1, 2),
        rotation0[6], rotation0[7], rotation0[8], -uArray(2, 2),
        0.0,          0.0,          0.0,          1.0);

      num::Vector3D testPoint3D1 = triangulateCalibratedImagePoint(
        c1Tc0_0, testPointCamera1, testPointCamera0);
      num::Vector3D testPoint3D0 = c1Tc0_0.invert() * testPoint3D1;
      if((testPoint3D0.z() > 0.0) && (testPoint3D1.z() > 0.0)) {
        return c1Tc0_0;
      }
      if((testPoint3D0.z() < 0.0) && (testPoint3D1.z() < 0.0)) {
        return c1Tc0_1;
      }

      num::Transform3D c1Tc0_2(
        rotation1[0], rotation1[1], rotation1[2], uArray(0, 2),
        rotation1[3], rotation1[4], rotation1[5], uArray(1, 2),
        rotation1[6], rotation1[7], rotation1[8], uArray(2, 2),
        0.0,          0.0,          0.0,          1.0);
      num::Transform3D c1Tc0_3(
        rotation1[0], rotation1[1], rotation1[2], -uArray(0, 2),
        rotation1[3], rotation1[4], rotation1[5], -uArray(1, 2),
        rotation1[6], rotation1[7], rotation1[8], -uArray(2, 2),
        0.0,          0.0,          0.0,          1.0);
      // Note: this call might throw...
      testPoint3D1 = triangulateCalibratedImagePoint(
        c1Tc0_2, testPointCamera1, testPointCamera0);
      testPoint3D0 = c1Tc0_2.invert() * testPoint3D1;
      if((testPoint3D0.z() > 0.0) && (testPoint3D1.z() > 0.0)) {
        return c1Tc0_2;
      }
      return c1Tc0_3;
#endif
    }
    

    dlr::numeric::Vector3D
    triangulateCalibratedImagePoint(
      dlr::numeric::Transform3D const& c0Tc1,
      dlr::numeric::Vector2D const& testPointCamera0,
      dlr::numeric::Vector2D const& testPointCamera1)
    {
      // This triangulation is much less efficient that the approach
      // described in [2], but is conceptually clearer, and has nicer
      // error characteristics.

      // Each test point defines a ray in the coordinate system of the
      // corresponding camera.
      geom::Ray3D ray0_0(
        num::Vector3D(0.0, 0.0, 0.0),
        num::Vector3D(testPointCamera0.x(), testPointCamera0.y(), 1.0));
      geom::Ray3D ray1_1(
        num::Vector3D(0.0, 0.0, 0.0),
        num::Vector3D(testPointCamera1.x(), testPointCamera1.y(), 1.0));

      // Transform one of the two rays from camera1 coordinates to
      // camera0 coordinates.
      geom::Ray3D ray1_0 = c0Tc1 * ray1_1;

      // And find the closest point of approach between the two rays.
      double distance0;
      double distance1;
      double residual;
      return geom::findIntersect(
        ray0_0, ray1_0, distance0, distance1, residual);
    }
    
    
    // This function is used internally by fivePointAlgorithm() to
    // generate a 10x20 matrix of coefficients of polynomial
    // constraints.
    dlr::numeric::Array2D<double>
    generateFivePointConstraintMatrix(
      dlr::numeric::Array2D<double> const& E0Array,
      dlr::numeric::Array2D<double> const& E1Array,
      dlr::numeric::Array2D<double> const& E2Array,
      dlr::numeric::Array2D<double> const& E3Array)
    {

      /*
        =====================================================
        ============== Start of long comment ================
        =====================================================

        # Here's maxima code that automatically generates our
        # constraint equations.  Save this as maximaCode.ma, and
        # execute the command
        #
        #   maxima -b maximaCode.ma
        #
        # to generate an intermedate text file called
        # maximaConstraints.txt, then procede to the python code
        # below.

        # ---------------- Start of Maxima code ------------------
        
        E0:matrix([e0rc00, e0rc01, e0rc02],
                  [e0rc10, e0rc11, e0rc12],
                  [e0rc20, e0rc21, e0rc22]);
        E1:matrix([e1rc00, e1rc01, e1rc02],
                  [e1rc10, e1rc11, e1rc12],
                  [e1rc20, e1rc21, e1rc22]);
        E2:matrix([e2rc00, e2rc01, e2rc02],
                  [e2rc10, e2rc11, e2rc12],
                  [e2rc20, e2rc21, e2rc22]);
        E3:matrix([e3rc00, e3rc01, e3rc02],
                  [e3rc10, e3rc11, e3rc12],
                  [e3rc20, e3rc21, e3rc22]);
        EE:x0*E0 + x1*E1 + x2*E2 + E3;

        C0:determinant(EE);
        C0E:expand(C0);
        
        EExEET:EE . transpose(EE);
        ETrace:EExEET[1][1] + EExEET[2][2] + EExEET[3][3];
        CC:2 * (EExEET . EE) - (ETrace * EE);
        CCE:expand(CC);

        Constraints:[CCE[1][1], CCE[1][2], CCE[1][3],
                     CCE[2][1], CCE[2][2], CCE[2][3],
                     CCE[3][1], CCE[3][2], CCE[3][3],
                     C0E];

        appendfile("maximaConstraints.txt");
        grind(Constraints);
        
        # ----------------- End of Maxima code -------------------

        # Here's python code that parses maximaConstraints.txt
        # (generated above) to create code for computing the
        # constraints matrix.  Save this code as pythonCode.py, and
        # execute the command
        #
        #   python pythonCode.py > cppSnippet.cpp
        # 
        # Then copy the contents of cppSnippet.cpp into the cut &
        # paste section below.

        # ---------------- Start of Python code ------------------
        import re
        constraintsRe = re.compile("\[.*\]", re.DOTALL)
        
        # This regex works as follows:
        #   (                  - Starts a group representing the coefficient.
        #     [+-]?              - Optional prefix of '+' or '-', as in
        #                          '-2*e0rc00'.
        #     (?:\d\*)?          - Optional leading multiplier, as in
        #                          '2*e0rc00'.
        #     e[e0123rc^*]*[0123] - A product of e?rc?? terms, as in
        #                           'e0rc00^2*e1rc21'.
        #   )                  - Ends the group representing the coefficient.
        #   (?:                - Starts a group.
        #     \*                 - Group has to start with a "*", as in
        #                          '*x0^3'.
        #     (x[x0123*^]*)      - A group representing the monomial, as in
        #                          'x0^2*x1'.
        #   )?                 - Closes the group, and makes it optional.
        termRe = re.compile(
          "([+-]?(?:\d\*)?e[e0123rc^*]*[0123])(?:\*(x[x0123*^]*))?")
        expectedMonomials = ['x0^3', 'x0^2*x1', 'x0^2*x2',
                             'x0*x1^2', 'x0*x1*x2', 'x0*x2^2',
                             'x1^3', 'x1^2*x2', 'x1*x2^2',
                             'x2^3',
                             'x0^2', 'x0*x1', 'x0*x2',
                             'x1^2', 'x1*x2', 'x2^2',
                             'x0', 'x1', 'x2',
                             None]
        
        inputFile = open('maximaConstraints.txt')
        inputText = inputFile.read()
        
        # This array represents the 10x20 constraint matrix.
        coefficientArray = map(lambda x: [''] * 20, range(10))
        
        # We're expecting to find 10 equations: 9 polynomials for the nine
        # terms in the "2 * E * transpose(E) * E ..." constraint, and one
        # polynomial for the determinant constraint.
        constraintsText = constraintsRe.search(inputText).group(0)[1:-1]
        constraintsList = constraintsText.split(',')
        
        # Iterate over all constraints.
        constraintNumber = 0
        for constraint in constraintsList:
          coefficientMap = {}
          
          # Reset this row of the constraint matrix.
          for monomial in expectedMonomials:
            coefficientMap[monomial] = ''
          # end for
          
          # Examine all terms of this polynomial, split each term into
          # coefficient and monomial parts (where coefficient might be
          # something like "e0rc12^2*e2rc21", and monomial might be
          # something like "x1*x2^2"), and add the coefficient to the
          # total coefficient for the monomial (since there may be
          # more than one term associated with each monomial).
          startPos = 0
          while 1:
            termMatch = termRe.search(constraint, startPos)
            if not termMatch:
              break
            # end if
            coefficient = termMatch.group(1)
            monomial = termMatch.group(2)
            if not coefficientMap.has_key(monomial):
              raise IOError, 'Unexpected monomial'
            # end if
            coefficientMap[monomial] = (
              coefficientMap[monomial] + coefficient.replace('^', '_'))
            startPos = termMatch.end()
          # end while
          
          # Now put the accumulated coefficients into the constraint matrix.
          for ii in range(len(expectedMonomials)):
            coefficient = coefficientMap[expectedMonomials[ii]]
            if coefficient != '':
              coefficientArray[constraintNumber][ii] = coefficient
            else:
              coefficientArray[constraintNumber][ii] = '0.0'
            # end if
          # end for
          constraintNumber = constraintNumber + 1
        # end for
        
        # Send C++ code to standard out.
        for ii in range(len(coefficientArray)):
          for jj in range(len(expectedMonomials)):
            print ('    AMatrix(%d, %d) = %s;'
                   % (ii, jj, coefficientArray[ii][jj]))
          # end for
        # end for
                
        # ----------------- End of Python code -------------------
        
        =====================================================
        ==============  End of long comment  ================
        =====================================================
       */

      // First set up the variables for the auto-generated code...

      // Individual matrix elements.  The name eXrcYZ means the (Y, Z)
      // element of matrix E'X'Array.
      double e0rc00 = E0Array(0, 0);
      double e0rc01 = E0Array(0, 1);
      double e0rc02 = E0Array(0, 2);
      double e0rc10 = E0Array(1, 0);
      double e0rc11 = E0Array(1, 1);
      double e0rc12 = E0Array(1, 2);
      double e0rc20 = E0Array(2, 0);
      double e0rc21 = E0Array(2, 1);
      double e0rc22 = E0Array(2, 2);
      double e1rc00 = E1Array(0, 0);
      double e1rc01 = E1Array(0, 1);
      double e1rc02 = E1Array(0, 2);
      double e1rc10 = E1Array(1, 0);
      double e1rc11 = E1Array(1, 1);
      double e1rc12 = E1Array(1, 2);
      double e1rc20 = E1Array(2, 0);
      double e1rc21 = E1Array(2, 1);
      double e1rc22 = E1Array(2, 2);
      double e2rc00 = E2Array(0, 0);
      double e2rc01 = E2Array(0, 1);
      double e2rc02 = E2Array(0, 2);
      double e2rc10 = E2Array(1, 0);
      double e2rc11 = E2Array(1, 1);
      double e2rc12 = E2Array(1, 2);
      double e2rc20 = E2Array(2, 0);
      double e2rc21 = E2Array(2, 1);
      double e2rc22 = E2Array(2, 2);
      double e3rc00 = E3Array(0, 0);
      double e3rc01 = E3Array(0, 1);
      double e3rc02 = E3Array(0, 2);
      double e3rc10 = E3Array(1, 0);
      double e3rc11 = E3Array(1, 1);
      double e3rc12 = E3Array(1, 2);
      double e3rc20 = E3Array(2, 0);
      double e3rc21 = E3Array(2, 1);
      double e3rc22 = E3Array(2, 2);

      // Squares of individual matrix elements.
      double e0rc00_2 = e0rc00 * e0rc00;
      double e0rc01_2 = e0rc01 * e0rc01;
      double e0rc02_2 = e0rc02 * e0rc02;
      double e0rc10_2 = e0rc10 * e0rc10;
      double e0rc11_2 = e0rc11 * e0rc11;
      double e0rc12_2 = e0rc12 * e0rc12;
      double e0rc20_2 = e0rc20 * e0rc20;
      double e0rc21_2 = e0rc21 * e0rc21;
      double e0rc22_2 = e0rc22 * e0rc22;
      double e1rc00_2 = e1rc00 * e1rc00;
      double e1rc01_2 = e1rc01 * e1rc01;
      double e1rc02_2 = e1rc02 * e1rc02;
      double e1rc10_2 = e1rc10 * e1rc10;
      double e1rc11_2 = e1rc11 * e1rc11;
      double e1rc12_2 = e1rc12 * e1rc12;
      double e1rc20_2 = e1rc20 * e1rc20;
      double e1rc21_2 = e1rc21 * e1rc21;
      double e1rc22_2 = e1rc22 * e1rc22;
      double e2rc00_2 = e2rc00 * e2rc00;
      double e2rc01_2 = e2rc01 * e2rc01;
      double e2rc02_2 = e2rc02 * e2rc02;
      double e2rc10_2 = e2rc10 * e2rc10;
      double e2rc11_2 = e2rc11 * e2rc11;
      double e2rc12_2 = e2rc12 * e2rc12;
      double e2rc20_2 = e2rc20 * e2rc20;
      double e2rc21_2 = e2rc21 * e2rc21;
      double e2rc22_2 = e2rc22 * e2rc22;
      double e3rc00_2 = e3rc00 * e3rc00;
      double e3rc01_2 = e3rc01 * e3rc01;
      double e3rc02_2 = e3rc02 * e3rc02;
      double e3rc10_2 = e3rc10 * e3rc10;
      double e3rc11_2 = e3rc11 * e3rc11;
      double e3rc12_2 = e3rc12 * e3rc12;
      double e3rc20_2 = e3rc20 * e3rc20;
      double e3rc21_2 = e3rc21 * e3rc21;
      double e3rc22_2 = e3rc22 * e3rc22;
      
      // Cubes of individual matrix elements.
      double e0rc00_3 = e0rc00_2 * e0rc00;
      double e0rc01_3 = e0rc01_2 * e0rc01;
      double e0rc02_3 = e0rc02_2 * e0rc02;
      double e0rc10_3 = e0rc10_2 * e0rc10;
      double e0rc11_3 = e0rc11_2 * e0rc11;
      double e0rc12_3 = e0rc12_2 * e0rc12;
      double e0rc20_3 = e0rc20_2 * e0rc20;
      double e0rc21_3 = e0rc21_2 * e0rc21;
      double e0rc22_3 = e0rc22_2 * e0rc22;
      double e1rc00_3 = e1rc00_2 * e1rc00;
      double e1rc01_3 = e1rc01_2 * e1rc01;
      double e1rc02_3 = e1rc02_2 * e1rc02;
      double e1rc10_3 = e1rc10_2 * e1rc10;
      double e1rc11_3 = e1rc11_2 * e1rc11;
      double e1rc12_3 = e1rc12_2 * e1rc12;
      double e1rc20_3 = e1rc20_2 * e1rc20;
      double e1rc21_3 = e1rc21_2 * e1rc21;
      double e1rc22_3 = e1rc22_2 * e1rc22;
      double e2rc00_3 = e2rc00_2 * e2rc00;
      double e2rc01_3 = e2rc01_2 * e2rc01;
      double e2rc02_3 = e2rc02_2 * e2rc02;
      double e2rc10_3 = e2rc10_2 * e2rc10;
      double e2rc11_3 = e2rc11_2 * e2rc11;
      double e2rc12_3 = e2rc12_2 * e2rc12;
      double e2rc20_3 = e2rc20_2 * e2rc20;
      double e2rc21_3 = e2rc21_2 * e2rc21;
      double e2rc22_3 = e2rc22_2 * e2rc22;
      double e3rc00_3 = e3rc00_2 * e3rc00;
      double e3rc01_3 = e3rc01_2 * e3rc01;
      double e3rc02_3 = e3rc02_2 * e3rc02;
      double e3rc10_3 = e3rc10_2 * e3rc10;
      double e3rc11_3 = e3rc11_2 * e3rc11;
      double e3rc12_3 = e3rc12_2 * e3rc12;
      double e3rc20_3 = e3rc20_2 * e3rc20;
      double e3rc21_3 = e3rc21_2 * e3rc21;
      double e3rc22_3 = e3rc22_2 * e3rc22;

      // Big matrix to accept all of the constraint coefficients.
      dlr::numeric::Array2D<double> AMatrix(10, 20);

      
      // ============= Begin cut & paste section ============= 

    AMatrix(0, 0) = -e0rc00*e0rc22_2+2*e0rc02*e0rc20*e0rc22-e0rc00*e0rc21_2+2*e0rc01*e0rc20*e0rc21+e0rc00*e0rc20_2-e0rc00*e0rc12_2+2*e0rc02*e0rc10*e0rc12-e0rc00*e0rc11_2+2*e0rc01*e0rc10*e0rc11+e0rc00*e0rc10_2+e0rc00*e0rc02_2+e0rc00*e0rc01_2+e0rc00_3;
    AMatrix(0, 1) = -2*e0rc00*e0rc22*e1rc22+2*e0rc02*e0rc20*e1rc22-2*e0rc00*e0rc21*e1rc21+2*e0rc01*e0rc20*e1rc21+2*e0rc02*e0rc22*e1rc20+2*e0rc01*e0rc21*e1rc20+2*e0rc00*e0rc20*e1rc20-2*e0rc00*e0rc12*e1rc12+2*e0rc02*e0rc10*e1rc12-2*e0rc00*e0rc11*e1rc11+2*e0rc01*e0rc10*e1rc11+2*e0rc02*e0rc12*e1rc10+2*e0rc01*e0rc11*e1rc10+2*e0rc00*e0rc10*e1rc10+2*e0rc20*e0rc22*e1rc02+2*e0rc10*e0rc12*e1rc02+2*e0rc00*e0rc02*e1rc02+2*e0rc20*e0rc21*e1rc01+2*e0rc10*e0rc11*e1rc01+2*e0rc00*e0rc01*e1rc01-e0rc22_2*e1rc00-e0rc21_2*e1rc00+e0rc20_2*e1rc00-e0rc12_2*e1rc00-e0rc11_2*e1rc00+e0rc10_2*e1rc00+e0rc02_2*e1rc00+e0rc01_2*e1rc00+3*e0rc00_2*e1rc00;
    AMatrix(0, 2) = -2*e0rc00*e0rc22*e2rc22+2*e0rc02*e0rc20*e2rc22-2*e0rc00*e0rc21*e2rc21+2*e0rc01*e0rc20*e2rc21+2*e0rc02*e0rc22*e2rc20+2*e0rc01*e0rc21*e2rc20+2*e0rc00*e0rc20*e2rc20-2*e0rc00*e0rc12*e2rc12+2*e0rc02*e0rc10*e2rc12-2*e0rc00*e0rc11*e2rc11+2*e0rc01*e0rc10*e2rc11+2*e0rc02*e0rc12*e2rc10+2*e0rc01*e0rc11*e2rc10+2*e0rc00*e0rc10*e2rc10+2*e0rc20*e0rc22*e2rc02+2*e0rc10*e0rc12*e2rc02+2*e0rc00*e0rc02*e2rc02+2*e0rc20*e0rc21*e2rc01+2*e0rc10*e0rc11*e2rc01+2*e0rc00*e0rc01*e2rc01-e0rc22_2*e2rc00-e0rc21_2*e2rc00+e0rc20_2*e2rc00-e0rc12_2*e2rc00-e0rc11_2*e2rc00+e0rc10_2*e2rc00+e0rc02_2*e2rc00+e0rc01_2*e2rc00+3*e0rc00_2*e2rc00;
    AMatrix(0, 3) = -e0rc00*e1rc22_2+2*e0rc02*e1rc20*e1rc22+2*e0rc20*e1rc02*e1rc22-2*e0rc22*e1rc00*e1rc22-e0rc00*e1rc21_2+2*e0rc01*e1rc20*e1rc21+2*e0rc20*e1rc01*e1rc21-2*e0rc21*e1rc00*e1rc21+e0rc00*e1rc20_2+2*e0rc22*e1rc02*e1rc20+2*e0rc21*e1rc01*e1rc20+2*e0rc20*e1rc00*e1rc20-e0rc00*e1rc12_2+2*e0rc02*e1rc10*e1rc12+2*e0rc10*e1rc02*e1rc12-2*e0rc12*e1rc00*e1rc12-e0rc00*e1rc11_2+2*e0rc01*e1rc10*e1rc11+2*e0rc10*e1rc01*e1rc11-2*e0rc11*e1rc00*e1rc11+e0rc00*e1rc10_2+2*e0rc12*e1rc02*e1rc10+2*e0rc11*e1rc01*e1rc10+2*e0rc10*e1rc00*e1rc10+e0rc00*e1rc02_2+2*e0rc02*e1rc00*e1rc02+e0rc00*e1rc01_2+2*e0rc01*e1rc00*e1rc01+3*e0rc00*e1rc00_2;
    AMatrix(0, 4) = -2*e0rc00*e1rc22*e2rc22+2*e0rc02*e1rc20*e2rc22+2*e0rc20*e1rc02*e2rc22-2*e0rc22*e1rc00*e2rc22-2*e0rc00*e1rc21*e2rc21+2*e0rc01*e1rc20*e2rc21+2*e0rc20*e1rc01*e2rc21-2*e0rc21*e1rc00*e2rc21+2*e0rc02*e1rc22*e2rc20+2*e0rc01*e1rc21*e2rc20+2*e0rc00*e1rc20*e2rc20+2*e0rc22*e1rc02*e2rc20+2*e0rc21*e1rc01*e2rc20+2*e0rc20*e1rc00*e2rc20-2*e0rc00*e1rc12*e2rc12+2*e0rc02*e1rc10*e2rc12+2*e0rc10*e1rc02*e2rc12-2*e0rc12*e1rc00*e2rc12-2*e0rc00*e1rc11*e2rc11+2*e0rc01*e1rc10*e2rc11+2*e0rc10*e1rc01*e2rc11-2*e0rc11*e1rc00*e2rc11+2*e0rc02*e1rc12*e2rc10+2*e0rc01*e1rc11*e2rc10+2*e0rc00*e1rc10*e2rc10+2*e0rc12*e1rc02*e2rc10+2*e0rc11*e1rc01*e2rc10+2*e0rc10*e1rc00*e2rc10+2*e0rc20*e1rc22*e2rc02+2*e0rc22*e1rc20*e2rc02+2*e0rc10*e1rc12*e2rc02+2*e0rc12*e1rc10*e2rc02+2*e0rc00*e1rc02*e2rc02+2*e0rc02*e1rc00*e2rc02+2*e0rc20*e1rc21*e2rc01+2*e0rc21*e1rc20*e2rc01+2*e0rc10*e1rc11*e2rc01+2*e0rc11*e1rc10*e2rc01+2*e0rc00*e1rc01*e2rc01+2*e0rc01*e1rc00*e2rc01-2*e0rc22*e1rc22*e2rc00-2*e0rc21*e1rc21*e2rc00+2*e0rc20*e1rc20*e2rc00-2*e0rc12*e1rc12*e2rc00-2*e0rc11*e1rc11*e2rc00+2*e0rc10*e1rc10*e2rc00+2*e0rc02*e1rc02*e2rc00+2*e0rc01*e1rc01*e2rc00+6*e0rc00*e1rc00*e2rc00;
    AMatrix(0, 5) = -e0rc00*e2rc22_2+2*e0rc02*e2rc20*e2rc22+2*e0rc20*e2rc02*e2rc22-2*e0rc22*e2rc00*e2rc22-e0rc00*e2rc21_2+2*e0rc01*e2rc20*e2rc21+2*e0rc20*e2rc01*e2rc21-2*e0rc21*e2rc00*e2rc21+e0rc00*e2rc20_2+2*e0rc22*e2rc02*e2rc20+2*e0rc21*e2rc01*e2rc20+2*e0rc20*e2rc00*e2rc20-e0rc00*e2rc12_2+2*e0rc02*e2rc10*e2rc12+2*e0rc10*e2rc02*e2rc12-2*e0rc12*e2rc00*e2rc12-e0rc00*e2rc11_2+2*e0rc01*e2rc10*e2rc11+2*e0rc10*e2rc01*e2rc11-2*e0rc11*e2rc00*e2rc11+e0rc00*e2rc10_2+2*e0rc12*e2rc02*e2rc10+2*e0rc11*e2rc01*e2rc10+2*e0rc10*e2rc00*e2rc10+e0rc00*e2rc02_2+2*e0rc02*e2rc00*e2rc02+e0rc00*e2rc01_2+2*e0rc01*e2rc00*e2rc01+3*e0rc00*e2rc00_2;
    AMatrix(0, 6) = -e1rc00*e1rc22_2+2*e1rc02*e1rc20*e1rc22-e1rc00*e1rc21_2+2*e1rc01*e1rc20*e1rc21+e1rc00*e1rc20_2-e1rc00*e1rc12_2+2*e1rc02*e1rc10*e1rc12-e1rc00*e1rc11_2+2*e1rc01*e1rc10*e1rc11+e1rc00*e1rc10_2+e1rc00*e1rc02_2+e1rc00*e1rc01_2+e1rc00_3;
    AMatrix(0, 7) = -2*e1rc00*e1rc22*e2rc22+2*e1rc02*e1rc20*e2rc22-2*e1rc00*e1rc21*e2rc21+2*e1rc01*e1rc20*e2rc21+2*e1rc02*e1rc22*e2rc20+2*e1rc01*e1rc21*e2rc20+2*e1rc00*e1rc20*e2rc20-2*e1rc00*e1rc12*e2rc12+2*e1rc02*e1rc10*e2rc12-2*e1rc00*e1rc11*e2rc11+2*e1rc01*e1rc10*e2rc11+2*e1rc02*e1rc12*e2rc10+2*e1rc01*e1rc11*e2rc10+2*e1rc00*e1rc10*e2rc10+2*e1rc20*e1rc22*e2rc02+2*e1rc10*e1rc12*e2rc02+2*e1rc00*e1rc02*e2rc02+2*e1rc20*e1rc21*e2rc01+2*e1rc10*e1rc11*e2rc01+2*e1rc00*e1rc01*e2rc01-e1rc22_2*e2rc00-e1rc21_2*e2rc00+e1rc20_2*e2rc00-e1rc12_2*e2rc00-e1rc11_2*e2rc00+e1rc10_2*e2rc00+e1rc02_2*e2rc00+e1rc01_2*e2rc00+3*e1rc00_2*e2rc00;
    AMatrix(0, 8) = -e1rc00*e2rc22_2+2*e1rc02*e2rc20*e2rc22+2*e1rc20*e2rc02*e2rc22-2*e1rc22*e2rc00*e2rc22-e1rc00*e2rc21_2+2*e1rc01*e2rc20*e2rc21+2*e1rc20*e2rc01*e2rc21-2*e1rc21*e2rc00*e2rc21+e1rc00*e2rc20_2+2*e1rc22*e2rc02*e2rc20+2*e1rc21*e2rc01*e2rc20+2*e1rc20*e2rc00*e2rc20-e1rc00*e2rc12_2+2*e1rc02*e2rc10*e2rc12+2*e1rc10*e2rc02*e2rc12-2*e1rc12*e2rc00*e2rc12-e1rc00*e2rc11_2+2*e1rc01*e2rc10*e2rc11+2*e1rc10*e2rc01*e2rc11-2*e1rc11*e2rc00*e2rc11+e1rc00*e2rc10_2+2*e1rc12*e2rc02*e2rc10+2*e1rc11*e2rc01*e2rc10+2*e1rc10*e2rc00*e2rc10+e1rc00*e2rc02_2+2*e1rc02*e2rc00*e2rc02+e1rc00*e2rc01_2+2*e1rc01*e2rc00*e2rc01+3*e1rc00*e2rc00_2;
    AMatrix(0, 9) = -e2rc00*e2rc22_2+2*e2rc02*e2rc20*e2rc22-e2rc00*e2rc21_2+2*e2rc01*e2rc20*e2rc21+e2rc00*e2rc20_2-e2rc00*e2rc12_2+2*e2rc02*e2rc10*e2rc12-e2rc00*e2rc11_2+2*e2rc01*e2rc10*e2rc11+e2rc00*e2rc10_2+e2rc00*e2rc02_2+e2rc00*e2rc01_2+e2rc00_3;
    AMatrix(0, 10) = -2*e0rc00*e0rc22*e3rc22+2*e0rc02*e0rc20*e3rc22-2*e0rc00*e0rc21*e3rc21+2*e0rc01*e0rc20*e3rc21+2*e0rc02*e0rc22*e3rc20+2*e0rc01*e0rc21*e3rc20+2*e0rc00*e0rc20*e3rc20-2*e0rc00*e0rc12*e3rc12+2*e0rc02*e0rc10*e3rc12-2*e0rc00*e0rc11*e3rc11+2*e0rc01*e0rc10*e3rc11+2*e0rc02*e0rc12*e3rc10+2*e0rc01*e0rc11*e3rc10+2*e0rc00*e0rc10*e3rc10+2*e0rc20*e0rc22*e3rc02+2*e0rc10*e0rc12*e3rc02+2*e0rc00*e0rc02*e3rc02+2*e0rc20*e0rc21*e3rc01+2*e0rc10*e0rc11*e3rc01+2*e0rc00*e0rc01*e3rc01-e0rc22_2*e3rc00-e0rc21_2*e3rc00+e0rc20_2*e3rc00-e0rc12_2*e3rc00-e0rc11_2*e3rc00+e0rc10_2*e3rc00+e0rc02_2*e3rc00+e0rc01_2*e3rc00+3*e0rc00_2*e3rc00;
    AMatrix(0, 11) = -2*e0rc00*e1rc22*e3rc22+2*e0rc02*e1rc20*e3rc22+2*e0rc20*e1rc02*e3rc22-2*e0rc22*e1rc00*e3rc22-2*e0rc00*e1rc21*e3rc21+2*e0rc01*e1rc20*e3rc21+2*e0rc20*e1rc01*e3rc21-2*e0rc21*e1rc00*e3rc21+2*e0rc02*e1rc22*e3rc20+2*e0rc01*e1rc21*e3rc20+2*e0rc00*e1rc20*e3rc20+2*e0rc22*e1rc02*e3rc20+2*e0rc21*e1rc01*e3rc20+2*e0rc20*e1rc00*e3rc20-2*e0rc00*e1rc12*e3rc12+2*e0rc02*e1rc10*e3rc12+2*e0rc10*e1rc02*e3rc12-2*e0rc12*e1rc00*e3rc12-2*e0rc00*e1rc11*e3rc11+2*e0rc01*e1rc10*e3rc11+2*e0rc10*e1rc01*e3rc11-2*e0rc11*e1rc00*e3rc11+2*e0rc02*e1rc12*e3rc10+2*e0rc01*e1rc11*e3rc10+2*e0rc00*e1rc10*e3rc10+2*e0rc12*e1rc02*e3rc10+2*e0rc11*e1rc01*e3rc10+2*e0rc10*e1rc00*e3rc10+2*e0rc20*e1rc22*e3rc02+2*e0rc22*e1rc20*e3rc02+2*e0rc10*e1rc12*e3rc02+2*e0rc12*e1rc10*e3rc02+2*e0rc00*e1rc02*e3rc02+2*e0rc02*e1rc00*e3rc02+2*e0rc20*e1rc21*e3rc01+2*e0rc21*e1rc20*e3rc01+2*e0rc10*e1rc11*e3rc01+2*e0rc11*e1rc10*e3rc01+2*e0rc00*e1rc01*e3rc01+2*e0rc01*e1rc00*e3rc01-2*e0rc22*e1rc22*e3rc00-2*e0rc21*e1rc21*e3rc00+2*e0rc20*e1rc20*e3rc00-2*e0rc12*e1rc12*e3rc00-2*e0rc11*e1rc11*e3rc00+2*e0rc10*e1rc10*e3rc00+2*e0rc02*e1rc02*e3rc00+2*e0rc01*e1rc01*e3rc00+6*e0rc00*e1rc00*e3rc00;
    AMatrix(0, 12) = -2*e0rc00*e2rc22*e3rc22+2*e0rc02*e2rc20*e3rc22+2*e0rc20*e2rc02*e3rc22-2*e0rc22*e2rc00*e3rc22-2*e0rc00*e2rc21*e3rc21+2*e0rc01*e2rc20*e3rc21+2*e0rc20*e2rc01*e3rc21-2*e0rc21*e2rc00*e3rc21+2*e0rc02*e2rc22*e3rc20+2*e0rc01*e2rc21*e3rc20+2*e0rc00*e2rc20*e3rc20+2*e0rc22*e2rc02*e3rc20+2*e0rc21*e2rc01*e3rc20+2*e0rc20*e2rc00*e3rc20-2*e0rc00*e2rc12*e3rc12+2*e0rc02*e2rc10*e3rc12+2*e0rc10*e2rc02*e3rc12-2*e0rc12*e2rc00*e3rc12-2*e0rc00*e2rc11*e3rc11+2*e0rc01*e2rc10*e3rc11+2*e0rc10*e2rc01*e3rc11-2*e0rc11*e2rc00*e3rc11+2*e0rc02*e2rc12*e3rc10+2*e0rc01*e2rc11*e3rc10+2*e0rc00*e2rc10*e3rc10+2*e0rc12*e2rc02*e3rc10+2*e0rc11*e2rc01*e3rc10+2*e0rc10*e2rc00*e3rc10+2*e0rc20*e2rc22*e3rc02+2*e0rc22*e2rc20*e3rc02+2*e0rc10*e2rc12*e3rc02+2*e0rc12*e2rc10*e3rc02+2*e0rc00*e2rc02*e3rc02+2*e0rc02*e2rc00*e3rc02+2*e0rc20*e2rc21*e3rc01+2*e0rc21*e2rc20*e3rc01+2*e0rc10*e2rc11*e3rc01+2*e0rc11*e2rc10*e3rc01+2*e0rc00*e2rc01*e3rc01+2*e0rc01*e2rc00*e3rc01-2*e0rc22*e2rc22*e3rc00-2*e0rc21*e2rc21*e3rc00+2*e0rc20*e2rc20*e3rc00-2*e0rc12*e2rc12*e3rc00-2*e0rc11*e2rc11*e3rc00+2*e0rc10*e2rc10*e3rc00+2*e0rc02*e2rc02*e3rc00+2*e0rc01*e2rc01*e3rc00+6*e0rc00*e2rc00*e3rc00;
    AMatrix(0, 13) = -2*e1rc00*e1rc22*e3rc22+2*e1rc02*e1rc20*e3rc22-2*e1rc00*e1rc21*e3rc21+2*e1rc01*e1rc20*e3rc21+2*e1rc02*e1rc22*e3rc20+2*e1rc01*e1rc21*e3rc20+2*e1rc00*e1rc20*e3rc20-2*e1rc00*e1rc12*e3rc12+2*e1rc02*e1rc10*e3rc12-2*e1rc00*e1rc11*e3rc11+2*e1rc01*e1rc10*e3rc11+2*e1rc02*e1rc12*e3rc10+2*e1rc01*e1rc11*e3rc10+2*e1rc00*e1rc10*e3rc10+2*e1rc20*e1rc22*e3rc02+2*e1rc10*e1rc12*e3rc02+2*e1rc00*e1rc02*e3rc02+2*e1rc20*e1rc21*e3rc01+2*e1rc10*e1rc11*e3rc01+2*e1rc00*e1rc01*e3rc01-e1rc22_2*e3rc00-e1rc21_2*e3rc00+e1rc20_2*e3rc00-e1rc12_2*e3rc00-e1rc11_2*e3rc00+e1rc10_2*e3rc00+e1rc02_2*e3rc00+e1rc01_2*e3rc00+3*e1rc00_2*e3rc00;
    AMatrix(0, 14) = -2*e1rc00*e2rc22*e3rc22+2*e1rc02*e2rc20*e3rc22+2*e1rc20*e2rc02*e3rc22-2*e1rc22*e2rc00*e3rc22-2*e1rc00*e2rc21*e3rc21+2*e1rc01*e2rc20*e3rc21+2*e1rc20*e2rc01*e3rc21-2*e1rc21*e2rc00*e3rc21+2*e1rc02*e2rc22*e3rc20+2*e1rc01*e2rc21*e3rc20+2*e1rc00*e2rc20*e3rc20+2*e1rc22*e2rc02*e3rc20+2*e1rc21*e2rc01*e3rc20+2*e1rc20*e2rc00*e3rc20-2*e1rc00*e2rc12*e3rc12+2*e1rc02*e2rc10*e3rc12+2*e1rc10*e2rc02*e3rc12-2*e1rc12*e2rc00*e3rc12-2*e1rc00*e2rc11*e3rc11+2*e1rc01*e2rc10*e3rc11+2*e1rc10*e2rc01*e3rc11-2*e1rc11*e2rc00*e3rc11+2*e1rc02*e2rc12*e3rc10+2*e1rc01*e2rc11*e3rc10+2*e1rc00*e2rc10*e3rc10+2*e1rc12*e2rc02*e3rc10+2*e1rc11*e2rc01*e3rc10+2*e1rc10*e2rc00*e3rc10+2*e1rc20*e2rc22*e3rc02+2*e1rc22*e2rc20*e3rc02+2*e1rc10*e2rc12*e3rc02+2*e1rc12*e2rc10*e3rc02+2*e1rc00*e2rc02*e3rc02+2*e1rc02*e2rc00*e3rc02+2*e1rc20*e2rc21*e3rc01+2*e1rc21*e2rc20*e3rc01+2*e1rc10*e2rc11*e3rc01+2*e1rc11*e2rc10*e3rc01+2*e1rc00*e2rc01*e3rc01+2*e1rc01*e2rc00*e3rc01-2*e1rc22*e2rc22*e3rc00-2*e1rc21*e2rc21*e3rc00+2*e1rc20*e2rc20*e3rc00-2*e1rc12*e2rc12*e3rc00-2*e1rc11*e2rc11*e3rc00+2*e1rc10*e2rc10*e3rc00+2*e1rc02*e2rc02*e3rc00+2*e1rc01*e2rc01*e3rc00+6*e1rc00*e2rc00*e3rc00;
    AMatrix(0, 15) = -2*e2rc00*e2rc22*e3rc22+2*e2rc02*e2rc20*e3rc22-2*e2rc00*e2rc21*e3rc21+2*e2rc01*e2rc20*e3rc21+2*e2rc02*e2rc22*e3rc20+2*e2rc01*e2rc21*e3rc20+2*e2rc00*e2rc20*e3rc20-2*e2rc00*e2rc12*e3rc12+2*e2rc02*e2rc10*e3rc12-2*e2rc00*e2rc11*e3rc11+2*e2rc01*e2rc10*e3rc11+2*e2rc02*e2rc12*e3rc10+2*e2rc01*e2rc11*e3rc10+2*e2rc00*e2rc10*e3rc10+2*e2rc20*e2rc22*e3rc02+2*e2rc10*e2rc12*e3rc02+2*e2rc00*e2rc02*e3rc02+2*e2rc20*e2rc21*e3rc01+2*e2rc10*e2rc11*e3rc01+2*e2rc00*e2rc01*e3rc01-e2rc22_2*e3rc00-e2rc21_2*e3rc00+e2rc20_2*e3rc00-e2rc12_2*e3rc00-e2rc11_2*e3rc00+e2rc10_2*e3rc00+e2rc02_2*e3rc00+e2rc01_2*e3rc00+3*e2rc00_2*e3rc00;
    AMatrix(0, 16) = -e0rc00*e3rc22_2+2*e0rc02*e3rc20*e3rc22+2*e0rc20*e3rc02*e3rc22-2*e0rc22*e3rc00*e3rc22-e0rc00*e3rc21_2+2*e0rc01*e3rc20*e3rc21+2*e0rc20*e3rc01*e3rc21-2*e0rc21*e3rc00*e3rc21+e0rc00*e3rc20_2+2*e0rc22*e3rc02*e3rc20+2*e0rc21*e3rc01*e3rc20+2*e0rc20*e3rc00*e3rc20-e0rc00*e3rc12_2+2*e0rc02*e3rc10*e3rc12+2*e0rc10*e3rc02*e3rc12-2*e0rc12*e3rc00*e3rc12-e0rc00*e3rc11_2+2*e0rc01*e3rc10*e3rc11+2*e0rc10*e3rc01*e3rc11-2*e0rc11*e3rc00*e3rc11+e0rc00*e3rc10_2+2*e0rc12*e3rc02*e3rc10+2*e0rc11*e3rc01*e3rc10+2*e0rc10*e3rc00*e3rc10+e0rc00*e3rc02_2+2*e0rc02*e3rc00*e3rc02+e0rc00*e3rc01_2+2*e0rc01*e3rc00*e3rc01+3*e0rc00*e3rc00_2;
    AMatrix(0, 17) = -e1rc00*e3rc22_2+2*e1rc02*e3rc20*e3rc22+2*e1rc20*e3rc02*e3rc22-2*e1rc22*e3rc00*e3rc22-e1rc00*e3rc21_2+2*e1rc01*e3rc20*e3rc21+2*e1rc20*e3rc01*e3rc21-2*e1rc21*e3rc00*e3rc21+e1rc00*e3rc20_2+2*e1rc22*e3rc02*e3rc20+2*e1rc21*e3rc01*e3rc20+2*e1rc20*e3rc00*e3rc20-e1rc00*e3rc12_2+2*e1rc02*e3rc10*e3rc12+2*e1rc10*e3rc02*e3rc12-2*e1rc12*e3rc00*e3rc12-e1rc00*e3rc11_2+2*e1rc01*e3rc10*e3rc11+2*e1rc10*e3rc01*e3rc11-2*e1rc11*e3rc00*e3rc11+e1rc00*e3rc10_2+2*e1rc12*e3rc02*e3rc10+2*e1rc11*e3rc01*e3rc10+2*e1rc10*e3rc00*e3rc10+e1rc00*e3rc02_2+2*e1rc02*e3rc00*e3rc02+e1rc00*e3rc01_2+2*e1rc01*e3rc00*e3rc01+3*e1rc00*e3rc00_2;
    AMatrix(0, 18) = -e2rc00*e3rc22_2+2*e2rc02*e3rc20*e3rc22+2*e2rc20*e3rc02*e3rc22-2*e2rc22*e3rc00*e3rc22-e2rc00*e3rc21_2+2*e2rc01*e3rc20*e3rc21+2*e2rc20*e3rc01*e3rc21-2*e2rc21*e3rc00*e3rc21+e2rc00*e3rc20_2+2*e2rc22*e3rc02*e3rc20+2*e2rc21*e3rc01*e3rc20+2*e2rc20*e3rc00*e3rc20-e2rc00*e3rc12_2+2*e2rc02*e3rc10*e3rc12+2*e2rc10*e3rc02*e3rc12-2*e2rc12*e3rc00*e3rc12-e2rc00*e3rc11_2+2*e2rc01*e3rc10*e3rc11+2*e2rc10*e3rc01*e3rc11-2*e2rc11*e3rc00*e3rc11+e2rc00*e3rc10_2+2*e2rc12*e3rc02*e3rc10+2*e2rc11*e3rc01*e3rc10+2*e2rc10*e3rc00*e3rc10+e2rc00*e3rc02_2+2*e2rc02*e3rc00*e3rc02+e2rc00*e3rc01_2+2*e2rc01*e3rc00*e3rc01+3*e2rc00*e3rc00_2;
    AMatrix(0, 19) = -e3rc00*e3rc22_2+2*e3rc02*e3rc20*e3rc22-e3rc00*e3rc21_2+2*e3rc01*e3rc20*e3rc21+e3rc00*e3rc20_2-e3rc00*e3rc12_2+2*e3rc02*e3rc10*e3rc12-e3rc00*e3rc11_2+2*e3rc01*e3rc10*e3rc11+e3rc00*e3rc10_2+e3rc00*e3rc02_2+e3rc00*e3rc01_2+e3rc00_3;
    AMatrix(1, 0) = -e0rc01*e0rc22_2+2*e0rc02*e0rc21*e0rc22+e0rc01*e0rc21_2+2*e0rc00*e0rc20*e0rc21-e0rc01*e0rc20_2-e0rc01*e0rc12_2+2*e0rc02*e0rc11*e0rc12+e0rc01*e0rc11_2+2*e0rc00*e0rc10*e0rc11-e0rc01*e0rc10_2+e0rc01*e0rc02_2+e0rc01_3+e0rc00_2*e0rc01;
    AMatrix(1, 1) = -2*e0rc01*e0rc22*e1rc22+2*e0rc02*e0rc21*e1rc22+2*e0rc02*e0rc22*e1rc21+2*e0rc01*e0rc21*e1rc21+2*e0rc00*e0rc20*e1rc21+2*e0rc00*e0rc21*e1rc20-2*e0rc01*e0rc20*e1rc20-2*e0rc01*e0rc12*e1rc12+2*e0rc02*e0rc11*e1rc12+2*e0rc02*e0rc12*e1rc11+2*e0rc01*e0rc11*e1rc11+2*e0rc00*e0rc10*e1rc11+2*e0rc00*e0rc11*e1rc10-2*e0rc01*e0rc10*e1rc10+2*e0rc21*e0rc22*e1rc02+2*e0rc11*e0rc12*e1rc02+2*e0rc01*e0rc02*e1rc02-e0rc22_2*e1rc01+e0rc21_2*e1rc01-e0rc20_2*e1rc01-e0rc12_2*e1rc01+e0rc11_2*e1rc01-e0rc10_2*e1rc01+e0rc02_2*e1rc01+3*e0rc01_2*e1rc01+e0rc00_2*e1rc01+2*e0rc20*e0rc21*e1rc00+2*e0rc10*e0rc11*e1rc00+2*e0rc00*e0rc01*e1rc00;
    AMatrix(1, 2) = -2*e0rc01*e0rc22*e2rc22+2*e0rc02*e0rc21*e2rc22+2*e0rc02*e0rc22*e2rc21+2*e0rc01*e0rc21*e2rc21+2*e0rc00*e0rc20*e2rc21+2*e0rc00*e0rc21*e2rc20-2*e0rc01*e0rc20*e2rc20-2*e0rc01*e0rc12*e2rc12+2*e0rc02*e0rc11*e2rc12+2*e0rc02*e0rc12*e2rc11+2*e0rc01*e0rc11*e2rc11+2*e0rc00*e0rc10*e2rc11+2*e0rc00*e0rc11*e2rc10-2*e0rc01*e0rc10*e2rc10+2*e0rc21*e0rc22*e2rc02+2*e0rc11*e0rc12*e2rc02+2*e0rc01*e0rc02*e2rc02-e0rc22_2*e2rc01+e0rc21_2*e2rc01-e0rc20_2*e2rc01-e0rc12_2*e2rc01+e0rc11_2*e2rc01-e0rc10_2*e2rc01+e0rc02_2*e2rc01+3*e0rc01_2*e2rc01+e0rc00_2*e2rc01+2*e0rc20*e0rc21*e2rc00+2*e0rc10*e0rc11*e2rc00+2*e0rc00*e0rc01*e2rc00;
    AMatrix(1, 3) = -e0rc01*e1rc22_2+2*e0rc02*e1rc21*e1rc22+2*e0rc21*e1rc02*e1rc22-2*e0rc22*e1rc01*e1rc22+e0rc01*e1rc21_2+2*e0rc00*e1rc20*e1rc21+2*e0rc22*e1rc02*e1rc21+2*e0rc21*e1rc01*e1rc21+2*e0rc20*e1rc00*e1rc21-e0rc01*e1rc20_2-2*e0rc20*e1rc01*e1rc20+2*e0rc21*e1rc00*e1rc20-e0rc01*e1rc12_2+2*e0rc02*e1rc11*e1rc12+2*e0rc11*e1rc02*e1rc12-2*e0rc12*e1rc01*e1rc12+e0rc01*e1rc11_2+2*e0rc00*e1rc10*e1rc11+2*e0rc12*e1rc02*e1rc11+2*e0rc11*e1rc01*e1rc11+2*e0rc10*e1rc00*e1rc11-e0rc01*e1rc10_2-2*e0rc10*e1rc01*e1rc10+2*e0rc11*e1rc00*e1rc10+e0rc01*e1rc02_2+2*e0rc02*e1rc01*e1rc02+3*e0rc01*e1rc01_2+2*e0rc00*e1rc00*e1rc01+e0rc01*e1rc00_2;
    AMatrix(1, 4) = -2*e0rc01*e1rc22*e2rc22+2*e0rc02*e1rc21*e2rc22+2*e0rc21*e1rc02*e2rc22-2*e0rc22*e1rc01*e2rc22+2*e0rc02*e1rc22*e2rc21+2*e0rc01*e1rc21*e2rc21+2*e0rc00*e1rc20*e2rc21+2*e0rc22*e1rc02*e2rc21+2*e0rc21*e1rc01*e2rc21+2*e0rc20*e1rc00*e2rc21+2*e0rc00*e1rc21*e2rc20-2*e0rc01*e1rc20*e2rc20-2*e0rc20*e1rc01*e2rc20+2*e0rc21*e1rc00*e2rc20-2*e0rc01*e1rc12*e2rc12+2*e0rc02*e1rc11*e2rc12+2*e0rc11*e1rc02*e2rc12-2*e0rc12*e1rc01*e2rc12+2*e0rc02*e1rc12*e2rc11+2*e0rc01*e1rc11*e2rc11+2*e0rc00*e1rc10*e2rc11+2*e0rc12*e1rc02*e2rc11+2*e0rc11*e1rc01*e2rc11+2*e0rc10*e1rc00*e2rc11+2*e0rc00*e1rc11*e2rc10-2*e0rc01*e1rc10*e2rc10-2*e0rc10*e1rc01*e2rc10+2*e0rc11*e1rc00*e2rc10+2*e0rc21*e1rc22*e2rc02+2*e0rc22*e1rc21*e2rc02+2*e0rc11*e1rc12*e2rc02+2*e0rc12*e1rc11*e2rc02+2*e0rc01*e1rc02*e2rc02+2*e0rc02*e1rc01*e2rc02-2*e0rc22*e1rc22*e2rc01+2*e0rc21*e1rc21*e2rc01-2*e0rc20*e1rc20*e2rc01-2*e0rc12*e1rc12*e2rc01+2*e0rc11*e1rc11*e2rc01-2*e0rc10*e1rc10*e2rc01+2*e0rc02*e1rc02*e2rc01+6*e0rc01*e1rc01*e2rc01+2*e0rc00*e1rc00*e2rc01+2*e0rc20*e1rc21*e2rc00+2*e0rc21*e1rc20*e2rc00+2*e0rc10*e1rc11*e2rc00+2*e0rc11*e1rc10*e2rc00+2*e0rc00*e1rc01*e2rc00+2*e0rc01*e1rc00*e2rc00;
    AMatrix(1, 5) = -e0rc01*e2rc22_2+2*e0rc02*e2rc21*e2rc22+2*e0rc21*e2rc02*e2rc22-2*e0rc22*e2rc01*e2rc22+e0rc01*e2rc21_2+2*e0rc00*e2rc20*e2rc21+2*e0rc22*e2rc02*e2rc21+2*e0rc21*e2rc01*e2rc21+2*e0rc20*e2rc00*e2rc21-e0rc01*e2rc20_2-2*e0rc20*e2rc01*e2rc20+2*e0rc21*e2rc00*e2rc20-e0rc01*e2rc12_2+2*e0rc02*e2rc11*e2rc12+2*e0rc11*e2rc02*e2rc12-2*e0rc12*e2rc01*e2rc12+e0rc01*e2rc11_2+2*e0rc00*e2rc10*e2rc11+2*e0rc12*e2rc02*e2rc11+2*e0rc11*e2rc01*e2rc11+2*e0rc10*e2rc00*e2rc11-e0rc01*e2rc10_2-2*e0rc10*e2rc01*e2rc10+2*e0rc11*e2rc00*e2rc10+e0rc01*e2rc02_2+2*e0rc02*e2rc01*e2rc02+3*e0rc01*e2rc01_2+2*e0rc00*e2rc00*e2rc01+e0rc01*e2rc00_2;
    AMatrix(1, 6) = -e1rc01*e1rc22_2+2*e1rc02*e1rc21*e1rc22+e1rc01*e1rc21_2+2*e1rc00*e1rc20*e1rc21-e1rc01*e1rc20_2-e1rc01*e1rc12_2+2*e1rc02*e1rc11*e1rc12+e1rc01*e1rc11_2+2*e1rc00*e1rc10*e1rc11-e1rc01*e1rc10_2+e1rc01*e1rc02_2+e1rc01_3+e1rc00_2*e1rc01;
    AMatrix(1, 7) = -2*e1rc01*e1rc22*e2rc22+2*e1rc02*e1rc21*e2rc22+2*e1rc02*e1rc22*e2rc21+2*e1rc01*e1rc21*e2rc21+2*e1rc00*e1rc20*e2rc21+2*e1rc00*e1rc21*e2rc20-2*e1rc01*e1rc20*e2rc20-2*e1rc01*e1rc12*e2rc12+2*e1rc02*e1rc11*e2rc12+2*e1rc02*e1rc12*e2rc11+2*e1rc01*e1rc11*e2rc11+2*e1rc00*e1rc10*e2rc11+2*e1rc00*e1rc11*e2rc10-2*e1rc01*e1rc10*e2rc10+2*e1rc21*e1rc22*e2rc02+2*e1rc11*e1rc12*e2rc02+2*e1rc01*e1rc02*e2rc02-e1rc22_2*e2rc01+e1rc21_2*e2rc01-e1rc20_2*e2rc01-e1rc12_2*e2rc01+e1rc11_2*e2rc01-e1rc10_2*e2rc01+e1rc02_2*e2rc01+3*e1rc01_2*e2rc01+e1rc00_2*e2rc01+2*e1rc20*e1rc21*e2rc00+2*e1rc10*e1rc11*e2rc00+2*e1rc00*e1rc01*e2rc00;
    AMatrix(1, 8) = -e1rc01*e2rc22_2+2*e1rc02*e2rc21*e2rc22+2*e1rc21*e2rc02*e2rc22-2*e1rc22*e2rc01*e2rc22+e1rc01*e2rc21_2+2*e1rc00*e2rc20*e2rc21+2*e1rc22*e2rc02*e2rc21+2*e1rc21*e2rc01*e2rc21+2*e1rc20*e2rc00*e2rc21-e1rc01*e2rc20_2-2*e1rc20*e2rc01*e2rc20+2*e1rc21*e2rc00*e2rc20-e1rc01*e2rc12_2+2*e1rc02*e2rc11*e2rc12+2*e1rc11*e2rc02*e2rc12-2*e1rc12*e2rc01*e2rc12+e1rc01*e2rc11_2+2*e1rc00*e2rc10*e2rc11+2*e1rc12*e2rc02*e2rc11+2*e1rc11*e2rc01*e2rc11+2*e1rc10*e2rc00*e2rc11-e1rc01*e2rc10_2-2*e1rc10*e2rc01*e2rc10+2*e1rc11*e2rc00*e2rc10+e1rc01*e2rc02_2+2*e1rc02*e2rc01*e2rc02+3*e1rc01*e2rc01_2+2*e1rc00*e2rc00*e2rc01+e1rc01*e2rc00_2;
    AMatrix(1, 9) = -e2rc01*e2rc22_2+2*e2rc02*e2rc21*e2rc22+e2rc01*e2rc21_2+2*e2rc00*e2rc20*e2rc21-e2rc01*e2rc20_2-e2rc01*e2rc12_2+2*e2rc02*e2rc11*e2rc12+e2rc01*e2rc11_2+2*e2rc00*e2rc10*e2rc11-e2rc01*e2rc10_2+e2rc01*e2rc02_2+e2rc01_3+e2rc00_2*e2rc01;
    AMatrix(1, 10) = -2*e0rc01*e0rc22*e3rc22+2*e0rc02*e0rc21*e3rc22+2*e0rc02*e0rc22*e3rc21+2*e0rc01*e0rc21*e3rc21+2*e0rc00*e0rc20*e3rc21+2*e0rc00*e0rc21*e3rc20-2*e0rc01*e0rc20*e3rc20-2*e0rc01*e0rc12*e3rc12+2*e0rc02*e0rc11*e3rc12+2*e0rc02*e0rc12*e3rc11+2*e0rc01*e0rc11*e3rc11+2*e0rc00*e0rc10*e3rc11+2*e0rc00*e0rc11*e3rc10-2*e0rc01*e0rc10*e3rc10+2*e0rc21*e0rc22*e3rc02+2*e0rc11*e0rc12*e3rc02+2*e0rc01*e0rc02*e3rc02-e0rc22_2*e3rc01+e0rc21_2*e3rc01-e0rc20_2*e3rc01-e0rc12_2*e3rc01+e0rc11_2*e3rc01-e0rc10_2*e3rc01+e0rc02_2*e3rc01+3*e0rc01_2*e3rc01+e0rc00_2*e3rc01+2*e0rc20*e0rc21*e3rc00+2*e0rc10*e0rc11*e3rc00+2*e0rc00*e0rc01*e3rc00;
    AMatrix(1, 11) = -2*e0rc01*e1rc22*e3rc22+2*e0rc02*e1rc21*e3rc22+2*e0rc21*e1rc02*e3rc22-2*e0rc22*e1rc01*e3rc22+2*e0rc02*e1rc22*e3rc21+2*e0rc01*e1rc21*e3rc21+2*e0rc00*e1rc20*e3rc21+2*e0rc22*e1rc02*e3rc21+2*e0rc21*e1rc01*e3rc21+2*e0rc20*e1rc00*e3rc21+2*e0rc00*e1rc21*e3rc20-2*e0rc01*e1rc20*e3rc20-2*e0rc20*e1rc01*e3rc20+2*e0rc21*e1rc00*e3rc20-2*e0rc01*e1rc12*e3rc12+2*e0rc02*e1rc11*e3rc12+2*e0rc11*e1rc02*e3rc12-2*e0rc12*e1rc01*e3rc12+2*e0rc02*e1rc12*e3rc11+2*e0rc01*e1rc11*e3rc11+2*e0rc00*e1rc10*e3rc11+2*e0rc12*e1rc02*e3rc11+2*e0rc11*e1rc01*e3rc11+2*e0rc10*e1rc00*e3rc11+2*e0rc00*e1rc11*e3rc10-2*e0rc01*e1rc10*e3rc10-2*e0rc10*e1rc01*e3rc10+2*e0rc11*e1rc00*e3rc10+2*e0rc21*e1rc22*e3rc02+2*e0rc22*e1rc21*e3rc02+2*e0rc11*e1rc12*e3rc02+2*e0rc12*e1rc11*e3rc02+2*e0rc01*e1rc02*e3rc02+2*e0rc02*e1rc01*e3rc02-2*e0rc22*e1rc22*e3rc01+2*e0rc21*e1rc21*e3rc01-2*e0rc20*e1rc20*e3rc01-2*e0rc12*e1rc12*e3rc01+2*e0rc11*e1rc11*e3rc01-2*e0rc10*e1rc10*e3rc01+2*e0rc02*e1rc02*e3rc01+6*e0rc01*e1rc01*e3rc01+2*e0rc00*e1rc00*e3rc01+2*e0rc20*e1rc21*e3rc00+2*e0rc21*e1rc20*e3rc00+2*e0rc10*e1rc11*e3rc00+2*e0rc11*e1rc10*e3rc00+2*e0rc00*e1rc01*e3rc00+2*e0rc01*e1rc00*e3rc00;
    AMatrix(1, 12) = -2*e0rc01*e2rc22*e3rc22+2*e0rc02*e2rc21*e3rc22+2*e0rc21*e2rc02*e3rc22-2*e0rc22*e2rc01*e3rc22+2*e0rc02*e2rc22*e3rc21+2*e0rc01*e2rc21*e3rc21+2*e0rc00*e2rc20*e3rc21+2*e0rc22*e2rc02*e3rc21+2*e0rc21*e2rc01*e3rc21+2*e0rc20*e2rc00*e3rc21+2*e0rc00*e2rc21*e3rc20-2*e0rc01*e2rc20*e3rc20-2*e0rc20*e2rc01*e3rc20+2*e0rc21*e2rc00*e3rc20-2*e0rc01*e2rc12*e3rc12+2*e0rc02*e2rc11*e3rc12+2*e0rc11*e2rc02*e3rc12-2*e0rc12*e2rc01*e3rc12+2*e0rc02*e2rc12*e3rc11+2*e0rc01*e2rc11*e3rc11+2*e0rc00*e2rc10*e3rc11+2*e0rc12*e2rc02*e3rc11+2*e0rc11*e2rc01*e3rc11+2*e0rc10*e2rc00*e3rc11+2*e0rc00*e2rc11*e3rc10-2*e0rc01*e2rc10*e3rc10-2*e0rc10*e2rc01*e3rc10+2*e0rc11*e2rc00*e3rc10+2*e0rc21*e2rc22*e3rc02+2*e0rc22*e2rc21*e3rc02+2*e0rc11*e2rc12*e3rc02+2*e0rc12*e2rc11*e3rc02+2*e0rc01*e2rc02*e3rc02+2*e0rc02*e2rc01*e3rc02-2*e0rc22*e2rc22*e3rc01+2*e0rc21*e2rc21*e3rc01-2*e0rc20*e2rc20*e3rc01-2*e0rc12*e2rc12*e3rc01+2*e0rc11*e2rc11*e3rc01-2*e0rc10*e2rc10*e3rc01+2*e0rc02*e2rc02*e3rc01+6*e0rc01*e2rc01*e3rc01+2*e0rc00*e2rc00*e3rc01+2*e0rc20*e2rc21*e3rc00+2*e0rc21*e2rc20*e3rc00+2*e0rc10*e2rc11*e3rc00+2*e0rc11*e2rc10*e3rc00+2*e0rc00*e2rc01*e3rc00+2*e0rc01*e2rc00*e3rc00;
    AMatrix(1, 13) = -2*e1rc01*e1rc22*e3rc22+2*e1rc02*e1rc21*e3rc22+2*e1rc02*e1rc22*e3rc21+2*e1rc01*e1rc21*e3rc21+2*e1rc00*e1rc20*e3rc21+2*e1rc00*e1rc21*e3rc20-2*e1rc01*e1rc20*e3rc20-2*e1rc01*e1rc12*e3rc12+2*e1rc02*e1rc11*e3rc12+2*e1rc02*e1rc12*e3rc11+2*e1rc01*e1rc11*e3rc11+2*e1rc00*e1rc10*e3rc11+2*e1rc00*e1rc11*e3rc10-2*e1rc01*e1rc10*e3rc10+2*e1rc21*e1rc22*e3rc02+2*e1rc11*e1rc12*e3rc02+2*e1rc01*e1rc02*e3rc02-e1rc22_2*e3rc01+e1rc21_2*e3rc01-e1rc20_2*e3rc01-e1rc12_2*e3rc01+e1rc11_2*e3rc01-e1rc10_2*e3rc01+e1rc02_2*e3rc01+3*e1rc01_2*e3rc01+e1rc00_2*e3rc01+2*e1rc20*e1rc21*e3rc00+2*e1rc10*e1rc11*e3rc00+2*e1rc00*e1rc01*e3rc00;
    AMatrix(1, 14) = -2*e1rc01*e2rc22*e3rc22+2*e1rc02*e2rc21*e3rc22+2*e1rc21*e2rc02*e3rc22-2*e1rc22*e2rc01*e3rc22+2*e1rc02*e2rc22*e3rc21+2*e1rc01*e2rc21*e3rc21+2*e1rc00*e2rc20*e3rc21+2*e1rc22*e2rc02*e3rc21+2*e1rc21*e2rc01*e3rc21+2*e1rc20*e2rc00*e3rc21+2*e1rc00*e2rc21*e3rc20-2*e1rc01*e2rc20*e3rc20-2*e1rc20*e2rc01*e3rc20+2*e1rc21*e2rc00*e3rc20-2*e1rc01*e2rc12*e3rc12+2*e1rc02*e2rc11*e3rc12+2*e1rc11*e2rc02*e3rc12-2*e1rc12*e2rc01*e3rc12+2*e1rc02*e2rc12*e3rc11+2*e1rc01*e2rc11*e3rc11+2*e1rc00*e2rc10*e3rc11+2*e1rc12*e2rc02*e3rc11+2*e1rc11*e2rc01*e3rc11+2*e1rc10*e2rc00*e3rc11+2*e1rc00*e2rc11*e3rc10-2*e1rc01*e2rc10*e3rc10-2*e1rc10*e2rc01*e3rc10+2*e1rc11*e2rc00*e3rc10+2*e1rc21*e2rc22*e3rc02+2*e1rc22*e2rc21*e3rc02+2*e1rc11*e2rc12*e3rc02+2*e1rc12*e2rc11*e3rc02+2*e1rc01*e2rc02*e3rc02+2*e1rc02*e2rc01*e3rc02-2*e1rc22*e2rc22*e3rc01+2*e1rc21*e2rc21*e3rc01-2*e1rc20*e2rc20*e3rc01-2*e1rc12*e2rc12*e3rc01+2*e1rc11*e2rc11*e3rc01-2*e1rc10*e2rc10*e3rc01+2*e1rc02*e2rc02*e3rc01+6*e1rc01*e2rc01*e3rc01+2*e1rc00*e2rc00*e3rc01+2*e1rc20*e2rc21*e3rc00+2*e1rc21*e2rc20*e3rc00+2*e1rc10*e2rc11*e3rc00+2*e1rc11*e2rc10*e3rc00+2*e1rc00*e2rc01*e3rc00+2*e1rc01*e2rc00*e3rc00;
    AMatrix(1, 15) = -2*e2rc01*e2rc22*e3rc22+2*e2rc02*e2rc21*e3rc22+2*e2rc02*e2rc22*e3rc21+2*e2rc01*e2rc21*e3rc21+2*e2rc00*e2rc20*e3rc21+2*e2rc00*e2rc21*e3rc20-2*e2rc01*e2rc20*e3rc20-2*e2rc01*e2rc12*e3rc12+2*e2rc02*e2rc11*e3rc12+2*e2rc02*e2rc12*e3rc11+2*e2rc01*e2rc11*e3rc11+2*e2rc00*e2rc10*e3rc11+2*e2rc00*e2rc11*e3rc10-2*e2rc01*e2rc10*e3rc10+2*e2rc21*e2rc22*e3rc02+2*e2rc11*e2rc12*e3rc02+2*e2rc01*e2rc02*e3rc02-e2rc22_2*e3rc01+e2rc21_2*e3rc01-e2rc20_2*e3rc01-e2rc12_2*e3rc01+e2rc11_2*e3rc01-e2rc10_2*e3rc01+e2rc02_2*e3rc01+3*e2rc01_2*e3rc01+e2rc00_2*e3rc01+2*e2rc20*e2rc21*e3rc00+2*e2rc10*e2rc11*e3rc00+2*e2rc00*e2rc01*e3rc00;
    AMatrix(1, 16) = -e0rc01*e3rc22_2+2*e0rc02*e3rc21*e3rc22+2*e0rc21*e3rc02*e3rc22-2*e0rc22*e3rc01*e3rc22+e0rc01*e3rc21_2+2*e0rc00*e3rc20*e3rc21+2*e0rc22*e3rc02*e3rc21+2*e0rc21*e3rc01*e3rc21+2*e0rc20*e3rc00*e3rc21-e0rc01*e3rc20_2-2*e0rc20*e3rc01*e3rc20+2*e0rc21*e3rc00*e3rc20-e0rc01*e3rc12_2+2*e0rc02*e3rc11*e3rc12+2*e0rc11*e3rc02*e3rc12-2*e0rc12*e3rc01*e3rc12+e0rc01*e3rc11_2+2*e0rc00*e3rc10*e3rc11+2*e0rc12*e3rc02*e3rc11+2*e0rc11*e3rc01*e3rc11+2*e0rc10*e3rc00*e3rc11-e0rc01*e3rc10_2-2*e0rc10*e3rc01*e3rc10+2*e0rc11*e3rc00*e3rc10+e0rc01*e3rc02_2+2*e0rc02*e3rc01*e3rc02+3*e0rc01*e3rc01_2+2*e0rc00*e3rc00*e3rc01+e0rc01*e3rc00_2;
    AMatrix(1, 17) = -e1rc01*e3rc22_2+2*e1rc02*e3rc21*e3rc22+2*e1rc21*e3rc02*e3rc22-2*e1rc22*e3rc01*e3rc22+e1rc01*e3rc21_2+2*e1rc00*e3rc20*e3rc21+2*e1rc22*e3rc02*e3rc21+2*e1rc21*e3rc01*e3rc21+2*e1rc20*e3rc00*e3rc21-e1rc01*e3rc20_2-2*e1rc20*e3rc01*e3rc20+2*e1rc21*e3rc00*e3rc20-e1rc01*e3rc12_2+2*e1rc02*e3rc11*e3rc12+2*e1rc11*e3rc02*e3rc12-2*e1rc12*e3rc01*e3rc12+e1rc01*e3rc11_2+2*e1rc00*e3rc10*e3rc11+2*e1rc12*e3rc02*e3rc11+2*e1rc11*e3rc01*e3rc11+2*e1rc10*e3rc00*e3rc11-e1rc01*e3rc10_2-2*e1rc10*e3rc01*e3rc10+2*e1rc11*e3rc00*e3rc10+e1rc01*e3rc02_2+2*e1rc02*e3rc01*e3rc02+3*e1rc01*e3rc01_2+2*e1rc00*e3rc00*e3rc01+e1rc01*e3rc00_2;
    AMatrix(1, 18) = -e2rc01*e3rc22_2+2*e2rc02*e3rc21*e3rc22+2*e2rc21*e3rc02*e3rc22-2*e2rc22*e3rc01*e3rc22+e2rc01*e3rc21_2+2*e2rc00*e3rc20*e3rc21+2*e2rc22*e3rc02*e3rc21+2*e2rc21*e3rc01*e3rc21+2*e2rc20*e3rc00*e3rc21-e2rc01*e3rc20_2-2*e2rc20*e3rc01*e3rc20+2*e2rc21*e3rc00*e3rc20-e2rc01*e3rc12_2+2*e2rc02*e3rc11*e3rc12+2*e2rc11*e3rc02*e3rc12-2*e2rc12*e3rc01*e3rc12+e2rc01*e3rc11_2+2*e2rc00*e3rc10*e3rc11+2*e2rc12*e3rc02*e3rc11+2*e2rc11*e3rc01*e3rc11+2*e2rc10*e3rc00*e3rc11-e2rc01*e3rc10_2-2*e2rc10*e3rc01*e3rc10+2*e2rc11*e3rc00*e3rc10+e2rc01*e3rc02_2+2*e2rc02*e3rc01*e3rc02+3*e2rc01*e3rc01_2+2*e2rc00*e3rc00*e3rc01+e2rc01*e3rc00_2;
    AMatrix(1, 19) = -e3rc01*e3rc22_2+2*e3rc02*e3rc21*e3rc22+e3rc01*e3rc21_2+2*e3rc00*e3rc20*e3rc21-e3rc01*e3rc20_2-e3rc01*e3rc12_2+2*e3rc02*e3rc11*e3rc12+e3rc01*e3rc11_2+2*e3rc00*e3rc10*e3rc11-e3rc01*e3rc10_2+e3rc01*e3rc02_2+e3rc01_3+e3rc00_2*e3rc01;
    AMatrix(2, 0) = +e0rc02*e0rc22_2+2*e0rc01*e0rc21*e0rc22+2*e0rc00*e0rc20*e0rc22-e0rc02*e0rc21_2-e0rc02*e0rc20_2+e0rc02*e0rc12_2+2*e0rc01*e0rc11*e0rc12+2*e0rc00*e0rc10*e0rc12-e0rc02*e0rc11_2-e0rc02*e0rc10_2+e0rc02_3+e0rc01_2*e0rc02+e0rc00_2*e0rc02;
    AMatrix(2, 1) = +2*e0rc02*e0rc22*e1rc22+2*e0rc01*e0rc21*e1rc22+2*e0rc00*e0rc20*e1rc22+2*e0rc01*e0rc22*e1rc21-2*e0rc02*e0rc21*e1rc21+2*e0rc00*e0rc22*e1rc20-2*e0rc02*e0rc20*e1rc20+2*e0rc02*e0rc12*e1rc12+2*e0rc01*e0rc11*e1rc12+2*e0rc00*e0rc10*e1rc12+2*e0rc01*e0rc12*e1rc11-2*e0rc02*e0rc11*e1rc11+2*e0rc00*e0rc12*e1rc10-2*e0rc02*e0rc10*e1rc10+e0rc22_2*e1rc02-e0rc21_2*e1rc02-e0rc20_2*e1rc02+e0rc12_2*e1rc02-e0rc11_2*e1rc02-e0rc10_2*e1rc02+3*e0rc02_2*e1rc02+e0rc01_2*e1rc02+e0rc00_2*e1rc02+2*e0rc21*e0rc22*e1rc01+2*e0rc11*e0rc12*e1rc01+2*e0rc01*e0rc02*e1rc01+2*e0rc20*e0rc22*e1rc00+2*e0rc10*e0rc12*e1rc00+2*e0rc00*e0rc02*e1rc00;
    AMatrix(2, 2) = +2*e0rc02*e0rc22*e2rc22+2*e0rc01*e0rc21*e2rc22+2*e0rc00*e0rc20*e2rc22+2*e0rc01*e0rc22*e2rc21-2*e0rc02*e0rc21*e2rc21+2*e0rc00*e0rc22*e2rc20-2*e0rc02*e0rc20*e2rc20+2*e0rc02*e0rc12*e2rc12+2*e0rc01*e0rc11*e2rc12+2*e0rc00*e0rc10*e2rc12+2*e0rc01*e0rc12*e2rc11-2*e0rc02*e0rc11*e2rc11+2*e0rc00*e0rc12*e2rc10-2*e0rc02*e0rc10*e2rc10+e0rc22_2*e2rc02-e0rc21_2*e2rc02-e0rc20_2*e2rc02+e0rc12_2*e2rc02-e0rc11_2*e2rc02-e0rc10_2*e2rc02+3*e0rc02_2*e2rc02+e0rc01_2*e2rc02+e0rc00_2*e2rc02+2*e0rc21*e0rc22*e2rc01+2*e0rc11*e0rc12*e2rc01+2*e0rc01*e0rc02*e2rc01+2*e0rc20*e0rc22*e2rc00+2*e0rc10*e0rc12*e2rc00+2*e0rc00*e0rc02*e2rc00;
    AMatrix(2, 3) = +e0rc02*e1rc22_2+2*e0rc01*e1rc21*e1rc22+2*e0rc00*e1rc20*e1rc22+2*e0rc22*e1rc02*e1rc22+2*e0rc21*e1rc01*e1rc22+2*e0rc20*e1rc00*e1rc22-e0rc02*e1rc21_2-2*e0rc21*e1rc02*e1rc21+2*e0rc22*e1rc01*e1rc21-e0rc02*e1rc20_2-2*e0rc20*e1rc02*e1rc20+2*e0rc22*e1rc00*e1rc20+e0rc02*e1rc12_2+2*e0rc01*e1rc11*e1rc12+2*e0rc00*e1rc10*e1rc12+2*e0rc12*e1rc02*e1rc12+2*e0rc11*e1rc01*e1rc12+2*e0rc10*e1rc00*e1rc12-e0rc02*e1rc11_2-2*e0rc11*e1rc02*e1rc11+2*e0rc12*e1rc01*e1rc11-e0rc02*e1rc10_2-2*e0rc10*e1rc02*e1rc10+2*e0rc12*e1rc00*e1rc10+3*e0rc02*e1rc02_2+2*e0rc01*e1rc01*e1rc02+2*e0rc00*e1rc00*e1rc02+e0rc02*e1rc01_2+e0rc02*e1rc00_2;
    AMatrix(2, 4) = +2*e0rc02*e1rc22*e2rc22+2*e0rc01*e1rc21*e2rc22+2*e0rc00*e1rc20*e2rc22+2*e0rc22*e1rc02*e2rc22+2*e0rc21*e1rc01*e2rc22+2*e0rc20*e1rc00*e2rc22+2*e0rc01*e1rc22*e2rc21-2*e0rc02*e1rc21*e2rc21-2*e0rc21*e1rc02*e2rc21+2*e0rc22*e1rc01*e2rc21+2*e0rc00*e1rc22*e2rc20-2*e0rc02*e1rc20*e2rc20-2*e0rc20*e1rc02*e2rc20+2*e0rc22*e1rc00*e2rc20+2*e0rc02*e1rc12*e2rc12+2*e0rc01*e1rc11*e2rc12+2*e0rc00*e1rc10*e2rc12+2*e0rc12*e1rc02*e2rc12+2*e0rc11*e1rc01*e2rc12+2*e0rc10*e1rc00*e2rc12+2*e0rc01*e1rc12*e2rc11-2*e0rc02*e1rc11*e2rc11-2*e0rc11*e1rc02*e2rc11+2*e0rc12*e1rc01*e2rc11+2*e0rc00*e1rc12*e2rc10-2*e0rc02*e1rc10*e2rc10-2*e0rc10*e1rc02*e2rc10+2*e0rc12*e1rc00*e2rc10+2*e0rc22*e1rc22*e2rc02-2*e0rc21*e1rc21*e2rc02-2*e0rc20*e1rc20*e2rc02+2*e0rc12*e1rc12*e2rc02-2*e0rc11*e1rc11*e2rc02-2*e0rc10*e1rc10*e2rc02+6*e0rc02*e1rc02*e2rc02+2*e0rc01*e1rc01*e2rc02+2*e0rc00*e1rc00*e2rc02+2*e0rc21*e1rc22*e2rc01+2*e0rc22*e1rc21*e2rc01+2*e0rc11*e1rc12*e2rc01+2*e0rc12*e1rc11*e2rc01+2*e0rc01*e1rc02*e2rc01+2*e0rc02*e1rc01*e2rc01+2*e0rc20*e1rc22*e2rc00+2*e0rc22*e1rc20*e2rc00+2*e0rc10*e1rc12*e2rc00+2*e0rc12*e1rc10*e2rc00+2*e0rc00*e1rc02*e2rc00+2*e0rc02*e1rc00*e2rc00;
    AMatrix(2, 5) = +e0rc02*e2rc22_2+2*e0rc01*e2rc21*e2rc22+2*e0rc00*e2rc20*e2rc22+2*e0rc22*e2rc02*e2rc22+2*e0rc21*e2rc01*e2rc22+2*e0rc20*e2rc00*e2rc22-e0rc02*e2rc21_2-2*e0rc21*e2rc02*e2rc21+2*e0rc22*e2rc01*e2rc21-e0rc02*e2rc20_2-2*e0rc20*e2rc02*e2rc20+2*e0rc22*e2rc00*e2rc20+e0rc02*e2rc12_2+2*e0rc01*e2rc11*e2rc12+2*e0rc00*e2rc10*e2rc12+2*e0rc12*e2rc02*e2rc12+2*e0rc11*e2rc01*e2rc12+2*e0rc10*e2rc00*e2rc12-e0rc02*e2rc11_2-2*e0rc11*e2rc02*e2rc11+2*e0rc12*e2rc01*e2rc11-e0rc02*e2rc10_2-2*e0rc10*e2rc02*e2rc10+2*e0rc12*e2rc00*e2rc10+3*e0rc02*e2rc02_2+2*e0rc01*e2rc01*e2rc02+2*e0rc00*e2rc00*e2rc02+e0rc02*e2rc01_2+e0rc02*e2rc00_2;
    AMatrix(2, 6) = +e1rc02*e1rc22_2+2*e1rc01*e1rc21*e1rc22+2*e1rc00*e1rc20*e1rc22-e1rc02*e1rc21_2-e1rc02*e1rc20_2+e1rc02*e1rc12_2+2*e1rc01*e1rc11*e1rc12+2*e1rc00*e1rc10*e1rc12-e1rc02*e1rc11_2-e1rc02*e1rc10_2+e1rc02_3+e1rc01_2*e1rc02+e1rc00_2*e1rc02;
    AMatrix(2, 7) = +2*e1rc02*e1rc22*e2rc22+2*e1rc01*e1rc21*e2rc22+2*e1rc00*e1rc20*e2rc22+2*e1rc01*e1rc22*e2rc21-2*e1rc02*e1rc21*e2rc21+2*e1rc00*e1rc22*e2rc20-2*e1rc02*e1rc20*e2rc20+2*e1rc02*e1rc12*e2rc12+2*e1rc01*e1rc11*e2rc12+2*e1rc00*e1rc10*e2rc12+2*e1rc01*e1rc12*e2rc11-2*e1rc02*e1rc11*e2rc11+2*e1rc00*e1rc12*e2rc10-2*e1rc02*e1rc10*e2rc10+e1rc22_2*e2rc02-e1rc21_2*e2rc02-e1rc20_2*e2rc02+e1rc12_2*e2rc02-e1rc11_2*e2rc02-e1rc10_2*e2rc02+3*e1rc02_2*e2rc02+e1rc01_2*e2rc02+e1rc00_2*e2rc02+2*e1rc21*e1rc22*e2rc01+2*e1rc11*e1rc12*e2rc01+2*e1rc01*e1rc02*e2rc01+2*e1rc20*e1rc22*e2rc00+2*e1rc10*e1rc12*e2rc00+2*e1rc00*e1rc02*e2rc00;
    AMatrix(2, 8) = +e1rc02*e2rc22_2+2*e1rc01*e2rc21*e2rc22+2*e1rc00*e2rc20*e2rc22+2*e1rc22*e2rc02*e2rc22+2*e1rc21*e2rc01*e2rc22+2*e1rc20*e2rc00*e2rc22-e1rc02*e2rc21_2-2*e1rc21*e2rc02*e2rc21+2*e1rc22*e2rc01*e2rc21-e1rc02*e2rc20_2-2*e1rc20*e2rc02*e2rc20+2*e1rc22*e2rc00*e2rc20+e1rc02*e2rc12_2+2*e1rc01*e2rc11*e2rc12+2*e1rc00*e2rc10*e2rc12+2*e1rc12*e2rc02*e2rc12+2*e1rc11*e2rc01*e2rc12+2*e1rc10*e2rc00*e2rc12-e1rc02*e2rc11_2-2*e1rc11*e2rc02*e2rc11+2*e1rc12*e2rc01*e2rc11-e1rc02*e2rc10_2-2*e1rc10*e2rc02*e2rc10+2*e1rc12*e2rc00*e2rc10+3*e1rc02*e2rc02_2+2*e1rc01*e2rc01*e2rc02+2*e1rc00*e2rc00*e2rc02+e1rc02*e2rc01_2+e1rc02*e2rc00_2;
    AMatrix(2, 9) = e2rc02*e2rc22_2+2*e2rc01*e2rc21*e2rc22+2*e2rc00*e2rc20*e2rc22-e2rc02*e2rc21_2-e2rc02*e2rc20_2+e2rc02*e2rc12_2+2*e2rc01*e2rc11*e2rc12+2*e2rc00*e2rc10*e2rc12-e2rc02*e2rc11_2-e2rc02*e2rc10_2+e2rc02_3+e2rc01_2*e2rc02+e2rc00_2*e2rc02;
    AMatrix(2, 10) = +2*e0rc02*e0rc22*e3rc22+2*e0rc01*e0rc21*e3rc22+2*e0rc00*e0rc20*e3rc22+2*e0rc01*e0rc22*e3rc21-2*e0rc02*e0rc21*e3rc21+2*e0rc00*e0rc22*e3rc20-2*e0rc02*e0rc20*e3rc20+2*e0rc02*e0rc12*e3rc12+2*e0rc01*e0rc11*e3rc12+2*e0rc00*e0rc10*e3rc12+2*e0rc01*e0rc12*e3rc11-2*e0rc02*e0rc11*e3rc11+2*e0rc00*e0rc12*e3rc10-2*e0rc02*e0rc10*e3rc10+e0rc22_2*e3rc02-e0rc21_2*e3rc02-e0rc20_2*e3rc02+e0rc12_2*e3rc02-e0rc11_2*e3rc02-e0rc10_2*e3rc02+3*e0rc02_2*e3rc02+e0rc01_2*e3rc02+e0rc00_2*e3rc02+2*e0rc21*e0rc22*e3rc01+2*e0rc11*e0rc12*e3rc01+2*e0rc01*e0rc02*e3rc01+2*e0rc20*e0rc22*e3rc00+2*e0rc10*e0rc12*e3rc00+2*e0rc00*e0rc02*e3rc00;
    AMatrix(2, 11) = +2*e0rc02*e1rc22*e3rc22+2*e0rc01*e1rc21*e3rc22+2*e0rc00*e1rc20*e3rc22+2*e0rc22*e1rc02*e3rc22+2*e0rc21*e1rc01*e3rc22+2*e0rc20*e1rc00*e3rc22+2*e0rc01*e1rc22*e3rc21-2*e0rc02*e1rc21*e3rc21-2*e0rc21*e1rc02*e3rc21+2*e0rc22*e1rc01*e3rc21+2*e0rc00*e1rc22*e3rc20-2*e0rc02*e1rc20*e3rc20-2*e0rc20*e1rc02*e3rc20+2*e0rc22*e1rc00*e3rc20+2*e0rc02*e1rc12*e3rc12+2*e0rc01*e1rc11*e3rc12+2*e0rc00*e1rc10*e3rc12+2*e0rc12*e1rc02*e3rc12+2*e0rc11*e1rc01*e3rc12+2*e0rc10*e1rc00*e3rc12+2*e0rc01*e1rc12*e3rc11-2*e0rc02*e1rc11*e3rc11-2*e0rc11*e1rc02*e3rc11+2*e0rc12*e1rc01*e3rc11+2*e0rc00*e1rc12*e3rc10-2*e0rc02*e1rc10*e3rc10-2*e0rc10*e1rc02*e3rc10+2*e0rc12*e1rc00*e3rc10+2*e0rc22*e1rc22*e3rc02-2*e0rc21*e1rc21*e3rc02-2*e0rc20*e1rc20*e3rc02+2*e0rc12*e1rc12*e3rc02-2*e0rc11*e1rc11*e3rc02-2*e0rc10*e1rc10*e3rc02+6*e0rc02*e1rc02*e3rc02+2*e0rc01*e1rc01*e3rc02+2*e0rc00*e1rc00*e3rc02+2*e0rc21*e1rc22*e3rc01+2*e0rc22*e1rc21*e3rc01+2*e0rc11*e1rc12*e3rc01+2*e0rc12*e1rc11*e3rc01+2*e0rc01*e1rc02*e3rc01+2*e0rc02*e1rc01*e3rc01+2*e0rc20*e1rc22*e3rc00+2*e0rc22*e1rc20*e3rc00+2*e0rc10*e1rc12*e3rc00+2*e0rc12*e1rc10*e3rc00+2*e0rc00*e1rc02*e3rc00+2*e0rc02*e1rc00*e3rc00;
    AMatrix(2, 12) = +2*e0rc02*e2rc22*e3rc22+2*e0rc01*e2rc21*e3rc22+2*e0rc00*e2rc20*e3rc22+2*e0rc22*e2rc02*e3rc22+2*e0rc21*e2rc01*e3rc22+2*e0rc20*e2rc00*e3rc22+2*e0rc01*e2rc22*e3rc21-2*e0rc02*e2rc21*e3rc21-2*e0rc21*e2rc02*e3rc21+2*e0rc22*e2rc01*e3rc21+2*e0rc00*e2rc22*e3rc20-2*e0rc02*e2rc20*e3rc20-2*e0rc20*e2rc02*e3rc20+2*e0rc22*e2rc00*e3rc20+2*e0rc02*e2rc12*e3rc12+2*e0rc01*e2rc11*e3rc12+2*e0rc00*e2rc10*e3rc12+2*e0rc12*e2rc02*e3rc12+2*e0rc11*e2rc01*e3rc12+2*e0rc10*e2rc00*e3rc12+2*e0rc01*e2rc12*e3rc11-2*e0rc02*e2rc11*e3rc11-2*e0rc11*e2rc02*e3rc11+2*e0rc12*e2rc01*e3rc11+2*e0rc00*e2rc12*e3rc10-2*e0rc02*e2rc10*e3rc10-2*e0rc10*e2rc02*e3rc10+2*e0rc12*e2rc00*e3rc10+2*e0rc22*e2rc22*e3rc02-2*e0rc21*e2rc21*e3rc02-2*e0rc20*e2rc20*e3rc02+2*e0rc12*e2rc12*e3rc02-2*e0rc11*e2rc11*e3rc02-2*e0rc10*e2rc10*e3rc02+6*e0rc02*e2rc02*e3rc02+2*e0rc01*e2rc01*e3rc02+2*e0rc00*e2rc00*e3rc02+2*e0rc21*e2rc22*e3rc01+2*e0rc22*e2rc21*e3rc01+2*e0rc11*e2rc12*e3rc01+2*e0rc12*e2rc11*e3rc01+2*e0rc01*e2rc02*e3rc01+2*e0rc02*e2rc01*e3rc01+2*e0rc20*e2rc22*e3rc00+2*e0rc22*e2rc20*e3rc00+2*e0rc10*e2rc12*e3rc00+2*e0rc12*e2rc10*e3rc00+2*e0rc00*e2rc02*e3rc00+2*e0rc02*e2rc00*e3rc00;
    AMatrix(2, 13) = +2*e1rc02*e1rc22*e3rc22+2*e1rc01*e1rc21*e3rc22+2*e1rc00*e1rc20*e3rc22+2*e1rc01*e1rc22*e3rc21-2*e1rc02*e1rc21*e3rc21+2*e1rc00*e1rc22*e3rc20-2*e1rc02*e1rc20*e3rc20+2*e1rc02*e1rc12*e3rc12+2*e1rc01*e1rc11*e3rc12+2*e1rc00*e1rc10*e3rc12+2*e1rc01*e1rc12*e3rc11-2*e1rc02*e1rc11*e3rc11+2*e1rc00*e1rc12*e3rc10-2*e1rc02*e1rc10*e3rc10+e1rc22_2*e3rc02-e1rc21_2*e3rc02-e1rc20_2*e3rc02+e1rc12_2*e3rc02-e1rc11_2*e3rc02-e1rc10_2*e3rc02+3*e1rc02_2*e3rc02+e1rc01_2*e3rc02+e1rc00_2*e3rc02+2*e1rc21*e1rc22*e3rc01+2*e1rc11*e1rc12*e3rc01+2*e1rc01*e1rc02*e3rc01+2*e1rc20*e1rc22*e3rc00+2*e1rc10*e1rc12*e3rc00+2*e1rc00*e1rc02*e3rc00;
    AMatrix(2, 14) = +2*e1rc02*e2rc22*e3rc22+2*e1rc01*e2rc21*e3rc22+2*e1rc00*e2rc20*e3rc22+2*e1rc22*e2rc02*e3rc22+2*e1rc21*e2rc01*e3rc22+2*e1rc20*e2rc00*e3rc22+2*e1rc01*e2rc22*e3rc21-2*e1rc02*e2rc21*e3rc21-2*e1rc21*e2rc02*e3rc21+2*e1rc22*e2rc01*e3rc21+2*e1rc00*e2rc22*e3rc20-2*e1rc02*e2rc20*e3rc20-2*e1rc20*e2rc02*e3rc20+2*e1rc22*e2rc00*e3rc20+2*e1rc02*e2rc12*e3rc12+2*e1rc01*e2rc11*e3rc12+2*e1rc00*e2rc10*e3rc12+2*e1rc12*e2rc02*e3rc12+2*e1rc11*e2rc01*e3rc12+2*e1rc10*e2rc00*e3rc12+2*e1rc01*e2rc12*e3rc11-2*e1rc02*e2rc11*e3rc11-2*e1rc11*e2rc02*e3rc11+2*e1rc12*e2rc01*e3rc11+2*e1rc00*e2rc12*e3rc10-2*e1rc02*e2rc10*e3rc10-2*e1rc10*e2rc02*e3rc10+2*e1rc12*e2rc00*e3rc10+2*e1rc22*e2rc22*e3rc02-2*e1rc21*e2rc21*e3rc02-2*e1rc20*e2rc20*e3rc02+2*e1rc12*e2rc12*e3rc02-2*e1rc11*e2rc11*e3rc02-2*e1rc10*e2rc10*e3rc02+6*e1rc02*e2rc02*e3rc02+2*e1rc01*e2rc01*e3rc02+2*e1rc00*e2rc00*e3rc02+2*e1rc21*e2rc22*e3rc01+2*e1rc22*e2rc21*e3rc01+2*e1rc11*e2rc12*e3rc01+2*e1rc12*e2rc11*e3rc01+2*e1rc01*e2rc02*e3rc01+2*e1rc02*e2rc01*e3rc01+2*e1rc20*e2rc22*e3rc00+2*e1rc22*e2rc20*e3rc00+2*e1rc10*e2rc12*e3rc00+2*e1rc12*e2rc10*e3rc00+2*e1rc00*e2rc02*e3rc00+2*e1rc02*e2rc00*e3rc00;
    AMatrix(2, 15) = +2*e2rc02*e2rc22*e3rc22+2*e2rc01*e2rc21*e3rc22+2*e2rc00*e2rc20*e3rc22+2*e2rc01*e2rc22*e3rc21-2*e2rc02*e2rc21*e3rc21+2*e2rc00*e2rc22*e3rc20-2*e2rc02*e2rc20*e3rc20+2*e2rc02*e2rc12*e3rc12+2*e2rc01*e2rc11*e3rc12+2*e2rc00*e2rc10*e3rc12+2*e2rc01*e2rc12*e3rc11-2*e2rc02*e2rc11*e3rc11+2*e2rc00*e2rc12*e3rc10-2*e2rc02*e2rc10*e3rc10+e2rc22_2*e3rc02-e2rc21_2*e3rc02-e2rc20_2*e3rc02+e2rc12_2*e3rc02-e2rc11_2*e3rc02-e2rc10_2*e3rc02+3*e2rc02_2*e3rc02+e2rc01_2*e3rc02+e2rc00_2*e3rc02+2*e2rc21*e2rc22*e3rc01+2*e2rc11*e2rc12*e3rc01+2*e2rc01*e2rc02*e3rc01+2*e2rc20*e2rc22*e3rc00+2*e2rc10*e2rc12*e3rc00+2*e2rc00*e2rc02*e3rc00;
    AMatrix(2, 16) = +e0rc02*e3rc22_2+2*e0rc01*e3rc21*e3rc22+2*e0rc00*e3rc20*e3rc22+2*e0rc22*e3rc02*e3rc22+2*e0rc21*e3rc01*e3rc22+2*e0rc20*e3rc00*e3rc22-e0rc02*e3rc21_2-2*e0rc21*e3rc02*e3rc21+2*e0rc22*e3rc01*e3rc21-e0rc02*e3rc20_2-2*e0rc20*e3rc02*e3rc20+2*e0rc22*e3rc00*e3rc20+e0rc02*e3rc12_2+2*e0rc01*e3rc11*e3rc12+2*e0rc00*e3rc10*e3rc12+2*e0rc12*e3rc02*e3rc12+2*e0rc11*e3rc01*e3rc12+2*e0rc10*e3rc00*e3rc12-e0rc02*e3rc11_2-2*e0rc11*e3rc02*e3rc11+2*e0rc12*e3rc01*e3rc11-e0rc02*e3rc10_2-2*e0rc10*e3rc02*e3rc10+2*e0rc12*e3rc00*e3rc10+3*e0rc02*e3rc02_2+2*e0rc01*e3rc01*e3rc02+2*e0rc00*e3rc00*e3rc02+e0rc02*e3rc01_2+e0rc02*e3rc00_2;
    AMatrix(2, 17) = +e1rc02*e3rc22_2+2*e1rc01*e3rc21*e3rc22+2*e1rc00*e3rc20*e3rc22+2*e1rc22*e3rc02*e3rc22+2*e1rc21*e3rc01*e3rc22+2*e1rc20*e3rc00*e3rc22-e1rc02*e3rc21_2-2*e1rc21*e3rc02*e3rc21+2*e1rc22*e3rc01*e3rc21-e1rc02*e3rc20_2-2*e1rc20*e3rc02*e3rc20+2*e1rc22*e3rc00*e3rc20+e1rc02*e3rc12_2+2*e1rc01*e3rc11*e3rc12+2*e1rc00*e3rc10*e3rc12+2*e1rc12*e3rc02*e3rc12+2*e1rc11*e3rc01*e3rc12+2*e1rc10*e3rc00*e3rc12-e1rc02*e3rc11_2-2*e1rc11*e3rc02*e3rc11+2*e1rc12*e3rc01*e3rc11-e1rc02*e3rc10_2-2*e1rc10*e3rc02*e3rc10+2*e1rc12*e3rc00*e3rc10+3*e1rc02*e3rc02_2+2*e1rc01*e3rc01*e3rc02+2*e1rc00*e3rc00*e3rc02+e1rc02*e3rc01_2+e1rc02*e3rc00_2;
    AMatrix(2, 18) = +e2rc02*e3rc22_2+2*e2rc01*e3rc21*e3rc22+2*e2rc00*e3rc20*e3rc22+2*e2rc22*e3rc02*e3rc22+2*e2rc21*e3rc01*e3rc22+2*e2rc20*e3rc00*e3rc22-e2rc02*e3rc21_2-2*e2rc21*e3rc02*e3rc21+2*e2rc22*e3rc01*e3rc21-e2rc02*e3rc20_2-2*e2rc20*e3rc02*e3rc20+2*e2rc22*e3rc00*e3rc20+e2rc02*e3rc12_2+2*e2rc01*e3rc11*e3rc12+2*e2rc00*e3rc10*e3rc12+2*e2rc12*e3rc02*e3rc12+2*e2rc11*e3rc01*e3rc12+2*e2rc10*e3rc00*e3rc12-e2rc02*e3rc11_2-2*e2rc11*e3rc02*e3rc11+2*e2rc12*e3rc01*e3rc11-e2rc02*e3rc10_2-2*e2rc10*e3rc02*e3rc10+2*e2rc12*e3rc00*e3rc10+3*e2rc02*e3rc02_2+2*e2rc01*e3rc01*e3rc02+2*e2rc00*e3rc00*e3rc02+e2rc02*e3rc01_2+e2rc02*e3rc00_2;
    AMatrix(2, 19) = +e3rc02*e3rc22_2+2*e3rc01*e3rc21*e3rc22+2*e3rc00*e3rc20*e3rc22-e3rc02*e3rc21_2-e3rc02*e3rc20_2+e3rc02*e3rc12_2+2*e3rc01*e3rc11*e3rc12+2*e3rc00*e3rc10*e3rc12-e3rc02*e3rc11_2-e3rc02*e3rc10_2+e3rc02_3+e3rc01_2*e3rc02+e3rc00_2*e3rc02;
    AMatrix(3, 0) = -e0rc10*e0rc22_2+2*e0rc12*e0rc20*e0rc22-e0rc10*e0rc21_2+2*e0rc11*e0rc20*e0rc21+e0rc10*e0rc20_2+e0rc10*e0rc12_2+2*e0rc00*e0rc02*e0rc12+e0rc10*e0rc11_2+2*e0rc00*e0rc01*e0rc11+e0rc10_3-e0rc02_2*e0rc10-e0rc01_2*e0rc10+e0rc00_2*e0rc10;
    AMatrix(3, 1) = -2*e0rc10*e0rc22*e1rc22+2*e0rc12*e0rc20*e1rc22-2*e0rc10*e0rc21*e1rc21+2*e0rc11*e0rc20*e1rc21+2*e0rc12*e0rc22*e1rc20+2*e0rc11*e0rc21*e1rc20+2*e0rc10*e0rc20*e1rc20+2*e0rc20*e0rc22*e1rc12+2*e0rc10*e0rc12*e1rc12+2*e0rc00*e0rc02*e1rc12+2*e0rc20*e0rc21*e1rc11+2*e0rc10*e0rc11*e1rc11+2*e0rc00*e0rc01*e1rc11-e0rc22_2*e1rc10-e0rc21_2*e1rc10+e0rc20_2*e1rc10+e0rc12_2*e1rc10+e0rc11_2*e1rc10+3*e0rc10_2*e1rc10-e0rc02_2*e1rc10-e0rc01_2*e1rc10+e0rc00_2*e1rc10+2*e0rc00*e0rc12*e1rc02-2*e0rc02*e0rc10*e1rc02+2*e0rc00*e0rc11*e1rc01-2*e0rc01*e0rc10*e1rc01+2*e0rc02*e0rc12*e1rc00+2*e0rc01*e0rc11*e1rc00+2*e0rc00*e0rc10*e1rc00;
    AMatrix(3, 2) = -2*e0rc10*e0rc22*e2rc22+2*e0rc12*e0rc20*e2rc22-2*e0rc10*e0rc21*e2rc21+2*e0rc11*e0rc20*e2rc21+2*e0rc12*e0rc22*e2rc20+2*e0rc11*e0rc21*e2rc20+2*e0rc10*e0rc20*e2rc20+2*e0rc20*e0rc22*e2rc12+2*e0rc10*e0rc12*e2rc12+2*e0rc00*e0rc02*e2rc12+2*e0rc20*e0rc21*e2rc11+2*e0rc10*e0rc11*e2rc11+2*e0rc00*e0rc01*e2rc11-e0rc22_2*e2rc10-e0rc21_2*e2rc10+e0rc20_2*e2rc10+e0rc12_2*e2rc10+e0rc11_2*e2rc10+3*e0rc10_2*e2rc10-e0rc02_2*e2rc10-e0rc01_2*e2rc10+e0rc00_2*e2rc10+2*e0rc00*e0rc12*e2rc02-2*e0rc02*e0rc10*e2rc02+2*e0rc00*e0rc11*e2rc01-2*e0rc01*e0rc10*e2rc01+2*e0rc02*e0rc12*e2rc00+2*e0rc01*e0rc11*e2rc00+2*e0rc00*e0rc10*e2rc00;
    AMatrix(3, 3) = -e0rc10*e1rc22_2+2*e0rc12*e1rc20*e1rc22+2*e0rc20*e1rc12*e1rc22-2*e0rc22*e1rc10*e1rc22-e0rc10*e1rc21_2+2*e0rc11*e1rc20*e1rc21+2*e0rc20*e1rc11*e1rc21-2*e0rc21*e1rc10*e1rc21+e0rc10*e1rc20_2+2*e0rc22*e1rc12*e1rc20+2*e0rc21*e1rc11*e1rc20+2*e0rc20*e1rc10*e1rc20+e0rc10*e1rc12_2+2*e0rc12*e1rc10*e1rc12+2*e0rc00*e1rc02*e1rc12+2*e0rc02*e1rc00*e1rc12+e0rc10*e1rc11_2+2*e0rc11*e1rc10*e1rc11+2*e0rc00*e1rc01*e1rc11+2*e0rc01*e1rc00*e1rc11+3*e0rc10*e1rc10_2-2*e0rc02*e1rc02*e1rc10-2*e0rc01*e1rc01*e1rc10+2*e0rc00*e1rc00*e1rc10-e0rc10*e1rc02_2+2*e0rc12*e1rc00*e1rc02-e0rc10*e1rc01_2+2*e0rc11*e1rc00*e1rc01+e0rc10*e1rc00_2;
    AMatrix(3, 4) = -2*e0rc10*e1rc22*e2rc22+2*e0rc12*e1rc20*e2rc22+2*e0rc20*e1rc12*e2rc22-2*e0rc22*e1rc10*e2rc22-2*e0rc10*e1rc21*e2rc21+2*e0rc11*e1rc20*e2rc21+2*e0rc20*e1rc11*e2rc21-2*e0rc21*e1rc10*e2rc21+2*e0rc12*e1rc22*e2rc20+2*e0rc11*e1rc21*e2rc20+2*e0rc10*e1rc20*e2rc20+2*e0rc22*e1rc12*e2rc20+2*e0rc21*e1rc11*e2rc20+2*e0rc20*e1rc10*e2rc20+2*e0rc20*e1rc22*e2rc12+2*e0rc22*e1rc20*e2rc12+2*e0rc10*e1rc12*e2rc12+2*e0rc12*e1rc10*e2rc12+2*e0rc00*e1rc02*e2rc12+2*e0rc02*e1rc00*e2rc12+2*e0rc20*e1rc21*e2rc11+2*e0rc21*e1rc20*e2rc11+2*e0rc10*e1rc11*e2rc11+2*e0rc11*e1rc10*e2rc11+2*e0rc00*e1rc01*e2rc11+2*e0rc01*e1rc00*e2rc11-2*e0rc22*e1rc22*e2rc10-2*e0rc21*e1rc21*e2rc10+2*e0rc20*e1rc20*e2rc10+2*e0rc12*e1rc12*e2rc10+2*e0rc11*e1rc11*e2rc10+6*e0rc10*e1rc10*e2rc10-2*e0rc02*e1rc02*e2rc10-2*e0rc01*e1rc01*e2rc10+2*e0rc00*e1rc00*e2rc10+2*e0rc00*e1rc12*e2rc02-2*e0rc02*e1rc10*e2rc02-2*e0rc10*e1rc02*e2rc02+2*e0rc12*e1rc00*e2rc02+2*e0rc00*e1rc11*e2rc01-2*e0rc01*e1rc10*e2rc01-2*e0rc10*e1rc01*e2rc01+2*e0rc11*e1rc00*e2rc01+2*e0rc02*e1rc12*e2rc00+2*e0rc01*e1rc11*e2rc00+2*e0rc00*e1rc10*e2rc00+2*e0rc12*e1rc02*e2rc00+2*e0rc11*e1rc01*e2rc00+2*e0rc10*e1rc00*e2rc00;
    AMatrix(3, 5) = -e0rc10*e2rc22_2+2*e0rc12*e2rc20*e2rc22+2*e0rc20*e2rc12*e2rc22-2*e0rc22*e2rc10*e2rc22-e0rc10*e2rc21_2+2*e0rc11*e2rc20*e2rc21+2*e0rc20*e2rc11*e2rc21-2*e0rc21*e2rc10*e2rc21+e0rc10*e2rc20_2+2*e0rc22*e2rc12*e2rc20+2*e0rc21*e2rc11*e2rc20+2*e0rc20*e2rc10*e2rc20+e0rc10*e2rc12_2+2*e0rc12*e2rc10*e2rc12+2*e0rc00*e2rc02*e2rc12+2*e0rc02*e2rc00*e2rc12+e0rc10*e2rc11_2+2*e0rc11*e2rc10*e2rc11+2*e0rc00*e2rc01*e2rc11+2*e0rc01*e2rc00*e2rc11+3*e0rc10*e2rc10_2-2*e0rc02*e2rc02*e2rc10-2*e0rc01*e2rc01*e2rc10+2*e0rc00*e2rc00*e2rc10-e0rc10*e2rc02_2+2*e0rc12*e2rc00*e2rc02-e0rc10*e2rc01_2+2*e0rc11*e2rc00*e2rc01+e0rc10*e2rc00_2;
    AMatrix(3, 6) = -e1rc10*e1rc22_2+2*e1rc12*e1rc20*e1rc22-e1rc10*e1rc21_2+2*e1rc11*e1rc20*e1rc21+e1rc10*e1rc20_2+e1rc10*e1rc12_2+2*e1rc00*e1rc02*e1rc12+e1rc10*e1rc11_2+2*e1rc00*e1rc01*e1rc11+e1rc10_3-e1rc02_2*e1rc10-e1rc01_2*e1rc10+e1rc00_2*e1rc10;
    AMatrix(3, 7) = -2*e1rc10*e1rc22*e2rc22+2*e1rc12*e1rc20*e2rc22-2*e1rc10*e1rc21*e2rc21+2*e1rc11*e1rc20*e2rc21+2*e1rc12*e1rc22*e2rc20+2*e1rc11*e1rc21*e2rc20+2*e1rc10*e1rc20*e2rc20+2*e1rc20*e1rc22*e2rc12+2*e1rc10*e1rc12*e2rc12+2*e1rc00*e1rc02*e2rc12+2*e1rc20*e1rc21*e2rc11+2*e1rc10*e1rc11*e2rc11+2*e1rc00*e1rc01*e2rc11-e1rc22_2*e2rc10-e1rc21_2*e2rc10+e1rc20_2*e2rc10+e1rc12_2*e2rc10+e1rc11_2*e2rc10+3*e1rc10_2*e2rc10-e1rc02_2*e2rc10-e1rc01_2*e2rc10+e1rc00_2*e2rc10+2*e1rc00*e1rc12*e2rc02-2*e1rc02*e1rc10*e2rc02+2*e1rc00*e1rc11*e2rc01-2*e1rc01*e1rc10*e2rc01+2*e1rc02*e1rc12*e2rc00+2*e1rc01*e1rc11*e2rc00+2*e1rc00*e1rc10*e2rc00;
    AMatrix(3, 8) = -e1rc10*e2rc22_2+2*e1rc12*e2rc20*e2rc22+2*e1rc20*e2rc12*e2rc22-2*e1rc22*e2rc10*e2rc22-e1rc10*e2rc21_2+2*e1rc11*e2rc20*e2rc21+2*e1rc20*e2rc11*e2rc21-2*e1rc21*e2rc10*e2rc21+e1rc10*e2rc20_2+2*e1rc22*e2rc12*e2rc20+2*e1rc21*e2rc11*e2rc20+2*e1rc20*e2rc10*e2rc20+e1rc10*e2rc12_2+2*e1rc12*e2rc10*e2rc12+2*e1rc00*e2rc02*e2rc12+2*e1rc02*e2rc00*e2rc12+e1rc10*e2rc11_2+2*e1rc11*e2rc10*e2rc11+2*e1rc00*e2rc01*e2rc11+2*e1rc01*e2rc00*e2rc11+3*e1rc10*e2rc10_2-2*e1rc02*e2rc02*e2rc10-2*e1rc01*e2rc01*e2rc10+2*e1rc00*e2rc00*e2rc10-e1rc10*e2rc02_2+2*e1rc12*e2rc00*e2rc02-e1rc10*e2rc01_2+2*e1rc11*e2rc00*e2rc01+e1rc10*e2rc00_2;
    AMatrix(3, 9) = -e2rc10*e2rc22_2+2*e2rc12*e2rc20*e2rc22-e2rc10*e2rc21_2+2*e2rc11*e2rc20*e2rc21+e2rc10*e2rc20_2+e2rc10*e2rc12_2+2*e2rc00*e2rc02*e2rc12+e2rc10*e2rc11_2+2*e2rc00*e2rc01*e2rc11+e2rc10_3-e2rc02_2*e2rc10-e2rc01_2*e2rc10+e2rc00_2*e2rc10;
    AMatrix(3, 10) = -2*e0rc10*e0rc22*e3rc22+2*e0rc12*e0rc20*e3rc22-2*e0rc10*e0rc21*e3rc21+2*e0rc11*e0rc20*e3rc21+2*e0rc12*e0rc22*e3rc20+2*e0rc11*e0rc21*e3rc20+2*e0rc10*e0rc20*e3rc20+2*e0rc20*e0rc22*e3rc12+2*e0rc10*e0rc12*e3rc12+2*e0rc00*e0rc02*e3rc12+2*e0rc20*e0rc21*e3rc11+2*e0rc10*e0rc11*e3rc11+2*e0rc00*e0rc01*e3rc11-e0rc22_2*e3rc10-e0rc21_2*e3rc10+e0rc20_2*e3rc10+e0rc12_2*e3rc10+e0rc11_2*e3rc10+3*e0rc10_2*e3rc10-e0rc02_2*e3rc10-e0rc01_2*e3rc10+e0rc00_2*e3rc10+2*e0rc00*e0rc12*e3rc02-2*e0rc02*e0rc10*e3rc02+2*e0rc00*e0rc11*e3rc01-2*e0rc01*e0rc10*e3rc01+2*e0rc02*e0rc12*e3rc00+2*e0rc01*e0rc11*e3rc00+2*e0rc00*e0rc10*e3rc00;
    AMatrix(3, 11) = -2*e0rc10*e1rc22*e3rc22+2*e0rc12*e1rc20*e3rc22+2*e0rc20*e1rc12*e3rc22-2*e0rc22*e1rc10*e3rc22-2*e0rc10*e1rc21*e3rc21+2*e0rc11*e1rc20*e3rc21+2*e0rc20*e1rc11*e3rc21-2*e0rc21*e1rc10*e3rc21+2*e0rc12*e1rc22*e3rc20+2*e0rc11*e1rc21*e3rc20+2*e0rc10*e1rc20*e3rc20+2*e0rc22*e1rc12*e3rc20+2*e0rc21*e1rc11*e3rc20+2*e0rc20*e1rc10*e3rc20+2*e0rc20*e1rc22*e3rc12+2*e0rc22*e1rc20*e3rc12+2*e0rc10*e1rc12*e3rc12+2*e0rc12*e1rc10*e3rc12+2*e0rc00*e1rc02*e3rc12+2*e0rc02*e1rc00*e3rc12+2*e0rc20*e1rc21*e3rc11+2*e0rc21*e1rc20*e3rc11+2*e0rc10*e1rc11*e3rc11+2*e0rc11*e1rc10*e3rc11+2*e0rc00*e1rc01*e3rc11+2*e0rc01*e1rc00*e3rc11-2*e0rc22*e1rc22*e3rc10-2*e0rc21*e1rc21*e3rc10+2*e0rc20*e1rc20*e3rc10+2*e0rc12*e1rc12*e3rc10+2*e0rc11*e1rc11*e3rc10+6*e0rc10*e1rc10*e3rc10-2*e0rc02*e1rc02*e3rc10-2*e0rc01*e1rc01*e3rc10+2*e0rc00*e1rc00*e3rc10+2*e0rc00*e1rc12*e3rc02-2*e0rc02*e1rc10*e3rc02-2*e0rc10*e1rc02*e3rc02+2*e0rc12*e1rc00*e3rc02+2*e0rc00*e1rc11*e3rc01-2*e0rc01*e1rc10*e3rc01-2*e0rc10*e1rc01*e3rc01+2*e0rc11*e1rc00*e3rc01+2*e0rc02*e1rc12*e3rc00+2*e0rc01*e1rc11*e3rc00+2*e0rc00*e1rc10*e3rc00+2*e0rc12*e1rc02*e3rc00+2*e0rc11*e1rc01*e3rc00+2*e0rc10*e1rc00*e3rc00;
    AMatrix(3, 12) = -2*e0rc10*e2rc22*e3rc22+2*e0rc12*e2rc20*e3rc22+2*e0rc20*e2rc12*e3rc22-2*e0rc22*e2rc10*e3rc22-2*e0rc10*e2rc21*e3rc21+2*e0rc11*e2rc20*e3rc21+2*e0rc20*e2rc11*e3rc21-2*e0rc21*e2rc10*e3rc21+2*e0rc12*e2rc22*e3rc20+2*e0rc11*e2rc21*e3rc20+2*e0rc10*e2rc20*e3rc20+2*e0rc22*e2rc12*e3rc20+2*e0rc21*e2rc11*e3rc20+2*e0rc20*e2rc10*e3rc20+2*e0rc20*e2rc22*e3rc12+2*e0rc22*e2rc20*e3rc12+2*e0rc10*e2rc12*e3rc12+2*e0rc12*e2rc10*e3rc12+2*e0rc00*e2rc02*e3rc12+2*e0rc02*e2rc00*e3rc12+2*e0rc20*e2rc21*e3rc11+2*e0rc21*e2rc20*e3rc11+2*e0rc10*e2rc11*e3rc11+2*e0rc11*e2rc10*e3rc11+2*e0rc00*e2rc01*e3rc11+2*e0rc01*e2rc00*e3rc11-2*e0rc22*e2rc22*e3rc10-2*e0rc21*e2rc21*e3rc10+2*e0rc20*e2rc20*e3rc10+2*e0rc12*e2rc12*e3rc10+2*e0rc11*e2rc11*e3rc10+6*e0rc10*e2rc10*e3rc10-2*e0rc02*e2rc02*e3rc10-2*e0rc01*e2rc01*e3rc10+2*e0rc00*e2rc00*e3rc10+2*e0rc00*e2rc12*e3rc02-2*e0rc02*e2rc10*e3rc02-2*e0rc10*e2rc02*e3rc02+2*e0rc12*e2rc00*e3rc02+2*e0rc00*e2rc11*e3rc01-2*e0rc01*e2rc10*e3rc01-2*e0rc10*e2rc01*e3rc01+2*e0rc11*e2rc00*e3rc01+2*e0rc02*e2rc12*e3rc00+2*e0rc01*e2rc11*e3rc00+2*e0rc00*e2rc10*e3rc00+2*e0rc12*e2rc02*e3rc00+2*e0rc11*e2rc01*e3rc00+2*e0rc10*e2rc00*e3rc00;
    AMatrix(3, 13) = -2*e1rc10*e1rc22*e3rc22+2*e1rc12*e1rc20*e3rc22-2*e1rc10*e1rc21*e3rc21+2*e1rc11*e1rc20*e3rc21+2*e1rc12*e1rc22*e3rc20+2*e1rc11*e1rc21*e3rc20+2*e1rc10*e1rc20*e3rc20+2*e1rc20*e1rc22*e3rc12+2*e1rc10*e1rc12*e3rc12+2*e1rc00*e1rc02*e3rc12+2*e1rc20*e1rc21*e3rc11+2*e1rc10*e1rc11*e3rc11+2*e1rc00*e1rc01*e3rc11-e1rc22_2*e3rc10-e1rc21_2*e3rc10+e1rc20_2*e3rc10+e1rc12_2*e3rc10+e1rc11_2*e3rc10+3*e1rc10_2*e3rc10-e1rc02_2*e3rc10-e1rc01_2*e3rc10+e1rc00_2*e3rc10+2*e1rc00*e1rc12*e3rc02-2*e1rc02*e1rc10*e3rc02+2*e1rc00*e1rc11*e3rc01-2*e1rc01*e1rc10*e3rc01+2*e1rc02*e1rc12*e3rc00+2*e1rc01*e1rc11*e3rc00+2*e1rc00*e1rc10*e3rc00;
    AMatrix(3, 14) = -2*e1rc10*e2rc22*e3rc22+2*e1rc12*e2rc20*e3rc22+2*e1rc20*e2rc12*e3rc22-2*e1rc22*e2rc10*e3rc22-2*e1rc10*e2rc21*e3rc21+2*e1rc11*e2rc20*e3rc21+2*e1rc20*e2rc11*e3rc21-2*e1rc21*e2rc10*e3rc21+2*e1rc12*e2rc22*e3rc20+2*e1rc11*e2rc21*e3rc20+2*e1rc10*e2rc20*e3rc20+2*e1rc22*e2rc12*e3rc20+2*e1rc21*e2rc11*e3rc20+2*e1rc20*e2rc10*e3rc20+2*e1rc20*e2rc22*e3rc12+2*e1rc22*e2rc20*e3rc12+2*e1rc10*e2rc12*e3rc12+2*e1rc12*e2rc10*e3rc12+2*e1rc00*e2rc02*e3rc12+2*e1rc02*e2rc00*e3rc12+2*e1rc20*e2rc21*e3rc11+2*e1rc21*e2rc20*e3rc11+2*e1rc10*e2rc11*e3rc11+2*e1rc11*e2rc10*e3rc11+2*e1rc00*e2rc01*e3rc11+2*e1rc01*e2rc00*e3rc11-2*e1rc22*e2rc22*e3rc10-2*e1rc21*e2rc21*e3rc10+2*e1rc20*e2rc20*e3rc10+2*e1rc12*e2rc12*e3rc10+2*e1rc11*e2rc11*e3rc10+6*e1rc10*e2rc10*e3rc10-2*e1rc02*e2rc02*e3rc10-2*e1rc01*e2rc01*e3rc10+2*e1rc00*e2rc00*e3rc10+2*e1rc00*e2rc12*e3rc02-2*e1rc02*e2rc10*e3rc02-2*e1rc10*e2rc02*e3rc02+2*e1rc12*e2rc00*e3rc02+2*e1rc00*e2rc11*e3rc01-2*e1rc01*e2rc10*e3rc01-2*e1rc10*e2rc01*e3rc01+2*e1rc11*e2rc00*e3rc01+2*e1rc02*e2rc12*e3rc00+2*e1rc01*e2rc11*e3rc00+2*e1rc00*e2rc10*e3rc00+2*e1rc12*e2rc02*e3rc00+2*e1rc11*e2rc01*e3rc00+2*e1rc10*e2rc00*e3rc00;
    AMatrix(3, 15) = -2*e2rc10*e2rc22*e3rc22+2*e2rc12*e2rc20*e3rc22-2*e2rc10*e2rc21*e3rc21+2*e2rc11*e2rc20*e3rc21+2*e2rc12*e2rc22*e3rc20+2*e2rc11*e2rc21*e3rc20+2*e2rc10*e2rc20*e3rc20+2*e2rc20*e2rc22*e3rc12+2*e2rc10*e2rc12*e3rc12+2*e2rc00*e2rc02*e3rc12+2*e2rc20*e2rc21*e3rc11+2*e2rc10*e2rc11*e3rc11+2*e2rc00*e2rc01*e3rc11-e2rc22_2*e3rc10-e2rc21_2*e3rc10+e2rc20_2*e3rc10+e2rc12_2*e3rc10+e2rc11_2*e3rc10+3*e2rc10_2*e3rc10-e2rc02_2*e3rc10-e2rc01_2*e3rc10+e2rc00_2*e3rc10+2*e2rc00*e2rc12*e3rc02-2*e2rc02*e2rc10*e3rc02+2*e2rc00*e2rc11*e3rc01-2*e2rc01*e2rc10*e3rc01+2*e2rc02*e2rc12*e3rc00+2*e2rc01*e2rc11*e3rc00+2*e2rc00*e2rc10*e3rc00;
    AMatrix(3, 16) = -e0rc10*e3rc22_2+2*e0rc12*e3rc20*e3rc22+2*e0rc20*e3rc12*e3rc22-2*e0rc22*e3rc10*e3rc22-e0rc10*e3rc21_2+2*e0rc11*e3rc20*e3rc21+2*e0rc20*e3rc11*e3rc21-2*e0rc21*e3rc10*e3rc21+e0rc10*e3rc20_2+2*e0rc22*e3rc12*e3rc20+2*e0rc21*e3rc11*e3rc20+2*e0rc20*e3rc10*e3rc20+e0rc10*e3rc12_2+2*e0rc12*e3rc10*e3rc12+2*e0rc00*e3rc02*e3rc12+2*e0rc02*e3rc00*e3rc12+e0rc10*e3rc11_2+2*e0rc11*e3rc10*e3rc11+2*e0rc00*e3rc01*e3rc11+2*e0rc01*e3rc00*e3rc11+3*e0rc10*e3rc10_2-2*e0rc02*e3rc02*e3rc10-2*e0rc01*e3rc01*e3rc10+2*e0rc00*e3rc00*e3rc10-e0rc10*e3rc02_2+2*e0rc12*e3rc00*e3rc02-e0rc10*e3rc01_2+2*e0rc11*e3rc00*e3rc01+e0rc10*e3rc00_2;
    AMatrix(3, 17) = -e1rc10*e3rc22_2+2*e1rc12*e3rc20*e3rc22+2*e1rc20*e3rc12*e3rc22-2*e1rc22*e3rc10*e3rc22-e1rc10*e3rc21_2+2*e1rc11*e3rc20*e3rc21+2*e1rc20*e3rc11*e3rc21-2*e1rc21*e3rc10*e3rc21+e1rc10*e3rc20_2+2*e1rc22*e3rc12*e3rc20+2*e1rc21*e3rc11*e3rc20+2*e1rc20*e3rc10*e3rc20+e1rc10*e3rc12_2+2*e1rc12*e3rc10*e3rc12+2*e1rc00*e3rc02*e3rc12+2*e1rc02*e3rc00*e3rc12+e1rc10*e3rc11_2+2*e1rc11*e3rc10*e3rc11+2*e1rc00*e3rc01*e3rc11+2*e1rc01*e3rc00*e3rc11+3*e1rc10*e3rc10_2-2*e1rc02*e3rc02*e3rc10-2*e1rc01*e3rc01*e3rc10+2*e1rc00*e3rc00*e3rc10-e1rc10*e3rc02_2+2*e1rc12*e3rc00*e3rc02-e1rc10*e3rc01_2+2*e1rc11*e3rc00*e3rc01+e1rc10*e3rc00_2;
    AMatrix(3, 18) = -e2rc10*e3rc22_2+2*e2rc12*e3rc20*e3rc22+2*e2rc20*e3rc12*e3rc22-2*e2rc22*e3rc10*e3rc22-e2rc10*e3rc21_2+2*e2rc11*e3rc20*e3rc21+2*e2rc20*e3rc11*e3rc21-2*e2rc21*e3rc10*e3rc21+e2rc10*e3rc20_2+2*e2rc22*e3rc12*e3rc20+2*e2rc21*e3rc11*e3rc20+2*e2rc20*e3rc10*e3rc20+e2rc10*e3rc12_2+2*e2rc12*e3rc10*e3rc12+2*e2rc00*e3rc02*e3rc12+2*e2rc02*e3rc00*e3rc12+e2rc10*e3rc11_2+2*e2rc11*e3rc10*e3rc11+2*e2rc00*e3rc01*e3rc11+2*e2rc01*e3rc00*e3rc11+3*e2rc10*e3rc10_2-2*e2rc02*e3rc02*e3rc10-2*e2rc01*e3rc01*e3rc10+2*e2rc00*e3rc00*e3rc10-e2rc10*e3rc02_2+2*e2rc12*e3rc00*e3rc02-e2rc10*e3rc01_2+2*e2rc11*e3rc00*e3rc01+e2rc10*e3rc00_2;
    AMatrix(3, 19) = -e3rc10*e3rc22_2+2*e3rc12*e3rc20*e3rc22-e3rc10*e3rc21_2+2*e3rc11*e3rc20*e3rc21+e3rc10*e3rc20_2+e3rc10*e3rc12_2+2*e3rc00*e3rc02*e3rc12+e3rc10*e3rc11_2+2*e3rc00*e3rc01*e3rc11+e3rc10_3-e3rc02_2*e3rc10-e3rc01_2*e3rc10+e3rc00_2*e3rc10;
    AMatrix(4, 0) = -e0rc11*e0rc22_2+2*e0rc12*e0rc21*e0rc22+e0rc11*e0rc21_2+2*e0rc10*e0rc20*e0rc21-e0rc11*e0rc20_2+e0rc11*e0rc12_2+2*e0rc01*e0rc02*e0rc12+e0rc11_3+e0rc10_2*e0rc11-e0rc02_2*e0rc11+e0rc01_2*e0rc11-e0rc00_2*e0rc11+2*e0rc00*e0rc01*e0rc10;
    AMatrix(4, 1) = -2*e0rc11*e0rc22*e1rc22+2*e0rc12*e0rc21*e1rc22+2*e0rc12*e0rc22*e1rc21+2*e0rc11*e0rc21*e1rc21+2*e0rc10*e0rc20*e1rc21+2*e0rc10*e0rc21*e1rc20-2*e0rc11*e0rc20*e1rc20+2*e0rc21*e0rc22*e1rc12+2*e0rc11*e0rc12*e1rc12+2*e0rc01*e0rc02*e1rc12-e0rc22_2*e1rc11+e0rc21_2*e1rc11-e0rc20_2*e1rc11+e0rc12_2*e1rc11+3*e0rc11_2*e1rc11+e0rc10_2*e1rc11-e0rc02_2*e1rc11+e0rc01_2*e1rc11-e0rc00_2*e1rc11+2*e0rc20*e0rc21*e1rc10+2*e0rc10*e0rc11*e1rc10+2*e0rc00*e0rc01*e1rc10+2*e0rc01*e0rc12*e1rc02-2*e0rc02*e0rc11*e1rc02+2*e0rc02*e0rc12*e1rc01+2*e0rc01*e0rc11*e1rc01+2*e0rc00*e0rc10*e1rc01-2*e0rc00*e0rc11*e1rc00+2*e0rc01*e0rc10*e1rc00;
    AMatrix(4, 2) = -2*e0rc11*e0rc22*e2rc22+2*e0rc12*e0rc21*e2rc22+2*e0rc12*e0rc22*e2rc21+2*e0rc11*e0rc21*e2rc21+2*e0rc10*e0rc20*e2rc21+2*e0rc10*e0rc21*e2rc20-2*e0rc11*e0rc20*e2rc20+2*e0rc21*e0rc22*e2rc12+2*e0rc11*e0rc12*e2rc12+2*e0rc01*e0rc02*e2rc12-e0rc22_2*e2rc11+e0rc21_2*e2rc11-e0rc20_2*e2rc11+e0rc12_2*e2rc11+3*e0rc11_2*e2rc11+e0rc10_2*e2rc11-e0rc02_2*e2rc11+e0rc01_2*e2rc11-e0rc00_2*e2rc11+2*e0rc20*e0rc21*e2rc10+2*e0rc10*e0rc11*e2rc10+2*e0rc00*e0rc01*e2rc10+2*e0rc01*e0rc12*e2rc02-2*e0rc02*e0rc11*e2rc02+2*e0rc02*e0rc12*e2rc01+2*e0rc01*e0rc11*e2rc01+2*e0rc00*e0rc10*e2rc01-2*e0rc00*e0rc11*e2rc00+2*e0rc01*e0rc10*e2rc00;
    AMatrix(4, 3) = -e0rc11*e1rc22_2+2*e0rc12*e1rc21*e1rc22+2*e0rc21*e1rc12*e1rc22-2*e0rc22*e1rc11*e1rc22+e0rc11*e1rc21_2+2*e0rc10*e1rc20*e1rc21+2*e0rc22*e1rc12*e1rc21+2*e0rc21*e1rc11*e1rc21+2*e0rc20*e1rc10*e1rc21-e0rc11*e1rc20_2-2*e0rc20*e1rc11*e1rc20+2*e0rc21*e1rc10*e1rc20+e0rc11*e1rc12_2+2*e0rc12*e1rc11*e1rc12+2*e0rc01*e1rc02*e1rc12+2*e0rc02*e1rc01*e1rc12+3*e0rc11*e1rc11_2+2*e0rc10*e1rc10*e1rc11-2*e0rc02*e1rc02*e1rc11+2*e0rc01*e1rc01*e1rc11-2*e0rc00*e1rc00*e1rc11+e0rc11*e1rc10_2+2*e0rc00*e1rc01*e1rc10+2*e0rc01*e1rc00*e1rc10-e0rc11*e1rc02_2+2*e0rc12*e1rc01*e1rc02+e0rc11*e1rc01_2+2*e0rc10*e1rc00*e1rc01-e0rc11*e1rc00_2;
    AMatrix(4, 4) = -2*e0rc11*e1rc22*e2rc22+2*e0rc12*e1rc21*e2rc22+2*e0rc21*e1rc12*e2rc22-2*e0rc22*e1rc11*e2rc22+2*e0rc12*e1rc22*e2rc21+2*e0rc11*e1rc21*e2rc21+2*e0rc10*e1rc20*e2rc21+2*e0rc22*e1rc12*e2rc21+2*e0rc21*e1rc11*e2rc21+2*e0rc20*e1rc10*e2rc21+2*e0rc10*e1rc21*e2rc20-2*e0rc11*e1rc20*e2rc20-2*e0rc20*e1rc11*e2rc20+2*e0rc21*e1rc10*e2rc20+2*e0rc21*e1rc22*e2rc12+2*e0rc22*e1rc21*e2rc12+2*e0rc11*e1rc12*e2rc12+2*e0rc12*e1rc11*e2rc12+2*e0rc01*e1rc02*e2rc12+2*e0rc02*e1rc01*e2rc12-2*e0rc22*e1rc22*e2rc11+2*e0rc21*e1rc21*e2rc11-2*e0rc20*e1rc20*e2rc11+2*e0rc12*e1rc12*e2rc11+6*e0rc11*e1rc11*e2rc11+2*e0rc10*e1rc10*e2rc11-2*e0rc02*e1rc02*e2rc11+2*e0rc01*e1rc01*e2rc11-2*e0rc00*e1rc00*e2rc11+2*e0rc20*e1rc21*e2rc10+2*e0rc21*e1rc20*e2rc10+2*e0rc10*e1rc11*e2rc10+2*e0rc11*e1rc10*e2rc10+2*e0rc00*e1rc01*e2rc10+2*e0rc01*e1rc00*e2rc10+2*e0rc01*e1rc12*e2rc02-2*e0rc02*e1rc11*e2rc02-2*e0rc11*e1rc02*e2rc02+2*e0rc12*e1rc01*e2rc02+2*e0rc02*e1rc12*e2rc01+2*e0rc01*e1rc11*e2rc01+2*e0rc00*e1rc10*e2rc01+2*e0rc12*e1rc02*e2rc01+2*e0rc11*e1rc01*e2rc01+2*e0rc10*e1rc00*e2rc01-2*e0rc00*e1rc11*e2rc00+2*e0rc01*e1rc10*e2rc00+2*e0rc10*e1rc01*e2rc00-2*e0rc11*e1rc00*e2rc00;
    AMatrix(4, 5) = -e0rc11*e2rc22_2+2*e0rc12*e2rc21*e2rc22+2*e0rc21*e2rc12*e2rc22-2*e0rc22*e2rc11*e2rc22+e0rc11*e2rc21_2+2*e0rc10*e2rc20*e2rc21+2*e0rc22*e2rc12*e2rc21+2*e0rc21*e2rc11*e2rc21+2*e0rc20*e2rc10*e2rc21-e0rc11*e2rc20_2-2*e0rc20*e2rc11*e2rc20+2*e0rc21*e2rc10*e2rc20+e0rc11*e2rc12_2+2*e0rc12*e2rc11*e2rc12+2*e0rc01*e2rc02*e2rc12+2*e0rc02*e2rc01*e2rc12+3*e0rc11*e2rc11_2+2*e0rc10*e2rc10*e2rc11-2*e0rc02*e2rc02*e2rc11+2*e0rc01*e2rc01*e2rc11-2*e0rc00*e2rc00*e2rc11+e0rc11*e2rc10_2+2*e0rc00*e2rc01*e2rc10+2*e0rc01*e2rc00*e2rc10-e0rc11*e2rc02_2+2*e0rc12*e2rc01*e2rc02+e0rc11*e2rc01_2+2*e0rc10*e2rc00*e2rc01-e0rc11*e2rc00_2;
    AMatrix(4, 6) = -e1rc11*e1rc22_2+2*e1rc12*e1rc21*e1rc22+e1rc11*e1rc21_2+2*e1rc10*e1rc20*e1rc21-e1rc11*e1rc20_2+e1rc11*e1rc12_2+2*e1rc01*e1rc02*e1rc12+e1rc11_3+e1rc10_2*e1rc11-e1rc02_2*e1rc11+e1rc01_2*e1rc11-e1rc00_2*e1rc11+2*e1rc00*e1rc01*e1rc10;
    AMatrix(4, 7) = -2*e1rc11*e1rc22*e2rc22+2*e1rc12*e1rc21*e2rc22+2*e1rc12*e1rc22*e2rc21+2*e1rc11*e1rc21*e2rc21+2*e1rc10*e1rc20*e2rc21+2*e1rc10*e1rc21*e2rc20-2*e1rc11*e1rc20*e2rc20+2*e1rc21*e1rc22*e2rc12+2*e1rc11*e1rc12*e2rc12+2*e1rc01*e1rc02*e2rc12-e1rc22_2*e2rc11+e1rc21_2*e2rc11-e1rc20_2*e2rc11+e1rc12_2*e2rc11+3*e1rc11_2*e2rc11+e1rc10_2*e2rc11-e1rc02_2*e2rc11+e1rc01_2*e2rc11-e1rc00_2*e2rc11+2*e1rc20*e1rc21*e2rc10+2*e1rc10*e1rc11*e2rc10+2*e1rc00*e1rc01*e2rc10+2*e1rc01*e1rc12*e2rc02-2*e1rc02*e1rc11*e2rc02+2*e1rc02*e1rc12*e2rc01+2*e1rc01*e1rc11*e2rc01+2*e1rc00*e1rc10*e2rc01-2*e1rc00*e1rc11*e2rc00+2*e1rc01*e1rc10*e2rc00;
    AMatrix(4, 8) = -e1rc11*e2rc22_2+2*e1rc12*e2rc21*e2rc22+2*e1rc21*e2rc12*e2rc22-2*e1rc22*e2rc11*e2rc22+e1rc11*e2rc21_2+2*e1rc10*e2rc20*e2rc21+2*e1rc22*e2rc12*e2rc21+2*e1rc21*e2rc11*e2rc21+2*e1rc20*e2rc10*e2rc21-e1rc11*e2rc20_2-2*e1rc20*e2rc11*e2rc20+2*e1rc21*e2rc10*e2rc20+e1rc11*e2rc12_2+2*e1rc12*e2rc11*e2rc12+2*e1rc01*e2rc02*e2rc12+2*e1rc02*e2rc01*e2rc12+3*e1rc11*e2rc11_2+2*e1rc10*e2rc10*e2rc11-2*e1rc02*e2rc02*e2rc11+2*e1rc01*e2rc01*e2rc11-2*e1rc00*e2rc00*e2rc11+e1rc11*e2rc10_2+2*e1rc00*e2rc01*e2rc10+2*e1rc01*e2rc00*e2rc10-e1rc11*e2rc02_2+2*e1rc12*e2rc01*e2rc02+e1rc11*e2rc01_2+2*e1rc10*e2rc00*e2rc01-e1rc11*e2rc00_2;
    AMatrix(4, 9) = -e2rc11*e2rc22_2+2*e2rc12*e2rc21*e2rc22+e2rc11*e2rc21_2+2*e2rc10*e2rc20*e2rc21-e2rc11*e2rc20_2+e2rc11*e2rc12_2+2*e2rc01*e2rc02*e2rc12+e2rc11_3+e2rc10_2*e2rc11-e2rc02_2*e2rc11+e2rc01_2*e2rc11-e2rc00_2*e2rc11+2*e2rc00*e2rc01*e2rc10;
    AMatrix(4, 10) = -2*e0rc11*e0rc22*e3rc22+2*e0rc12*e0rc21*e3rc22+2*e0rc12*e0rc22*e3rc21+2*e0rc11*e0rc21*e3rc21+2*e0rc10*e0rc20*e3rc21+2*e0rc10*e0rc21*e3rc20-2*e0rc11*e0rc20*e3rc20+2*e0rc21*e0rc22*e3rc12+2*e0rc11*e0rc12*e3rc12+2*e0rc01*e0rc02*e3rc12-e0rc22_2*e3rc11+e0rc21_2*e3rc11-e0rc20_2*e3rc11+e0rc12_2*e3rc11+3*e0rc11_2*e3rc11+e0rc10_2*e3rc11-e0rc02_2*e3rc11+e0rc01_2*e3rc11-e0rc00_2*e3rc11+2*e0rc20*e0rc21*e3rc10+2*e0rc10*e0rc11*e3rc10+2*e0rc00*e0rc01*e3rc10+2*e0rc01*e0rc12*e3rc02-2*e0rc02*e0rc11*e3rc02+2*e0rc02*e0rc12*e3rc01+2*e0rc01*e0rc11*e3rc01+2*e0rc00*e0rc10*e3rc01-2*e0rc00*e0rc11*e3rc00+2*e0rc01*e0rc10*e3rc00;
    AMatrix(4, 11) = -2*e0rc11*e1rc22*e3rc22+2*e0rc12*e1rc21*e3rc22+2*e0rc21*e1rc12*e3rc22-2*e0rc22*e1rc11*e3rc22+2*e0rc12*e1rc22*e3rc21+2*e0rc11*e1rc21*e3rc21+2*e0rc10*e1rc20*e3rc21+2*e0rc22*e1rc12*e3rc21+2*e0rc21*e1rc11*e3rc21+2*e0rc20*e1rc10*e3rc21+2*e0rc10*e1rc21*e3rc20-2*e0rc11*e1rc20*e3rc20-2*e0rc20*e1rc11*e3rc20+2*e0rc21*e1rc10*e3rc20+2*e0rc21*e1rc22*e3rc12+2*e0rc22*e1rc21*e3rc12+2*e0rc11*e1rc12*e3rc12+2*e0rc12*e1rc11*e3rc12+2*e0rc01*e1rc02*e3rc12+2*e0rc02*e1rc01*e3rc12-2*e0rc22*e1rc22*e3rc11+2*e0rc21*e1rc21*e3rc11-2*e0rc20*e1rc20*e3rc11+2*e0rc12*e1rc12*e3rc11+6*e0rc11*e1rc11*e3rc11+2*e0rc10*e1rc10*e3rc11-2*e0rc02*e1rc02*e3rc11+2*e0rc01*e1rc01*e3rc11-2*e0rc00*e1rc00*e3rc11+2*e0rc20*e1rc21*e3rc10+2*e0rc21*e1rc20*e3rc10+2*e0rc10*e1rc11*e3rc10+2*e0rc11*e1rc10*e3rc10+2*e0rc00*e1rc01*e3rc10+2*e0rc01*e1rc00*e3rc10+2*e0rc01*e1rc12*e3rc02-2*e0rc02*e1rc11*e3rc02-2*e0rc11*e1rc02*e3rc02+2*e0rc12*e1rc01*e3rc02+2*e0rc02*e1rc12*e3rc01+2*e0rc01*e1rc11*e3rc01+2*e0rc00*e1rc10*e3rc01+2*e0rc12*e1rc02*e3rc01+2*e0rc11*e1rc01*e3rc01+2*e0rc10*e1rc00*e3rc01-2*e0rc00*e1rc11*e3rc00+2*e0rc01*e1rc10*e3rc00+2*e0rc10*e1rc01*e3rc00-2*e0rc11*e1rc00*e3rc00;
    AMatrix(4, 12) = -2*e0rc11*e2rc22*e3rc22+2*e0rc12*e2rc21*e3rc22+2*e0rc21*e2rc12*e3rc22-2*e0rc22*e2rc11*e3rc22+2*e0rc12*e2rc22*e3rc21+2*e0rc11*e2rc21*e3rc21+2*e0rc10*e2rc20*e3rc21+2*e0rc22*e2rc12*e3rc21+2*e0rc21*e2rc11*e3rc21+2*e0rc20*e2rc10*e3rc21+2*e0rc10*e2rc21*e3rc20-2*e0rc11*e2rc20*e3rc20-2*e0rc20*e2rc11*e3rc20+2*e0rc21*e2rc10*e3rc20+2*e0rc21*e2rc22*e3rc12+2*e0rc22*e2rc21*e3rc12+2*e0rc11*e2rc12*e3rc12+2*e0rc12*e2rc11*e3rc12+2*e0rc01*e2rc02*e3rc12+2*e0rc02*e2rc01*e3rc12-2*e0rc22*e2rc22*e3rc11+2*e0rc21*e2rc21*e3rc11-2*e0rc20*e2rc20*e3rc11+2*e0rc12*e2rc12*e3rc11+6*e0rc11*e2rc11*e3rc11+2*e0rc10*e2rc10*e3rc11-2*e0rc02*e2rc02*e3rc11+2*e0rc01*e2rc01*e3rc11-2*e0rc00*e2rc00*e3rc11+2*e0rc20*e2rc21*e3rc10+2*e0rc21*e2rc20*e3rc10+2*e0rc10*e2rc11*e3rc10+2*e0rc11*e2rc10*e3rc10+2*e0rc00*e2rc01*e3rc10+2*e0rc01*e2rc00*e3rc10+2*e0rc01*e2rc12*e3rc02-2*e0rc02*e2rc11*e3rc02-2*e0rc11*e2rc02*e3rc02+2*e0rc12*e2rc01*e3rc02+2*e0rc02*e2rc12*e3rc01+2*e0rc01*e2rc11*e3rc01+2*e0rc00*e2rc10*e3rc01+2*e0rc12*e2rc02*e3rc01+2*e0rc11*e2rc01*e3rc01+2*e0rc10*e2rc00*e3rc01-2*e0rc00*e2rc11*e3rc00+2*e0rc01*e2rc10*e3rc00+2*e0rc10*e2rc01*e3rc00-2*e0rc11*e2rc00*e3rc00;
    AMatrix(4, 13) = -2*e1rc11*e1rc22*e3rc22+2*e1rc12*e1rc21*e3rc22+2*e1rc12*e1rc22*e3rc21+2*e1rc11*e1rc21*e3rc21+2*e1rc10*e1rc20*e3rc21+2*e1rc10*e1rc21*e3rc20-2*e1rc11*e1rc20*e3rc20+2*e1rc21*e1rc22*e3rc12+2*e1rc11*e1rc12*e3rc12+2*e1rc01*e1rc02*e3rc12-e1rc22_2*e3rc11+e1rc21_2*e3rc11-e1rc20_2*e3rc11+e1rc12_2*e3rc11+3*e1rc11_2*e3rc11+e1rc10_2*e3rc11-e1rc02_2*e3rc11+e1rc01_2*e3rc11-e1rc00_2*e3rc11+2*e1rc20*e1rc21*e3rc10+2*e1rc10*e1rc11*e3rc10+2*e1rc00*e1rc01*e3rc10+2*e1rc01*e1rc12*e3rc02-2*e1rc02*e1rc11*e3rc02+2*e1rc02*e1rc12*e3rc01+2*e1rc01*e1rc11*e3rc01+2*e1rc00*e1rc10*e3rc01-2*e1rc00*e1rc11*e3rc00+2*e1rc01*e1rc10*e3rc00;
    AMatrix(4, 14) = -2*e1rc11*e2rc22*e3rc22+2*e1rc12*e2rc21*e3rc22+2*e1rc21*e2rc12*e3rc22-2*e1rc22*e2rc11*e3rc22+2*e1rc12*e2rc22*e3rc21+2*e1rc11*e2rc21*e3rc21+2*e1rc10*e2rc20*e3rc21+2*e1rc22*e2rc12*e3rc21+2*e1rc21*e2rc11*e3rc21+2*e1rc20*e2rc10*e3rc21+2*e1rc10*e2rc21*e3rc20-2*e1rc11*e2rc20*e3rc20-2*e1rc20*e2rc11*e3rc20+2*e1rc21*e2rc10*e3rc20+2*e1rc21*e2rc22*e3rc12+2*e1rc22*e2rc21*e3rc12+2*e1rc11*e2rc12*e3rc12+2*e1rc12*e2rc11*e3rc12+2*e1rc01*e2rc02*e3rc12+2*e1rc02*e2rc01*e3rc12-2*e1rc22*e2rc22*e3rc11+2*e1rc21*e2rc21*e3rc11-2*e1rc20*e2rc20*e3rc11+2*e1rc12*e2rc12*e3rc11+6*e1rc11*e2rc11*e3rc11+2*e1rc10*e2rc10*e3rc11-2*e1rc02*e2rc02*e3rc11+2*e1rc01*e2rc01*e3rc11-2*e1rc00*e2rc00*e3rc11+2*e1rc20*e2rc21*e3rc10+2*e1rc21*e2rc20*e3rc10+2*e1rc10*e2rc11*e3rc10+2*e1rc11*e2rc10*e3rc10+2*e1rc00*e2rc01*e3rc10+2*e1rc01*e2rc00*e3rc10+2*e1rc01*e2rc12*e3rc02-2*e1rc02*e2rc11*e3rc02-2*e1rc11*e2rc02*e3rc02+2*e1rc12*e2rc01*e3rc02+2*e1rc02*e2rc12*e3rc01+2*e1rc01*e2rc11*e3rc01+2*e1rc00*e2rc10*e3rc01+2*e1rc12*e2rc02*e3rc01+2*e1rc11*e2rc01*e3rc01+2*e1rc10*e2rc00*e3rc01-2*e1rc00*e2rc11*e3rc00+2*e1rc01*e2rc10*e3rc00+2*e1rc10*e2rc01*e3rc00-2*e1rc11*e2rc00*e3rc00;
    AMatrix(4, 15) = -2*e2rc11*e2rc22*e3rc22+2*e2rc12*e2rc21*e3rc22+2*e2rc12*e2rc22*e3rc21+2*e2rc11*e2rc21*e3rc21+2*e2rc10*e2rc20*e3rc21+2*e2rc10*e2rc21*e3rc20-2*e2rc11*e2rc20*e3rc20+2*e2rc21*e2rc22*e3rc12+2*e2rc11*e2rc12*e3rc12+2*e2rc01*e2rc02*e3rc12-e2rc22_2*e3rc11+e2rc21_2*e3rc11-e2rc20_2*e3rc11+e2rc12_2*e3rc11+3*e2rc11_2*e3rc11+e2rc10_2*e3rc11-e2rc02_2*e3rc11+e2rc01_2*e3rc11-e2rc00_2*e3rc11+2*e2rc20*e2rc21*e3rc10+2*e2rc10*e2rc11*e3rc10+2*e2rc00*e2rc01*e3rc10+2*e2rc01*e2rc12*e3rc02-2*e2rc02*e2rc11*e3rc02+2*e2rc02*e2rc12*e3rc01+2*e2rc01*e2rc11*e3rc01+2*e2rc00*e2rc10*e3rc01-2*e2rc00*e2rc11*e3rc00+2*e2rc01*e2rc10*e3rc00;
    AMatrix(4, 16) = -e0rc11*e3rc22_2+2*e0rc12*e3rc21*e3rc22+2*e0rc21*e3rc12*e3rc22-2*e0rc22*e3rc11*e3rc22+e0rc11*e3rc21_2+2*e0rc10*e3rc20*e3rc21+2*e0rc22*e3rc12*e3rc21+2*e0rc21*e3rc11*e3rc21+2*e0rc20*e3rc10*e3rc21-e0rc11*e3rc20_2-2*e0rc20*e3rc11*e3rc20+2*e0rc21*e3rc10*e3rc20+e0rc11*e3rc12_2+2*e0rc12*e3rc11*e3rc12+2*e0rc01*e3rc02*e3rc12+2*e0rc02*e3rc01*e3rc12+3*e0rc11*e3rc11_2+2*e0rc10*e3rc10*e3rc11-2*e0rc02*e3rc02*e3rc11+2*e0rc01*e3rc01*e3rc11-2*e0rc00*e3rc00*e3rc11+e0rc11*e3rc10_2+2*e0rc00*e3rc01*e3rc10+2*e0rc01*e3rc00*e3rc10-e0rc11*e3rc02_2+2*e0rc12*e3rc01*e3rc02+e0rc11*e3rc01_2+2*e0rc10*e3rc00*e3rc01-e0rc11*e3rc00_2;
    AMatrix(4, 17) = -e1rc11*e3rc22_2+2*e1rc12*e3rc21*e3rc22+2*e1rc21*e3rc12*e3rc22-2*e1rc22*e3rc11*e3rc22+e1rc11*e3rc21_2+2*e1rc10*e3rc20*e3rc21+2*e1rc22*e3rc12*e3rc21+2*e1rc21*e3rc11*e3rc21+2*e1rc20*e3rc10*e3rc21-e1rc11*e3rc20_2-2*e1rc20*e3rc11*e3rc20+2*e1rc21*e3rc10*e3rc20+e1rc11*e3rc12_2+2*e1rc12*e3rc11*e3rc12+2*e1rc01*e3rc02*e3rc12+2*e1rc02*e3rc01*e3rc12+3*e1rc11*e3rc11_2+2*e1rc10*e3rc10*e3rc11-2*e1rc02*e3rc02*e3rc11+2*e1rc01*e3rc01*e3rc11-2*e1rc00*e3rc00*e3rc11+e1rc11*e3rc10_2+2*e1rc00*e3rc01*e3rc10+2*e1rc01*e3rc00*e3rc10-e1rc11*e3rc02_2+2*e1rc12*e3rc01*e3rc02+e1rc11*e3rc01_2+2*e1rc10*e3rc00*e3rc01-e1rc11*e3rc00_2;
    AMatrix(4, 18) = -e2rc11*e3rc22_2+2*e2rc12*e3rc21*e3rc22+2*e2rc21*e3rc12*e3rc22-2*e2rc22*e3rc11*e3rc22+e2rc11*e3rc21_2+2*e2rc10*e3rc20*e3rc21+2*e2rc22*e3rc12*e3rc21+2*e2rc21*e3rc11*e3rc21+2*e2rc20*e3rc10*e3rc21-e2rc11*e3rc20_2-2*e2rc20*e3rc11*e3rc20+2*e2rc21*e3rc10*e3rc20+e2rc11*e3rc12_2+2*e2rc12*e3rc11*e3rc12+2*e2rc01*e3rc02*e3rc12+2*e2rc02*e3rc01*e3rc12+3*e2rc11*e3rc11_2+2*e2rc10*e3rc10*e3rc11-2*e2rc02*e3rc02*e3rc11+2*e2rc01*e3rc01*e3rc11-2*e2rc00*e3rc00*e3rc11+e2rc11*e3rc10_2+2*e2rc00*e3rc01*e3rc10+2*e2rc01*e3rc00*e3rc10-e2rc11*e3rc02_2+2*e2rc12*e3rc01*e3rc02+e2rc11*e3rc01_2+2*e2rc10*e3rc00*e3rc01-e2rc11*e3rc00_2;
    AMatrix(4, 19) = -e3rc11*e3rc22_2+2*e3rc12*e3rc21*e3rc22+e3rc11*e3rc21_2+2*e3rc10*e3rc20*e3rc21-e3rc11*e3rc20_2+e3rc11*e3rc12_2+2*e3rc01*e3rc02*e3rc12+e3rc11_3+e3rc10_2*e3rc11-e3rc02_2*e3rc11+e3rc01_2*e3rc11-e3rc00_2*e3rc11+2*e3rc00*e3rc01*e3rc10;
    AMatrix(5, 0) = +e0rc12*e0rc22_2+2*e0rc11*e0rc21*e0rc22+2*e0rc10*e0rc20*e0rc22-e0rc12*e0rc21_2-e0rc12*e0rc20_2+e0rc12_3+e0rc11_2*e0rc12+e0rc10_2*e0rc12+e0rc02_2*e0rc12-e0rc01_2*e0rc12-e0rc00_2*e0rc12+2*e0rc01*e0rc02*e0rc11+2*e0rc00*e0rc02*e0rc10;
    AMatrix(5, 1) = +2*e0rc12*e0rc22*e1rc22+2*e0rc11*e0rc21*e1rc22+2*e0rc10*e0rc20*e1rc22+2*e0rc11*e0rc22*e1rc21-2*e0rc12*e0rc21*e1rc21+2*e0rc10*e0rc22*e1rc20-2*e0rc12*e0rc20*e1rc20+e0rc22_2*e1rc12-e0rc21_2*e1rc12-e0rc20_2*e1rc12+3*e0rc12_2*e1rc12+e0rc11_2*e1rc12+e0rc10_2*e1rc12+e0rc02_2*e1rc12-e0rc01_2*e1rc12-e0rc00_2*e1rc12+2*e0rc21*e0rc22*e1rc11+2*e0rc11*e0rc12*e1rc11+2*e0rc01*e0rc02*e1rc11+2*e0rc20*e0rc22*e1rc10+2*e0rc10*e0rc12*e1rc10+2*e0rc00*e0rc02*e1rc10+2*e0rc02*e0rc12*e1rc02+2*e0rc01*e0rc11*e1rc02+2*e0rc00*e0rc10*e1rc02-2*e0rc01*e0rc12*e1rc01+2*e0rc02*e0rc11*e1rc01-2*e0rc00*e0rc12*e1rc00+2*e0rc02*e0rc10*e1rc00;
    AMatrix(5, 2) = +2*e0rc12*e0rc22*e2rc22+2*e0rc11*e0rc21*e2rc22+2*e0rc10*e0rc20*e2rc22+2*e0rc11*e0rc22*e2rc21-2*e0rc12*e0rc21*e2rc21+2*e0rc10*e0rc22*e2rc20-2*e0rc12*e0rc20*e2rc20+e0rc22_2*e2rc12-e0rc21_2*e2rc12-e0rc20_2*e2rc12+3*e0rc12_2*e2rc12+e0rc11_2*e2rc12+e0rc10_2*e2rc12+e0rc02_2*e2rc12-e0rc01_2*e2rc12-e0rc00_2*e2rc12+2*e0rc21*e0rc22*e2rc11+2*e0rc11*e0rc12*e2rc11+2*e0rc01*e0rc02*e2rc11+2*e0rc20*e0rc22*e2rc10+2*e0rc10*e0rc12*e2rc10+2*e0rc00*e0rc02*e2rc10+2*e0rc02*e0rc12*e2rc02+2*e0rc01*e0rc11*e2rc02+2*e0rc00*e0rc10*e2rc02-2*e0rc01*e0rc12*e2rc01+2*e0rc02*e0rc11*e2rc01-2*e0rc00*e0rc12*e2rc00+2*e0rc02*e0rc10*e2rc00;
    AMatrix(5, 3) = +e0rc12*e1rc22_2+2*e0rc11*e1rc21*e1rc22+2*e0rc10*e1rc20*e1rc22+2*e0rc22*e1rc12*e1rc22+2*e0rc21*e1rc11*e1rc22+2*e0rc20*e1rc10*e1rc22-e0rc12*e1rc21_2-2*e0rc21*e1rc12*e1rc21+2*e0rc22*e1rc11*e1rc21-e0rc12*e1rc20_2-2*e0rc20*e1rc12*e1rc20+2*e0rc22*e1rc10*e1rc20+3*e0rc12*e1rc12_2+2*e0rc11*e1rc11*e1rc12+2*e0rc10*e1rc10*e1rc12+2*e0rc02*e1rc02*e1rc12-2*e0rc01*e1rc01*e1rc12-2*e0rc00*e1rc00*e1rc12+e0rc12*e1rc11_2+2*e0rc01*e1rc02*e1rc11+2*e0rc02*e1rc01*e1rc11+e0rc12*e1rc10_2+2*e0rc00*e1rc02*e1rc10+2*e0rc02*e1rc00*e1rc10+e0rc12*e1rc02_2+2*e0rc11*e1rc01*e1rc02+2*e0rc10*e1rc00*e1rc02-e0rc12*e1rc01_2-e0rc12*e1rc00_2;
    AMatrix(5, 4) = +2*e0rc12*e1rc22*e2rc22+2*e0rc11*e1rc21*e2rc22+2*e0rc10*e1rc20*e2rc22+2*e0rc22*e1rc12*e2rc22+2*e0rc21*e1rc11*e2rc22+2*e0rc20*e1rc10*e2rc22+2*e0rc11*e1rc22*e2rc21-2*e0rc12*e1rc21*e2rc21-2*e0rc21*e1rc12*e2rc21+2*e0rc22*e1rc11*e2rc21+2*e0rc10*e1rc22*e2rc20-2*e0rc12*e1rc20*e2rc20-2*e0rc20*e1rc12*e2rc20+2*e0rc22*e1rc10*e2rc20+2*e0rc22*e1rc22*e2rc12-2*e0rc21*e1rc21*e2rc12-2*e0rc20*e1rc20*e2rc12+6*e0rc12*e1rc12*e2rc12+2*e0rc11*e1rc11*e2rc12+2*e0rc10*e1rc10*e2rc12+2*e0rc02*e1rc02*e2rc12-2*e0rc01*e1rc01*e2rc12-2*e0rc00*e1rc00*e2rc12+2*e0rc21*e1rc22*e2rc11+2*e0rc22*e1rc21*e2rc11+2*e0rc11*e1rc12*e2rc11+2*e0rc12*e1rc11*e2rc11+2*e0rc01*e1rc02*e2rc11+2*e0rc02*e1rc01*e2rc11+2*e0rc20*e1rc22*e2rc10+2*e0rc22*e1rc20*e2rc10+2*e0rc10*e1rc12*e2rc10+2*e0rc12*e1rc10*e2rc10+2*e0rc00*e1rc02*e2rc10+2*e0rc02*e1rc00*e2rc10+2*e0rc02*e1rc12*e2rc02+2*e0rc01*e1rc11*e2rc02+2*e0rc00*e1rc10*e2rc02+2*e0rc12*e1rc02*e2rc02+2*e0rc11*e1rc01*e2rc02+2*e0rc10*e1rc00*e2rc02-2*e0rc01*e1rc12*e2rc01+2*e0rc02*e1rc11*e2rc01+2*e0rc11*e1rc02*e2rc01-2*e0rc12*e1rc01*e2rc01-2*e0rc00*e1rc12*e2rc00+2*e0rc02*e1rc10*e2rc00+2*e0rc10*e1rc02*e2rc00-2*e0rc12*e1rc00*e2rc00;
    AMatrix(5, 5) = +e0rc12*e2rc22_2+2*e0rc11*e2rc21*e2rc22+2*e0rc10*e2rc20*e2rc22+2*e0rc22*e2rc12*e2rc22+2*e0rc21*e2rc11*e2rc22+2*e0rc20*e2rc10*e2rc22-e0rc12*e2rc21_2-2*e0rc21*e2rc12*e2rc21+2*e0rc22*e2rc11*e2rc21-e0rc12*e2rc20_2-2*e0rc20*e2rc12*e2rc20+2*e0rc22*e2rc10*e2rc20+3*e0rc12*e2rc12_2+2*e0rc11*e2rc11*e2rc12+2*e0rc10*e2rc10*e2rc12+2*e0rc02*e2rc02*e2rc12-2*e0rc01*e2rc01*e2rc12-2*e0rc00*e2rc00*e2rc12+e0rc12*e2rc11_2+2*e0rc01*e2rc02*e2rc11+2*e0rc02*e2rc01*e2rc11+e0rc12*e2rc10_2+2*e0rc00*e2rc02*e2rc10+2*e0rc02*e2rc00*e2rc10+e0rc12*e2rc02_2+2*e0rc11*e2rc01*e2rc02+2*e0rc10*e2rc00*e2rc02-e0rc12*e2rc01_2-e0rc12*e2rc00_2;
    AMatrix(5, 6) = +e1rc12*e1rc22_2+2*e1rc11*e1rc21*e1rc22+2*e1rc10*e1rc20*e1rc22-e1rc12*e1rc21_2-e1rc12*e1rc20_2+e1rc12_3+e1rc11_2*e1rc12+e1rc10_2*e1rc12+e1rc02_2*e1rc12-e1rc01_2*e1rc12-e1rc00_2*e1rc12+2*e1rc01*e1rc02*e1rc11+2*e1rc00*e1rc02*e1rc10;
    AMatrix(5, 7) = +2*e1rc12*e1rc22*e2rc22+2*e1rc11*e1rc21*e2rc22+2*e1rc10*e1rc20*e2rc22+2*e1rc11*e1rc22*e2rc21-2*e1rc12*e1rc21*e2rc21+2*e1rc10*e1rc22*e2rc20-2*e1rc12*e1rc20*e2rc20+e1rc22_2*e2rc12-e1rc21_2*e2rc12-e1rc20_2*e2rc12+3*e1rc12_2*e2rc12+e1rc11_2*e2rc12+e1rc10_2*e2rc12+e1rc02_2*e2rc12-e1rc01_2*e2rc12-e1rc00_2*e2rc12+2*e1rc21*e1rc22*e2rc11+2*e1rc11*e1rc12*e2rc11+2*e1rc01*e1rc02*e2rc11+2*e1rc20*e1rc22*e2rc10+2*e1rc10*e1rc12*e2rc10+2*e1rc00*e1rc02*e2rc10+2*e1rc02*e1rc12*e2rc02+2*e1rc01*e1rc11*e2rc02+2*e1rc00*e1rc10*e2rc02-2*e1rc01*e1rc12*e2rc01+2*e1rc02*e1rc11*e2rc01-2*e1rc00*e1rc12*e2rc00+2*e1rc02*e1rc10*e2rc00;
    AMatrix(5, 8) = +e1rc12*e2rc22_2+2*e1rc11*e2rc21*e2rc22+2*e1rc10*e2rc20*e2rc22+2*e1rc22*e2rc12*e2rc22+2*e1rc21*e2rc11*e2rc22+2*e1rc20*e2rc10*e2rc22-e1rc12*e2rc21_2-2*e1rc21*e2rc12*e2rc21+2*e1rc22*e2rc11*e2rc21-e1rc12*e2rc20_2-2*e1rc20*e2rc12*e2rc20+2*e1rc22*e2rc10*e2rc20+3*e1rc12*e2rc12_2+2*e1rc11*e2rc11*e2rc12+2*e1rc10*e2rc10*e2rc12+2*e1rc02*e2rc02*e2rc12-2*e1rc01*e2rc01*e2rc12-2*e1rc00*e2rc00*e2rc12+e1rc12*e2rc11_2+2*e1rc01*e2rc02*e2rc11+2*e1rc02*e2rc01*e2rc11+e1rc12*e2rc10_2+2*e1rc00*e2rc02*e2rc10+2*e1rc02*e2rc00*e2rc10+e1rc12*e2rc02_2+2*e1rc11*e2rc01*e2rc02+2*e1rc10*e2rc00*e2rc02-e1rc12*e2rc01_2-e1rc12*e2rc00_2;
    AMatrix(5, 9) = e2rc12*e2rc22_2+2*e2rc11*e2rc21*e2rc22+2*e2rc10*e2rc20*e2rc22-e2rc12*e2rc21_2-e2rc12*e2rc20_2+e2rc12_3+e2rc11_2*e2rc12+e2rc10_2*e2rc12+e2rc02_2*e2rc12-e2rc01_2*e2rc12-e2rc00_2*e2rc12+2*e2rc01*e2rc02*e2rc11+2*e2rc00*e2rc02*e2rc10;
    AMatrix(5, 10) = +2*e0rc12*e0rc22*e3rc22+2*e0rc11*e0rc21*e3rc22+2*e0rc10*e0rc20*e3rc22+2*e0rc11*e0rc22*e3rc21-2*e0rc12*e0rc21*e3rc21+2*e0rc10*e0rc22*e3rc20-2*e0rc12*e0rc20*e3rc20+e0rc22_2*e3rc12-e0rc21_2*e3rc12-e0rc20_2*e3rc12+3*e0rc12_2*e3rc12+e0rc11_2*e3rc12+e0rc10_2*e3rc12+e0rc02_2*e3rc12-e0rc01_2*e3rc12-e0rc00_2*e3rc12+2*e0rc21*e0rc22*e3rc11+2*e0rc11*e0rc12*e3rc11+2*e0rc01*e0rc02*e3rc11+2*e0rc20*e0rc22*e3rc10+2*e0rc10*e0rc12*e3rc10+2*e0rc00*e0rc02*e3rc10+2*e0rc02*e0rc12*e3rc02+2*e0rc01*e0rc11*e3rc02+2*e0rc00*e0rc10*e3rc02-2*e0rc01*e0rc12*e3rc01+2*e0rc02*e0rc11*e3rc01-2*e0rc00*e0rc12*e3rc00+2*e0rc02*e0rc10*e3rc00;
    AMatrix(5, 11) = +2*e0rc12*e1rc22*e3rc22+2*e0rc11*e1rc21*e3rc22+2*e0rc10*e1rc20*e3rc22+2*e0rc22*e1rc12*e3rc22+2*e0rc21*e1rc11*e3rc22+2*e0rc20*e1rc10*e3rc22+2*e0rc11*e1rc22*e3rc21-2*e0rc12*e1rc21*e3rc21-2*e0rc21*e1rc12*e3rc21+2*e0rc22*e1rc11*e3rc21+2*e0rc10*e1rc22*e3rc20-2*e0rc12*e1rc20*e3rc20-2*e0rc20*e1rc12*e3rc20+2*e0rc22*e1rc10*e3rc20+2*e0rc22*e1rc22*e3rc12-2*e0rc21*e1rc21*e3rc12-2*e0rc20*e1rc20*e3rc12+6*e0rc12*e1rc12*e3rc12+2*e0rc11*e1rc11*e3rc12+2*e0rc10*e1rc10*e3rc12+2*e0rc02*e1rc02*e3rc12-2*e0rc01*e1rc01*e3rc12-2*e0rc00*e1rc00*e3rc12+2*e0rc21*e1rc22*e3rc11+2*e0rc22*e1rc21*e3rc11+2*e0rc11*e1rc12*e3rc11+2*e0rc12*e1rc11*e3rc11+2*e0rc01*e1rc02*e3rc11+2*e0rc02*e1rc01*e3rc11+2*e0rc20*e1rc22*e3rc10+2*e0rc22*e1rc20*e3rc10+2*e0rc10*e1rc12*e3rc10+2*e0rc12*e1rc10*e3rc10+2*e0rc00*e1rc02*e3rc10+2*e0rc02*e1rc00*e3rc10+2*e0rc02*e1rc12*e3rc02+2*e0rc01*e1rc11*e3rc02+2*e0rc00*e1rc10*e3rc02+2*e0rc12*e1rc02*e3rc02+2*e0rc11*e1rc01*e3rc02+2*e0rc10*e1rc00*e3rc02-2*e0rc01*e1rc12*e3rc01+2*e0rc02*e1rc11*e3rc01+2*e0rc11*e1rc02*e3rc01-2*e0rc12*e1rc01*e3rc01-2*e0rc00*e1rc12*e3rc00+2*e0rc02*e1rc10*e3rc00+2*e0rc10*e1rc02*e3rc00-2*e0rc12*e1rc00*e3rc00;
    AMatrix(5, 12) = +2*e0rc12*e2rc22*e3rc22+2*e0rc11*e2rc21*e3rc22+2*e0rc10*e2rc20*e3rc22+2*e0rc22*e2rc12*e3rc22+2*e0rc21*e2rc11*e3rc22+2*e0rc20*e2rc10*e3rc22+2*e0rc11*e2rc22*e3rc21-2*e0rc12*e2rc21*e3rc21-2*e0rc21*e2rc12*e3rc21+2*e0rc22*e2rc11*e3rc21+2*e0rc10*e2rc22*e3rc20-2*e0rc12*e2rc20*e3rc20-2*e0rc20*e2rc12*e3rc20+2*e0rc22*e2rc10*e3rc20+2*e0rc22*e2rc22*e3rc12-2*e0rc21*e2rc21*e3rc12-2*e0rc20*e2rc20*e3rc12+6*e0rc12*e2rc12*e3rc12+2*e0rc11*e2rc11*e3rc12+2*e0rc10*e2rc10*e3rc12+2*e0rc02*e2rc02*e3rc12-2*e0rc01*e2rc01*e3rc12-2*e0rc00*e2rc00*e3rc12+2*e0rc21*e2rc22*e3rc11+2*e0rc22*e2rc21*e3rc11+2*e0rc11*e2rc12*e3rc11+2*e0rc12*e2rc11*e3rc11+2*e0rc01*e2rc02*e3rc11+2*e0rc02*e2rc01*e3rc11+2*e0rc20*e2rc22*e3rc10+2*e0rc22*e2rc20*e3rc10+2*e0rc10*e2rc12*e3rc10+2*e0rc12*e2rc10*e3rc10+2*e0rc00*e2rc02*e3rc10+2*e0rc02*e2rc00*e3rc10+2*e0rc02*e2rc12*e3rc02+2*e0rc01*e2rc11*e3rc02+2*e0rc00*e2rc10*e3rc02+2*e0rc12*e2rc02*e3rc02+2*e0rc11*e2rc01*e3rc02+2*e0rc10*e2rc00*e3rc02-2*e0rc01*e2rc12*e3rc01+2*e0rc02*e2rc11*e3rc01+2*e0rc11*e2rc02*e3rc01-2*e0rc12*e2rc01*e3rc01-2*e0rc00*e2rc12*e3rc00+2*e0rc02*e2rc10*e3rc00+2*e0rc10*e2rc02*e3rc00-2*e0rc12*e2rc00*e3rc00;
    AMatrix(5, 13) = +2*e1rc12*e1rc22*e3rc22+2*e1rc11*e1rc21*e3rc22+2*e1rc10*e1rc20*e3rc22+2*e1rc11*e1rc22*e3rc21-2*e1rc12*e1rc21*e3rc21+2*e1rc10*e1rc22*e3rc20-2*e1rc12*e1rc20*e3rc20+e1rc22_2*e3rc12-e1rc21_2*e3rc12-e1rc20_2*e3rc12+3*e1rc12_2*e3rc12+e1rc11_2*e3rc12+e1rc10_2*e3rc12+e1rc02_2*e3rc12-e1rc01_2*e3rc12-e1rc00_2*e3rc12+2*e1rc21*e1rc22*e3rc11+2*e1rc11*e1rc12*e3rc11+2*e1rc01*e1rc02*e3rc11+2*e1rc20*e1rc22*e3rc10+2*e1rc10*e1rc12*e3rc10+2*e1rc00*e1rc02*e3rc10+2*e1rc02*e1rc12*e3rc02+2*e1rc01*e1rc11*e3rc02+2*e1rc00*e1rc10*e3rc02-2*e1rc01*e1rc12*e3rc01+2*e1rc02*e1rc11*e3rc01-2*e1rc00*e1rc12*e3rc00+2*e1rc02*e1rc10*e3rc00;
    AMatrix(5, 14) = +2*e1rc12*e2rc22*e3rc22+2*e1rc11*e2rc21*e3rc22+2*e1rc10*e2rc20*e3rc22+2*e1rc22*e2rc12*e3rc22+2*e1rc21*e2rc11*e3rc22+2*e1rc20*e2rc10*e3rc22+2*e1rc11*e2rc22*e3rc21-2*e1rc12*e2rc21*e3rc21-2*e1rc21*e2rc12*e3rc21+2*e1rc22*e2rc11*e3rc21+2*e1rc10*e2rc22*e3rc20-2*e1rc12*e2rc20*e3rc20-2*e1rc20*e2rc12*e3rc20+2*e1rc22*e2rc10*e3rc20+2*e1rc22*e2rc22*e3rc12-2*e1rc21*e2rc21*e3rc12-2*e1rc20*e2rc20*e3rc12+6*e1rc12*e2rc12*e3rc12+2*e1rc11*e2rc11*e3rc12+2*e1rc10*e2rc10*e3rc12+2*e1rc02*e2rc02*e3rc12-2*e1rc01*e2rc01*e3rc12-2*e1rc00*e2rc00*e3rc12+2*e1rc21*e2rc22*e3rc11+2*e1rc22*e2rc21*e3rc11+2*e1rc11*e2rc12*e3rc11+2*e1rc12*e2rc11*e3rc11+2*e1rc01*e2rc02*e3rc11+2*e1rc02*e2rc01*e3rc11+2*e1rc20*e2rc22*e3rc10+2*e1rc22*e2rc20*e3rc10+2*e1rc10*e2rc12*e3rc10+2*e1rc12*e2rc10*e3rc10+2*e1rc00*e2rc02*e3rc10+2*e1rc02*e2rc00*e3rc10+2*e1rc02*e2rc12*e3rc02+2*e1rc01*e2rc11*e3rc02+2*e1rc00*e2rc10*e3rc02+2*e1rc12*e2rc02*e3rc02+2*e1rc11*e2rc01*e3rc02+2*e1rc10*e2rc00*e3rc02-2*e1rc01*e2rc12*e3rc01+2*e1rc02*e2rc11*e3rc01+2*e1rc11*e2rc02*e3rc01-2*e1rc12*e2rc01*e3rc01-2*e1rc00*e2rc12*e3rc00+2*e1rc02*e2rc10*e3rc00+2*e1rc10*e2rc02*e3rc00-2*e1rc12*e2rc00*e3rc00;
    AMatrix(5, 15) = +2*e2rc12*e2rc22*e3rc22+2*e2rc11*e2rc21*e3rc22+2*e2rc10*e2rc20*e3rc22+2*e2rc11*e2rc22*e3rc21-2*e2rc12*e2rc21*e3rc21+2*e2rc10*e2rc22*e3rc20-2*e2rc12*e2rc20*e3rc20+e2rc22_2*e3rc12-e2rc21_2*e3rc12-e2rc20_2*e3rc12+3*e2rc12_2*e3rc12+e2rc11_2*e3rc12+e2rc10_2*e3rc12+e2rc02_2*e3rc12-e2rc01_2*e3rc12-e2rc00_2*e3rc12+2*e2rc21*e2rc22*e3rc11+2*e2rc11*e2rc12*e3rc11+2*e2rc01*e2rc02*e3rc11+2*e2rc20*e2rc22*e3rc10+2*e2rc10*e2rc12*e3rc10+2*e2rc00*e2rc02*e3rc10+2*e2rc02*e2rc12*e3rc02+2*e2rc01*e2rc11*e3rc02+2*e2rc00*e2rc10*e3rc02-2*e2rc01*e2rc12*e3rc01+2*e2rc02*e2rc11*e3rc01-2*e2rc00*e2rc12*e3rc00+2*e2rc02*e2rc10*e3rc00;
    AMatrix(5, 16) = +e0rc12*e3rc22_2+2*e0rc11*e3rc21*e3rc22+2*e0rc10*e3rc20*e3rc22+2*e0rc22*e3rc12*e3rc22+2*e0rc21*e3rc11*e3rc22+2*e0rc20*e3rc10*e3rc22-e0rc12*e3rc21_2-2*e0rc21*e3rc12*e3rc21+2*e0rc22*e3rc11*e3rc21-e0rc12*e3rc20_2-2*e0rc20*e3rc12*e3rc20+2*e0rc22*e3rc10*e3rc20+3*e0rc12*e3rc12_2+2*e0rc11*e3rc11*e3rc12+2*e0rc10*e3rc10*e3rc12+2*e0rc02*e3rc02*e3rc12-2*e0rc01*e3rc01*e3rc12-2*e0rc00*e3rc00*e3rc12+e0rc12*e3rc11_2+2*e0rc01*e3rc02*e3rc11+2*e0rc02*e3rc01*e3rc11+e0rc12*e3rc10_2+2*e0rc00*e3rc02*e3rc10+2*e0rc02*e3rc00*e3rc10+e0rc12*e3rc02_2+2*e0rc11*e3rc01*e3rc02+2*e0rc10*e3rc00*e3rc02-e0rc12*e3rc01_2-e0rc12*e3rc00_2;
    AMatrix(5, 17) = +e1rc12*e3rc22_2+2*e1rc11*e3rc21*e3rc22+2*e1rc10*e3rc20*e3rc22+2*e1rc22*e3rc12*e3rc22+2*e1rc21*e3rc11*e3rc22+2*e1rc20*e3rc10*e3rc22-e1rc12*e3rc21_2-2*e1rc21*e3rc12*e3rc21+2*e1rc22*e3rc11*e3rc21-e1rc12*e3rc20_2-2*e1rc20*e3rc12*e3rc20+2*e1rc22*e3rc10*e3rc20+3*e1rc12*e3rc12_2+2*e1rc11*e3rc11*e3rc12+2*e1rc10*e3rc10*e3rc12+2*e1rc02*e3rc02*e3rc12-2*e1rc01*e3rc01*e3rc12-2*e1rc00*e3rc00*e3rc12+e1rc12*e3rc11_2+2*e1rc01*e3rc02*e3rc11+2*e1rc02*e3rc01*e3rc11+e1rc12*e3rc10_2+2*e1rc00*e3rc02*e3rc10+2*e1rc02*e3rc00*e3rc10+e1rc12*e3rc02_2+2*e1rc11*e3rc01*e3rc02+2*e1rc10*e3rc00*e3rc02-e1rc12*e3rc01_2-e1rc12*e3rc00_2;
    AMatrix(5, 18) = +e2rc12*e3rc22_2+2*e2rc11*e3rc21*e3rc22+2*e2rc10*e3rc20*e3rc22+2*e2rc22*e3rc12*e3rc22+2*e2rc21*e3rc11*e3rc22+2*e2rc20*e3rc10*e3rc22-e2rc12*e3rc21_2-2*e2rc21*e3rc12*e3rc21+2*e2rc22*e3rc11*e3rc21-e2rc12*e3rc20_2-2*e2rc20*e3rc12*e3rc20+2*e2rc22*e3rc10*e3rc20+3*e2rc12*e3rc12_2+2*e2rc11*e3rc11*e3rc12+2*e2rc10*e3rc10*e3rc12+2*e2rc02*e3rc02*e3rc12-2*e2rc01*e3rc01*e3rc12-2*e2rc00*e3rc00*e3rc12+e2rc12*e3rc11_2+2*e2rc01*e3rc02*e3rc11+2*e2rc02*e3rc01*e3rc11+e2rc12*e3rc10_2+2*e2rc00*e3rc02*e3rc10+2*e2rc02*e3rc00*e3rc10+e2rc12*e3rc02_2+2*e2rc11*e3rc01*e3rc02+2*e2rc10*e3rc00*e3rc02-e2rc12*e3rc01_2-e2rc12*e3rc00_2;
    AMatrix(5, 19) = +e3rc12*e3rc22_2+2*e3rc11*e3rc21*e3rc22+2*e3rc10*e3rc20*e3rc22-e3rc12*e3rc21_2-e3rc12*e3rc20_2+e3rc12_3+e3rc11_2*e3rc12+e3rc10_2*e3rc12+e3rc02_2*e3rc12-e3rc01_2*e3rc12-e3rc00_2*e3rc12+2*e3rc01*e3rc02*e3rc11+2*e3rc00*e3rc02*e3rc10;
    AMatrix(6, 0) = +e0rc20*e0rc22_2+2*e0rc10*e0rc12*e0rc22+2*e0rc00*e0rc02*e0rc22+e0rc20*e0rc21_2+2*e0rc10*e0rc11*e0rc21+2*e0rc00*e0rc01*e0rc21+e0rc20_3-e0rc12_2*e0rc20-e0rc11_2*e0rc20+e0rc10_2*e0rc20-e0rc02_2*e0rc20-e0rc01_2*e0rc20+e0rc00_2*e0rc20;
    AMatrix(6, 1) = +2*e0rc20*e0rc22*e1rc22+2*e0rc10*e0rc12*e1rc22+2*e0rc00*e0rc02*e1rc22+2*e0rc20*e0rc21*e1rc21+2*e0rc10*e0rc11*e1rc21+2*e0rc00*e0rc01*e1rc21+e0rc22_2*e1rc20+e0rc21_2*e1rc20+3*e0rc20_2*e1rc20-e0rc12_2*e1rc20-e0rc11_2*e1rc20+e0rc10_2*e1rc20-e0rc02_2*e1rc20-e0rc01_2*e1rc20+e0rc00_2*e1rc20+2*e0rc10*e0rc22*e1rc12-2*e0rc12*e0rc20*e1rc12+2*e0rc10*e0rc21*e1rc11-2*e0rc11*e0rc20*e1rc11+2*e0rc12*e0rc22*e1rc10+2*e0rc11*e0rc21*e1rc10+2*e0rc10*e0rc20*e1rc10+2*e0rc00*e0rc22*e1rc02-2*e0rc02*e0rc20*e1rc02+2*e0rc00*e0rc21*e1rc01-2*e0rc01*e0rc20*e1rc01+2*e0rc02*e0rc22*e1rc00+2*e0rc01*e0rc21*e1rc00+2*e0rc00*e0rc20*e1rc00;
    AMatrix(6, 2) = +2*e0rc20*e0rc22*e2rc22+2*e0rc10*e0rc12*e2rc22+2*e0rc00*e0rc02*e2rc22+2*e0rc20*e0rc21*e2rc21+2*e0rc10*e0rc11*e2rc21+2*e0rc00*e0rc01*e2rc21+e0rc22_2*e2rc20+e0rc21_2*e2rc20+3*e0rc20_2*e2rc20-e0rc12_2*e2rc20-e0rc11_2*e2rc20+e0rc10_2*e2rc20-e0rc02_2*e2rc20-e0rc01_2*e2rc20+e0rc00_2*e2rc20+2*e0rc10*e0rc22*e2rc12-2*e0rc12*e0rc20*e2rc12+2*e0rc10*e0rc21*e2rc11-2*e0rc11*e0rc20*e2rc11+2*e0rc12*e0rc22*e2rc10+2*e0rc11*e0rc21*e2rc10+2*e0rc10*e0rc20*e2rc10+2*e0rc00*e0rc22*e2rc02-2*e0rc02*e0rc20*e2rc02+2*e0rc00*e0rc21*e2rc01-2*e0rc01*e0rc20*e2rc01+2*e0rc02*e0rc22*e2rc00+2*e0rc01*e0rc21*e2rc00+2*e0rc00*e0rc20*e2rc00;
    AMatrix(6, 3) = +e0rc20*e1rc22_2+2*e0rc22*e1rc20*e1rc22+2*e0rc10*e1rc12*e1rc22+2*e0rc12*e1rc10*e1rc22+2*e0rc00*e1rc02*e1rc22+2*e0rc02*e1rc00*e1rc22+e0rc20*e1rc21_2+2*e0rc21*e1rc20*e1rc21+2*e0rc10*e1rc11*e1rc21+2*e0rc11*e1rc10*e1rc21+2*e0rc00*e1rc01*e1rc21+2*e0rc01*e1rc00*e1rc21+3*e0rc20*e1rc20_2-2*e0rc12*e1rc12*e1rc20-2*e0rc11*e1rc11*e1rc20+2*e0rc10*e1rc10*e1rc20-2*e0rc02*e1rc02*e1rc20-2*e0rc01*e1rc01*e1rc20+2*e0rc00*e1rc00*e1rc20-e0rc20*e1rc12_2+2*e0rc22*e1rc10*e1rc12-e0rc20*e1rc11_2+2*e0rc21*e1rc10*e1rc11+e0rc20*e1rc10_2-e0rc20*e1rc02_2+2*e0rc22*e1rc00*e1rc02-e0rc20*e1rc01_2+2*e0rc21*e1rc00*e1rc01+e0rc20*e1rc00_2;
    AMatrix(6, 4) = +2*e0rc20*e1rc22*e2rc22+2*e0rc22*e1rc20*e2rc22+2*e0rc10*e1rc12*e2rc22+2*e0rc12*e1rc10*e2rc22+2*e0rc00*e1rc02*e2rc22+2*e0rc02*e1rc00*e2rc22+2*e0rc20*e1rc21*e2rc21+2*e0rc21*e1rc20*e2rc21+2*e0rc10*e1rc11*e2rc21+2*e0rc11*e1rc10*e2rc21+2*e0rc00*e1rc01*e2rc21+2*e0rc01*e1rc00*e2rc21+2*e0rc22*e1rc22*e2rc20+2*e0rc21*e1rc21*e2rc20+6*e0rc20*e1rc20*e2rc20-2*e0rc12*e1rc12*e2rc20-2*e0rc11*e1rc11*e2rc20+2*e0rc10*e1rc10*e2rc20-2*e0rc02*e1rc02*e2rc20-2*e0rc01*e1rc01*e2rc20+2*e0rc00*e1rc00*e2rc20+2*e0rc10*e1rc22*e2rc12-2*e0rc12*e1rc20*e2rc12-2*e0rc20*e1rc12*e2rc12+2*e0rc22*e1rc10*e2rc12+2*e0rc10*e1rc21*e2rc11-2*e0rc11*e1rc20*e2rc11-2*e0rc20*e1rc11*e2rc11+2*e0rc21*e1rc10*e2rc11+2*e0rc12*e1rc22*e2rc10+2*e0rc11*e1rc21*e2rc10+2*e0rc10*e1rc20*e2rc10+2*e0rc22*e1rc12*e2rc10+2*e0rc21*e1rc11*e2rc10+2*e0rc20*e1rc10*e2rc10+2*e0rc00*e1rc22*e2rc02-2*e0rc02*e1rc20*e2rc02-2*e0rc20*e1rc02*e2rc02+2*e0rc22*e1rc00*e2rc02+2*e0rc00*e1rc21*e2rc01-2*e0rc01*e1rc20*e2rc01-2*e0rc20*e1rc01*e2rc01+2*e0rc21*e1rc00*e2rc01+2*e0rc02*e1rc22*e2rc00+2*e0rc01*e1rc21*e2rc00+2*e0rc00*e1rc20*e2rc00+2*e0rc22*e1rc02*e2rc00+2*e0rc21*e1rc01*e2rc00+2*e0rc20*e1rc00*e2rc00;
    AMatrix(6, 5) = +e0rc20*e2rc22_2+2*e0rc22*e2rc20*e2rc22+2*e0rc10*e2rc12*e2rc22+2*e0rc12*e2rc10*e2rc22+2*e0rc00*e2rc02*e2rc22+2*e0rc02*e2rc00*e2rc22+e0rc20*e2rc21_2+2*e0rc21*e2rc20*e2rc21+2*e0rc10*e2rc11*e2rc21+2*e0rc11*e2rc10*e2rc21+2*e0rc00*e2rc01*e2rc21+2*e0rc01*e2rc00*e2rc21+3*e0rc20*e2rc20_2-2*e0rc12*e2rc12*e2rc20-2*e0rc11*e2rc11*e2rc20+2*e0rc10*e2rc10*e2rc20-2*e0rc02*e2rc02*e2rc20-2*e0rc01*e2rc01*e2rc20+2*e0rc00*e2rc00*e2rc20-e0rc20*e2rc12_2+2*e0rc22*e2rc10*e2rc12-e0rc20*e2rc11_2+2*e0rc21*e2rc10*e2rc11+e0rc20*e2rc10_2-e0rc20*e2rc02_2+2*e0rc22*e2rc00*e2rc02-e0rc20*e2rc01_2+2*e0rc21*e2rc00*e2rc01+e0rc20*e2rc00_2;
    AMatrix(6, 6) = +e1rc20*e1rc22_2+2*e1rc10*e1rc12*e1rc22+2*e1rc00*e1rc02*e1rc22+e1rc20*e1rc21_2+2*e1rc10*e1rc11*e1rc21+2*e1rc00*e1rc01*e1rc21+e1rc20_3-e1rc12_2*e1rc20-e1rc11_2*e1rc20+e1rc10_2*e1rc20-e1rc02_2*e1rc20-e1rc01_2*e1rc20+e1rc00_2*e1rc20;
    AMatrix(6, 7) = +2*e1rc20*e1rc22*e2rc22+2*e1rc10*e1rc12*e2rc22+2*e1rc00*e1rc02*e2rc22+2*e1rc20*e1rc21*e2rc21+2*e1rc10*e1rc11*e2rc21+2*e1rc00*e1rc01*e2rc21+e1rc22_2*e2rc20+e1rc21_2*e2rc20+3*e1rc20_2*e2rc20-e1rc12_2*e2rc20-e1rc11_2*e2rc20+e1rc10_2*e2rc20-e1rc02_2*e2rc20-e1rc01_2*e2rc20+e1rc00_2*e2rc20+2*e1rc10*e1rc22*e2rc12-2*e1rc12*e1rc20*e2rc12+2*e1rc10*e1rc21*e2rc11-2*e1rc11*e1rc20*e2rc11+2*e1rc12*e1rc22*e2rc10+2*e1rc11*e1rc21*e2rc10+2*e1rc10*e1rc20*e2rc10+2*e1rc00*e1rc22*e2rc02-2*e1rc02*e1rc20*e2rc02+2*e1rc00*e1rc21*e2rc01-2*e1rc01*e1rc20*e2rc01+2*e1rc02*e1rc22*e2rc00+2*e1rc01*e1rc21*e2rc00+2*e1rc00*e1rc20*e2rc00;
    AMatrix(6, 8) = +e1rc20*e2rc22_2+2*e1rc22*e2rc20*e2rc22+2*e1rc10*e2rc12*e2rc22+2*e1rc12*e2rc10*e2rc22+2*e1rc00*e2rc02*e2rc22+2*e1rc02*e2rc00*e2rc22+e1rc20*e2rc21_2+2*e1rc21*e2rc20*e2rc21+2*e1rc10*e2rc11*e2rc21+2*e1rc11*e2rc10*e2rc21+2*e1rc00*e2rc01*e2rc21+2*e1rc01*e2rc00*e2rc21+3*e1rc20*e2rc20_2-2*e1rc12*e2rc12*e2rc20-2*e1rc11*e2rc11*e2rc20+2*e1rc10*e2rc10*e2rc20-2*e1rc02*e2rc02*e2rc20-2*e1rc01*e2rc01*e2rc20+2*e1rc00*e2rc00*e2rc20-e1rc20*e2rc12_2+2*e1rc22*e2rc10*e2rc12-e1rc20*e2rc11_2+2*e1rc21*e2rc10*e2rc11+e1rc20*e2rc10_2-e1rc20*e2rc02_2+2*e1rc22*e2rc00*e2rc02-e1rc20*e2rc01_2+2*e1rc21*e2rc00*e2rc01+e1rc20*e2rc00_2;
    AMatrix(6, 9) = e2rc20*e2rc22_2+2*e2rc10*e2rc12*e2rc22+2*e2rc00*e2rc02*e2rc22+e2rc20*e2rc21_2+2*e2rc10*e2rc11*e2rc21+2*e2rc00*e2rc01*e2rc21+e2rc20_3-e2rc12_2*e2rc20-e2rc11_2*e2rc20+e2rc10_2*e2rc20-e2rc02_2*e2rc20-e2rc01_2*e2rc20+e2rc00_2*e2rc20;
    AMatrix(6, 10) = +2*e0rc20*e0rc22*e3rc22+2*e0rc10*e0rc12*e3rc22+2*e0rc00*e0rc02*e3rc22+2*e0rc20*e0rc21*e3rc21+2*e0rc10*e0rc11*e3rc21+2*e0rc00*e0rc01*e3rc21+e0rc22_2*e3rc20+e0rc21_2*e3rc20+3*e0rc20_2*e3rc20-e0rc12_2*e3rc20-e0rc11_2*e3rc20+e0rc10_2*e3rc20-e0rc02_2*e3rc20-e0rc01_2*e3rc20+e0rc00_2*e3rc20+2*e0rc10*e0rc22*e3rc12-2*e0rc12*e0rc20*e3rc12+2*e0rc10*e0rc21*e3rc11-2*e0rc11*e0rc20*e3rc11+2*e0rc12*e0rc22*e3rc10+2*e0rc11*e0rc21*e3rc10+2*e0rc10*e0rc20*e3rc10+2*e0rc00*e0rc22*e3rc02-2*e0rc02*e0rc20*e3rc02+2*e0rc00*e0rc21*e3rc01-2*e0rc01*e0rc20*e3rc01+2*e0rc02*e0rc22*e3rc00+2*e0rc01*e0rc21*e3rc00+2*e0rc00*e0rc20*e3rc00;
    AMatrix(6, 11) = +2*e0rc20*e1rc22*e3rc22+2*e0rc22*e1rc20*e3rc22+2*e0rc10*e1rc12*e3rc22+2*e0rc12*e1rc10*e3rc22+2*e0rc00*e1rc02*e3rc22+2*e0rc02*e1rc00*e3rc22+2*e0rc20*e1rc21*e3rc21+2*e0rc21*e1rc20*e3rc21+2*e0rc10*e1rc11*e3rc21+2*e0rc11*e1rc10*e3rc21+2*e0rc00*e1rc01*e3rc21+2*e0rc01*e1rc00*e3rc21+2*e0rc22*e1rc22*e3rc20+2*e0rc21*e1rc21*e3rc20+6*e0rc20*e1rc20*e3rc20-2*e0rc12*e1rc12*e3rc20-2*e0rc11*e1rc11*e3rc20+2*e0rc10*e1rc10*e3rc20-2*e0rc02*e1rc02*e3rc20-2*e0rc01*e1rc01*e3rc20+2*e0rc00*e1rc00*e3rc20+2*e0rc10*e1rc22*e3rc12-2*e0rc12*e1rc20*e3rc12-2*e0rc20*e1rc12*e3rc12+2*e0rc22*e1rc10*e3rc12+2*e0rc10*e1rc21*e3rc11-2*e0rc11*e1rc20*e3rc11-2*e0rc20*e1rc11*e3rc11+2*e0rc21*e1rc10*e3rc11+2*e0rc12*e1rc22*e3rc10+2*e0rc11*e1rc21*e3rc10+2*e0rc10*e1rc20*e3rc10+2*e0rc22*e1rc12*e3rc10+2*e0rc21*e1rc11*e3rc10+2*e0rc20*e1rc10*e3rc10+2*e0rc00*e1rc22*e3rc02-2*e0rc02*e1rc20*e3rc02-2*e0rc20*e1rc02*e3rc02+2*e0rc22*e1rc00*e3rc02+2*e0rc00*e1rc21*e3rc01-2*e0rc01*e1rc20*e3rc01-2*e0rc20*e1rc01*e3rc01+2*e0rc21*e1rc00*e3rc01+2*e0rc02*e1rc22*e3rc00+2*e0rc01*e1rc21*e3rc00+2*e0rc00*e1rc20*e3rc00+2*e0rc22*e1rc02*e3rc00+2*e0rc21*e1rc01*e3rc00+2*e0rc20*e1rc00*e3rc00;
    AMatrix(6, 12) = +2*e0rc20*e2rc22*e3rc22+2*e0rc22*e2rc20*e3rc22+2*e0rc10*e2rc12*e3rc22+2*e0rc12*e2rc10*e3rc22+2*e0rc00*e2rc02*e3rc22+2*e0rc02*e2rc00*e3rc22+2*e0rc20*e2rc21*e3rc21+2*e0rc21*e2rc20*e3rc21+2*e0rc10*e2rc11*e3rc21+2*e0rc11*e2rc10*e3rc21+2*e0rc00*e2rc01*e3rc21+2*e0rc01*e2rc00*e3rc21+2*e0rc22*e2rc22*e3rc20+2*e0rc21*e2rc21*e3rc20+6*e0rc20*e2rc20*e3rc20-2*e0rc12*e2rc12*e3rc20-2*e0rc11*e2rc11*e3rc20+2*e0rc10*e2rc10*e3rc20-2*e0rc02*e2rc02*e3rc20-2*e0rc01*e2rc01*e3rc20+2*e0rc00*e2rc00*e3rc20+2*e0rc10*e2rc22*e3rc12-2*e0rc12*e2rc20*e3rc12-2*e0rc20*e2rc12*e3rc12+2*e0rc22*e2rc10*e3rc12+2*e0rc10*e2rc21*e3rc11-2*e0rc11*e2rc20*e3rc11-2*e0rc20*e2rc11*e3rc11+2*e0rc21*e2rc10*e3rc11+2*e0rc12*e2rc22*e3rc10+2*e0rc11*e2rc21*e3rc10+2*e0rc10*e2rc20*e3rc10+2*e0rc22*e2rc12*e3rc10+2*e0rc21*e2rc11*e3rc10+2*e0rc20*e2rc10*e3rc10+2*e0rc00*e2rc22*e3rc02-2*e0rc02*e2rc20*e3rc02-2*e0rc20*e2rc02*e3rc02+2*e0rc22*e2rc00*e3rc02+2*e0rc00*e2rc21*e3rc01-2*e0rc01*e2rc20*e3rc01-2*e0rc20*e2rc01*e3rc01+2*e0rc21*e2rc00*e3rc01+2*e0rc02*e2rc22*e3rc00+2*e0rc01*e2rc21*e3rc00+2*e0rc00*e2rc20*e3rc00+2*e0rc22*e2rc02*e3rc00+2*e0rc21*e2rc01*e3rc00+2*e0rc20*e2rc00*e3rc00;
    AMatrix(6, 13) = +2*e1rc20*e1rc22*e3rc22+2*e1rc10*e1rc12*e3rc22+2*e1rc00*e1rc02*e3rc22+2*e1rc20*e1rc21*e3rc21+2*e1rc10*e1rc11*e3rc21+2*e1rc00*e1rc01*e3rc21+e1rc22_2*e3rc20+e1rc21_2*e3rc20+3*e1rc20_2*e3rc20-e1rc12_2*e3rc20-e1rc11_2*e3rc20+e1rc10_2*e3rc20-e1rc02_2*e3rc20-e1rc01_2*e3rc20+e1rc00_2*e3rc20+2*e1rc10*e1rc22*e3rc12-2*e1rc12*e1rc20*e3rc12+2*e1rc10*e1rc21*e3rc11-2*e1rc11*e1rc20*e3rc11+2*e1rc12*e1rc22*e3rc10+2*e1rc11*e1rc21*e3rc10+2*e1rc10*e1rc20*e3rc10+2*e1rc00*e1rc22*e3rc02-2*e1rc02*e1rc20*e3rc02+2*e1rc00*e1rc21*e3rc01-2*e1rc01*e1rc20*e3rc01+2*e1rc02*e1rc22*e3rc00+2*e1rc01*e1rc21*e3rc00+2*e1rc00*e1rc20*e3rc00;
    AMatrix(6, 14) = +2*e1rc20*e2rc22*e3rc22+2*e1rc22*e2rc20*e3rc22+2*e1rc10*e2rc12*e3rc22+2*e1rc12*e2rc10*e3rc22+2*e1rc00*e2rc02*e3rc22+2*e1rc02*e2rc00*e3rc22+2*e1rc20*e2rc21*e3rc21+2*e1rc21*e2rc20*e3rc21+2*e1rc10*e2rc11*e3rc21+2*e1rc11*e2rc10*e3rc21+2*e1rc00*e2rc01*e3rc21+2*e1rc01*e2rc00*e3rc21+2*e1rc22*e2rc22*e3rc20+2*e1rc21*e2rc21*e3rc20+6*e1rc20*e2rc20*e3rc20-2*e1rc12*e2rc12*e3rc20-2*e1rc11*e2rc11*e3rc20+2*e1rc10*e2rc10*e3rc20-2*e1rc02*e2rc02*e3rc20-2*e1rc01*e2rc01*e3rc20+2*e1rc00*e2rc00*e3rc20+2*e1rc10*e2rc22*e3rc12-2*e1rc12*e2rc20*e3rc12-2*e1rc20*e2rc12*e3rc12+2*e1rc22*e2rc10*e3rc12+2*e1rc10*e2rc21*e3rc11-2*e1rc11*e2rc20*e3rc11-2*e1rc20*e2rc11*e3rc11+2*e1rc21*e2rc10*e3rc11+2*e1rc12*e2rc22*e3rc10+2*e1rc11*e2rc21*e3rc10+2*e1rc10*e2rc20*e3rc10+2*e1rc22*e2rc12*e3rc10+2*e1rc21*e2rc11*e3rc10+2*e1rc20*e2rc10*e3rc10+2*e1rc00*e2rc22*e3rc02-2*e1rc02*e2rc20*e3rc02-2*e1rc20*e2rc02*e3rc02+2*e1rc22*e2rc00*e3rc02+2*e1rc00*e2rc21*e3rc01-2*e1rc01*e2rc20*e3rc01-2*e1rc20*e2rc01*e3rc01+2*e1rc21*e2rc00*e3rc01+2*e1rc02*e2rc22*e3rc00+2*e1rc01*e2rc21*e3rc00+2*e1rc00*e2rc20*e3rc00+2*e1rc22*e2rc02*e3rc00+2*e1rc21*e2rc01*e3rc00+2*e1rc20*e2rc00*e3rc00;
    AMatrix(6, 15) = +2*e2rc20*e2rc22*e3rc22+2*e2rc10*e2rc12*e3rc22+2*e2rc00*e2rc02*e3rc22+2*e2rc20*e2rc21*e3rc21+2*e2rc10*e2rc11*e3rc21+2*e2rc00*e2rc01*e3rc21+e2rc22_2*e3rc20+e2rc21_2*e3rc20+3*e2rc20_2*e3rc20-e2rc12_2*e3rc20-e2rc11_2*e3rc20+e2rc10_2*e3rc20-e2rc02_2*e3rc20-e2rc01_2*e3rc20+e2rc00_2*e3rc20+2*e2rc10*e2rc22*e3rc12-2*e2rc12*e2rc20*e3rc12+2*e2rc10*e2rc21*e3rc11-2*e2rc11*e2rc20*e3rc11+2*e2rc12*e2rc22*e3rc10+2*e2rc11*e2rc21*e3rc10+2*e2rc10*e2rc20*e3rc10+2*e2rc00*e2rc22*e3rc02-2*e2rc02*e2rc20*e3rc02+2*e2rc00*e2rc21*e3rc01-2*e2rc01*e2rc20*e3rc01+2*e2rc02*e2rc22*e3rc00+2*e2rc01*e2rc21*e3rc00+2*e2rc00*e2rc20*e3rc00;
    AMatrix(6, 16) = +e0rc20*e3rc22_2+2*e0rc22*e3rc20*e3rc22+2*e0rc10*e3rc12*e3rc22+2*e0rc12*e3rc10*e3rc22+2*e0rc00*e3rc02*e3rc22+2*e0rc02*e3rc00*e3rc22+e0rc20*e3rc21_2+2*e0rc21*e3rc20*e3rc21+2*e0rc10*e3rc11*e3rc21+2*e0rc11*e3rc10*e3rc21+2*e0rc00*e3rc01*e3rc21+2*e0rc01*e3rc00*e3rc21+3*e0rc20*e3rc20_2-2*e0rc12*e3rc12*e3rc20-2*e0rc11*e3rc11*e3rc20+2*e0rc10*e3rc10*e3rc20-2*e0rc02*e3rc02*e3rc20-2*e0rc01*e3rc01*e3rc20+2*e0rc00*e3rc00*e3rc20-e0rc20*e3rc12_2+2*e0rc22*e3rc10*e3rc12-e0rc20*e3rc11_2+2*e0rc21*e3rc10*e3rc11+e0rc20*e3rc10_2-e0rc20*e3rc02_2+2*e0rc22*e3rc00*e3rc02-e0rc20*e3rc01_2+2*e0rc21*e3rc00*e3rc01+e0rc20*e3rc00_2;
    AMatrix(6, 17) = +e1rc20*e3rc22_2+2*e1rc22*e3rc20*e3rc22+2*e1rc10*e3rc12*e3rc22+2*e1rc12*e3rc10*e3rc22+2*e1rc00*e3rc02*e3rc22+2*e1rc02*e3rc00*e3rc22+e1rc20*e3rc21_2+2*e1rc21*e3rc20*e3rc21+2*e1rc10*e3rc11*e3rc21+2*e1rc11*e3rc10*e3rc21+2*e1rc00*e3rc01*e3rc21+2*e1rc01*e3rc00*e3rc21+3*e1rc20*e3rc20_2-2*e1rc12*e3rc12*e3rc20-2*e1rc11*e3rc11*e3rc20+2*e1rc10*e3rc10*e3rc20-2*e1rc02*e3rc02*e3rc20-2*e1rc01*e3rc01*e3rc20+2*e1rc00*e3rc00*e3rc20-e1rc20*e3rc12_2+2*e1rc22*e3rc10*e3rc12-e1rc20*e3rc11_2+2*e1rc21*e3rc10*e3rc11+e1rc20*e3rc10_2-e1rc20*e3rc02_2+2*e1rc22*e3rc00*e3rc02-e1rc20*e3rc01_2+2*e1rc21*e3rc00*e3rc01+e1rc20*e3rc00_2;
    AMatrix(6, 18) = +e2rc20*e3rc22_2+2*e2rc22*e3rc20*e3rc22+2*e2rc10*e3rc12*e3rc22+2*e2rc12*e3rc10*e3rc22+2*e2rc00*e3rc02*e3rc22+2*e2rc02*e3rc00*e3rc22+e2rc20*e3rc21_2+2*e2rc21*e3rc20*e3rc21+2*e2rc10*e3rc11*e3rc21+2*e2rc11*e3rc10*e3rc21+2*e2rc00*e3rc01*e3rc21+2*e2rc01*e3rc00*e3rc21+3*e2rc20*e3rc20_2-2*e2rc12*e3rc12*e3rc20-2*e2rc11*e3rc11*e3rc20+2*e2rc10*e3rc10*e3rc20-2*e2rc02*e3rc02*e3rc20-2*e2rc01*e3rc01*e3rc20+2*e2rc00*e3rc00*e3rc20-e2rc20*e3rc12_2+2*e2rc22*e3rc10*e3rc12-e2rc20*e3rc11_2+2*e2rc21*e3rc10*e3rc11+e2rc20*e3rc10_2-e2rc20*e3rc02_2+2*e2rc22*e3rc00*e3rc02-e2rc20*e3rc01_2+2*e2rc21*e3rc00*e3rc01+e2rc20*e3rc00_2;
    AMatrix(6, 19) = +e3rc20*e3rc22_2+2*e3rc10*e3rc12*e3rc22+2*e3rc00*e3rc02*e3rc22+e3rc20*e3rc21_2+2*e3rc10*e3rc11*e3rc21+2*e3rc00*e3rc01*e3rc21+e3rc20_3-e3rc12_2*e3rc20-e3rc11_2*e3rc20+e3rc10_2*e3rc20-e3rc02_2*e3rc20-e3rc01_2*e3rc20+e3rc00_2*e3rc20;
    AMatrix(7, 0) = +e0rc21*e0rc22_2+2*e0rc11*e0rc12*e0rc22+2*e0rc01*e0rc02*e0rc22+e0rc21_3+e0rc20_2*e0rc21-e0rc12_2*e0rc21+e0rc11_2*e0rc21-e0rc10_2*e0rc21-e0rc02_2*e0rc21+e0rc01_2*e0rc21-e0rc00_2*e0rc21+2*e0rc10*e0rc11*e0rc20+2*e0rc00*e0rc01*e0rc20;
    AMatrix(7, 1) = +2*e0rc21*e0rc22*e1rc22+2*e0rc11*e0rc12*e1rc22+2*e0rc01*e0rc02*e1rc22+e0rc22_2*e1rc21+3*e0rc21_2*e1rc21+e0rc20_2*e1rc21-e0rc12_2*e1rc21+e0rc11_2*e1rc21-e0rc10_2*e1rc21-e0rc02_2*e1rc21+e0rc01_2*e1rc21-e0rc00_2*e1rc21+2*e0rc20*e0rc21*e1rc20+2*e0rc10*e0rc11*e1rc20+2*e0rc00*e0rc01*e1rc20+2*e0rc11*e0rc22*e1rc12-2*e0rc12*e0rc21*e1rc12+2*e0rc12*e0rc22*e1rc11+2*e0rc11*e0rc21*e1rc11+2*e0rc10*e0rc20*e1rc11-2*e0rc10*e0rc21*e1rc10+2*e0rc11*e0rc20*e1rc10+2*e0rc01*e0rc22*e1rc02-2*e0rc02*e0rc21*e1rc02+2*e0rc02*e0rc22*e1rc01+2*e0rc01*e0rc21*e1rc01+2*e0rc00*e0rc20*e1rc01-2*e0rc00*e0rc21*e1rc00+2*e0rc01*e0rc20*e1rc00;
    AMatrix(7, 2) = +2*e0rc21*e0rc22*e2rc22+2*e0rc11*e0rc12*e2rc22+2*e0rc01*e0rc02*e2rc22+e0rc22_2*e2rc21+3*e0rc21_2*e2rc21+e0rc20_2*e2rc21-e0rc12_2*e2rc21+e0rc11_2*e2rc21-e0rc10_2*e2rc21-e0rc02_2*e2rc21+e0rc01_2*e2rc21-e0rc00_2*e2rc21+2*e0rc20*e0rc21*e2rc20+2*e0rc10*e0rc11*e2rc20+2*e0rc00*e0rc01*e2rc20+2*e0rc11*e0rc22*e2rc12-2*e0rc12*e0rc21*e2rc12+2*e0rc12*e0rc22*e2rc11+2*e0rc11*e0rc21*e2rc11+2*e0rc10*e0rc20*e2rc11-2*e0rc10*e0rc21*e2rc10+2*e0rc11*e0rc20*e2rc10+2*e0rc01*e0rc22*e2rc02-2*e0rc02*e0rc21*e2rc02+2*e0rc02*e0rc22*e2rc01+2*e0rc01*e0rc21*e2rc01+2*e0rc00*e0rc20*e2rc01-2*e0rc00*e0rc21*e2rc00+2*e0rc01*e0rc20*e2rc00;
    AMatrix(7, 3) = +e0rc21*e1rc22_2+2*e0rc22*e1rc21*e1rc22+2*e0rc11*e1rc12*e1rc22+2*e0rc12*e1rc11*e1rc22+2*e0rc01*e1rc02*e1rc22+2*e0rc02*e1rc01*e1rc22+3*e0rc21*e1rc21_2+2*e0rc20*e1rc20*e1rc21-2*e0rc12*e1rc12*e1rc21+2*e0rc11*e1rc11*e1rc21-2*e0rc10*e1rc10*e1rc21-2*e0rc02*e1rc02*e1rc21+2*e0rc01*e1rc01*e1rc21-2*e0rc00*e1rc00*e1rc21+e0rc21*e1rc20_2+2*e0rc10*e1rc11*e1rc20+2*e0rc11*e1rc10*e1rc20+2*e0rc00*e1rc01*e1rc20+2*e0rc01*e1rc00*e1rc20-e0rc21*e1rc12_2+2*e0rc22*e1rc11*e1rc12+e0rc21*e1rc11_2+2*e0rc20*e1rc10*e1rc11-e0rc21*e1rc10_2-e0rc21*e1rc02_2+2*e0rc22*e1rc01*e1rc02+e0rc21*e1rc01_2+2*e0rc20*e1rc00*e1rc01-e0rc21*e1rc00_2;
    AMatrix(7, 4) = +2*e0rc21*e1rc22*e2rc22+2*e0rc22*e1rc21*e2rc22+2*e0rc11*e1rc12*e2rc22+2*e0rc12*e1rc11*e2rc22+2*e0rc01*e1rc02*e2rc22+2*e0rc02*e1rc01*e2rc22+2*e0rc22*e1rc22*e2rc21+6*e0rc21*e1rc21*e2rc21+2*e0rc20*e1rc20*e2rc21-2*e0rc12*e1rc12*e2rc21+2*e0rc11*e1rc11*e2rc21-2*e0rc10*e1rc10*e2rc21-2*e0rc02*e1rc02*e2rc21+2*e0rc01*e1rc01*e2rc21-2*e0rc00*e1rc00*e2rc21+2*e0rc20*e1rc21*e2rc20+2*e0rc21*e1rc20*e2rc20+2*e0rc10*e1rc11*e2rc20+2*e0rc11*e1rc10*e2rc20+2*e0rc00*e1rc01*e2rc20+2*e0rc01*e1rc00*e2rc20+2*e0rc11*e1rc22*e2rc12-2*e0rc12*e1rc21*e2rc12-2*e0rc21*e1rc12*e2rc12+2*e0rc22*e1rc11*e2rc12+2*e0rc12*e1rc22*e2rc11+2*e0rc11*e1rc21*e2rc11+2*e0rc10*e1rc20*e2rc11+2*e0rc22*e1rc12*e2rc11+2*e0rc21*e1rc11*e2rc11+2*e0rc20*e1rc10*e2rc11-2*e0rc10*e1rc21*e2rc10+2*e0rc11*e1rc20*e2rc10+2*e0rc20*e1rc11*e2rc10-2*e0rc21*e1rc10*e2rc10+2*e0rc01*e1rc22*e2rc02-2*e0rc02*e1rc21*e2rc02-2*e0rc21*e1rc02*e2rc02+2*e0rc22*e1rc01*e2rc02+2*e0rc02*e1rc22*e2rc01+2*e0rc01*e1rc21*e2rc01+2*e0rc00*e1rc20*e2rc01+2*e0rc22*e1rc02*e2rc01+2*e0rc21*e1rc01*e2rc01+2*e0rc20*e1rc00*e2rc01-2*e0rc00*e1rc21*e2rc00+2*e0rc01*e1rc20*e2rc00+2*e0rc20*e1rc01*e2rc00-2*e0rc21*e1rc00*e2rc00;
    AMatrix(7, 5) = +e0rc21*e2rc22_2+2*e0rc22*e2rc21*e2rc22+2*e0rc11*e2rc12*e2rc22+2*e0rc12*e2rc11*e2rc22+2*e0rc01*e2rc02*e2rc22+2*e0rc02*e2rc01*e2rc22+3*e0rc21*e2rc21_2+2*e0rc20*e2rc20*e2rc21-2*e0rc12*e2rc12*e2rc21+2*e0rc11*e2rc11*e2rc21-2*e0rc10*e2rc10*e2rc21-2*e0rc02*e2rc02*e2rc21+2*e0rc01*e2rc01*e2rc21-2*e0rc00*e2rc00*e2rc21+e0rc21*e2rc20_2+2*e0rc10*e2rc11*e2rc20+2*e0rc11*e2rc10*e2rc20+2*e0rc00*e2rc01*e2rc20+2*e0rc01*e2rc00*e2rc20-e0rc21*e2rc12_2+2*e0rc22*e2rc11*e2rc12+e0rc21*e2rc11_2+2*e0rc20*e2rc10*e2rc11-e0rc21*e2rc10_2-e0rc21*e2rc02_2+2*e0rc22*e2rc01*e2rc02+e0rc21*e2rc01_2+2*e0rc20*e2rc00*e2rc01-e0rc21*e2rc00_2;
    AMatrix(7, 6) = +e1rc21*e1rc22_2+2*e1rc11*e1rc12*e1rc22+2*e1rc01*e1rc02*e1rc22+e1rc21_3+e1rc20_2*e1rc21-e1rc12_2*e1rc21+e1rc11_2*e1rc21-e1rc10_2*e1rc21-e1rc02_2*e1rc21+e1rc01_2*e1rc21-e1rc00_2*e1rc21+2*e1rc10*e1rc11*e1rc20+2*e1rc00*e1rc01*e1rc20;
    AMatrix(7, 7) = +2*e1rc21*e1rc22*e2rc22+2*e1rc11*e1rc12*e2rc22+2*e1rc01*e1rc02*e2rc22+e1rc22_2*e2rc21+3*e1rc21_2*e2rc21+e1rc20_2*e2rc21-e1rc12_2*e2rc21+e1rc11_2*e2rc21-e1rc10_2*e2rc21-e1rc02_2*e2rc21+e1rc01_2*e2rc21-e1rc00_2*e2rc21+2*e1rc20*e1rc21*e2rc20+2*e1rc10*e1rc11*e2rc20+2*e1rc00*e1rc01*e2rc20+2*e1rc11*e1rc22*e2rc12-2*e1rc12*e1rc21*e2rc12+2*e1rc12*e1rc22*e2rc11+2*e1rc11*e1rc21*e2rc11+2*e1rc10*e1rc20*e2rc11-2*e1rc10*e1rc21*e2rc10+2*e1rc11*e1rc20*e2rc10+2*e1rc01*e1rc22*e2rc02-2*e1rc02*e1rc21*e2rc02+2*e1rc02*e1rc22*e2rc01+2*e1rc01*e1rc21*e2rc01+2*e1rc00*e1rc20*e2rc01-2*e1rc00*e1rc21*e2rc00+2*e1rc01*e1rc20*e2rc00;
    AMatrix(7, 8) = +e1rc21*e2rc22_2+2*e1rc22*e2rc21*e2rc22+2*e1rc11*e2rc12*e2rc22+2*e1rc12*e2rc11*e2rc22+2*e1rc01*e2rc02*e2rc22+2*e1rc02*e2rc01*e2rc22+3*e1rc21*e2rc21_2+2*e1rc20*e2rc20*e2rc21-2*e1rc12*e2rc12*e2rc21+2*e1rc11*e2rc11*e2rc21-2*e1rc10*e2rc10*e2rc21-2*e1rc02*e2rc02*e2rc21+2*e1rc01*e2rc01*e2rc21-2*e1rc00*e2rc00*e2rc21+e1rc21*e2rc20_2+2*e1rc10*e2rc11*e2rc20+2*e1rc11*e2rc10*e2rc20+2*e1rc00*e2rc01*e2rc20+2*e1rc01*e2rc00*e2rc20-e1rc21*e2rc12_2+2*e1rc22*e2rc11*e2rc12+e1rc21*e2rc11_2+2*e1rc20*e2rc10*e2rc11-e1rc21*e2rc10_2-e1rc21*e2rc02_2+2*e1rc22*e2rc01*e2rc02+e1rc21*e2rc01_2+2*e1rc20*e2rc00*e2rc01-e1rc21*e2rc00_2;
    AMatrix(7, 9) = e2rc21*e2rc22_2+2*e2rc11*e2rc12*e2rc22+2*e2rc01*e2rc02*e2rc22+e2rc21_3+e2rc20_2*e2rc21-e2rc12_2*e2rc21+e2rc11_2*e2rc21-e2rc10_2*e2rc21-e2rc02_2*e2rc21+e2rc01_2*e2rc21-e2rc00_2*e2rc21+2*e2rc10*e2rc11*e2rc20+2*e2rc00*e2rc01*e2rc20;
    AMatrix(7, 10) = +2*e0rc21*e0rc22*e3rc22+2*e0rc11*e0rc12*e3rc22+2*e0rc01*e0rc02*e3rc22+e0rc22_2*e3rc21+3*e0rc21_2*e3rc21+e0rc20_2*e3rc21-e0rc12_2*e3rc21+e0rc11_2*e3rc21-e0rc10_2*e3rc21-e0rc02_2*e3rc21+e0rc01_2*e3rc21-e0rc00_2*e3rc21+2*e0rc20*e0rc21*e3rc20+2*e0rc10*e0rc11*e3rc20+2*e0rc00*e0rc01*e3rc20+2*e0rc11*e0rc22*e3rc12-2*e0rc12*e0rc21*e3rc12+2*e0rc12*e0rc22*e3rc11+2*e0rc11*e0rc21*e3rc11+2*e0rc10*e0rc20*e3rc11-2*e0rc10*e0rc21*e3rc10+2*e0rc11*e0rc20*e3rc10+2*e0rc01*e0rc22*e3rc02-2*e0rc02*e0rc21*e3rc02+2*e0rc02*e0rc22*e3rc01+2*e0rc01*e0rc21*e3rc01+2*e0rc00*e0rc20*e3rc01-2*e0rc00*e0rc21*e3rc00+2*e0rc01*e0rc20*e3rc00;
    AMatrix(7, 11) = +2*e0rc21*e1rc22*e3rc22+2*e0rc22*e1rc21*e3rc22+2*e0rc11*e1rc12*e3rc22+2*e0rc12*e1rc11*e3rc22+2*e0rc01*e1rc02*e3rc22+2*e0rc02*e1rc01*e3rc22+2*e0rc22*e1rc22*e3rc21+6*e0rc21*e1rc21*e3rc21+2*e0rc20*e1rc20*e3rc21-2*e0rc12*e1rc12*e3rc21+2*e0rc11*e1rc11*e3rc21-2*e0rc10*e1rc10*e3rc21-2*e0rc02*e1rc02*e3rc21+2*e0rc01*e1rc01*e3rc21-2*e0rc00*e1rc00*e3rc21+2*e0rc20*e1rc21*e3rc20+2*e0rc21*e1rc20*e3rc20+2*e0rc10*e1rc11*e3rc20+2*e0rc11*e1rc10*e3rc20+2*e0rc00*e1rc01*e3rc20+2*e0rc01*e1rc00*e3rc20+2*e0rc11*e1rc22*e3rc12-2*e0rc12*e1rc21*e3rc12-2*e0rc21*e1rc12*e3rc12+2*e0rc22*e1rc11*e3rc12+2*e0rc12*e1rc22*e3rc11+2*e0rc11*e1rc21*e3rc11+2*e0rc10*e1rc20*e3rc11+2*e0rc22*e1rc12*e3rc11+2*e0rc21*e1rc11*e3rc11+2*e0rc20*e1rc10*e3rc11-2*e0rc10*e1rc21*e3rc10+2*e0rc11*e1rc20*e3rc10+2*e0rc20*e1rc11*e3rc10-2*e0rc21*e1rc10*e3rc10+2*e0rc01*e1rc22*e3rc02-2*e0rc02*e1rc21*e3rc02-2*e0rc21*e1rc02*e3rc02+2*e0rc22*e1rc01*e3rc02+2*e0rc02*e1rc22*e3rc01+2*e0rc01*e1rc21*e3rc01+2*e0rc00*e1rc20*e3rc01+2*e0rc22*e1rc02*e3rc01+2*e0rc21*e1rc01*e3rc01+2*e0rc20*e1rc00*e3rc01-2*e0rc00*e1rc21*e3rc00+2*e0rc01*e1rc20*e3rc00+2*e0rc20*e1rc01*e3rc00-2*e0rc21*e1rc00*e3rc00;
    AMatrix(7, 12) = +2*e0rc21*e2rc22*e3rc22+2*e0rc22*e2rc21*e3rc22+2*e0rc11*e2rc12*e3rc22+2*e0rc12*e2rc11*e3rc22+2*e0rc01*e2rc02*e3rc22+2*e0rc02*e2rc01*e3rc22+2*e0rc22*e2rc22*e3rc21+6*e0rc21*e2rc21*e3rc21+2*e0rc20*e2rc20*e3rc21-2*e0rc12*e2rc12*e3rc21+2*e0rc11*e2rc11*e3rc21-2*e0rc10*e2rc10*e3rc21-2*e0rc02*e2rc02*e3rc21+2*e0rc01*e2rc01*e3rc21-2*e0rc00*e2rc00*e3rc21+2*e0rc20*e2rc21*e3rc20+2*e0rc21*e2rc20*e3rc20+2*e0rc10*e2rc11*e3rc20+2*e0rc11*e2rc10*e3rc20+2*e0rc00*e2rc01*e3rc20+2*e0rc01*e2rc00*e3rc20+2*e0rc11*e2rc22*e3rc12-2*e0rc12*e2rc21*e3rc12-2*e0rc21*e2rc12*e3rc12+2*e0rc22*e2rc11*e3rc12+2*e0rc12*e2rc22*e3rc11+2*e0rc11*e2rc21*e3rc11+2*e0rc10*e2rc20*e3rc11+2*e0rc22*e2rc12*e3rc11+2*e0rc21*e2rc11*e3rc11+2*e0rc20*e2rc10*e3rc11-2*e0rc10*e2rc21*e3rc10+2*e0rc11*e2rc20*e3rc10+2*e0rc20*e2rc11*e3rc10-2*e0rc21*e2rc10*e3rc10+2*e0rc01*e2rc22*e3rc02-2*e0rc02*e2rc21*e3rc02-2*e0rc21*e2rc02*e3rc02+2*e0rc22*e2rc01*e3rc02+2*e0rc02*e2rc22*e3rc01+2*e0rc01*e2rc21*e3rc01+2*e0rc00*e2rc20*e3rc01+2*e0rc22*e2rc02*e3rc01+2*e0rc21*e2rc01*e3rc01+2*e0rc20*e2rc00*e3rc01-2*e0rc00*e2rc21*e3rc00+2*e0rc01*e2rc20*e3rc00+2*e0rc20*e2rc01*e3rc00-2*e0rc21*e2rc00*e3rc00;
    AMatrix(7, 13) = +2*e1rc21*e1rc22*e3rc22+2*e1rc11*e1rc12*e3rc22+2*e1rc01*e1rc02*e3rc22+e1rc22_2*e3rc21+3*e1rc21_2*e3rc21+e1rc20_2*e3rc21-e1rc12_2*e3rc21+e1rc11_2*e3rc21-e1rc10_2*e3rc21-e1rc02_2*e3rc21+e1rc01_2*e3rc21-e1rc00_2*e3rc21+2*e1rc20*e1rc21*e3rc20+2*e1rc10*e1rc11*e3rc20+2*e1rc00*e1rc01*e3rc20+2*e1rc11*e1rc22*e3rc12-2*e1rc12*e1rc21*e3rc12+2*e1rc12*e1rc22*e3rc11+2*e1rc11*e1rc21*e3rc11+2*e1rc10*e1rc20*e3rc11-2*e1rc10*e1rc21*e3rc10+2*e1rc11*e1rc20*e3rc10+2*e1rc01*e1rc22*e3rc02-2*e1rc02*e1rc21*e3rc02+2*e1rc02*e1rc22*e3rc01+2*e1rc01*e1rc21*e3rc01+2*e1rc00*e1rc20*e3rc01-2*e1rc00*e1rc21*e3rc00+2*e1rc01*e1rc20*e3rc00;
    AMatrix(7, 14) = +2*e1rc21*e2rc22*e3rc22+2*e1rc22*e2rc21*e3rc22+2*e1rc11*e2rc12*e3rc22+2*e1rc12*e2rc11*e3rc22+2*e1rc01*e2rc02*e3rc22+2*e1rc02*e2rc01*e3rc22+2*e1rc22*e2rc22*e3rc21+6*e1rc21*e2rc21*e3rc21+2*e1rc20*e2rc20*e3rc21-2*e1rc12*e2rc12*e3rc21+2*e1rc11*e2rc11*e3rc21-2*e1rc10*e2rc10*e3rc21-2*e1rc02*e2rc02*e3rc21+2*e1rc01*e2rc01*e3rc21-2*e1rc00*e2rc00*e3rc21+2*e1rc20*e2rc21*e3rc20+2*e1rc21*e2rc20*e3rc20+2*e1rc10*e2rc11*e3rc20+2*e1rc11*e2rc10*e3rc20+2*e1rc00*e2rc01*e3rc20+2*e1rc01*e2rc00*e3rc20+2*e1rc11*e2rc22*e3rc12-2*e1rc12*e2rc21*e3rc12-2*e1rc21*e2rc12*e3rc12+2*e1rc22*e2rc11*e3rc12+2*e1rc12*e2rc22*e3rc11+2*e1rc11*e2rc21*e3rc11+2*e1rc10*e2rc20*e3rc11+2*e1rc22*e2rc12*e3rc11+2*e1rc21*e2rc11*e3rc11+2*e1rc20*e2rc10*e3rc11-2*e1rc10*e2rc21*e3rc10+2*e1rc11*e2rc20*e3rc10+2*e1rc20*e2rc11*e3rc10-2*e1rc21*e2rc10*e3rc10+2*e1rc01*e2rc22*e3rc02-2*e1rc02*e2rc21*e3rc02-2*e1rc21*e2rc02*e3rc02+2*e1rc22*e2rc01*e3rc02+2*e1rc02*e2rc22*e3rc01+2*e1rc01*e2rc21*e3rc01+2*e1rc00*e2rc20*e3rc01+2*e1rc22*e2rc02*e3rc01+2*e1rc21*e2rc01*e3rc01+2*e1rc20*e2rc00*e3rc01-2*e1rc00*e2rc21*e3rc00+2*e1rc01*e2rc20*e3rc00+2*e1rc20*e2rc01*e3rc00-2*e1rc21*e2rc00*e3rc00;
    AMatrix(7, 15) = +2*e2rc21*e2rc22*e3rc22+2*e2rc11*e2rc12*e3rc22+2*e2rc01*e2rc02*e3rc22+e2rc22_2*e3rc21+3*e2rc21_2*e3rc21+e2rc20_2*e3rc21-e2rc12_2*e3rc21+e2rc11_2*e3rc21-e2rc10_2*e3rc21-e2rc02_2*e3rc21+e2rc01_2*e3rc21-e2rc00_2*e3rc21+2*e2rc20*e2rc21*e3rc20+2*e2rc10*e2rc11*e3rc20+2*e2rc00*e2rc01*e3rc20+2*e2rc11*e2rc22*e3rc12-2*e2rc12*e2rc21*e3rc12+2*e2rc12*e2rc22*e3rc11+2*e2rc11*e2rc21*e3rc11+2*e2rc10*e2rc20*e3rc11-2*e2rc10*e2rc21*e3rc10+2*e2rc11*e2rc20*e3rc10+2*e2rc01*e2rc22*e3rc02-2*e2rc02*e2rc21*e3rc02+2*e2rc02*e2rc22*e3rc01+2*e2rc01*e2rc21*e3rc01+2*e2rc00*e2rc20*e3rc01-2*e2rc00*e2rc21*e3rc00+2*e2rc01*e2rc20*e3rc00;
    AMatrix(7, 16) = +e0rc21*e3rc22_2+2*e0rc22*e3rc21*e3rc22+2*e0rc11*e3rc12*e3rc22+2*e0rc12*e3rc11*e3rc22+2*e0rc01*e3rc02*e3rc22+2*e0rc02*e3rc01*e3rc22+3*e0rc21*e3rc21_2+2*e0rc20*e3rc20*e3rc21-2*e0rc12*e3rc12*e3rc21+2*e0rc11*e3rc11*e3rc21-2*e0rc10*e3rc10*e3rc21-2*e0rc02*e3rc02*e3rc21+2*e0rc01*e3rc01*e3rc21-2*e0rc00*e3rc00*e3rc21+e0rc21*e3rc20_2+2*e0rc10*e3rc11*e3rc20+2*e0rc11*e3rc10*e3rc20+2*e0rc00*e3rc01*e3rc20+2*e0rc01*e3rc00*e3rc20-e0rc21*e3rc12_2+2*e0rc22*e3rc11*e3rc12+e0rc21*e3rc11_2+2*e0rc20*e3rc10*e3rc11-e0rc21*e3rc10_2-e0rc21*e3rc02_2+2*e0rc22*e3rc01*e3rc02+e0rc21*e3rc01_2+2*e0rc20*e3rc00*e3rc01-e0rc21*e3rc00_2;
    AMatrix(7, 17) = +e1rc21*e3rc22_2+2*e1rc22*e3rc21*e3rc22+2*e1rc11*e3rc12*e3rc22+2*e1rc12*e3rc11*e3rc22+2*e1rc01*e3rc02*e3rc22+2*e1rc02*e3rc01*e3rc22+3*e1rc21*e3rc21_2+2*e1rc20*e3rc20*e3rc21-2*e1rc12*e3rc12*e3rc21+2*e1rc11*e3rc11*e3rc21-2*e1rc10*e3rc10*e3rc21-2*e1rc02*e3rc02*e3rc21+2*e1rc01*e3rc01*e3rc21-2*e1rc00*e3rc00*e3rc21+e1rc21*e3rc20_2+2*e1rc10*e3rc11*e3rc20+2*e1rc11*e3rc10*e3rc20+2*e1rc00*e3rc01*e3rc20+2*e1rc01*e3rc00*e3rc20-e1rc21*e3rc12_2+2*e1rc22*e3rc11*e3rc12+e1rc21*e3rc11_2+2*e1rc20*e3rc10*e3rc11-e1rc21*e3rc10_2-e1rc21*e3rc02_2+2*e1rc22*e3rc01*e3rc02+e1rc21*e3rc01_2+2*e1rc20*e3rc00*e3rc01-e1rc21*e3rc00_2;
    AMatrix(7, 18) = +e2rc21*e3rc22_2+2*e2rc22*e3rc21*e3rc22+2*e2rc11*e3rc12*e3rc22+2*e2rc12*e3rc11*e3rc22+2*e2rc01*e3rc02*e3rc22+2*e2rc02*e3rc01*e3rc22+3*e2rc21*e3rc21_2+2*e2rc20*e3rc20*e3rc21-2*e2rc12*e3rc12*e3rc21+2*e2rc11*e3rc11*e3rc21-2*e2rc10*e3rc10*e3rc21-2*e2rc02*e3rc02*e3rc21+2*e2rc01*e3rc01*e3rc21-2*e2rc00*e3rc00*e3rc21+e2rc21*e3rc20_2+2*e2rc10*e3rc11*e3rc20+2*e2rc11*e3rc10*e3rc20+2*e2rc00*e3rc01*e3rc20+2*e2rc01*e3rc00*e3rc20-e2rc21*e3rc12_2+2*e2rc22*e3rc11*e3rc12+e2rc21*e3rc11_2+2*e2rc20*e3rc10*e3rc11-e2rc21*e3rc10_2-e2rc21*e3rc02_2+2*e2rc22*e3rc01*e3rc02+e2rc21*e3rc01_2+2*e2rc20*e3rc00*e3rc01-e2rc21*e3rc00_2;
    AMatrix(7, 19) = +e3rc21*e3rc22_2+2*e3rc11*e3rc12*e3rc22+2*e3rc01*e3rc02*e3rc22+e3rc21_3+e3rc20_2*e3rc21-e3rc12_2*e3rc21+e3rc11_2*e3rc21-e3rc10_2*e3rc21-e3rc02_2*e3rc21+e3rc01_2*e3rc21-e3rc00_2*e3rc21+2*e3rc10*e3rc11*e3rc20+2*e3rc00*e3rc01*e3rc20;
    AMatrix(8, 0) = +e0rc22_3+e0rc21_2*e0rc22+e0rc20_2*e0rc22+e0rc12_2*e0rc22-e0rc11_2*e0rc22-e0rc10_2*e0rc22+e0rc02_2*e0rc22-e0rc01_2*e0rc22-e0rc00_2*e0rc22+2*e0rc11*e0rc12*e0rc21+2*e0rc01*e0rc02*e0rc21+2*e0rc10*e0rc12*e0rc20+2*e0rc00*e0rc02*e0rc20;
    AMatrix(8, 1) = +3*e0rc22_2*e1rc22+e0rc21_2*e1rc22+e0rc20_2*e1rc22+e0rc12_2*e1rc22-e0rc11_2*e1rc22-e0rc10_2*e1rc22+e0rc02_2*e1rc22-e0rc01_2*e1rc22-e0rc00_2*e1rc22+2*e0rc21*e0rc22*e1rc21+2*e0rc11*e0rc12*e1rc21+2*e0rc01*e0rc02*e1rc21+2*e0rc20*e0rc22*e1rc20+2*e0rc10*e0rc12*e1rc20+2*e0rc00*e0rc02*e1rc20+2*e0rc12*e0rc22*e1rc12+2*e0rc11*e0rc21*e1rc12+2*e0rc10*e0rc20*e1rc12-2*e0rc11*e0rc22*e1rc11+2*e0rc12*e0rc21*e1rc11-2*e0rc10*e0rc22*e1rc10+2*e0rc12*e0rc20*e1rc10+2*e0rc02*e0rc22*e1rc02+2*e0rc01*e0rc21*e1rc02+2*e0rc00*e0rc20*e1rc02-2*e0rc01*e0rc22*e1rc01+2*e0rc02*e0rc21*e1rc01-2*e0rc00*e0rc22*e1rc00+2*e0rc02*e0rc20*e1rc00;
    AMatrix(8, 2) = +3*e0rc22_2*e2rc22+e0rc21_2*e2rc22+e0rc20_2*e2rc22+e0rc12_2*e2rc22-e0rc11_2*e2rc22-e0rc10_2*e2rc22+e0rc02_2*e2rc22-e0rc01_2*e2rc22-e0rc00_2*e2rc22+2*e0rc21*e0rc22*e2rc21+2*e0rc11*e0rc12*e2rc21+2*e0rc01*e0rc02*e2rc21+2*e0rc20*e0rc22*e2rc20+2*e0rc10*e0rc12*e2rc20+2*e0rc00*e0rc02*e2rc20+2*e0rc12*e0rc22*e2rc12+2*e0rc11*e0rc21*e2rc12+2*e0rc10*e0rc20*e2rc12-2*e0rc11*e0rc22*e2rc11+2*e0rc12*e0rc21*e2rc11-2*e0rc10*e0rc22*e2rc10+2*e0rc12*e0rc20*e2rc10+2*e0rc02*e0rc22*e2rc02+2*e0rc01*e0rc21*e2rc02+2*e0rc00*e0rc20*e2rc02-2*e0rc01*e0rc22*e2rc01+2*e0rc02*e0rc21*e2rc01-2*e0rc00*e0rc22*e2rc00+2*e0rc02*e0rc20*e2rc00;
    AMatrix(8, 3) = +3*e0rc22*e1rc22_2+2*e0rc21*e1rc21*e1rc22+2*e0rc20*e1rc20*e1rc22+2*e0rc12*e1rc12*e1rc22-2*e0rc11*e1rc11*e1rc22-2*e0rc10*e1rc10*e1rc22+2*e0rc02*e1rc02*e1rc22-2*e0rc01*e1rc01*e1rc22-2*e0rc00*e1rc00*e1rc22+e0rc22*e1rc21_2+2*e0rc11*e1rc12*e1rc21+2*e0rc12*e1rc11*e1rc21+2*e0rc01*e1rc02*e1rc21+2*e0rc02*e1rc01*e1rc21+e0rc22*e1rc20_2+2*e0rc10*e1rc12*e1rc20+2*e0rc12*e1rc10*e1rc20+2*e0rc00*e1rc02*e1rc20+2*e0rc02*e1rc00*e1rc20+e0rc22*e1rc12_2+2*e0rc21*e1rc11*e1rc12+2*e0rc20*e1rc10*e1rc12-e0rc22*e1rc11_2-e0rc22*e1rc10_2+e0rc22*e1rc02_2+2*e0rc21*e1rc01*e1rc02+2*e0rc20*e1rc00*e1rc02-e0rc22*e1rc01_2-e0rc22*e1rc00_2;
    AMatrix(8, 4) = +6*e0rc22*e1rc22*e2rc22+2*e0rc21*e1rc21*e2rc22+2*e0rc20*e1rc20*e2rc22+2*e0rc12*e1rc12*e2rc22-2*e0rc11*e1rc11*e2rc22-2*e0rc10*e1rc10*e2rc22+2*e0rc02*e1rc02*e2rc22-2*e0rc01*e1rc01*e2rc22-2*e0rc00*e1rc00*e2rc22+2*e0rc21*e1rc22*e2rc21+2*e0rc22*e1rc21*e2rc21+2*e0rc11*e1rc12*e2rc21+2*e0rc12*e1rc11*e2rc21+2*e0rc01*e1rc02*e2rc21+2*e0rc02*e1rc01*e2rc21+2*e0rc20*e1rc22*e2rc20+2*e0rc22*e1rc20*e2rc20+2*e0rc10*e1rc12*e2rc20+2*e0rc12*e1rc10*e2rc20+2*e0rc00*e1rc02*e2rc20+2*e0rc02*e1rc00*e2rc20+2*e0rc12*e1rc22*e2rc12+2*e0rc11*e1rc21*e2rc12+2*e0rc10*e1rc20*e2rc12+2*e0rc22*e1rc12*e2rc12+2*e0rc21*e1rc11*e2rc12+2*e0rc20*e1rc10*e2rc12-2*e0rc11*e1rc22*e2rc11+2*e0rc12*e1rc21*e2rc11+2*e0rc21*e1rc12*e2rc11-2*e0rc22*e1rc11*e2rc11-2*e0rc10*e1rc22*e2rc10+2*e0rc12*e1rc20*e2rc10+2*e0rc20*e1rc12*e2rc10-2*e0rc22*e1rc10*e2rc10+2*e0rc02*e1rc22*e2rc02+2*e0rc01*e1rc21*e2rc02+2*e0rc00*e1rc20*e2rc02+2*e0rc22*e1rc02*e2rc02+2*e0rc21*e1rc01*e2rc02+2*e0rc20*e1rc00*e2rc02-2*e0rc01*e1rc22*e2rc01+2*e0rc02*e1rc21*e2rc01+2*e0rc21*e1rc02*e2rc01-2*e0rc22*e1rc01*e2rc01-2*e0rc00*e1rc22*e2rc00+2*e0rc02*e1rc20*e2rc00+2*e0rc20*e1rc02*e2rc00-2*e0rc22*e1rc00*e2rc00;
    AMatrix(8, 5) = +3*e0rc22*e2rc22_2+2*e0rc21*e2rc21*e2rc22+2*e0rc20*e2rc20*e2rc22+2*e0rc12*e2rc12*e2rc22-2*e0rc11*e2rc11*e2rc22-2*e0rc10*e2rc10*e2rc22+2*e0rc02*e2rc02*e2rc22-2*e0rc01*e2rc01*e2rc22-2*e0rc00*e2rc00*e2rc22+e0rc22*e2rc21_2+2*e0rc11*e2rc12*e2rc21+2*e0rc12*e2rc11*e2rc21+2*e0rc01*e2rc02*e2rc21+2*e0rc02*e2rc01*e2rc21+e0rc22*e2rc20_2+2*e0rc10*e2rc12*e2rc20+2*e0rc12*e2rc10*e2rc20+2*e0rc00*e2rc02*e2rc20+2*e0rc02*e2rc00*e2rc20+e0rc22*e2rc12_2+2*e0rc21*e2rc11*e2rc12+2*e0rc20*e2rc10*e2rc12-e0rc22*e2rc11_2-e0rc22*e2rc10_2+e0rc22*e2rc02_2+2*e0rc21*e2rc01*e2rc02+2*e0rc20*e2rc00*e2rc02-e0rc22*e2rc01_2-e0rc22*e2rc00_2;
    AMatrix(8, 6) = +e1rc22_3+e1rc21_2*e1rc22+e1rc20_2*e1rc22+e1rc12_2*e1rc22-e1rc11_2*e1rc22-e1rc10_2*e1rc22+e1rc02_2*e1rc22-e1rc01_2*e1rc22-e1rc00_2*e1rc22+2*e1rc11*e1rc12*e1rc21+2*e1rc01*e1rc02*e1rc21+2*e1rc10*e1rc12*e1rc20+2*e1rc00*e1rc02*e1rc20;
    AMatrix(8, 7) = +3*e1rc22_2*e2rc22+e1rc21_2*e2rc22+e1rc20_2*e2rc22+e1rc12_2*e2rc22-e1rc11_2*e2rc22-e1rc10_2*e2rc22+e1rc02_2*e2rc22-e1rc01_2*e2rc22-e1rc00_2*e2rc22+2*e1rc21*e1rc22*e2rc21+2*e1rc11*e1rc12*e2rc21+2*e1rc01*e1rc02*e2rc21+2*e1rc20*e1rc22*e2rc20+2*e1rc10*e1rc12*e2rc20+2*e1rc00*e1rc02*e2rc20+2*e1rc12*e1rc22*e2rc12+2*e1rc11*e1rc21*e2rc12+2*e1rc10*e1rc20*e2rc12-2*e1rc11*e1rc22*e2rc11+2*e1rc12*e1rc21*e2rc11-2*e1rc10*e1rc22*e2rc10+2*e1rc12*e1rc20*e2rc10+2*e1rc02*e1rc22*e2rc02+2*e1rc01*e1rc21*e2rc02+2*e1rc00*e1rc20*e2rc02-2*e1rc01*e1rc22*e2rc01+2*e1rc02*e1rc21*e2rc01-2*e1rc00*e1rc22*e2rc00+2*e1rc02*e1rc20*e2rc00;
    AMatrix(8, 8) = +3*e1rc22*e2rc22_2+2*e1rc21*e2rc21*e2rc22+2*e1rc20*e2rc20*e2rc22+2*e1rc12*e2rc12*e2rc22-2*e1rc11*e2rc11*e2rc22-2*e1rc10*e2rc10*e2rc22+2*e1rc02*e2rc02*e2rc22-2*e1rc01*e2rc01*e2rc22-2*e1rc00*e2rc00*e2rc22+e1rc22*e2rc21_2+2*e1rc11*e2rc12*e2rc21+2*e1rc12*e2rc11*e2rc21+2*e1rc01*e2rc02*e2rc21+2*e1rc02*e2rc01*e2rc21+e1rc22*e2rc20_2+2*e1rc10*e2rc12*e2rc20+2*e1rc12*e2rc10*e2rc20+2*e1rc00*e2rc02*e2rc20+2*e1rc02*e2rc00*e2rc20+e1rc22*e2rc12_2+2*e1rc21*e2rc11*e2rc12+2*e1rc20*e2rc10*e2rc12-e1rc22*e2rc11_2-e1rc22*e2rc10_2+e1rc22*e2rc02_2+2*e1rc21*e2rc01*e2rc02+2*e1rc20*e2rc00*e2rc02-e1rc22*e2rc01_2-e1rc22*e2rc00_2;
    AMatrix(8, 9) = e2rc22_3+e2rc21_2*e2rc22+e2rc20_2*e2rc22+e2rc12_2*e2rc22-e2rc11_2*e2rc22-e2rc10_2*e2rc22+e2rc02_2*e2rc22-e2rc01_2*e2rc22-e2rc00_2*e2rc22+2*e2rc11*e2rc12*e2rc21+2*e2rc01*e2rc02*e2rc21+2*e2rc10*e2rc12*e2rc20+2*e2rc00*e2rc02*e2rc20;
    AMatrix(8, 10) = +3*e0rc22_2*e3rc22+e0rc21_2*e3rc22+e0rc20_2*e3rc22+e0rc12_2*e3rc22-e0rc11_2*e3rc22-e0rc10_2*e3rc22+e0rc02_2*e3rc22-e0rc01_2*e3rc22-e0rc00_2*e3rc22+2*e0rc21*e0rc22*e3rc21+2*e0rc11*e0rc12*e3rc21+2*e0rc01*e0rc02*e3rc21+2*e0rc20*e0rc22*e3rc20+2*e0rc10*e0rc12*e3rc20+2*e0rc00*e0rc02*e3rc20+2*e0rc12*e0rc22*e3rc12+2*e0rc11*e0rc21*e3rc12+2*e0rc10*e0rc20*e3rc12-2*e0rc11*e0rc22*e3rc11+2*e0rc12*e0rc21*e3rc11-2*e0rc10*e0rc22*e3rc10+2*e0rc12*e0rc20*e3rc10+2*e0rc02*e0rc22*e3rc02+2*e0rc01*e0rc21*e3rc02+2*e0rc00*e0rc20*e3rc02-2*e0rc01*e0rc22*e3rc01+2*e0rc02*e0rc21*e3rc01-2*e0rc00*e0rc22*e3rc00+2*e0rc02*e0rc20*e3rc00;
    AMatrix(8, 11) = +6*e0rc22*e1rc22*e3rc22+2*e0rc21*e1rc21*e3rc22+2*e0rc20*e1rc20*e3rc22+2*e0rc12*e1rc12*e3rc22-2*e0rc11*e1rc11*e3rc22-2*e0rc10*e1rc10*e3rc22+2*e0rc02*e1rc02*e3rc22-2*e0rc01*e1rc01*e3rc22-2*e0rc00*e1rc00*e3rc22+2*e0rc21*e1rc22*e3rc21+2*e0rc22*e1rc21*e3rc21+2*e0rc11*e1rc12*e3rc21+2*e0rc12*e1rc11*e3rc21+2*e0rc01*e1rc02*e3rc21+2*e0rc02*e1rc01*e3rc21+2*e0rc20*e1rc22*e3rc20+2*e0rc22*e1rc20*e3rc20+2*e0rc10*e1rc12*e3rc20+2*e0rc12*e1rc10*e3rc20+2*e0rc00*e1rc02*e3rc20+2*e0rc02*e1rc00*e3rc20+2*e0rc12*e1rc22*e3rc12+2*e0rc11*e1rc21*e3rc12+2*e0rc10*e1rc20*e3rc12+2*e0rc22*e1rc12*e3rc12+2*e0rc21*e1rc11*e3rc12+2*e0rc20*e1rc10*e3rc12-2*e0rc11*e1rc22*e3rc11+2*e0rc12*e1rc21*e3rc11+2*e0rc21*e1rc12*e3rc11-2*e0rc22*e1rc11*e3rc11-2*e0rc10*e1rc22*e3rc10+2*e0rc12*e1rc20*e3rc10+2*e0rc20*e1rc12*e3rc10-2*e0rc22*e1rc10*e3rc10+2*e0rc02*e1rc22*e3rc02+2*e0rc01*e1rc21*e3rc02+2*e0rc00*e1rc20*e3rc02+2*e0rc22*e1rc02*e3rc02+2*e0rc21*e1rc01*e3rc02+2*e0rc20*e1rc00*e3rc02-2*e0rc01*e1rc22*e3rc01+2*e0rc02*e1rc21*e3rc01+2*e0rc21*e1rc02*e3rc01-2*e0rc22*e1rc01*e3rc01-2*e0rc00*e1rc22*e3rc00+2*e0rc02*e1rc20*e3rc00+2*e0rc20*e1rc02*e3rc00-2*e0rc22*e1rc00*e3rc00;
    AMatrix(8, 12) = +6*e0rc22*e2rc22*e3rc22+2*e0rc21*e2rc21*e3rc22+2*e0rc20*e2rc20*e3rc22+2*e0rc12*e2rc12*e3rc22-2*e0rc11*e2rc11*e3rc22-2*e0rc10*e2rc10*e3rc22+2*e0rc02*e2rc02*e3rc22-2*e0rc01*e2rc01*e3rc22-2*e0rc00*e2rc00*e3rc22+2*e0rc21*e2rc22*e3rc21+2*e0rc22*e2rc21*e3rc21+2*e0rc11*e2rc12*e3rc21+2*e0rc12*e2rc11*e3rc21+2*e0rc01*e2rc02*e3rc21+2*e0rc02*e2rc01*e3rc21+2*e0rc20*e2rc22*e3rc20+2*e0rc22*e2rc20*e3rc20+2*e0rc10*e2rc12*e3rc20+2*e0rc12*e2rc10*e3rc20+2*e0rc00*e2rc02*e3rc20+2*e0rc02*e2rc00*e3rc20+2*e0rc12*e2rc22*e3rc12+2*e0rc11*e2rc21*e3rc12+2*e0rc10*e2rc20*e3rc12+2*e0rc22*e2rc12*e3rc12+2*e0rc21*e2rc11*e3rc12+2*e0rc20*e2rc10*e3rc12-2*e0rc11*e2rc22*e3rc11+2*e0rc12*e2rc21*e3rc11+2*e0rc21*e2rc12*e3rc11-2*e0rc22*e2rc11*e3rc11-2*e0rc10*e2rc22*e3rc10+2*e0rc12*e2rc20*e3rc10+2*e0rc20*e2rc12*e3rc10-2*e0rc22*e2rc10*e3rc10+2*e0rc02*e2rc22*e3rc02+2*e0rc01*e2rc21*e3rc02+2*e0rc00*e2rc20*e3rc02+2*e0rc22*e2rc02*e3rc02+2*e0rc21*e2rc01*e3rc02+2*e0rc20*e2rc00*e3rc02-2*e0rc01*e2rc22*e3rc01+2*e0rc02*e2rc21*e3rc01+2*e0rc21*e2rc02*e3rc01-2*e0rc22*e2rc01*e3rc01-2*e0rc00*e2rc22*e3rc00+2*e0rc02*e2rc20*e3rc00+2*e0rc20*e2rc02*e3rc00-2*e0rc22*e2rc00*e3rc00;
    AMatrix(8, 13) = +3*e1rc22_2*e3rc22+e1rc21_2*e3rc22+e1rc20_2*e3rc22+e1rc12_2*e3rc22-e1rc11_2*e3rc22-e1rc10_2*e3rc22+e1rc02_2*e3rc22-e1rc01_2*e3rc22-e1rc00_2*e3rc22+2*e1rc21*e1rc22*e3rc21+2*e1rc11*e1rc12*e3rc21+2*e1rc01*e1rc02*e3rc21+2*e1rc20*e1rc22*e3rc20+2*e1rc10*e1rc12*e3rc20+2*e1rc00*e1rc02*e3rc20+2*e1rc12*e1rc22*e3rc12+2*e1rc11*e1rc21*e3rc12+2*e1rc10*e1rc20*e3rc12-2*e1rc11*e1rc22*e3rc11+2*e1rc12*e1rc21*e3rc11-2*e1rc10*e1rc22*e3rc10+2*e1rc12*e1rc20*e3rc10+2*e1rc02*e1rc22*e3rc02+2*e1rc01*e1rc21*e3rc02+2*e1rc00*e1rc20*e3rc02-2*e1rc01*e1rc22*e3rc01+2*e1rc02*e1rc21*e3rc01-2*e1rc00*e1rc22*e3rc00+2*e1rc02*e1rc20*e3rc00;
    AMatrix(8, 14) = +6*e1rc22*e2rc22*e3rc22+2*e1rc21*e2rc21*e3rc22+2*e1rc20*e2rc20*e3rc22+2*e1rc12*e2rc12*e3rc22-2*e1rc11*e2rc11*e3rc22-2*e1rc10*e2rc10*e3rc22+2*e1rc02*e2rc02*e3rc22-2*e1rc01*e2rc01*e3rc22-2*e1rc00*e2rc00*e3rc22+2*e1rc21*e2rc22*e3rc21+2*e1rc22*e2rc21*e3rc21+2*e1rc11*e2rc12*e3rc21+2*e1rc12*e2rc11*e3rc21+2*e1rc01*e2rc02*e3rc21+2*e1rc02*e2rc01*e3rc21+2*e1rc20*e2rc22*e3rc20+2*e1rc22*e2rc20*e3rc20+2*e1rc10*e2rc12*e3rc20+2*e1rc12*e2rc10*e3rc20+2*e1rc00*e2rc02*e3rc20+2*e1rc02*e2rc00*e3rc20+2*e1rc12*e2rc22*e3rc12+2*e1rc11*e2rc21*e3rc12+2*e1rc10*e2rc20*e3rc12+2*e1rc22*e2rc12*e3rc12+2*e1rc21*e2rc11*e3rc12+2*e1rc20*e2rc10*e3rc12-2*e1rc11*e2rc22*e3rc11+2*e1rc12*e2rc21*e3rc11+2*e1rc21*e2rc12*e3rc11-2*e1rc22*e2rc11*e3rc11-2*e1rc10*e2rc22*e3rc10+2*e1rc12*e2rc20*e3rc10+2*e1rc20*e2rc12*e3rc10-2*e1rc22*e2rc10*e3rc10+2*e1rc02*e2rc22*e3rc02+2*e1rc01*e2rc21*e3rc02+2*e1rc00*e2rc20*e3rc02+2*e1rc22*e2rc02*e3rc02+2*e1rc21*e2rc01*e3rc02+2*e1rc20*e2rc00*e3rc02-2*e1rc01*e2rc22*e3rc01+2*e1rc02*e2rc21*e3rc01+2*e1rc21*e2rc02*e3rc01-2*e1rc22*e2rc01*e3rc01-2*e1rc00*e2rc22*e3rc00+2*e1rc02*e2rc20*e3rc00+2*e1rc20*e2rc02*e3rc00-2*e1rc22*e2rc00*e3rc00;
    AMatrix(8, 15) = +3*e2rc22_2*e3rc22+e2rc21_2*e3rc22+e2rc20_2*e3rc22+e2rc12_2*e3rc22-e2rc11_2*e3rc22-e2rc10_2*e3rc22+e2rc02_2*e3rc22-e2rc01_2*e3rc22-e2rc00_2*e3rc22+2*e2rc21*e2rc22*e3rc21+2*e2rc11*e2rc12*e3rc21+2*e2rc01*e2rc02*e3rc21+2*e2rc20*e2rc22*e3rc20+2*e2rc10*e2rc12*e3rc20+2*e2rc00*e2rc02*e3rc20+2*e2rc12*e2rc22*e3rc12+2*e2rc11*e2rc21*e3rc12+2*e2rc10*e2rc20*e3rc12-2*e2rc11*e2rc22*e3rc11+2*e2rc12*e2rc21*e3rc11-2*e2rc10*e2rc22*e3rc10+2*e2rc12*e2rc20*e3rc10+2*e2rc02*e2rc22*e3rc02+2*e2rc01*e2rc21*e3rc02+2*e2rc00*e2rc20*e3rc02-2*e2rc01*e2rc22*e3rc01+2*e2rc02*e2rc21*e3rc01-2*e2rc00*e2rc22*e3rc00+2*e2rc02*e2rc20*e3rc00;
    AMatrix(8, 16) = +3*e0rc22*e3rc22_2+2*e0rc21*e3rc21*e3rc22+2*e0rc20*e3rc20*e3rc22+2*e0rc12*e3rc12*e3rc22-2*e0rc11*e3rc11*e3rc22-2*e0rc10*e3rc10*e3rc22+2*e0rc02*e3rc02*e3rc22-2*e0rc01*e3rc01*e3rc22-2*e0rc00*e3rc00*e3rc22+e0rc22*e3rc21_2+2*e0rc11*e3rc12*e3rc21+2*e0rc12*e3rc11*e3rc21+2*e0rc01*e3rc02*e3rc21+2*e0rc02*e3rc01*e3rc21+e0rc22*e3rc20_2+2*e0rc10*e3rc12*e3rc20+2*e0rc12*e3rc10*e3rc20+2*e0rc00*e3rc02*e3rc20+2*e0rc02*e3rc00*e3rc20+e0rc22*e3rc12_2+2*e0rc21*e3rc11*e3rc12+2*e0rc20*e3rc10*e3rc12-e0rc22*e3rc11_2-e0rc22*e3rc10_2+e0rc22*e3rc02_2+2*e0rc21*e3rc01*e3rc02+2*e0rc20*e3rc00*e3rc02-e0rc22*e3rc01_2-e0rc22*e3rc00_2;
    AMatrix(8, 17) = +3*e1rc22*e3rc22_2+2*e1rc21*e3rc21*e3rc22+2*e1rc20*e3rc20*e3rc22+2*e1rc12*e3rc12*e3rc22-2*e1rc11*e3rc11*e3rc22-2*e1rc10*e3rc10*e3rc22+2*e1rc02*e3rc02*e3rc22-2*e1rc01*e3rc01*e3rc22-2*e1rc00*e3rc00*e3rc22+e1rc22*e3rc21_2+2*e1rc11*e3rc12*e3rc21+2*e1rc12*e3rc11*e3rc21+2*e1rc01*e3rc02*e3rc21+2*e1rc02*e3rc01*e3rc21+e1rc22*e3rc20_2+2*e1rc10*e3rc12*e3rc20+2*e1rc12*e3rc10*e3rc20+2*e1rc00*e3rc02*e3rc20+2*e1rc02*e3rc00*e3rc20+e1rc22*e3rc12_2+2*e1rc21*e3rc11*e3rc12+2*e1rc20*e3rc10*e3rc12-e1rc22*e3rc11_2-e1rc22*e3rc10_2+e1rc22*e3rc02_2+2*e1rc21*e3rc01*e3rc02+2*e1rc20*e3rc00*e3rc02-e1rc22*e3rc01_2-e1rc22*e3rc00_2;
    AMatrix(8, 18) = +3*e2rc22*e3rc22_2+2*e2rc21*e3rc21*e3rc22+2*e2rc20*e3rc20*e3rc22+2*e2rc12*e3rc12*e3rc22-2*e2rc11*e3rc11*e3rc22-2*e2rc10*e3rc10*e3rc22+2*e2rc02*e3rc02*e3rc22-2*e2rc01*e3rc01*e3rc22-2*e2rc00*e3rc00*e3rc22+e2rc22*e3rc21_2+2*e2rc11*e3rc12*e3rc21+2*e2rc12*e3rc11*e3rc21+2*e2rc01*e3rc02*e3rc21+2*e2rc02*e3rc01*e3rc21+e2rc22*e3rc20_2+2*e2rc10*e3rc12*e3rc20+2*e2rc12*e3rc10*e3rc20+2*e2rc00*e3rc02*e3rc20+2*e2rc02*e3rc00*e3rc20+e2rc22*e3rc12_2+2*e2rc21*e3rc11*e3rc12+2*e2rc20*e3rc10*e3rc12-e2rc22*e3rc11_2-e2rc22*e3rc10_2+e2rc22*e3rc02_2+2*e2rc21*e3rc01*e3rc02+2*e2rc20*e3rc00*e3rc02-e2rc22*e3rc01_2-e2rc22*e3rc00_2;
    AMatrix(8, 19) = +e3rc22_3+e3rc21_2*e3rc22+e3rc20_2*e3rc22+e3rc12_2*e3rc22-e3rc11_2*e3rc22-e3rc10_2*e3rc22+e3rc02_2*e3rc22-e3rc01_2*e3rc22-e3rc00_2*e3rc22+2*e3rc11*e3rc12*e3rc21+2*e3rc01*e3rc02*e3rc21+2*e3rc10*e3rc12*e3rc20+2*e3rc00*e3rc02*e3rc20;
    AMatrix(9, 0) = +e0rc00*e0rc11*e0rc22-e0rc01*e0rc10*e0rc22-e0rc00*e0rc12*e0rc21+e0rc02*e0rc10*e0rc21+e0rc01*e0rc12*e0rc20-e0rc02*e0rc11*e0rc20;
    AMatrix(9, 1) = +e0rc00*e0rc11*e1rc22-e0rc01*e0rc10*e1rc22-e0rc00*e0rc12*e1rc21+e0rc02*e0rc10*e1rc21+e0rc01*e0rc12*e1rc20-e0rc02*e0rc11*e1rc20-e0rc00*e0rc21*e1rc12+e0rc01*e0rc20*e1rc12+e0rc00*e0rc22*e1rc11-e0rc02*e0rc20*e1rc11-e0rc01*e0rc22*e1rc10+e0rc02*e0rc21*e1rc10+e0rc10*e0rc21*e1rc02-e0rc11*e0rc20*e1rc02-e0rc10*e0rc22*e1rc01+e0rc12*e0rc20*e1rc01+e0rc11*e0rc22*e1rc00-e0rc12*e0rc21*e1rc00;
    AMatrix(9, 2) = +e0rc00*e0rc11*e2rc22-e0rc01*e0rc10*e2rc22-e0rc00*e0rc12*e2rc21+e0rc02*e0rc10*e2rc21+e0rc01*e0rc12*e2rc20-e0rc02*e0rc11*e2rc20-e0rc00*e0rc21*e2rc12+e0rc01*e0rc20*e2rc12+e0rc00*e0rc22*e2rc11-e0rc02*e0rc20*e2rc11-e0rc01*e0rc22*e2rc10+e0rc02*e0rc21*e2rc10+e0rc10*e0rc21*e2rc02-e0rc11*e0rc20*e2rc02-e0rc10*e0rc22*e2rc01+e0rc12*e0rc20*e2rc01+e0rc11*e0rc22*e2rc00-e0rc12*e0rc21*e2rc00;
    AMatrix(9, 3) = +e0rc00*e1rc11*e1rc22-e0rc01*e1rc10*e1rc22-e0rc10*e1rc01*e1rc22+e0rc11*e1rc00*e1rc22-e0rc00*e1rc12*e1rc21+e0rc02*e1rc10*e1rc21+e0rc10*e1rc02*e1rc21-e0rc12*e1rc00*e1rc21+e0rc01*e1rc12*e1rc20-e0rc02*e1rc11*e1rc20-e0rc11*e1rc02*e1rc20+e0rc12*e1rc01*e1rc20+e0rc20*e1rc01*e1rc12-e0rc21*e1rc00*e1rc12-e0rc20*e1rc02*e1rc11+e0rc22*e1rc00*e1rc11+e0rc21*e1rc02*e1rc10-e0rc22*e1rc01*e1rc10;
    AMatrix(9, 4) = +e0rc00*e1rc11*e2rc22-e0rc01*e1rc10*e2rc22-e0rc10*e1rc01*e2rc22+e0rc11*e1rc00*e2rc22-e0rc00*e1rc12*e2rc21+e0rc02*e1rc10*e2rc21+e0rc10*e1rc02*e2rc21-e0rc12*e1rc00*e2rc21+e0rc01*e1rc12*e2rc20-e0rc02*e1rc11*e2rc20-e0rc11*e1rc02*e2rc20+e0rc12*e1rc01*e2rc20-e0rc00*e1rc21*e2rc12+e0rc01*e1rc20*e2rc12+e0rc20*e1rc01*e2rc12-e0rc21*e1rc00*e2rc12+e0rc00*e1rc22*e2rc11-e0rc02*e1rc20*e2rc11-e0rc20*e1rc02*e2rc11+e0rc22*e1rc00*e2rc11-e0rc01*e1rc22*e2rc10+e0rc02*e1rc21*e2rc10+e0rc21*e1rc02*e2rc10-e0rc22*e1rc01*e2rc10+e0rc10*e1rc21*e2rc02-e0rc11*e1rc20*e2rc02-e0rc20*e1rc11*e2rc02+e0rc21*e1rc10*e2rc02-e0rc10*e1rc22*e2rc01+e0rc12*e1rc20*e2rc01+e0rc20*e1rc12*e2rc01-e0rc22*e1rc10*e2rc01+e0rc11*e1rc22*e2rc00-e0rc12*e1rc21*e2rc00-e0rc21*e1rc12*e2rc00+e0rc22*e1rc11*e2rc00;
    AMatrix(9, 5) = +e0rc00*e2rc11*e2rc22-e0rc01*e2rc10*e2rc22-e0rc10*e2rc01*e2rc22+e0rc11*e2rc00*e2rc22-e0rc00*e2rc12*e2rc21+e0rc02*e2rc10*e2rc21+e0rc10*e2rc02*e2rc21-e0rc12*e2rc00*e2rc21+e0rc01*e2rc12*e2rc20-e0rc02*e2rc11*e2rc20-e0rc11*e2rc02*e2rc20+e0rc12*e2rc01*e2rc20+e0rc20*e2rc01*e2rc12-e0rc21*e2rc00*e2rc12-e0rc20*e2rc02*e2rc11+e0rc22*e2rc00*e2rc11+e0rc21*e2rc02*e2rc10-e0rc22*e2rc01*e2rc10;
    AMatrix(9, 6) = +e1rc00*e1rc11*e1rc22-e1rc01*e1rc10*e1rc22-e1rc00*e1rc12*e1rc21+e1rc02*e1rc10*e1rc21+e1rc01*e1rc12*e1rc20-e1rc02*e1rc11*e1rc20;
    AMatrix(9, 7) = +e1rc00*e1rc11*e2rc22-e1rc01*e1rc10*e2rc22-e1rc00*e1rc12*e2rc21+e1rc02*e1rc10*e2rc21+e1rc01*e1rc12*e2rc20-e1rc02*e1rc11*e2rc20-e1rc00*e1rc21*e2rc12+e1rc01*e1rc20*e2rc12+e1rc00*e1rc22*e2rc11-e1rc02*e1rc20*e2rc11-e1rc01*e1rc22*e2rc10+e1rc02*e1rc21*e2rc10+e1rc10*e1rc21*e2rc02-e1rc11*e1rc20*e2rc02-e1rc10*e1rc22*e2rc01+e1rc12*e1rc20*e2rc01+e1rc11*e1rc22*e2rc00-e1rc12*e1rc21*e2rc00;
    AMatrix(9, 8) = +e1rc00*e2rc11*e2rc22-e1rc01*e2rc10*e2rc22-e1rc10*e2rc01*e2rc22+e1rc11*e2rc00*e2rc22-e1rc00*e2rc12*e2rc21+e1rc02*e2rc10*e2rc21+e1rc10*e2rc02*e2rc21-e1rc12*e2rc00*e2rc21+e1rc01*e2rc12*e2rc20-e1rc02*e2rc11*e2rc20-e1rc11*e2rc02*e2rc20+e1rc12*e2rc01*e2rc20+e1rc20*e2rc01*e2rc12-e1rc21*e2rc00*e2rc12-e1rc20*e2rc02*e2rc11+e1rc22*e2rc00*e2rc11+e1rc21*e2rc02*e2rc10-e1rc22*e2rc01*e2rc10;
    AMatrix(9, 9) = e2rc00*e2rc11*e2rc22-e2rc01*e2rc10*e2rc22-e2rc00*e2rc12*e2rc21+e2rc02*e2rc10*e2rc21+e2rc01*e2rc12*e2rc20-e2rc02*e2rc11*e2rc20;
    AMatrix(9, 10) = +e0rc00*e0rc11*e3rc22-e0rc01*e0rc10*e3rc22-e0rc00*e0rc12*e3rc21+e0rc02*e0rc10*e3rc21+e0rc01*e0rc12*e3rc20-e0rc02*e0rc11*e3rc20-e0rc00*e0rc21*e3rc12+e0rc01*e0rc20*e3rc12+e0rc00*e0rc22*e3rc11-e0rc02*e0rc20*e3rc11-e0rc01*e0rc22*e3rc10+e0rc02*e0rc21*e3rc10+e0rc10*e0rc21*e3rc02-e0rc11*e0rc20*e3rc02-e0rc10*e0rc22*e3rc01+e0rc12*e0rc20*e3rc01+e0rc11*e0rc22*e3rc00-e0rc12*e0rc21*e3rc00;
    AMatrix(9, 11) = +e0rc00*e1rc11*e3rc22-e0rc01*e1rc10*e3rc22-e0rc10*e1rc01*e3rc22+e0rc11*e1rc00*e3rc22-e0rc00*e1rc12*e3rc21+e0rc02*e1rc10*e3rc21+e0rc10*e1rc02*e3rc21-e0rc12*e1rc00*e3rc21+e0rc01*e1rc12*e3rc20-e0rc02*e1rc11*e3rc20-e0rc11*e1rc02*e3rc20+e0rc12*e1rc01*e3rc20-e0rc00*e1rc21*e3rc12+e0rc01*e1rc20*e3rc12+e0rc20*e1rc01*e3rc12-e0rc21*e1rc00*e3rc12+e0rc00*e1rc22*e3rc11-e0rc02*e1rc20*e3rc11-e0rc20*e1rc02*e3rc11+e0rc22*e1rc00*e3rc11-e0rc01*e1rc22*e3rc10+e0rc02*e1rc21*e3rc10+e0rc21*e1rc02*e3rc10-e0rc22*e1rc01*e3rc10+e0rc10*e1rc21*e3rc02-e0rc11*e1rc20*e3rc02-e0rc20*e1rc11*e3rc02+e0rc21*e1rc10*e3rc02-e0rc10*e1rc22*e3rc01+e0rc12*e1rc20*e3rc01+e0rc20*e1rc12*e3rc01-e0rc22*e1rc10*e3rc01+e0rc11*e1rc22*e3rc00-e0rc12*e1rc21*e3rc00-e0rc21*e1rc12*e3rc00+e0rc22*e1rc11*e3rc00;
    AMatrix(9, 12) = +e0rc00*e2rc11*e3rc22-e0rc01*e2rc10*e3rc22-e0rc10*e2rc01*e3rc22+e0rc11*e2rc00*e3rc22-e0rc00*e2rc12*e3rc21+e0rc02*e2rc10*e3rc21+e0rc10*e2rc02*e3rc21-e0rc12*e2rc00*e3rc21+e0rc01*e2rc12*e3rc20-e0rc02*e2rc11*e3rc20-e0rc11*e2rc02*e3rc20+e0rc12*e2rc01*e3rc20-e0rc00*e2rc21*e3rc12+e0rc01*e2rc20*e3rc12+e0rc20*e2rc01*e3rc12-e0rc21*e2rc00*e3rc12+e0rc00*e2rc22*e3rc11-e0rc02*e2rc20*e3rc11-e0rc20*e2rc02*e3rc11+e0rc22*e2rc00*e3rc11-e0rc01*e2rc22*e3rc10+e0rc02*e2rc21*e3rc10+e0rc21*e2rc02*e3rc10-e0rc22*e2rc01*e3rc10+e0rc10*e2rc21*e3rc02-e0rc11*e2rc20*e3rc02-e0rc20*e2rc11*e3rc02+e0rc21*e2rc10*e3rc02-e0rc10*e2rc22*e3rc01+e0rc12*e2rc20*e3rc01+e0rc20*e2rc12*e3rc01-e0rc22*e2rc10*e3rc01+e0rc11*e2rc22*e3rc00-e0rc12*e2rc21*e3rc00-e0rc21*e2rc12*e3rc00+e0rc22*e2rc11*e3rc00;
    AMatrix(9, 13) = +e1rc00*e1rc11*e3rc22-e1rc01*e1rc10*e3rc22-e1rc00*e1rc12*e3rc21+e1rc02*e1rc10*e3rc21+e1rc01*e1rc12*e3rc20-e1rc02*e1rc11*e3rc20-e1rc00*e1rc21*e3rc12+e1rc01*e1rc20*e3rc12+e1rc00*e1rc22*e3rc11-e1rc02*e1rc20*e3rc11-e1rc01*e1rc22*e3rc10+e1rc02*e1rc21*e3rc10+e1rc10*e1rc21*e3rc02-e1rc11*e1rc20*e3rc02-e1rc10*e1rc22*e3rc01+e1rc12*e1rc20*e3rc01+e1rc11*e1rc22*e3rc00-e1rc12*e1rc21*e3rc00;
    AMatrix(9, 14) = +e1rc00*e2rc11*e3rc22-e1rc01*e2rc10*e3rc22-e1rc10*e2rc01*e3rc22+e1rc11*e2rc00*e3rc22-e1rc00*e2rc12*e3rc21+e1rc02*e2rc10*e3rc21+e1rc10*e2rc02*e3rc21-e1rc12*e2rc00*e3rc21+e1rc01*e2rc12*e3rc20-e1rc02*e2rc11*e3rc20-e1rc11*e2rc02*e3rc20+e1rc12*e2rc01*e3rc20-e1rc00*e2rc21*e3rc12+e1rc01*e2rc20*e3rc12+e1rc20*e2rc01*e3rc12-e1rc21*e2rc00*e3rc12+e1rc00*e2rc22*e3rc11-e1rc02*e2rc20*e3rc11-e1rc20*e2rc02*e3rc11+e1rc22*e2rc00*e3rc11-e1rc01*e2rc22*e3rc10+e1rc02*e2rc21*e3rc10+e1rc21*e2rc02*e3rc10-e1rc22*e2rc01*e3rc10+e1rc10*e2rc21*e3rc02-e1rc11*e2rc20*e3rc02-e1rc20*e2rc11*e3rc02+e1rc21*e2rc10*e3rc02-e1rc10*e2rc22*e3rc01+e1rc12*e2rc20*e3rc01+e1rc20*e2rc12*e3rc01-e1rc22*e2rc10*e3rc01+e1rc11*e2rc22*e3rc00-e1rc12*e2rc21*e3rc00-e1rc21*e2rc12*e3rc00+e1rc22*e2rc11*e3rc00;
    AMatrix(9, 15) = +e2rc00*e2rc11*e3rc22-e2rc01*e2rc10*e3rc22-e2rc00*e2rc12*e3rc21+e2rc02*e2rc10*e3rc21+e2rc01*e2rc12*e3rc20-e2rc02*e2rc11*e3rc20-e2rc00*e2rc21*e3rc12+e2rc01*e2rc20*e3rc12+e2rc00*e2rc22*e3rc11-e2rc02*e2rc20*e3rc11-e2rc01*e2rc22*e3rc10+e2rc02*e2rc21*e3rc10+e2rc10*e2rc21*e3rc02-e2rc11*e2rc20*e3rc02-e2rc10*e2rc22*e3rc01+e2rc12*e2rc20*e3rc01+e2rc11*e2rc22*e3rc00-e2rc12*e2rc21*e3rc00;
    AMatrix(9, 16) = +e0rc00*e3rc11*e3rc22-e0rc01*e3rc10*e3rc22-e0rc10*e3rc01*e3rc22+e0rc11*e3rc00*e3rc22-e0rc00*e3rc12*e3rc21+e0rc02*e3rc10*e3rc21+e0rc10*e3rc02*e3rc21-e0rc12*e3rc00*e3rc21+e0rc01*e3rc12*e3rc20-e0rc02*e3rc11*e3rc20-e0rc11*e3rc02*e3rc20+e0rc12*e3rc01*e3rc20+e0rc20*e3rc01*e3rc12-e0rc21*e3rc00*e3rc12-e0rc20*e3rc02*e3rc11+e0rc22*e3rc00*e3rc11+e0rc21*e3rc02*e3rc10-e0rc22*e3rc01*e3rc10;
    AMatrix(9, 17) = +e1rc00*e3rc11*e3rc22-e1rc01*e3rc10*e3rc22-e1rc10*e3rc01*e3rc22+e1rc11*e3rc00*e3rc22-e1rc00*e3rc12*e3rc21+e1rc02*e3rc10*e3rc21+e1rc10*e3rc02*e3rc21-e1rc12*e3rc00*e3rc21+e1rc01*e3rc12*e3rc20-e1rc02*e3rc11*e3rc20-e1rc11*e3rc02*e3rc20+e1rc12*e3rc01*e3rc20+e1rc20*e3rc01*e3rc12-e1rc21*e3rc00*e3rc12-e1rc20*e3rc02*e3rc11+e1rc22*e3rc00*e3rc11+e1rc21*e3rc02*e3rc10-e1rc22*e3rc01*e3rc10;
    AMatrix(9, 18) = +e2rc00*e3rc11*e3rc22-e2rc01*e3rc10*e3rc22-e2rc10*e3rc01*e3rc22+e2rc11*e3rc00*e3rc22-e2rc00*e3rc12*e3rc21+e2rc02*e3rc10*e3rc21+e2rc10*e3rc02*e3rc21-e2rc12*e3rc00*e3rc21+e2rc01*e3rc12*e3rc20-e2rc02*e3rc11*e3rc20-e2rc11*e3rc02*e3rc20+e2rc12*e3rc01*e3rc20+e2rc20*e3rc01*e3rc12-e2rc21*e3rc00*e3rc12-e2rc20*e3rc02*e3rc11+e2rc22*e3rc00*e3rc11+e2rc21*e3rc02*e3rc10-e2rc22*e3rc01*e3rc10;
    AMatrix(9, 19) = +e3rc00*e3rc11*e3rc22-e3rc01*e3rc10*e3rc22-e3rc00*e3rc12*e3rc21+e3rc02*e3rc10*e3rc21+e3rc01*e3rc12*e3rc20-e3rc02*e3rc11*e3rc20;

      
      // ============= End cut & paste section =============

      // Warning(xxx): The action matrix in Stewenius & Nister's paper
      // doesn't appear to correspond to degree-then-lexicographic
      // order of monomials.  The right solution to this is to
      // generate the correct action matrix in fivePointAlgorithm.h,
      // but instead we temporarily shuffle the columns of our
      // constraint matrix to match the action matrix.
      num::Array2D<double> A2Matrix(AMatrix.rows(), AMatrix.columns());
      num::Array1D<int> shuffle(
        "[0, 1, 3, 6, 2, 4, 7, 5, 8, 9, 10, 11, 13, 12, 14, 15, 16, 17, 18, 19]"
        );
      for(size_t rowIndex = 0; rowIndex < AMatrix.rows(); ++rowIndex) {
        for(size_t columnIndex = 0; columnIndex < AMatrix.columns();
            ++columnIndex) {
          A2Matrix(rowIndex, columnIndex) =
            AMatrix(rowIndex, shuffle[columnIndex]);
        }
      }
      return A2Matrix;
    }

  } // namespace computerVision
    
} // namespace dlr
