/**
***************************************************************************
* @file dlrComputerVision/featureAssociation.h
*
* Header file declaring functions that pair features from two disjoint
* sets based on a user supplied similarity measure.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef DLR_COMPUTERVISION_FEATUREASSOCIATION_H
#define DLR_COMPUTERVISION_FEATUREASSOCIATION_H

#include <dlrNumeric/array2D.h>

namespace dlr {

  namespace computerVision {

    /** 
     * This function template implements the feature association
     * algorithm of Guy Scott and H. Christopher Longuet-Higgins, as
     * described in [1].
     *
     * Template argument Functor must implement a similarity measure
     * for compariing pairs of features.  For example, calling
     *
     *   similarityFunctor(*sequenc0Begin, *sequence1Begin)
     *
     * Should return a double precision similarity measure between 0.0
     * and 1.0.  Scott's suggested similarity measure is:
     *
     *   g = exp(-r_ij^2 / 2*sigma^2),
     *
     * where g is the similarity result, r_ij is the distance between
     * point i in the first feature set and point j in the second
     * feature set, and sigma is a parameter setting how sensitive the
     * algorithm is to increasing distance between matched features
     * (large alpha gives low sensitivity, small alpha gives high
     * sensitivity).
     *
     * [1] G. L. Scott and H. C. Longuet Higgins, "An Algorithm for
     * Associating the Features of Two Images," Proceedings of
     * Biological Sciences, Vol. 244, No. 1309, pp. 21-26, April,
     * 1991.
     *
     * 
     * @param sequence0Begin This argument is an STL style iterator
     * pointing to the first element of the first feature sequence.
     * 
     * @param sequence0End This argument is an STL style iterator
     * pointing one-past-the-last element of the first feature
     * sequence.
     * 
     * @param sequence1Begin This argument is an STL style iterator
     * pointing to the first element of the second feature sequence.
     * 
     * @param sequence1End This argument is an STL style iterator
     * pointing one-past-the-last element of the second feature
     * sequence.
     * 
     * @param similarityFunctor This argument specifies a functor that
     * takes two features and computes their similarity.
     * 
     * @return The return value is a vector of pairs of indices in
     * which each pair of indices indicates a pair of corresponding
     * features.  The first element of the pair is an index into
     * features sequence 0, while the second is an index into feature
     * sequence 1.  The pairs will be arranged so that the sequence 0
     * indices are in ascending order.
     *
     * Note(xxx): Provide a way of returning similarity values.
     */
    template<class Iterator0, class Iterator1, class Functor>
    std::vector< std::pair<size_t, size_t> >
    associateFeaturesScott91(Iterator0 sequence0Begin, Iterator0 sequence0End,
                             Iterator1 sequence1Begin, Iterator1 sequence1End,
                             Functor similarityFunctor);
    
  } // namespace computerVision
    
} // namespace dlr


/* ============ Definitions of inline & template functions ============ */


#include <cmath>
#include <dlrLinearAlgebra/linearAlgebra.h>
#include <dlrNumeric/maxRecorder.h>
#include <dlrNumeric/utilities.h>


namespace dlr {

  namespace computerVision {


    // This function template implements the feature association
    // algorithm of Guy Scott and H. Christopher Longuet-Higgins.
    template<class Iterator0, class Iterator1, class Functor>
    std::vector< std::pair<size_t, size_t> >
    associateFeaturesScott91(Iterator0 sequence0Begin, Iterator0 sequence0End,
                             Iterator1 sequence1Begin, Iterator1 sequence1End,
                             Functor similarityFunctor)
    {
      // Count the number of features in each sequence.  Even if this
      // operation is O(n), it will be dominated by the more expensive
      // steps below.
      size_t sequence0Length = sequence0End - sequence0Begin;
      size_t sequence1Length = sequence1End - sequence1Begin;

      // Compute the similarity matrix.
      dlr::numeric::Array2D<double> GMatrix(sequence0Length, sequence1Length);
      Iterator0 begin0 = sequence0Begin;
      for(size_t rr = 0; rr < sequence0Length; ++rr) {
        Iterator1 begin1 = sequence1Begin;
        for(size_t cc = 0; cc < sequence1Length; ++cc) {
          GMatrix(rr, cc) = similarityFunctor(*begin0, *begin1);
          ++begin1;
        }
        ++begin0;
      }

      // Compute a new similarity matrix, comprising only rotations
      // and reflections, that is "similar" to G (see the paper for
      // more details).
      dlr::numeric::Array2D<double> uMatrix;
      dlr::numeric::Array1D<double> sigmaArray;
      dlr::numeric::Array2D<double> vTransposeMatrix;
      dlr::linearAlgebra::singularValueDecomposition(
        GMatrix, uMatrix, sigmaArray, vTransposeMatrix);
      dlr::numeric::Array2D<double> PMatrix =
        dlr::numeric::matrixMultiply(uMatrix, vTransposeMatrix);

      // Find the max elements for each row and column of P.
      std::vector< dlr::numeric::MaxRecorder<double, size_t> > rowMaxes(
        sequence0Length);
      std::vector< dlr::numeric::MaxRecorder<double, size_t> > columnMaxes(
        sequence1Length);
      for(size_t rr = 0; rr < sequence0Length; ++rr) {
        for(size_t cc = 0; cc < sequence1Length; ++cc) {
          double similarity = PMatrix(rr, cc);
          rowMaxes[rr].test(similarity, cc);
          columnMaxes[cc].test(similarity, rr);
        }
      }

      // Valid correspondences are those for which the similarity is
      // max of both row and column.
      std::vector< std::pair<size_t, size_t> > result;
      for(size_t rr = 0; rr < sequence0Length; ++rr) {
        size_t bestColumn = rowMaxes[rr].getPayload();
        if(columnMaxes[bestColumn].getPayload() == rr) {
          result.push_back(std::make_pair(rr, bestColumn));
        }
      }

      return result;
    }

  } // namespace computerVision
    
} // namespace dlr

#endif /* #ifndef DLR_COMPUTERVISION_FEATUREASSOCIATION_H */
