/**
***************************************************************************
* @file stereoRectifyTest.cpp
*
* Source file defining tests for stereoRectify().
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrComputerVision/stereoRectify.h>
#include <dlrNumeric/rotations.h>
#include <dlrNumeric/utilities.h>
#include <dlrTest/testFixture.h>

namespace num = dlr::numeric;

namespace dlr {

  namespace computerVision {
    
    class StereoRectifyTest
      : public TestFixture<StereoRectifyTest> {

    public:

      StereoRectifyTest();
      ~StereoRectifyTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      // Tests.
      void testStereoRectify();

    private:

      const double m_defaultTolerance;
      
    }; // class StereoRectifyTest


    /* ============== Member Function Definititions ============== */

    StereoRectifyTest::
    StereoRectifyTest()
      : TestFixture<StereoRectifyTest>("StereoRectifyTest"),
        m_defaultTolerance(1.0E-10)
    {
      DLR_TEST_REGISTER_MEMBER(testStereoRectify);
    }


    void
    StereoRectifyTest::
    testStereoRectify()
    {
      // Set up a pair of (arbitrary) cameras.
      CameraIntrinsicsPinhole intrinsics0(640, 480, 500.0, 1.0, 1.2, 300, 220);
      CameraIntrinsicsPinhole intrinsics1(640, 480, 500.0, 1.1, 0.9, 350, 260);

      // Pick arbitrary, and slightly different, orientations for the
      // two cameras.
      num::Transform3D camera0Tworld =
        num::rollPitchYawToTransform3D(num::Vector3D(0.1, 0.7, -0.3));
      num::Transform3D camera1Tworld =
        num::rollPitchYawToTransform3D(num::Vector3D(-0.1, 0.5, -0.1));

      // Make cameras be looking coords origin, from a distance of,
      // say 5m.  Camera0 on the left.
      camera0Tworld.setValue<0, 3>(0.01);
      camera0Tworld.setValue<1, 3>(0.04);
      camera0Tworld.setValue<2, 3>(5.2);
      camera1Tworld.setValue<0, 3>(-0.01);
      camera1Tworld.setValue<1, 3>(-0.06);
      camera1Tworld.setValue<2, 3>(4.9);

      // Invert, since we'll need the inverses later.
      num::Transform3D worldTcamera0 = camera0Tworld.invert();
      num::Transform3D worldTcamera1 = camera1Tworld.invert();

      // Do the rectification.
      CameraIntrinsicsPinhole rectifiedIntrinsics0;
      CameraIntrinsicsPinhole rectifiedIntrinsics1;
      num::Transform3D rcamera0Tworld;
      num::Transform3D rcamera1Tworld;
      num::Transform2D image0Trimage0;
      num::Transform2D image1Trimage1;
      stereoRectify(intrinsics0, intrinsics1, camera0Tworld, camera1Tworld,
                    rectifiedIntrinsics0, rectifiedIntrinsics1,
                    rcamera0Tworld, rcamera1Tworld,
                    image0Trimage0, image1Trimage1);

      // Check that camera centers are still at the same place.
      num::Transform3D worldTrcamera0 = rcamera0Tworld.invert();
      num::Transform3D worldTrcamera1 = rcamera1Tworld.invert();
      num::Vector3D center0(
        worldTcamera0(0, 3), worldTcamera0(1, 3), worldTcamera0(2, 3));
      num::Vector3D center1(
        worldTcamera1(0, 3), worldTcamera1(1, 3), worldTcamera1(2, 3));
      num::Vector3D rcenter0(
        worldTrcamera0(0, 3), worldTrcamera0(1, 3), worldTrcamera0(2, 3));
      num::Vector3D rcenter1(
        worldTrcamera1(0, 3), worldTrcamera1(1, 3), worldTrcamera1(2, 3));

      DLR_TEST_ASSERT(
        approximatelyEqual(center0.x(), rcenter0.x(), m_defaultTolerance));
      DLR_TEST_ASSERT(
        approximatelyEqual(center0.y(), rcenter0.y(), m_defaultTolerance));
      DLR_TEST_ASSERT(
        approximatelyEqual(center0.z(), rcenter0.z(), m_defaultTolerance));
      DLR_TEST_ASSERT(
        approximatelyEqual(center1.x(), rcenter1.x(), m_defaultTolerance));
      DLR_TEST_ASSERT(
        approximatelyEqual(center1.y(), rcenter1.y(), m_defaultTolerance));
      DLR_TEST_ASSERT(
        approximatelyEqual(center1.z(), rcenter1.z(), m_defaultTolerance));


      // Check that the rectified rotation matrices are the same.
      for(size_t row = 0; row < 3; ++row) {
        for(size_t column = 0; column < 3; ++column) {
          DLR_TEST_ASSERT(
            approximatelyEqual(worldTrcamera0(row, column),
                               worldTrcamera1(row, column),
                               m_defaultTolerance));
        }
      }

      // Check that the rectified rotation matrices are orthonormal.
      num::Array2D<double> rArray(3, 3);
      for(size_t row = 0; row < 3; ++row) {
        for(size_t column = 0; column < 3; ++column) {
          rArray(row, column) = worldTrcamera0(row, column);
        }
      }
      num::Array2D<double> rrt = matrixMultiply(rArray, rArray.transpose());
      for(size_t row = 0; row < 3; ++row) {
        for(size_t column = 0; column < 3; ++column) {
          if(row == column) {
            DLR_TEST_ASSERT(
              approximatelyEqual(rrt(row, column), 1.0, m_defaultTolerance));
          } else {
            DLR_TEST_ASSERT(
              approximatelyEqual(rrt(row, column), 0.0, m_defaultTolerance));
          }
        }
      }

      // Check that camera X axes are parallel to the vector
      // connecting their rotation centers.
      num::Vector3D baseline_world = rcenter1 - rcenter0;
      baseline_world /= num::magnitude(baseline_world);
      num::Vector3D xAxis(
        rcamera1Tworld(0, 0), rcamera1Tworld(0, 1), rcamera1Tworld(0, 2));
      num::Vector3D crossProduct = num::cross(baseline_world, xAxis);
      double crossMag = num::magnitude(crossProduct);
      double dotProduct = num::dot(baseline_world, xAxis);
      DLR_TEST_ASSERT(approximatelyEqual(crossMag, 0.0, m_defaultTolerance));
      DLR_TEST_ASSERT(approximatelyEqual(dotProduct, 1.0, m_defaultTolerance));
      
      // Pick some points against which to test.
      std::vector<num::Vector3D> testPoints;
      testPoints.push_back(num::Vector3D(0.0, 0.0, 0.0));
      testPoints.push_back(num::Vector3D(1.0, 0.0, 0.0));
      testPoints.push_back(num::Vector3D(-1.0, 0.0, 0.0));
      testPoints.push_back(num::Vector3D(0.0, 1.0, 0.0));
      testPoints.push_back(num::Vector3D(0.0, -1.0, 0.0));
      testPoints.push_back(num::Vector3D(0.0, 0.0, 1.0));
      testPoints.push_back(num::Vector3D(0.0, 0.0, -1.0));
      testPoints.push_back(num::Vector3D(-10.0, 5.0, 2.0));
      testPoints.push_back(num::Vector3D(5.0, -7.0, -2.0));

      // Check against each point in turn.
      for(size_t ii = 0; ii < testPoints.size(); ++ii) {
        num::Vector3D testPoint = testPoints[ii];

        // Project into all cameras.
        num::Vector3D p_camera0 = camera0Tworld * testPoint;
        num::Vector3D p_camera1 = camera1Tworld * testPoint;
        num::Vector3D p_rcamera0 = rcamera0Tworld * testPoint;
        num::Vector3D p_rcamera1 = rcamera1Tworld * testPoint;
        num::Vector2D p_image0 = intrinsics0.project(p_camera0);
        num::Vector2D p_image1 = intrinsics1.project(p_camera1);
        num::Vector2D p_rimage0 = rectifiedIntrinsics0.project(p_rcamera0);
        num::Vector2D p_rimage1 = rectifiedIntrinsics1.project(p_rcamera1);

        // Check that rectified projections are on the same scan line.
        DLR_TEST_ASSERT(
          approximatelyEqual(p_rimage0.y(), p_rimage1.y(), m_defaultTolerance));

        // Check that mappings from rectified to unrectified images
        // are correct.
        num::Vector2D pHat_image0 = image0Trimage0 * p_rimage0;
        num::Vector2D pHat_image1 = image1Trimage1 * p_rimage1;
        DLR_TEST_ASSERT(approximatelyEqual(pHat_image0.x(), p_image0.x(),
                                           m_defaultTolerance));
        DLR_TEST_ASSERT(approximatelyEqual(pHat_image0.y(), p_image0.y(),
                                           m_defaultTolerance));
        DLR_TEST_ASSERT(approximatelyEqual(pHat_image1.x(), p_image1.x(),
                                           m_defaultTolerance));
        DLR_TEST_ASSERT(approximatelyEqual(pHat_image1.y(), p_image1.y(),
                                           m_defaultTolerance));
      }
    }

  } // namespace computerVision

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::StereoRectifyTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::StereoRectifyTest currentTest;

}

#endif

