/**
***************************************************************************
* @file featureAssociationTest.cpp
*
* Source file defining tests for featureAssociation functions.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <algorithm>
#include <dlrComputerVision/featureAssociation.h>
#include <dlrTest/testFixture.h>

namespace num = dlr::numeric;

namespace {

  const double s_sigma = 4.0;

  class SimilarityFunctor {
  public:
    
    SimilarityFunctor(double sigma) : m_sigma(sigma) {}
    ~SimilarityFunctor() {}

    double operator()(num::Vector2D const& feature0,
                      num::Vector2D const& feature1) {
      double distance = num::magnitude(feature1 - feature0);
      return std::exp(-(distance * distance) / (2.0 * m_sigma * m_sigma));
    }
    
  private:
    double m_sigma;
  };

}


namespace dlr {

  namespace computerVision {
    
    class FeatureAssociationTest
      : public TestFixture<FeatureAssociationTest> {

    public:

      FeatureAssociationTest();
      ~FeatureAssociationTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      // Tests.
      void testFeatureAssociation();
      
    private:

    }; // class FeatureAssociationTest


    /* ============== Member Function Definititions ============== */

    FeatureAssociationTest::
    FeatureAssociationTest()
      : TestFixture<FeatureAssociationTest>("FeatureAssociationTest")
    {
      DLR_TEST_REGISTER_MEMBER(testFeatureAssociation);
    }


    void
    FeatureAssociationTest::
    testFeatureAssociation()
    {
      // Make two sequences of features.
      std::vector<num::Vector2D> features0;
      for(size_t ii = 0; ii < 4; ++ii) {
        features0.push_back(
          num::Vector2D(static_cast<double>(ii), static_cast<double>(ii)));
      }
      std::vector<num::Vector2D> features1;
      for(size_t ii = 0; ii < 4; ++ii) {
        features1.push_back(features0[ii] + num::Vector2D(2.75, 2.0));        
      }

      // Do the association, but put strong emphasis on proximity..
      std::vector< std::pair<size_t, size_t> > correspondences =
        associateFeaturesScott91(features0.begin(), features0.end(),
                                 features1.begin(), features1.end(),
                                 SimilarityFunctor(1.0));

      // Correspondences should form based on shortest distance.
      bool twoGoesToZero = false;
      bool threeGoesToOne = false;
      for(size_t ii = 0; ii < correspondences.size(); ++ii) {
        if((correspondences[ii].first == 2)
           && (correspondences[ii].second == 0)) {
          twoGoesToZero = true;
        }
        if((correspondences[ii].first == 3)
           && (correspondences[ii].second == 1)) {
          threeGoesToOne = true;
        }
      }
      DLR_TEST_ASSERT(twoGoesToZero);
      DLR_TEST_ASSERT(threeGoesToOne);

      // Do the association, but put allow long-distance relationships.
      correspondences =
        associateFeaturesScott91(features0.begin(), features0.end(),
                                 features1.begin(), features1.end(),
                                 SimilarityFunctor(4.0));

      // Now correspondence should form between the first, second,
      // third pairs in the two sequences.
      DLR_TEST_ASSERT(correspondences.size() == features0.size());
      for(size_t ii = 0; ii < 4; ++ii) {
        DLR_TEST_ASSERT(correspondences[ii].first == ii);
        DLR_TEST_ASSERT(correspondences[ii].second == ii);
      }

      // Repeat the above, but reverse the order of features1.
      std::reverse(features1.begin(), features1.end());
      correspondences =
        associateFeaturesScott91(features0.begin(), features0.end(),
                                 features1.begin(), features1.end(),
                                 SimilarityFunctor(4.0));

      // Now correspondence should form between the first, second,
      // third pairs in the two sequences.
      DLR_TEST_ASSERT(correspondences.size() == features0.size());
      for(size_t ii = 0; ii < 4; ++ii) {
        DLR_TEST_ASSERT(correspondences[ii].first == ii);
        DLR_TEST_ASSERT(correspondences[ii].second
                        == features0.size() - ii - 1);
      }
      
    }
  
  } // namespace computerVision

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::FeatureAssociationTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::FeatureAssociationTest currentTest;

}

#endif

