/**
***************************************************************************
* @file eightPointAlgorithmTest.cpp
*
* Source file defining tests for eightPointAlgorithm().
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrComputerVision/test/testImages.h>
#include <dlrComputerVision/eightPointAlgorithm.h>
#include <dlrComputerVision/imageIO.h>
#include <dlrTest/testFixture.h>

namespace num = dlr::numeric;

namespace dlr {

  namespace computerVision {
    
    class EightPointAlgorithmTest
      : public TestFixture<EightPointAlgorithmTest> {

    public:

      EightPointAlgorithmTest();
      ~EightPointAlgorithmTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      // Tests.
      void testEightPointAlgorithm();
      void testNormalizePointSequence();
      
    private:

      void
      getTestPoints(std::vector<num::Vector2D>& uVector,
                    std::vector<num::Vector2D>& uPrimeVector,
                    num::Array2D<double>& FOrig);
      
      bool
      isApproximatelyEqual(const Array1D<double>& array0,
                           const Array1D<double>& array1);

      bool
      isApproximatelyEqual(const Array2D<double>& array0,
                           const Array2D<double>& array1);

      
      double m_defaultTolerance;
      
    }; // class EightPointAlgorithmTest


    /* ============== Member Function Definititions ============== */

    EightPointAlgorithmTest::
    EightPointAlgorithmTest()
      : TestFixture<EightPointAlgorithmTest>("EightPointAlgorithmTest"),
        m_defaultTolerance(1.0E-10)
    {
      DLR_TEST_REGISTER_MEMBER(testEightPointAlgorithm);
      DLR_TEST_REGISTER_MEMBER(testNormalizePointSequence);
    }


    void
    EightPointAlgorithmTest::
    testEightPointAlgorithm()
    {
      num::Array2D<double> FOrig;
      std::vector<num::Vector2D> uVector;
      std::vector<num::Vector2D> uPrimeVector;
      this->getTestPoints(uVector, uPrimeVector, FOrig);
      
      // Try to recover FOrig.
      num::Array2D<double> FEstimate = eightPointAlgorithm(
        uVector.begin(), uVector.end(), uPrimeVector.begin());

      // Make sure FEstimate has the appropriate qualities:
      for(size_t ii = 0; ii < uVector.size(); ++ii) {
        num::Array1D<double> uu(3);
        uu[0] = uVector[ii].x();
        uu[1] = uVector[ii].y();
        uu[2] = 1.0;
        num::Array1D<double> uPrime(3);
        uPrime[0] = uPrimeVector[ii].x();
        uPrime[1] = uPrimeVector[ii].y();
        uPrime[2] = 1.0;
        
        num::Array1D<double> lineCoeffs = matrixMultiply(FEstimate, uu);
        double residual = num::dot(uPrime, lineCoeffs);
        DLR_TEST_ASSERT(std::fabs(residual) < m_defaultTolerance);
      }
    }


    void
    EightPointAlgorithmTest::
    testNormalizePointSequence()
    {
      // Create a series of input points, ostensibly coming from one image.
      num::Array2D<double> inputPoints("[[ 12.0,  22.0, 1.0],"
                                       " [ 11.0,  22.0, 1.0],"
                                       " [ 10.0,  22.0, 1.0],"
                                       " [ -1.0,  22.0, 1.0],"
                                       " [ -2.0,  22.0, 1.0],"
                                       " [ 12.0,  21.0, 1.0],"
                                       " [ 11.0,  21.0, 1.0],"
                                       " [ 10.0,  21.0, 1.0],"
                                       " [-11.0,  21.0, 1.0],"
                                       " [-12.0,  21.0, 1.0],"
                                       " [ 12.0,  20.0, 1.0],"
                                       " [ 11.0,  20.0, 1.0],"
                                       " [ 10.0,  20.0, 1.0],"
                                       " [ -1.0,  20.0, 1.0],"
                                       " [ -2.0,  20.0, 1.0],"
                                       " [ 12.0, -1.0, 1.0],"
                                       " [ 11.0, -1.0, 1.0],"
                                       " [ 10.0, -21.0, 1.0],"
                                       " [-11.0, -21.0, 1.0],"
                                       " [-12.0, -1.0, 1.0],"
                                       " [ 12.0, -2.0, 1.0],"
                                       " [ 11.0, -22.0, 1.0],"
                                       " [ 10.0, -2.0, 1.0],"
                                       " [ -1.0, -2.0, 1.0],"
                                       " [ -2.0, -22.0, 1.0]]");

      
      // Normalize these points.
      num::Array2D<double> outputPoints;
      num::Array2D<double> transform;
      normalizePointSequence(inputPoints, outputPoints, transform);

      // Warning(xxx): mean and covariance are not normalized. Check this
      // out later.
      //
      // // Check that the normalization worked.
      // num::Array1D<double> meanArray;
      // num::Array2D<double> covarianceArray;
      // num::getMeanAndCovariance(outputPoints, meanArray, covarianceArray);
      // DLR_TEST_ASSERT(this->isApproximatelyEqual(
      //                   meanArray, num::zeros(3, type_tag<double>())));
      // DLR_TEST_ASSERT(this->isApproximatelyEqual(
      //                   covarianceArray,
      //                   num::identity(3, 3, type_tag<double>())));

      // Check that the transform and output points match.
      DLR_TEST_ASSERT(outputPoints.rows() == inputPoints.rows());
      DLR_TEST_ASSERT(outputPoints.columns() == inputPoints.columns());
      DLR_TEST_ASSERT(transform.rows() == 3);
      DLR_TEST_ASSERT(transform.columns() == 3);
      for(size_t ii = 0; ii < inputPoints.rows(); ++ii) {
        DLR_TEST_ASSERT(
          this->isApproximatelyEqual(
            num::matrixMultiply(transform, inputPoints.getRow(ii)),
            outputPoints.getRow(ii)));
      }

      // Check that the output points have the desired property.
      num::Array2D<double> uuT =
        num::matrixMultiply(outputPoints.transpose(), outputPoints);
      num::Array2D<double> expectedOuterProductSum =
        (static_cast<double>(inputPoints.rows())
         * num::identity(3, 3, type_tag<double>()));
      DLR_TEST_ASSERT(this->isApproximatelyEqual(uuT, expectedOuterProductSum));
    }


    void
    EightPointAlgorithmTest::
    getTestPoints(std::vector<num::Vector2D>& uVector,
                  std::vector<num::Vector2D>& uPrimeVector,
                  num::Array2D<double>& FOrig)
    {
      // Create a rank-2 homogeneous 2D transform to serve as our
      // fundamental matrix.
      FOrig = num::Array2D<double>("[[1.0, -1.0, 3.0],"
                                   " [1.0, 0.0, -1.0],"
                                   " [1.0, -1.0, 3.0]]");

      // Create a series of "u" points, ostensibly coming from one image.
      uVector.clear();
      uVector.push_back(num::Vector2D(2.0, 2.0));
      uVector.push_back(num::Vector2D(1.0, 2.0));
      uVector.push_back(num::Vector2D(0.0, 2.0));
      uVector.push_back(num::Vector2D(-1.0, 2.0));
      uVector.push_back(num::Vector2D(-2.0, 2.0));
      uVector.push_back(num::Vector2D(2.0, 1.0));
      uVector.push_back(num::Vector2D(1.0, 1.0));
      uVector.push_back(num::Vector2D(0.0, 1.0));
      uVector.push_back(num::Vector2D(-1.0, 1.0));
      uVector.push_back(num::Vector2D(-2.0, 1.0));
      uVector.push_back(num::Vector2D(2.0, 0.0));
      uVector.push_back(num::Vector2D(1.0, 0.0));
      uVector.push_back(num::Vector2D(0.0, 0.0));
      uVector.push_back(num::Vector2D(-1.0, 0.0));
      uVector.push_back(num::Vector2D(-2.0, 0.0));
      uVector.push_back(num::Vector2D(2.0, -1.0));
      uVector.push_back(num::Vector2D(1.0, -1.0));
      uVector.push_back(num::Vector2D(0.0, -1.0));
      uVector.push_back(num::Vector2D(-1.0, -1.0));
      uVector.push_back(num::Vector2D(-2.0, -1.0));
      uVector.push_back(num::Vector2D(2.0, -2.0));
      uVector.push_back(num::Vector2D(1.0, -2.0));
      uVector.push_back(num::Vector2D(0.0, -2.0));
      uVector.push_back(num::Vector2D(-1.0, -2.0));
      uVector.push_back(num::Vector2D(-2.0, -2.0));

      for(size_t ii = 0; ii < uVector.size(); ++ii) {
        uVector[ii] += num::Vector2D(10.0, 20.0);
      }
      
      // Create a series of matching "u'" points, ostensibly coming
      // from the other image.
      uPrimeVector.resize(uVector.size());
      for(size_t ii = 0; ii < uVector.size(); ++ii) {
        num::Array1D<double> inputPoint(3);
        inputPoint[0] = uVector[ii].x();
        inputPoint[1] = uVector[ii].y();
        inputPoint[2] = 1.0;
        num::Array1D<double> lineCoeffs = matrixMultiply(FOrig, inputPoint);

        // By definition of Fundamental Matrix, dot_(uPrime,
        // lineCoeffs) must equal 0, where dot_() is a dot product
        // that includes the third element of the 2D homogeneous
        // coords.
        //
        // if
        //
        //   lineCoords == [c_0, c_1, c_2],
        // 
        // and
        // 
        //   uPrime == [alpha * k_0, (1.0 - alpha) * k_0, 1]
        // 
        // then
        //
        //   c_0 * alpha * k_0 + c_1 * (1.0 - alpha) * k_0 + c_2 = 0
        // 
        //   k_0 * ((c_0 - c_1) * alpha + c_1) + c_2 = 0
        //
        //   k_0 = -c_2 / ((c_0 - c_1) * alpha + c_1)

        // Choose 0 < alpha < 1, with alpha not being a "clean"
        // fraction.  This makes us less likely to stumble onto an
        // unlucky combination of u and alpha that makes the solution
        // for k_0 singular.
        double alpha =
          static_cast<double>((ii % (uVector.size() / 2)) + 1.351)
          / (uVector.size() / 2 + 2);

        // Solve for k_0 and set the value of  uPrime.
        double k_0 = (-1.0 * lineCoeffs[2]
                      / ((lineCoeffs[0] - lineCoeffs[1]) * alpha
                         + lineCoeffs[1]));
        uPrimeVector[ii].setValue(alpha * k_0, (1.0 - alpha) * k_0);

        // Sanity check.
        double dotProduct = (uPrimeVector[ii].x() * lineCoeffs[0]
                             + uPrimeVector[ii].y() * lineCoeffs[1]
                             + lineCoeffs[2]);
        if(dotProduct >= 1.0E-8) {
          // Should never get here.
          DLR_THROW(dlr::LogicException,
                    "EightPointAlgorithmTest::getTestPoints()",
                    "Poorly computed corresponding point!");
        }
      }
    }
    

    bool
    EightPointAlgorithmTest::
    isApproximatelyEqual(const Array1D<double>& array0,
                         const Array1D<double>& array1)
    {
      if(array0.size() != array1.size()) {
        return false;
      }
      return std::equal(array0.begin(), array0.end(), array1.begin(),
                        ApproximatelyEqualFunctor<double>(1.0E-10));
    }

  
    bool
    EightPointAlgorithmTest::
    isApproximatelyEqual(const Array2D<double>& array0,
                         const Array2D<double>& array1)
    {
      if(array0.rows() != array1.rows()) {
        return false;
      }
      if(array0.columns() != array1.columns()) {
        return false;
      }
      return std::equal(array0.begin(), array0.end(), array1.begin(),
                        ApproximatelyEqualFunctor<double>(1.0E-10));
    }
  
  } // namespace computerVision

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::EightPointAlgorithmTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::EightPointAlgorithmTest currentTest;

}

#endif

