/**
***************************************************************************
* @file filterTest.cpp
*
* Source file defining tests for ColorspaceConverter classes.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 1179 $
* $Date: 2009-07-28 00:04:21 -0400 (Tue, 28 Jul 2009) $
***************************************************************************
**/

#include <dlrComputerVision/image.h>
#include <dlrComputerVision/colorspaceConverter.h>
#include <dlrComputerVision/pixelRGB.h>
#include <dlrTest/testFixture.h>

namespace dlr {

  namespace computerVision {

    class ColorspaceConverterTest
      : public TestFixture<ColorspaceConverterTest> {

    public:

      ColorspaceConverterTest();
      ~ColorspaceConverterTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      // Tests.
      void testRGB8ToGRAY8();
      void testRGB8ToBGRA8();
      void testRGB8ToRGBA8();
      void testRGB8ToHSV_FLOAT64();
      void testRGB8ToYIQ_FLOAT64();
      void testRGB_FLOAT64ToYIQ_FLOAT64();
      void testBGRA8ToRGB8();
      void testRGBA8ToRGB8();

    private:

    }; // class ColorspaceConverterTest


    /* ============== Member Function Definititions ============== */

    ColorspaceConverterTest::
    ColorspaceConverterTest()
      : TestFixture<ColorspaceConverterTest>("ColorspaceConverterTest")
    {
      DLR_TEST_REGISTER_MEMBER(testRGB8ToGRAY8);
      DLR_TEST_REGISTER_MEMBER(testRGB8ToBGRA8);
      DLR_TEST_REGISTER_MEMBER(testRGB8ToRGBA8);
      DLR_TEST_REGISTER_MEMBER(testRGB8ToHSV_FLOAT64);
      DLR_TEST_REGISTER_MEMBER(testRGB8ToYIQ_FLOAT64);
      DLR_TEST_REGISTER_MEMBER(testRGB_FLOAT64ToYIQ_FLOAT64);
      DLR_TEST_REGISTER_MEMBER(testRGBA8ToRGB8);
      DLR_TEST_REGISTER_MEMBER(testRGBA8ToRGB8);
    }


    void
    ColorspaceConverterTest::
    testRGB8ToGRAY8()
    {
      ColorspaceConverter<RGB8, GRAY8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            double redDbl = redValue;
            double greenDbl = greenValue;
            double blueDbl = blueValue;
            double grayDbl = 0.3 * redDbl + 0.59 * greenDbl + 0.11 * blueDbl;
            UnsignedInt8 grayValue = static_cast<UnsignedInt8>(grayDbl + 0.5);

            PixelRGB8 inputPixel(static_cast<UnsignedInt8>(redValue),
								 static_cast<UnsignedInt8>(greenValue), 
								 static_cast<UnsignedInt8>(blueValue));
            DLR_TEST_ASSERT(converter(inputPixel) == grayValue);
          }
        }
      }
      // Special case tests follow.
      PixelRGB8 inputPixel(255, 255, 255);
      DLR_TEST_ASSERT(converter(inputPixel) == 255);
    }


    void
    ColorspaceConverterTest::
    testRGB8ToBGRA8()
    {
      ColorspaceConverter<RGB8, BGRA8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            PixelRGB8 inputPixel(
              static_cast<UnsignedInt8>(redValue),
              static_cast<UnsignedInt8>(greenValue),
              static_cast<UnsignedInt8>(blueValue));
            PixelBGRA8 referencePixel(
              static_cast<UnsignedInt8>(blueValue),
              static_cast<UnsignedInt8>(greenValue),
              static_cast<UnsignedInt8>(redValue),
              255);
            DLR_TEST_ASSERT(converter(inputPixel) == referencePixel);
          }
        }
      }
    }

 
    void
    ColorspaceConverterTest::
    testRGB8ToRGBA8()
    {
      ColorspaceConverter<RGB8, RGBA8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            PixelRGB8 inputPixel(
              static_cast<UnsignedInt8>(redValue),
              static_cast<UnsignedInt8>(greenValue),
              static_cast<UnsignedInt8>(blueValue));
            PixelRGBA8 referencePixel(
              static_cast<UnsignedInt8>(redValue),
              static_cast<UnsignedInt8>(greenValue),
              static_cast<UnsignedInt8>(blueValue),
              255);
            DLR_TEST_ASSERT(converter(inputPixel) == referencePixel);
          }
        }
      }
    }

 
    void
    ColorspaceConverterTest::
    testRGB8ToHSV_FLOAT64()
    {
      std::vector<PixelRGB8> inputPixels;
      std::vector< PixelHSV<Float64> > targetPixels;

      inputPixels.push_back(PixelRGB8(192, 128, 108));
      inputPixels.push_back(PixelRGB8(177, 192, 108));
      inputPixels.push_back(PixelRGB8(108, 192, 134));
      inputPixels.push_back(PixelRGB8(108, 167, 192));
      inputPixels.push_back(PixelRGB8(132, 108, 192));
      inputPixels.push_back(PixelRGB8(192, 108, 169));

      // HSV values computed manually using:
      // 
      //   v = max(r,g,b) / 255.0.
      //   s = (0, if v == 0
      //        (max(r, g, b) - min(r, g, b)) / max, otherwise.
      //   h = {0, if max == min
      //        (60 * (g - b) / (max - min) mod 360) / 360, if max == r
      //        (60 * (b - r) / (max - min) + 120) / 360, if max == g
      //        (60 * (r - g) / (max - min) + 240) / 360, if max == b
      targetPixels.push_back(PixelHSV<Float64>(0.0397, 0.4375, 0.7529));
      targetPixels.push_back(PixelHSV<Float64>(0.1964, 0.4375, 0.7529));
      targetPixels.push_back(PixelHSV<Float64>(0.3849, 0.4375, 0.7529));
      targetPixels.push_back(PixelHSV<Float64>(0.5496, 0.4375, 0.7529));
      targetPixels.push_back(PixelHSV<Float64>(0.7143, 0.4375, 0.7529));
      targetPixels.push_back(PixelHSV<Float64>(0.8790, 0.4375, 0.7529));

      ColorspaceConverter<RGB8, HSV_FLOAT64> converter;

      for(size_t ii = 0; ii < inputPixels.size(); ++ii) {
	PixelHSV<Float64> outputPixel = converter(inputPixels[ii]);
	DLR_TEST_ASSERT(approximatelyEqual(
          outputPixel.hue, targetPixels[ii].hue, 1.0E-4));
	DLR_TEST_ASSERT(approximatelyEqual(
	  outputPixel.saturation, targetPixels[ii].saturation, 1.0E-4));
	DLR_TEST_ASSERT(approximatelyEqual(
	  outputPixel.value, targetPixels[ii].value, 1.0E-4));
      }
    }

 
    void
    ColorspaceConverterTest::
    testRGB8ToYIQ_FLOAT64()
    {
      std::vector<PixelRGB8> inputPixels;

      inputPixels.push_back(PixelRGB8(192, 128, 108));
      inputPixels.push_back(PixelRGB8(177, 192, 108));
      inputPixels.push_back(PixelRGB8(108, 192, 134));
      inputPixels.push_back(PixelRGB8(108, 167, 192));
      inputPixels.push_back(PixelRGB8(132, 108, 192));
      inputPixels.push_back(PixelRGB8(192, 108, 169));

      std::vector<PixelYIQFloat64> targetPixels(inputPixels.size());
      for(unsigned int ii = 0; ii < inputPixels.size(); ++ii) {
        double red = inputPixels[ii].red / 255.0;
        double green = inputPixels[ii].green / 255.0;
        double blue = inputPixels[ii].blue / 255.0;
        targetPixels[ii].luma =
          red * 0.299 + green * 0.587 + blue * 0.114;
        targetPixels[ii].inPhase =
          red * 0.595716 + green * -0.274453 + blue * -0.321263;
        targetPixels[ii].quadrature =
          red * 0.211456 + green * -0.522591 + blue * 0.311135;
      }
            
      ColorspaceConverter<RGB8, YIQ_FLOAT64> converter;

      const double tolerance = 1.0E-9;
      for(size_t ii = 0; ii < inputPixels.size(); ++ii) {
	PixelYIQ<Float64> outputPixel = converter(inputPixels[ii]);
	DLR_TEST_ASSERT(approximatelyEqual(
          outputPixel.luma, targetPixels[ii].luma, tolerance));
	DLR_TEST_ASSERT(approximatelyEqual(
	  outputPixel.inPhase, targetPixels[ii].inPhase, tolerance));
	DLR_TEST_ASSERT(approximatelyEqual(
	  outputPixel.quadrature, targetPixels[ii].quadrature, tolerance));
      }
    }

 
    void
    ColorspaceConverterTest::
    testRGB_FLOAT64ToYIQ_FLOAT64()
    {
      std::vector<PixelRGBFloat64> inputPixels;

      inputPixels.push_back(
        PixelRGBFloat64(192 / 255.0, 128 / 255.0, 108 / 255.0));
      inputPixels.push_back(
        PixelRGBFloat64(177 / 255.0, 192 / 255.0, 108 / 255.0));
      inputPixels.push_back(
        PixelRGBFloat64(108 / 255.0, 192 / 255.0, 134 / 255.0));
      inputPixels.push_back(
        PixelRGBFloat64(108 / 255.0, 167 / 255.0, 192 / 255.0));
      inputPixels.push_back(
        PixelRGBFloat64(132 / 255.0, 108 / 255.0, 192 / 255.0));
      inputPixels.push_back(
        PixelRGBFloat64(192 / 255.0, 108 / 255.0, 169 / 255.0));

      std::vector<PixelYIQFloat64> targetPixels(inputPixels.size());
      for(unsigned int ii = 0; ii < inputPixels.size(); ++ii) {
        double red = inputPixels[ii].red;
        double green = inputPixels[ii].green;
        double blue = inputPixels[ii].blue;
        targetPixels[ii].luma =
          red * 0.299 + green * 0.587 + blue * 0.114;
        targetPixels[ii].inPhase =
          red * 0.595716 + green * -0.274453 + blue * -0.321263;
        targetPixels[ii].quadrature =
          red * 0.211456 + green * -0.522591 + blue * 0.311135;
      }
            
      ColorspaceConverter<RGB_FLOAT64, YIQ_FLOAT64> converter;

      const double tolerance = 1.0E-9;
      for(size_t ii = 0; ii < inputPixels.size(); ++ii) {
	PixelYIQ<Float64> outputPixel = converter(inputPixels[ii]);
	DLR_TEST_ASSERT(approximatelyEqual(
          outputPixel.luma, targetPixels[ii].luma, tolerance));
	DLR_TEST_ASSERT(approximatelyEqual(
	  outputPixel.inPhase, targetPixels[ii].inPhase, tolerance));
	DLR_TEST_ASSERT(approximatelyEqual(
	  outputPixel.quadrature, targetPixels[ii].quadrature, tolerance));
      }
    }

 
    void
    ColorspaceConverterTest::
    testBGRA8ToRGB8()
    {
      ColorspaceConverter<BGRA8, RGB8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            for(UnsignedInt16 alphaValue = 0; alphaValue < 256;
                alphaValue += 7) {
              PixelBGRA8 inputPixel(
                static_cast<UnsignedInt8>(blueValue),
                static_cast<UnsignedInt8>(greenValue),
                static_cast<UnsignedInt8>(redValue),
                static_cast<UnsignedInt8>(alphaValue));
              PixelRGB8 referencePixel(
                static_cast<UnsignedInt8>(redValue),
                static_cast<UnsignedInt8>(greenValue),
                static_cast<UnsignedInt8>(blueValue));
              DLR_TEST_ASSERT(converter(inputPixel) == referencePixel);
            }
          }
        }
      }
    }
  
    void
    ColorspaceConverterTest::
    testRGBA8ToRGB8()
    {
      ColorspaceConverter<RGBA8, RGB8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            for(UnsignedInt16 alphaValue = 0; alphaValue < 256;
                alphaValue += 7) {
              PixelRGBA8 inputPixel(
                static_cast<UnsignedInt8>(redValue),
                static_cast<UnsignedInt8>(greenValue),
                static_cast<UnsignedInt8>(blueValue),
                static_cast<UnsignedInt8>(alphaValue));
              PixelRGB8 referencePixel(
                static_cast<UnsignedInt8>(redValue),
                static_cast<UnsignedInt8>(greenValue),
                static_cast<UnsignedInt8>(blueValue));
              DLR_TEST_ASSERT(converter(inputPixel) == referencePixel);
            }
          }
        }
      }
    }
  
  } // namespace computerVision

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::ColorspaceConverterTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::ColorspaceConverterTest currentTest;

}

#endif
