/**
***************************************************************************
* @file dlrComputerVision/naiveSnake.h
*
* Header file declaring a naive snakes implementation.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef DLR_COMPUTERVISION_NAIVESNAKE_H
#define DLR_COMPUTERVISION_NAIVESNAKE_H

#include <vector>
#include <dlrComputerVision/image.h>
#include <dlrNumeric/vector2D.h>

namespace dlr {

  namespace computerVision {

    enum SnakeStrategy {
      SNAKE_EUCLIDEAN_DISTANCE
    };


    class Snake {
    public:

      Snake();


      ~Snake() {}


      void
      enableCornerAdditionAndDeletion(bool enableFlag=true);


      std::vector<double>
      getBetaVector() {return m_betaVector;}
      
      
      bool
      isConverged() {return m_isConverged;}

        
      std::vector<Vector2D>
      run();

      
      std::vector<Vector2D>
      runOneIteration();
      
      
      void
      setBendingConstant(double beta);


      void
      setCornerAdditionAngle(double theta);


      void
      setCornerDeletionAngle(double theta);


      void
      setClosedCurve(bool isClosed=true)
        {m_isClosed = isClosed; m_isConverged = false;}


      void
      setExternalForceConstant(double kappa)
        {m_kappa = kappa; m_isConverged = false;}

    
      // void
      // setFixedEndpoints(bool isFixed=true)
      //   {m_isFixed = isFixed; m_isConverged = false;}


      void
      setInterestImage(const Image<GRAY1>& interestImage);


      void
      setMaxIterations(size_t maxIterations) {m_maxIterations = maxIterations;}


      void
      setMaxSnakeSize(size_t maxSize) {m_maxSnakeSize = maxSize;}


      void
      setMaximumSpanLength(size_t spanLength)
        {m_maxSpanLength = spanLength; m_isConverged = false;}


      void
      setMinimumSpanLength(size_t spanLength)
        {m_minSpanLength = spanLength; m_isConverged = false;}
      

      void
      setSeedPoints(const std::vector<Vector2D>& seedPoints);


      void
      setSeedPoints(const std::vector<Vector2D>& seedPoints,
                    const std::vector<bool> cornerFlags);
      

      void
      setStepsPerIteration(size_t numSteps) {m_contourIterations = numSteps;}
    

      void
      setStretchingConstant(double alpha)
        {m_alpha = alpha; m_isConverged = false;}


      void
      setViscosityConstant(double gamma)
        {m_gamma = gamma; m_isConverged = false;}


    private:

      void
      addResampledSpan(std::vector<Vector2D>& snake,
                       std::vector<double>& betaVector,
                       const Vector2D& newPoint,
                       double newBeta,
                       bool isLast=false);


      void
      adjustBetas(const std::vector<Vector2D>& snake,
                  std::vector<double>& betaVector);

      
      Array2D<double>
      buildForceBalanceMatrix(size_t numberOfSnakePoints);

      
      void
      buildForceBalanceRHS(const std::vector<Vector2D>& snake,
                           Array1D<double>& xRHS,
                           Array1D<double>& yRHS);

      
      bool
      isConverged(const std::vector<Vector2D>& snake,
                  const std::vector<Vector2D>& oldSnake);
    

      std::pair< std::vector<Vector2D>, std::vector<double> >
      resampleSnake();

      void
      updateSnakePosition(std::vector<Vector2D>& snake);
    
      
      double m_alpha;
      double m_beta;
      std::vector<double> m_betaVector;
      size_t m_contourIterations;
      double m_cornerAdditionThreshold;
      double m_cornerDeletionThreshold;
      Array2D<double> m_externalForceGradientX;
      Array2D<double> m_externalForceGradientY;
      Array2D<double> m_forceBalanceMatrix;
      double m_gamma;
      bool m_isClosed;
      bool m_isConverged;
      bool m_isFixed;
      double m_kappa;
      size_t m_maxIterations;
      size_t m_maxSnakeSize;
      double m_maxSpanLength;
      double m_minSpanLength;
      std::vector<Vector2D> m_snake;
    };

  } // namespace computerVision
  
} // namespace dlr


/* ============ Definitions of inline & template functions ============ */


#include <cmath>

namespace dlr {

  namespace computerVision {

    
  } // namespace computerVision

} // namespace dlr

#endif /* #ifndef DLR_COMPUTERVISION_NAIVESNAKE_H */
