/**
***************************************************************************
* @file dlrComputerVision/kernels.h
*
* Header file declaring some stock kernels to be used in client code.
*
* Copyright (C) 2006-2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef _DLRCOMPUTERVISION_KERNELS_H_
#define _DLRCOMPUTERVISION_KERNELS_H_

#include <dlrComputerVision/kernel.h>

namespace dlr {

  namespace computerVision {
    
    /** 
     * This function generates and returns a separable Gaussian kernel
     * of user-specified standard deviation.  The kernel will be sized
     * so that the number of rows is equal to the smallest odd number
     * greater than 6.0 * rowSigma, and the number of columns is equal
     * to the smallest odd number greater than 6.0 * columnSigma.  The
     * kernel will be normalized prior to return, so that the sum of
     * its elements is equal to one.
     *
     * Use this function like this:
     *
     * @code
     *   Kernel<double> kernel = getGaussianKernel<double>(1.0, 1.0);
     * @endcode
     * 
     * @param rowSigma This argument specifies the standard deviation
     * of the kernel in the Y direction.
     *
     * @param columnSigma This argument specifies the standard deviation of
     * the kernel in the X direction.
     * 
     * @return The return value is the kernel itself.
     */
    template<class KERNEL_TYPE>
    Kernel<KERNEL_TYPE>
    getGaussianKernel(double rowSigma, double columnSigma);


    /** 
     * This function generates and returns a separable Gaussian kernel
     * with the same variance along each axis.  The kernel will be
     * normalized prior to return, so that the sum of its elements is
     * equal to one.
     *
     * Use this function like this:
     *
     * @code
     *   Kernel<double> kernel = getGaussianKernel<double>(5, 5);
     * @endcode
     * 
     * @param rows This argument specifies how many rows the kernel
     * should have.
     * 
     * @param columns This argument specifies how many columns the
     * kernel should have.
     * 
     * @param rowSigma This argument specifies the standard deviation
     * of the kernel in the Y direction.  If sigma is less than 0.0,
     * then it will be automatically reset to rows / 6.0.
     *
     * @param columnSigma This argument specifies the standard deviation of
     * the kernel in the X direction.  If sigma is less than 0.0, then
     * it will be automatically reset columns / 6.0.
     * 
     * @return The return value is the kernel itself.
     */
    template<class KERNEL_TYPE>
    Kernel<KERNEL_TYPE>
    getGaussianKernel(size_t rows, size_t columns,
                      double rowSigma=-1.0, double columnSigma=-1.0);


  } // namespace computerVision

} // namespace dlr


/* =============== Implementation follows =============== */

#include <dlrNumeric/utilities.h>

namespace dlr {

  namespace computerVision {

    /// @cond privateCode
    namespace privateCode {
      
      template <class TYPE>
      Array1D<TYPE>
      getGaussian1D(size_t size, double sigma, bool normalize=false)
      {
        const double myPi = 3.14159265359;
        Array1D<TYPE> result(size);
        double x = (1.0 - static_cast<double>(size))/2.0;
        double twoSigmaSq = 2.0 * sigma * sigma;
        double k = 1.0 / (std::sqrt(2.0 * myPi) * sigma);
        for(size_t index0 = 0; index0 < size; ++index0) {
          result[index0] = k * exp(-x * x / twoSigmaSq);
          x += 1.0;
        }
        if(normalize) {
          result /= sum(result);
        }
        return result;
      }
      
    } // namespace privateCode
    /// @endcond


    // This function generates and returns a separable Gaussian kernel.
    template<class KERNEL_TYPE>
    Kernel<KERNEL_TYPE>
    getGaussianKernel(double rowSigma, double columnSigma)
    {
      size_t rows = static_cast<size_t>(6.0 * rowSigma + 1.0);
      size_t columns = static_cast<size_t>(6.0 * columnSigma + 1.0);
      if(rows % 2 == 0) {
        ++rows;
      }
      if(columns % 2 == 0) {
        ++columns;
      }
      return getGaussianKernel<KERNEL_TYPE>(
        rows, columns, rowSigma, columnSigma);
    }

    
    // This function generates and returns a separable Gaussian kernel.
    template<class KERNEL_TYPE>
    Kernel<KERNEL_TYPE>
    getGaussianKernel(size_t rows, size_t columns,
                      double rowSigma, double columnSigma)
    {
      // Argument checking.
      if(rows == 0 || columns == 0) {
        DLR_THROW(ValueException, "getGaussianKernel()",
                  "Arguments rows and columns may not have value of zero.");
      }
      if(rowSigma < 0.0) {
        rowSigma = rows / 6.0;
      }
      if(columnSigma < 0.0) {
        columnSigma = columns / 6.0;
      }

      Array1D<KERNEL_TYPE> rowComponent =
        privateCode::getGaussian1D<KERNEL_TYPE>(columns, columnSigma, true);
      Array1D<KERNEL_TYPE> columnComponent =
        privateCode::getGaussian1D<KERNEL_TYPE>(rows, rowSigma, true);
      return Kernel<KERNEL_TYPE>(rowComponent, columnComponent);
    }


  } // namespace computerVision
  
} // namespace dlr

#endif /* #ifndef _DLRCOMPUTERVISION_KERNEL_H_ */
