/**
***************************************************************************
* @file dlrComputerVision/kernel.h
*
* Header file declaring Kernel class.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 975 $
* $Date: 2007-12-30 01:57:17 -0500 (Sun, 30 Dec 2007) $
***************************************************************************
*/

#ifndef _DLRCOMPUTERVISION_KERNEL_H_
#define _DLRCOMPUTERVISION_KERNEL_H_

#include <dlrNumeric/array2D.h>

namespace dlr {

  namespace computerVision {
    
    /**
     ** This class template represents a 2D convolution kernel.  The template
     ** parameter indicates the type of the element in the kernel.
     **/
    template <class ELEMENT_TYPE>
    class Kernel
    {
    public:

      /* ******** Public typedefs ******** */

      typedef ELEMENT_TYPE ElementType;


      /* ******** Public member functions ******** */

      /** 
       * Default constructor initializes to zero size.
       */
      Kernel();

    
      /** 
       * The copy constructor does a deep copy.
       * 
       * @param source The Kernel instance to be copied.
       */
      Kernel(const Kernel<ELEMENT_TYPE>& source);

    
      /** 
       * This constructor allows us to implicitly make a Kernel
       * instance from an Array2D.  As with the copy constructor, the
       * newly created kernel has its own copy of the data in the copied
       * array.
       * 
       * @param source The Array2D instance to be copied.
       */
      Kernel(const Array2D<ElementType>& source);

    
      /**
       * Construct a kernel around external data.  Kernels constructed in
       * this way will copy the data from the input pointer.
       * 
       * @param rows Number of rows in the kernel after successful
       * construction.
       *
       * @param columns Number of columns in the kernel after successful
       * construction.
       *
       * @param dataPtr A C-style array of ElementType from which the newly
       * constructed Kernel should copy its data.
       */
      Kernel(size_t rows, size_t columns, ElementType* const dataPtr);


      /** 
       * This constructor allows us to implicitly make a separable
       * Kernel instance from a pair of Array1D instances.  As with the
       * copy constructor, the newly created kernel has its own copy of
       * the data from the copied arrays.
       * 
       * @param rowArray This argument specifies the portion of the
       * kernel which will be applied along image rows.
       * 
       * @param columnArray This argument specifies the portion of the
       * kernel which will be applied along image columns.
       */
      Kernel(const Array1D<ElementType>& rowArray,
             const Array1D<ElementType>& columnArray);

    
      /**
       * Destroys the Kernel instance and deletes the internal data
       * store.
       */
      virtual
      ~Kernel();


      /** 
       * Return a copy of the kernel data in an Array2D object.
       * 
       * @return The return value is an Array2D instance.  It does not
       * reference the same physical memory as the kernel.
       */
      Array2D<ElementType>
      getArray2D() const;
    

      /** 
       * This method is only valid for separable kernels; it returns
       * the separable kernel component which is parallel to the columns
       * of the image.
       * 
       * @return The return value is an Array1D instance representing an
       * N-row by 1-column kernel component.
       */
      Array1D<ElementType>
      getColumnComponent() const;


      /** 
       * This member function returns the number of columns in the kernel.
       * 
       * @return The return value is the width, in columns, of the kernel.
       */
      size_t getColumns() const;


      /** 
       * This method is only valid for separable kernels; it returns
       * the separable kernel component which is parallel to the rows
       * of the image.
       * 
       * @return The return value is an Array1D instance representing a
       * 1-row by M-column kernel component.
       */
      Array1D<ElementType>
      getRowComponent() const;


      /** 
       * This member function returns the number of rows in the kernel.
       * 
       * @return The return value is the height, in rows, of the kernel.
       */
      size_t getRows() const;

    
      /** 
       * This method returns true if the kernel is separable, false
       * otherwise.
       * 
       * @return The return value indicates whether or no the kernel is
       * separable.
       */
      bool
      isSeparable() const {return m_isSeparable;}


      /** 
       * This method allows the user to set the contents of the kernel.
       * This does a deep copy of the input array.
       * 
       * @param dataArray This argument specifies the kernel data to be
       * copied.
       */
      void
      setArray2D(const Array2D<ElementType>& dataArray);


      /** 
       * This method allows the user to set the contents of a separable
       * kernel.  Calling this method forces the kernel instance to
       * become separable.  This method does a deep copy of the input
       * arrays.
       * 
       * @param rowArray This argument specifies the separable kernel
       * component which is parallel to the rows of the image.  It
       * represents a 1-row by M-column kernel component.
       * 
       * @param columnArray This argument specifies the separable kernel
       * component which is parallel to the columns of the image.  It
       * represents an N-row by 1-column kernel component.
       */
      void
      setSeparableComponents(const Array1D<ElementType>& rowArray,
                             const Array1D<ElementType>& columnArray);


    private:

      Array2D<ElementType> m_data;
      bool m_isSeparable;
      size_t m_separableColumns;
      size_t m_separableRows;

    };

  } // namespace computerVision

} // namespace dlr


/* =============== Implementation follows =============== */


namespace dlr {

  namespace computerVision {

    // Default constructor initializes to zero size.
    template <class ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    Kernel()
      : m_data(),
        m_isSeparable(false),
        m_separableColumns(0),
        m_separableRows(0)
    {
      // Empty.
    }
    
    
    // The copy constructor does a shallow copy.
    template <class ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    Kernel(const Kernel<ELEMENT_TYPE> &source)
      : m_data(source.m_data.copy()),
        m_isSeparable(source.m_isSeparable),
        m_separableColumns(source.m_separableColumns),
        m_separableRows(source.m_separableRows)
    {
      // Empty.
    }
      
    
    // This constructor allows us to implicitly make a Kernel instance
    // from an Array2D.
    template <class ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    Kernel(const Array2D<ElementType> &source)
      : m_data(source.copy()),
        m_isSeparable(false),
        m_separableColumns(0),
        m_separableRows(0)
    {
      // Empty.
    }
    

    // Construct a kernel around external data.
    template <class ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    Kernel(size_t rows, size_t columns, ElementType* const dataPtr)
      : m_data(rows, columns),
        m_isSeparable(false),
        m_separableColumns(0),
        m_separableRows(0)
    {
      m_data.copy(dataPtr);
    }


    // This constructor allows us to implicitly make a separable
    // Kernel instance from a pair of Array1D instances.
    template <class ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    Kernel(const Array1D<ElementType>& rowArray,
           const Array1D<ElementType>& columnArray)
      : m_data(1, rowArray.size() + columnArray.size()),
        m_isSeparable(true),
        m_separableColumns(rowArray.size()),
        m_separableRows(columnArray.size())
    {
      std::copy(rowArray.begin(), rowArray.end(), m_data.begin());
      std::copy(columnArray.begin(), columnArray.end(),
                m_data.begin() + rowArray.size());
    }


    // Destroys the Kernel instance and deletes the internal data
    // store.
    template <class ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    ~Kernel()
    {
      // Empty.
    }


    // Return a copy of the kernel data in an Array2D object.
    template <class ELEMENT_TYPE>
    Array2D<ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    getArray2D() const
    {
      if(m_isSeparable) {
        Array2D<ELEMENT_TYPE> synthesizedArray(
          m_separableRows, m_separableColumns);
        typename Array2D<ELEMENT_TYPE>::iterator resultIterator =
          synthesizedArray.begin();
        for(size_t rowIndex = 0; rowIndex < m_separableRows; ++rowIndex) {
          for(size_t columnIndex = 0; columnIndex < m_separableColumns;
              ++columnIndex) {
            *resultIterator =
              m_data[columnIndex] * m_data[m_separableColumns + rowIndex];
	    ++resultIterator;
          }
        }
        return synthesizedArray;
      }
      // Not separable.
      return m_data.copy();
    }
    

    // This method is only valid for separable kernels; it returns
    // the separable kernel component which is parallel to the columns
    // of the image.
    template <class ELEMENT_TYPE>
    Array1D<ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    getColumnComponent() const
    {
      Array1D<ELEMENT_TYPE> columnComponent(m_separableRows);
      columnComponent.copy(m_data.begin() + m_separableColumns);
      return columnComponent;
    }

    
    // This member function returns the number of columns in the kernel.
    template <class ELEMENT_TYPE>
    size_t
    Kernel<ELEMENT_TYPE>::
    getColumns() const
    {
      if(this->isSeparable()) {
        return m_separableColumns;
      }
      return m_data.columns();
    }


    // This method is only valid for separable kernels; it returns
    // the separable kernel component which is parallel to the rows
    // of the image.
    template <class ELEMENT_TYPE>
    Array1D<ELEMENT_TYPE>
    Kernel<ELEMENT_TYPE>::
    getRowComponent() const
    {
      Array1D<ELEMENT_TYPE> rowComponent(m_separableColumns);
      rowComponent.copy(m_data.begin());
      return rowComponent;
    }


    // This member function returns the number of rows in the kernel.
    template <class ELEMENT_TYPE>
    size_t
    Kernel<ELEMENT_TYPE>::
    getRows() const
    {
      if(this->isSeparable()) {
        return m_separableRows;
      }
      return m_data.rows();
    }

  
    // This method allows the user to set the contents of the kernel.
    template <class ELEMENT_TYPE>
    void
    Kernel<ELEMENT_TYPE>::
    setArray2D(const Array2D<ElementType>& dataArray)
    {
      m_isSeparable = false;
      m_data = dataArray.copy();
      m_separableColumns = 0;
      m_separableRows = 0;
    }


    // This method allows the user to set the contents of a separable
    // kernel.
    template <class ELEMENT_TYPE>
    void
    Kernel<ELEMENT_TYPE>::
    setSeparableComponents(const Array1D<ElementType>& rowArray,
                           const Array1D<ElementType>& columnArray)
    {
      m_isSeparable = true;
      m_data.reinit(1, rowArray.size() + columnArray.size());
      std::copy(rowArray.begin(), rowArray.end(), m_data.begin());
      std::copy(columnArray.begin(), columnArray.end(),
		m_data.begin() + rowArray.size());
      m_separableColumns = rowArray.size();
      m_separableRows = columnArray.size();
    }
    
  } // namespace computerVision

} // namespace dlr

#endif /* #ifndef _DLRCOMPUTERVISION_KERNEL_H_ */
