/**
***************************************************************************
* @file dlrComputerVision/imageWarper.h
*
* Header file declaring ImageWarper class.
*
* Copyright (C) 2009 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 931 $
* $Date: 2007-05-30 13:51:26 -0400 (Wed, 30 May 2007) $
***************************************************************************
*/

#ifndef DLR_COMPUTERVISION_IMAGEWARPER_H
#define DLR_COMPUTERVISION_IMAGEWARPER_H

#include <dlrComputerVision/image.h>
#include <dlrNumeric/array2D.h>

namespace dlr {

  namespace computerVision {
    
    /**
     **/
    template <class NumericType, class TransformFunctor>
    class ImageWarper
    {
    public:

      /* ******** Public member functions ******** */

      /** 
       * 
       */
      ImageWarper();

    
      /** 
       * 
       * 
       * @param inputRows This argument 
       * 
       * @param inputColumns This argument 
       * 
       * @param outputRows This argument 
       * 
       * @param outputColumns This argument 
       * 
       * @param transformer This argument 
       */
      ImageWarper(size_t inputRows, size_t inputColumns,
                  size_t outputRows, size_t outputColumns,
                  TransformFunctor transformer);

    
      /**
       * Destroys the ImageWarper instance and deletes the internal data
       * store.
       */
      virtual
      ~ImageWarper();


      /** 
       * 
       * 
       * @param inputImage This argument 
       * 
       * @param defaultValue This argument 
       * 
       * @return The return value 
       */
      template <ImageFormat InputFormat, ImageFormat OutputFormat>
      Image<OutputFormat>
      warpImage(Image<InputFormat> const& inputImage,
                typename Image<OutputFormat>::PixelType defaultValue) const;

    private:

      struct SampleInfo {
        NumericType c00;
        NumericType c01;
        NumericType c10;
        NumericType c11;
        size_t index00;
        bool isInBounds;
      };

      size_t m_inputColumns;
      size_t m_inputRows;
      Array2D<SampleInfo> m_lookupTable;

    };

  } // namespace computerVision

} // namespace dlr


/* =============== Implementation follows =============== */

#include <dlrNumeric/vector2D.h>

namespace dlr {

  namespace computerVision {


    template<class NumericType, class TransformFunctor>
    ImageWarper<NumericType, TransformFunctor>::
    ImageWarper()
      : m_inputColumns(0),
        m_inputRows(0),
        m_lookupTable()
    {
      // Empty.
    }

    
    template<class NumericType, class TransformFunctor>
    ImageWarper<NumericType, TransformFunctor>::
    ImageWarper(size_t inputRows, size_t inputColumns,
                size_t outputRows, size_t outputColumns,
                TransformFunctor transformer)
      : m_inputColumns(inputColumns),
        m_inputRows(inputRows),
        m_lookupTable(outputRows, outputColumns)
    {
      // Note(xxx): Need to make this loop respect NumericType.
      for(size_t row = 0; row < outputRows; ++row) {
        for(size_t column = 0; column < outputColumns; ++column) {
          numeric::Vector2D outputCoord(column, row);
          numeric::Vector2D inputCoord = transformer(outputCoord);
          SampleInfo& sampleInfo = m_lookupTable(row, column);
          if((inputCoord.y() >= 0.0)
             && (inputCoord.x() >= 0.0)
             && (inputCoord.y() < static_cast<double>(inputRows - 1))
             && (inputCoord.x() < static_cast<double>(inputColumns - 1))) {
            double tmpDbl;
            double xFrac = modf(inputCoord.x(), &tmpDbl);
            size_t i0 = static_cast<size_t>(tmpDbl);
            double yFrac = modf(inputCoord.y(), &tmpDbl);
            size_t j0 = static_cast<size_t>(tmpDbl);
            double oneMinusXFrac = 1.0 - xFrac;
            double oneMinusYFrac = 1.0 - yFrac;
            sampleInfo.c00 = oneMinusXFrac * oneMinusYFrac;
            sampleInfo.c01 = xFrac * oneMinusYFrac;
            sampleInfo.c10 = oneMinusXFrac * yFrac;
            sampleInfo.c11 = xFrac * yFrac;
            sampleInfo.index00 = inputColumns * j0 + i0;
            sampleInfo.isInBounds = true;
          } else {
            sampleInfo.isInBounds = false;
          }
        }
      }
    }
    

    // Destroys the ImageWarper instance and deletes the internal data
    // store.
    template<class NumericType, class TransformFunctor>
    ImageWarper<NumericType, TransformFunctor>::
    ~ImageWarper()
    {
      // Empty.
    }



    template<class NumericType, class TransformFunctor>
    template <ImageFormat InputFormat, ImageFormat OutputFormat>
    Image<OutputFormat>
    ImageWarper<NumericType, TransformFunctor>::
    warpImage(Image<InputFormat> const& inputImage,
              typename Image<OutputFormat>::PixelType defaultValue) const
    {
      if((inputImage.rows() != m_inputRows)
         || (inputImage.columns() != m_inputColumns)) {
        std::ostringstream message;
        message
          << "InputImage (" << inputImage.rows() << "x" << inputImage.columns()
          << ") doesn't match expected dimensions (" << m_inputRows << "x"
          << m_inputColumns << ").";
        DLR_THROW(ValueException, "ImageWarper::warpImage()",
                  message.str().c_str());
      }
      Image<OutputFormat> outputImage(
        m_lookupTable.rows(), m_lookupTable.columns());
      for(size_t ii = 0; ii < m_lookupTable.size(); ++ii) {
        SampleInfo const& sampleInfo = m_lookupTable(ii);
        if(sampleInfo.isInBounds) {
          typename Image<OutputFormat>::PixelType& outputPixel =
            outputImage[ii];
          size_t inputIndex = sampleInfo.index00;
          
          outputPixel = sampleInfo.c00 * inputImage[inputIndex];
          ++inputIndex;
          outputPixel += sampleInfo.c01 * inputImage[inputIndex];
          inputIndex += m_inputColumns;
          outputPixel += sampleInfo.c11 * inputImage[inputIndex];
          --inputIndex;
          outputPixel += sampleInfo.c10 * inputImage[inputIndex];
        } else {
          outputImage[ii] = defaultValue;
        }
      }
      return outputImage;
    }

  } // namespace computerVision

} // namespace dlr

#endif /* #ifndef DLR_COMPUTERVISION_IMAGEWARPER_H */
