/**
***************************************************************************
* @file dlrComputerVision/cameraIntrinsics.h
*
* Header file declaring a parent class from which to derive classes which
* represent camera intrinsic parameters.
*
* Copyright (C) 2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef DLR_COMPUTERVISION_CAMERAINTRINSICS_H
#define DLR_COMPUTERVISION_CAMERAINTRINSICS_H

#include <dlrGeometry/ray3D.h>
#include <dlrNumeric/index2D.h>
#include <dlrNumeric/vector2D.h>
#include <dlrNumeric/vector3D.h>

namespace dlr {

  namespace computerVision {

    /**
     ** This abstract base class defines an interface for classes that
     ** describe camera projection and distortion parameters.  There
     ** are two coordinate systems referenced in the documentation for
     ** this class:
     **
     **  The "camera coordinate system" has its origin at the camera
     **  focus, its Z axis pointing in the direction of camera gaze,
     **  its X axis pointing to the right (from the camera's
     **  perspective), and its Y axis pointing down. The upper left
     **  corner of the image has pixel coordinate (0, 0).  The center
     **  of the upper left pixel of the image has pixel coordinate
     **  (0.5, 0.5).  The upper left corner of the next-to-leftmost
     **  pixel in the top row is at pixel coordinate (1, 0).
     **/
    class CameraIntrinsics {
    public:

      /** 
       * The default constructor currently does nothing.
       */
      CameraIntrinsics() {};


      /**
       * The destructor currently does nothing.
       */
      virtual
      ~CameraIntrinsics() {}


      /** 
       * This function should be overridden by derived classes so that
       * it takes points in 3D camera coordinates and projects them
       * into image pixel coordinates.
       * 
       * @param point This argument is the 3D point to be projected.
       * 
       * @return The return value is the resulting pixel coordinate.
       */
      virtual dlr::numeric::Vector2D
      project(const dlr::numeric::Vector3D& point) const = 0;


      /** 
       * This function returns a ray in 3D camera coordinates starting
       * at the camera focus, and passing through the center of the
       * specified pixel.
       * 
       * @param pixelPosition This argument is the pixel through which
       * the ray should pass.
       * 
       * @param normalize This argument indicates whether the ray
       * should be normalized to unit length before being returned.
       * 
       * @return The return value is the resulting ray.
       */
      virtual inline geometry::Ray3D
      reverseProject(const dlr::numeric::Index2D& pixelPosition,
                     bool normalize = true) const;


      /** 
       * This function should be overridden by derived classes to
       * return a ray in 3D camera coordinates starting at the camera
       * focus and passing through the specified pixel position.
       * 
       * @param pixelPosition This argument is the point in pixel
       * coordinates through which the returned ray should pass.
       * 
       * @param normalize This argument indicates whether the ray
       * should be normalized to unit length before being returned.
       * 
       * @return The return value is the resulting ray.
       */
      virtual geometry::Ray3D
      reverseProject(const dlr::numeric::Vector2D& pixelPosition,
                     bool normalize = true) const = 0;


    protected:

    };

  } // namespace computerVision
  
} // namespace dlr


/* ============ Definitions of inline & template functions ============ */


#include <cmath>

namespace dlr {

  namespace computerVision {

    // This function returns a ray in 3D camera coordinates starting
    // at the camera focus, and passing through the center of the
    // specified pixel.
    inline dlr::geometry::Ray3D
    CameraIntrinsics::
    reverseProject(const dlr::numeric::Index2D& pixelPosition,
                   bool normalize) const
    {
      return this->reverseProject(
        dlr::numeric::Vector2D(pixelPosition.getColumn() + 0.5,
                               pixelPosition.getRow() + 0.5),
        normalize);
    }

  } // namespace computerVision
  
} // namespace dlr

#endif /* #ifndef DLR_COMPUTERVISION_CAMERAINTRINSICS_H */
