/**
***************************************************************************
* @file dlrComputerVision/utilities.h
*
* Header file declaring utility functions for dlrComputerVision library.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 997 $
* $Date: 2008-01-16 03:17:44 -0500 (Wed, 16 Jan 2008) $
***************************************************************************
*/

#ifndef _DLRCOMPUTERVISION_UTILITIES_H_
#define _DLRCOMPUTERVISION_UTILITIES_H_

#include <dlrComputerVision/colorspaceConverter.h>
#include <dlrComputerVision/image.h>

namespace dlr {

  namespace computerVision {
    
    /**
     * This function returns by reference an image which either shares
     * or copies the data from the input array.
     *
     * If possible, this function returns an Image which references the
     * same memory as the input array, but in which each pixel is the
     * aggregate of the appropriate number of elements from the array.
     * For example, if this function is called with template argument
     * RGB8, and an Nx(3*M) array of UnsignedInt8 is passed as the
     * argument, then the return value will be an NxM Image<RGB8> which
     * references the same memory as the argument.  Imagine that the
     * first three elements of the first row of the argument are 12, 14,
     * and 72.  In this case, the upper-left RGB value of the returned
     * image image is {12, 14, 72}.  This memory sharing only works if
     * the compiler does not "pad" the pixel struct to byte align its
     * members.
     *
     * If memory sharing is not possible, then this function tests the
     * size of argument outputImage, reinitializes it only if the size
     * does not match the size of the input array, and then copies the
     * data from inputArray to outputImage.
     *
     * WARNING: If data sharing is possible, the returned image does no
     * memory management.  It is only valid until the original input
     * data is destroyed.
     *
     * If you use this function in conjunction with
     * dissociateColorComponents(), you can simply ignore the question
     * of whether the data sharing works or not.  For example, you might
     * use associateColorComponents to get an Image to operate on, do
     * all of your image processing, and then use
     * dissocateColorComponents to get back to a flat array.  If data
     * sharing is possible, all of your operations will have taken place
     * in the original array.  If data sharing isn't possible, then the
     * two functions will take care of copying back and forth between
     * the image and the flat array.
     * 
     * @param inputArray This argument is the array from which to
     * construct the return image.
     * 
     * @param outputImage This argument is the image which will be
     * modified or copied to.
     *
     * @return The return value is true if the returned image shares
     * data with the input array, false otherwise.
     */
    template<ImageFormat FORMAT>
    bool
    associateColorComponents(
      Array2D<typename ImageFormatTraits<FORMAT>::ComponentType>& inputArray,
      Image<FORMAT>& outputImage);


    /**
     * @deprecated {Please use the two-argument version of
     * associateColorComponents() instead.}
     *
     * This function tries to return an Image which references the
     * same memory as the input array, but in which each pixel is the
     * aggregate of the appropriate number of elements from the array.
     * If it is not possible to do so, then this function will throw a
     * LogicException.  This function is deprecated.  Instead, ues the
     * two-argument form of associateColorComponents().
     * 
     * WARNING: The returned image does no memory management.  It is only
     * valid until the original input image is destroyed.
     * 
     * @param inputArray This argument is the array from which to
     * construct the return image.
     * 
     * @return The return value is an image pointing to the input array
     * data, but in which the color components are lumped into pixels.
     */
    template<ImageFormat FORMAT>
    Image<FORMAT>
    associateColorComponents(
      Array2D<typename ImageFormatTraits<FORMAT>::ComponentType>& inputArray);


    /** 
     * This function takes an image in one colorspace and generates a
     * corresponding image in a second colorspace.  Use this function as
     * follows:
     *
     *   Image<GRAY8> = convertColorspace<GRAY8, RGB16>(myRGB16Image);
     *
     * or equivalently, the second template argument can be left implicit,
     *
     *   Image<GRAY8> = convertColorspace<GRAY8>(myRGB16Image);
     *
     * This function only works for image formats in which there's a
     * one-to-one mapping between input and output pixel types.  When
     * converting between formats which don't match this requirement,
     * such as when converting from RGB8 to YUV420, please use a
     * different routine.
     *
     * @param inputImage This argument is the image to be converted.
     * 
     * @return The return value is an image in the converted colorspace.
     */
    template<ImageFormat OUTPUT_FORMAT, ImageFormat INPUT_FORMAT>
    Image<OUTPUT_FORMAT>
    convertColorspace(const Image<INPUT_FORMAT>& inputImage);


    /**
     * This function returns by reference an array which either shares
     * or copies the data from the input image.
     *
     * If possible, this function returns an Array2D which references
     * the same memory as the input image, but in which each pixel has
     * been "flattened" so that the returned array has a separate
     * element for each color component of each pixel.  For example, if
     * this function is called with an NxM Image<RGB8> as its argument,
     * the returned value will be an Nx(3*M) array of 8-bit unsigned
     * ints which references the same memorty.  Imagine that the
     * upper-left RGB value of the image is {12, 14, 72}.  In this case,
     * the first three elements of first row of the returned image will
     * be 12, 14, and 72.  This memory sharing only works if the
     * compiler does not "pad" the pixel struct to byte align its
     * members.
     *
     * If memory sharing is not possible, then this function tests the
     * size of argument outputArray, reinitializes it only if the size
     * does not match the size of the input image, and then copies the
     * data from inputImage to outputArray.
     *
     * WARNING: If data sharing is possible, the returned array does no
     * memory management.  It is only valid until the original input
     * data is destroyed.
     *
     * If you use this function in conjunction with
     * associateColorComponents(), you can simply ignore the question
     * of whether the data sharing works or not.  For example, you might
     * use associateColorComponents to get an Image to operate on, do
     * all of your image processing, and then use
     * dissocateColorComponents to get back to a flat array.  If data
     * sharing is possible, all of your operations will have taken place
     * in the original array.  If data sharing isn't possible, then the
     * two functions will take care of copying back and forth between
     * the image and the flat array.
     * 
     * @param inputImage This argument is the image from which to
     * construct the return array.
     * 
     * @outputArray This argument is the array which will be modified or
     * copied to.
     *
     * @return The return value is true if the returned array shares
     * data with the input image, false otherwise.
     */
    template<ImageFormat FORMAT>
    bool
    dissociateColorComponents(
      Image<FORMAT>& inputImage,
      Array2D<typename ImageFormatTraits<FORMAT>::ComponentType>& outputArray);


    /** 
     * @deprecated {Please use the two-argument version of
     * dissociateColorComponents() instead.}
     *
     * This function tries to return an Array2D which references the
     * same memory as the input image, but in which each pixel has
     * been "flattened" so that the returned array has a separate
     * element for each color component of each pixel.  If it is not
     * possible to do so, then this function will throw a
     * LogicException.  This function is deprecated.  Instead, ues the
     * two-argument form of dissociateColorComponents().
     * 
     * WARNING: The returned array does no memory management.  It is only
     * valid until the original input image is destroyed.
     * 
     * @param inputImage This argument is the image from which to
     * construct the return array.
     * 
     * @return The return value is a 2D array pointing to the input
     * image data, but in which the color components are not lumped into
     * pixels.
     */
    template<ImageFormat FORMAT>
    Array2D<typename ImageFormatTraits<FORMAT>::ComponentType>
    dissociateColorComponents(Image<FORMAT>& inputImage);


    /**
     * This function subsamples its input to create a new, smaller
     * image.  The number of rows in the resulting image is the largest
     * integer, r, such that rowStep * r <= inputImage.rows().  The
     * number of columns in the resulting image is the largest integer,
     * c, such that columnStep * c <= inputImage.columns().  The pixel
     * values in the resulting image are the values at the intersection
     * of every rowStep-th row and columnStep-th column of inputImage,
     * starting with the value at (row, column) = (0, 0), which is
     * always copied directly in to pixel (0, 0) of the result image.
     * In other words, element (0, 0) of the resulting image is equal to
     * element (0, 0) of inputImage, element (0, 1) of the resulting
     * image is equal to element (0, columnStep) of inputImage, element
     * (0, 2) of the resulting image is equal to element (0, 2 *
     * columnStep) of inputImage, element (2, 4) of the resulting image
     * is equal to element (2 * rowStep, 4 * columnStep) in inputImage.
     *
     * @param inputImage This argument is the image to be subsampled.
     *
     * @param rowStep This argument controls which rows of inputImage
     * contribute to the result.
     *
     * @param columnStep This argument controls which columns of inputImage
     * contribute to the result.
     *
     * @return The return value is the subsampled image.
     */
    template<ImageFormat Format>
    Image<Format>
    subsample(const Image<Format>& inputImage,
              size_t rowStep = 2,
              size_t columnStep = 2);


#if 0
    /**
     * This function interpolates its input to create a new, larger
     * image.  The number of rows in the resulting image is twice the
     * number of rows in the input image.  The number of columns in the
     * resulting image is twice the number of columns in the input
     * image.  Where output pixels overlap input pixels, they are copied
     * directly from the input image.  Where output pixels fall in
     * between input pixels, they are constructed by bilinear
     * interpolation.  The pixel at (0, 0) in the output image falls
     * directly on top of pixel (0, 0) in the input iamage.  (2, 2) in
     * the output image falls on top of (1, 1) in the input image, and
     * so forth.
     *
     * @param inputImage This argument is the image to be interpolated.
     *
     * @return The return value is the supersampled image.
     */
    template<ImageFormat InputFormat,
             ImageFormat OutputFormat,
             ImageFormat IntermediateFormat>
    Image<OutputFormat>
    supersample(const Image<InputFormat>& inputImage,
                typename ImageFormatTraits<OutputFormat>::PixelType,
                typename ImageFormatTraits<IntermediateFormat>::PixelType);
#endif
  
    
    /** 
     * This function creates a new array and copies into it the pixel
     * data from the input image.  If the image format is one which has
     * multiple interleaved color components, such as RGB8, then the
     * returned array will have individual elements for each component
     * of each pixel.  For example, if the first row of an HSV8 image is
     * [{h0, s0, v0}, {h1, s1, v1}, {h2, s2, v2}, ... ], then the first
     * row of the array returned by toArray will be [h0, s0, v0, h1, s1,
     * v1, h2, s2, v2, ...].
     * 
     * @param inputImage This argument is the image to be copied.
     * 
     * @return The return value is an array containing the copied data.
     */
    template <class Type, ImageFormat FORMAT>
    Array2D<Type>
    toArray(const Image<FORMAT>& inputImage);

  } // namespace computerVision    

} // namespace dlr


/* ============ Definitions of inline & template functions ============ */


namespace dlr {

  namespace computerVision {

    /// @cond privateCode
    namespace privateCode {

      // This function simplifies template specialization of
      // associateColorComponents, which makes things easier on the
      // compiler.
      template<ImageFormat Format, class ComponentType>
      inline bool
      associateColorComponents(
	Array2D<ComponentType>& inputArray,
	Image<Format>& outputImage)
      {
	// Some typedefs for notational convenience.
	typedef typename ImageFormatTraits<Format>::PixelType PixelType;
	typedef typename Image<Format>::iterator ImageIterator;
	typedef typename Array2D<ComponentType>::iterator ArrayIterator;
      
	// Argument checking.
	if(inputArray.columns()
	   % ImageFormatTraits<Format>::getNumberOfComponents()
	   != 0) {
	  DLR_THROW(ValueException, "associateColorComponents()",
		    "Input image must have a number of columns which is "
		    "evenly divisible by the number of color components.");
	}

	// Calculate output image width.
	size_t numberOfOutputColumns = inputArray.columns()
	  / ImageFormatTraits<Format>::getNumberOfComponents();

	bool returnValue;
	if(PixelType::isContiguous()) {
	  // Build the output image.
	  outputImage = Image<Format>(
	    inputArray.rows(), numberOfOutputColumns,
	    reinterpret_cast<PixelType*>(inputArray.data()));
	  returnValue = true;
	} else {
	  if(outputImage.rows() != inputArray.rows()
	     || outputImage.columns() != numberOfOutputColumns) {
	    outputImage = Image<Format>(
	      inputArray.rows(), numberOfOutputColumns);
	  }
	  ArrayIterator arrayIterator = inputArray.begin();
	  for(ImageIterator imageIterator = outputImage.begin();
	      imageIterator != outputImage.end(); ++imageIterator) {
	    imageIterator->copyFromIterator(arrayIterator);
	    ++imageIterator;
	  }
	  returnValue = false;
	}
	return returnValue;
      }

  
      // Explicit specializations of
      // privateCode::associateColorComponents() are needed for those
      // image formats which use builtin types as pixels.
      template<>
      inline bool
      associateColorComponents(
	Array2D<bool>& inputArray,
	Image<GRAY1>& outputImage)
      {
	outputImage = inputArray;
	return true;
      }


      template<>
      inline bool
      associateColorComponents(Array2D<UnsignedInt8>& inputArray,
			       Image<GRAY8>& outputImage)
      {
	outputImage = inputArray;
	return true;
      }

    
      template<>
      inline bool
      associateColorComponents(Array2D<UnsignedInt16>& inputArray,
			       Image<GRAY16>& outputImage)
      {
	outputImage = inputArray;
	return true;
      }

    
      template<>
      inline bool
      associateColorComponents(Array2D<UnsignedInt32>& inputArray,
			       Image<GRAY32>& outputImage)
      {
	outputImage = inputArray;
	return true;
      }

    
      template<>
      inline bool
      associateColorComponents(Array2D<Int16>& inputArray,
			       Image<GRAY_SIGNED16>& outputImage)
      {
	outputImage = inputArray;
	return true;
      }

    
      template<>
      inline bool
      associateColorComponents(Array2D<Int32>& inputArray,
			       Image<GRAY_SIGNED32>& outputImage)
      {
	outputImage = inputArray;
	return true;
      }

    
      template<>
      inline bool
      associateColorComponents(Array2D<Float32>& inputArray,
			       Image<GRAY_FLOAT32>& outputImage)
      {
	outputImage = inputArray;
	return true;
      }

    
      template<>
      inline bool
      associateColorComponents(Array2D<Float64>& inputArray,
			       Image<GRAY_FLOAT64>& outputImage)
      {
	outputImage = inputArray;
	return true;
      }

      // This function returns by reference an array which either shares
      // or copies the data from the input image.
      template<ImageFormat Format, class ComponentType>
      inline bool
      dissociateColorComponents(
	Image<Format>& inputImage,
	Array2D<ComponentType>& outputArray)
      {
	// Some typedefs for notational convenience.
	typedef typename ImageFormatTraits<Format>::PixelType PixelType;
	typedef typename Image<Format>::iterator ImageIterator;
	typedef typename Array2D<ComponentType>::iterator ArrayIterator;

	// Calculate output array width.
	size_t numberOfOutputColumns = inputImage.columns()
	  * ImageFormatTraits<Format>::getNumberOfComponents();
    
	// Now build the output array.
	bool returnValue;
	if(PixelType::isContiguous()) {
	  outputArray = Array2D<ComponentType>(
	    inputImage.rows(), numberOfOutputColumns,
	    reinterpret_cast<ComponentType*>(inputImage.data()));
	  returnValue = true;
	} else {
	  if(outputArray.rows() != inputImage.rows()
	     || outputArray.columns() != numberOfOutputColumns) {
	    outputArray = Array2D<ComponentType>(
	      inputImage.rows(), numberOfOutputColumns);
	  }
	  ArrayIterator arrayIterator = outputArray.begin();
	  for(ImageIterator imageIterator = inputImage.begin();
	      imageIterator != inputImage.end(); ++imageIterator) {
	    imageIterator->copyToIterator(arrayIterator);
	    ++imageIterator;
	  }
	  returnValue = false;
	}
	return returnValue;
      }
  
  
      // Explicit specializations of dissociateColorComponents() are needed
      // for those image formats which use builtin types as pixels.
      template<>
      inline bool
      dissociateColorComponents(Image<GRAY1>& inputImage,
				Array2D<bool>& outputArray)
      {
	outputArray = inputImage;
	return true;
      }

    
      template<>
      inline bool
      dissociateColorComponents(Image<GRAY8>& inputImage,
				Array2D<UnsignedInt8>& outputArray)
      {
	outputArray = inputImage;
	return true;
      }

    
      template<>
      inline bool
      dissociateColorComponents(Image<GRAY16>& inputImage,
				Array2D<UnsignedInt16>& outputArray)
      {
	outputArray = inputImage;
	return true;
      }

    
      template<>
      inline bool
      dissociateColorComponents(Image<GRAY32>& inputImage,
				Array2D<UnsignedInt32>& outputArray)
      {
	outputArray = inputImage;
	return true;
      }

    
      template<>
      inline bool
      dissociateColorComponents(Image<GRAY_SIGNED16>& inputImage,
				Array2D<Int16>& outputArray)
      {
	outputArray = inputImage;
	return true;
      }

    
      template<>
      inline bool
      dissociateColorComponents(Image<GRAY_SIGNED32>& inputImage,
				Array2D<Int32>& outputArray)
      {
	outputArray = inputImage;
	return true;
      }

    
      template<>
      inline bool
      dissociateColorComponents(Image<GRAY_FLOAT32>& inputImage,
				Array2D<Float32>& outputArray)
      {
	outputArray = inputImage;
	return true;
      }

    
      template<>
      inline bool
      dissociateColorComponents(Image<GRAY_FLOAT64>& inputImage,
				Array2D<Float64>& outputArray)
      {
	outputArray = inputImage;
	return true;
      }

    } // namespace privateCode
    /// @endcond


    // This function returns by reference an image which either shares
    // or copies the data from the input array.
    template<ImageFormat FORMAT>
    bool
    associateColorComponents(
      Array2D<typename ImageFormatTraits<FORMAT>::ComponentType>& inputArray,
      Image<FORMAT>& outputImage)
    {
      return privateCode::associateColorComponents(inputArray, outputImage);
    }

    
    // This deprecated function tries to return an Image which
    // references the same memory as the input array, but in which each
    // pixel is the aggregate of the appropriate number of elements from
    // the array.  If it is not possible to do so, then this function
    template<ImageFormat FORMAT>
    Image<FORMAT>
    associateColorComponents(
      Array2D<typename ImageFormatTraits<FORMAT>::ComponentType>& inputArray)
    {
      typedef typename ImageFormatTraits<FORMAT>::PixelType PixelType;
      if(!PixelType::isContiguous()) {
        DLR_THROW(dlr::LogicException, "associateColorComponents()",
                  "The deprecated single-argument version of "
                  "associateColorComponents() can only be use with "
                  "contiguous pixel types.");
      }

      Image<FORMAT> outputImage;
      associateColorComponents(inputArray, outputImage);
      return outputImage;
    }


    // This function takes an image in one colorspace and generates a
    // corresponding image in a second colorspace.
    template<ImageFormat OUTPUT_FORMAT, ImageFormat INPUT_FORMAT>
    Image<OUTPUT_FORMAT>
    convertColorspace(const Image<INPUT_FORMAT>& inputImage)
    {
      Image<OUTPUT_FORMAT> outputImage(
	inputImage.rows(), inputImage.columns());
      ColorspaceConverter<INPUT_FORMAT, OUTPUT_FORMAT> converter;
      std::transform(inputImage.begin(), inputImage.end(),
                     outputImage.begin(), converter);
      return outputImage;
    }


    // This function returns by reference an array which either shares
    // or copies the data from the input image.
    template<ImageFormat FORMAT>
    bool
    dissociateColorComponents(
      Image<FORMAT>& inputImage,
      Array2D<typename ImageFormatTraits<FORMAT>::ComponentType>& outputArray)
    {
      return privateCode::dissociateColorComponents(inputImage, outputArray);
    }
  

    // This deprecated function tries to return an Array2D which
    // references the same memory as the input image, but in which each
    // pixel has been "flattened" so that the returned array has a
    // separate element for each color component of each pixel.  If it
    template<ImageFormat FORMAT>
    Array2D<typename ImageFormatTraits<FORMAT>::ComponentType>
    dissociateColorComponents(Image<FORMAT>& inputImage)
    {
      typedef typename ImageFormatTraits<FORMAT>::PixelType PixelType;
      if(!PixelType::isContiguous()) {
        DLR_THROW(dlr::LogicException, "dissociateColorComponents()",
                  "The deprecated single-argument version of "
                  "dissociateColorComponents() can only be use with "
                  "contiguous pixel types.");
      }
      Array2D<typename ImageFormatTraits<FORMAT>::ComponentType> outputArray;
      dissociateColorComponents(inputImage, outputArray);
      return outputArray;
    }

  
    // This function subsamples its input to create a new, smaller
    // image.
    template<ImageFormat Format>
    Image<Format>
    subsample(const Image<Format>& inputImage,
              size_t rowStep,
              size_t columnStep)
    {
      Image<Format> outputImage(inputImage.rows() / rowStep,
                                inputImage.columns() / columnStep);
      for(size_t outputRowIndex = 0; outputRowIndex < outputImage.rows();
          ++outputRowIndex) {
        Array1D<typename Image<Format>::PixelType> inputRow =
          inputImage.row(rowStep * outputRowIndex);
        Array1D<typename Image<Format>::PixelType> outputRow =
          outputImage.row(outputRowIndex);
        size_t inputColumnIndex = 0;
        for(size_t outputColumnIndex = 0;
            outputColumnIndex < outputImage.columns();
            ++outputColumnIndex) {
          outputRow[outputColumnIndex] = inputRow[inputColumnIndex];
          inputColumnIndex += columnStep;
        }
      }
      return outputImage;
    }


#if 0
    // This function interpolates its input to create a new, larger
    // image.
    template<ImageFormat InputFormat,
             ImageFormat OutputFormat,
             ImageFormat IntermediateFormat>
    Image<OutputFormat>
    supersample(const Image<InputFormat>& inputImage,
                typename ImageFormatTraits<OutputFormat>::PixelType,
                typename ImageFormatTraits<IntermediateFormat>::PixelType)
    {
      Image<OutputFormat> outputImage(inputImage.rows() * 2,
                                      inputImage.columns() * 2);
      ColorspaceConverter<InputFormat, IntermediateFormat> converter0;
      ColorspaceConverter<IntermediateFormat, OutputFormat> converter1;
      for(size_t inputRowIndex = 0; inputRowIndex < (inputImage.rows() - 1);
          ++inputRowIndex) {
        typename ImageFormatTraits<IntermediateFormat>::PixelType tempPixel;
        Array1D<typename Image<InputFormat>::PixelType> inputRow0 =
          inputImage.row(inputRowIndex);
        Array1D<typename Image<InputFormat>::PixelType> inputRow1 =
          inputImage.row(inputRowIndex + 1);
        Array1D<typename Image<OutputFormat>::PixelType> outputRow0 =
          outputImage.row(2 * inputRowIndex);
        Array1D<typename Image<OutputFormat>::PixelType> outputRow1 =
          outputImage.row(2 * inputRowIndex + 1);
        size_t outputColumnIndex = 0;
        for(size_t inputColumnIndex = 0;
            inputColumnIndex < (inputImage.columns() - 1);
            ++inputColumnIndex) {
          outputRow0[outputColumnIndex] = inputRow0[inputColumnIndex];
        
          tempPixel = converter0(inputRow0[inputColumnIndex]);
          tempPixel += converter0(inputRow0[inputColumnIndex + 1]);
          tempPixel /= 2;
          outputRow0[outputColumnIndex + 1] = converter1(tempPixel);

          tempPixel = converter0(inputRow0[inputColumnIndex]);
          tempPixel += converter0(inputRow1[inputColumnIndex]);
          tempPixel /= 2;
          outputRow1[outputColumnIndex] = converter1(tempPixel);

          tempPixel = converter0(inputRow0[inputColumnIndex]);
          tempPixel += converter0(inputRow0[inputColumnIndex + 1]);
          tempPixel += converter0(inputRow1[inputColumnIndex]);
          tempPixel += converter0(inputRow1[inputColumnIndex + 1]);
          tempPixel /= 4;
          outputRow1[outputColumnIndex + 1] = converter1(tempPixel);

          outputColumnIndex += 2;
        }
        // Handle pixels at end of row.
        outputRow0[outputColumnIndex] = inputRow0[inputRow0.size() - 1];
        outputRow0[outputColumnIndex + 1] = outputRow0[outputColumnIndex];

        tempPixel = converter0(inputRow0[inputRow0.size() - 1]);
        tempPixel += converter0(inputRow1[inputRow0.size() - 1]);
        tempPixel /= 2;
        outputRow1[outputColumnIndex] = converter1(tempPixel);
        outputRow1[outputColumnIndex + 1] = outputRow1[outputColumnIndex];
      }
      // Handle final rows.
      xxx;
      return outputImage;
    }
#endif
  

    // This function creates a new array and copies into it the pixel
    // data from the input image.
    template <class Type, ImageFormat FORMAT>
    Array2D<Type>
    toArray(const Image<FORMAT>& inputImage)
    {
      // A typedef for notational convenience.
      typedef typename ImageFormatTraits<FORMAT>::ComponentType ComponentType;

      size_t numberOfOutputColumns = inputImage.columns()
        * ImageFormatTraits<FORMAT>::getNumberOfComponents();

      Array2D<ComponentType> tempArray;
      dissociateColorComponents(
        const_cast< Image<FORMAT>& >(inputImage), tempArray);

      Array2D<Type> returnValue(inputImage.rows(), numberOfOutputColumns);
      returnValue.copy(tempArray);
      return returnValue;
    }
  
  } // namespace computerVision    
  
} // namespace dlr

#endif /* #ifndef _DLRCOMPUTERVISION_UTILITIES_H_ */
