/**
***************************************************************************
* @file cameraIntrinsicsPinholeTest.cpp
*
* Source file defining tests for the CameraIntrinsicsPinhole class.
*
* Copyright (C) 2006-2007 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
**/

#include <dlrCommon/functional.h>
#include <dlrComputerVision/cameraIntrinsicsPinhole.h>
#include <dlrTest/testFixture.h>

using namespace dlr::common;
using namespace dlr::computerVision;
using namespace dlr::geometry;
using namespace dlr::numeric;
using namespace dlr::test;


class CameraIntrinsicsPinholeTest
  : public TestFixture<CameraIntrinsicsPinholeTest> {

public:

  CameraIntrinsicsPinholeTest();
  ~CameraIntrinsicsPinholeTest() {}

  void setUp(const std::string& testName) {}
  void tearDown(const std::string& testName) {}

  // Tests.
  void testConstructor__void();
  void testConstructor__args();
  void testProject();
  void testReverseProject();

private:

  double m_defaultTolerance;
  
}; // class CameraIntrinsicsPinholeTest


/* ============== Member Function Definititions ============== */

CameraIntrinsicsPinholeTest::
CameraIntrinsicsPinholeTest()
  : TestFixture<CameraIntrinsicsPinholeTest>("CameraIntrinsicsPinholeTest"),
    m_defaultTolerance(1.0E-10)    
{
  DLR_TEST_REGISTER_MEMBER(testConstructor__void);
  DLR_TEST_REGISTER_MEMBER(testConstructor__args);
  DLR_TEST_REGISTER_MEMBER(testProject);
  DLR_TEST_REGISTER_MEMBER(testReverseProject);
}


void
CameraIntrinsicsPinholeTest::
testConstructor__void()
{
  // Pass.
}


void
CameraIntrinsicsPinholeTest::
testConstructor__args()
{
  // Tested in testReverseProject().
}


void
CameraIntrinsicsPinholeTest::
testProject()
{
  // Arbitrary camera params.
  size_t numPixelsX = 320;
  size_t numPixelsY = 240;
  double focalLength = 0.03;
  double pixelSizeX = 0.001;
  double pixelSizeY = 0.002;
  double centerU = 100.0;
  double centerV = 125.0;
  CameraIntrinsicsPinhole intrinsics(numPixelsX, numPixelsY,
                                     0.03, 0.001, 0.002, 100, 125);

  for(double zCoord = 1.0; zCoord < 10.0; zCoord += 0.7) {
    for(double yCoord = -1.0; yCoord < 1.0; yCoord += 0.1) {
      for(double xCoord = -1.0; xCoord < 1.0; xCoord += 0.1) {
        Vector3D cameraCoord(xCoord, yCoord, zCoord);
        Vector2D pixelCoord = intrinsics.project(cameraCoord);
        double referenceU =
          centerU + (xCoord * focalLength) / (pixelSizeX * zCoord);
        double referenceV =
          centerV + (yCoord * focalLength) / (pixelSizeY * zCoord);
        DLR_TEST_ASSERT(
          approximatelyEqual(pixelCoord.x(), referenceU, m_defaultTolerance));
        DLR_TEST_ASSERT(
          approximatelyEqual(pixelCoord.y(), referenceV, m_defaultTolerance));
      }
    }
  }
}


void
CameraIntrinsicsPinholeTest::
testReverseProject()
{
  // We test by round trip against convertWorldPointToPixel(), which
  // has its own independent test.

  // Arbitrary camera params.
  size_t numPixelsX = 320;
  size_t numPixelsY = 240;
  CameraIntrinsicsPinhole intrinsics(numPixelsX, numPixelsY,
                                     0.03, 0.001, 0.002, 100, 125);

  for(double vCoord = 0.0; vCoord < numPixelsY; vCoord += 1.2) {
    for(double uCoord = 0.0; uCoord < numPixelsX; uCoord += 1.2) {
      Vector2D pixelCoord(uCoord, vCoord);
      Ray3D ray = intrinsics.reverseProject(pixelCoord);
      Vector3D pointOnRay = ray.getOrigin() + 12.0 * ray.getDirectionVector();
      Vector2D recoveredPixelCoord = intrinsics.project(pointOnRay);
      DLR_TEST_ASSERT(approximatelyEqual(
                        recoveredPixelCoord.x(), pixelCoord.x(),
                        m_defaultTolerance));
      DLR_TEST_ASSERT(approximatelyEqual(
                        recoveredPixelCoord.y(), pixelCoord.y(),
                        m_defaultTolerance));
    }
  }
}


#if 0

int main(int argc, char** argv)
{
  CameraIntrinsicsPinholeTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  CameraIntrinsicsPinholeTest currentTest;

}

#endif
