/**
***************************************************************************
* @file dlrComputerVision/segmenterFelzenszwalb.h
*
* Header file declaring an image segmentation class.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef DLR_COMPUTERVISION_SEGMENTERFELZENSZWALB_H
#define DLR_COMPUTERVISION_SEGMENTERFELZENSZWALB_H

#include <vector>
#include <dlrComputerVision/disjointSet.h>
#include <dlrComputerVision/filter.h>
#include <dlrComputerVision/image.h>
#include <dlrComputerVision/kernels.h>
#include <dlrComputerVision/utilities.h>

namespace dlr {

  namespace computerVision {

    /**
     ** This class implements the image segmentation algorithm
     ** described [1].  Essentially grouping pixels based on local
     ** differences so that segmented regions have similar local
     ** variances.
     **
     ** [1] Felzenszwalb, P., and Huttenlocher, D., Efficient
     ** Graph-Based Image Segmentation, International Journal of
     ** Computer Vision, Volume 59, Number 2, September 2004.
     **/
    class SegmenterFelzenszwalb {
    public:

      struct Edge {
        size_t end0;
        size_t end1;
        float weight;
      };

      
      SegmenterFelzenszwalb(float k = 200.0f,
                            float sigma = 0.8f,
                            size_t minSegmentSize = 20);

      
      virtual
      ~SegmenterFelzenszwalb() {}


      virtual Array2D<UnsignedInt32>
      getLabelArray();

      
      virtual Array2D<UnsignedInt32>
      getLabelArray(UnsignedInt32& numberOfSegments,
                    std::vector<size_t>& segmentSizes);


      template <ImageFormat FORMAT>
      void
      segment(const Image<FORMAT>& inputImage);

    protected:

      typedef privateCode::DisjointSet<float> Segment;


      inline float
      getCost(const Segment& C_i, const Segment& C_j);


      std::vector<SegmenterFelzenszwalb::Edge>
      getEdges(const Image<GRAY_FLOAT32>& inImage);

      
      inline void
      setEdge(Edge& edge, size_t index0, size_t index1,
              Image<GRAY_FLOAT32> inImage);

      
      inline void
      updateCost(Segment& C_i, float weight);


      numeric::Index2D m_imageSize;
      float m_k;
      size_t m_minimumSegmentSize;
      numeric::Array1D<Segment> m_segmentation;
      float m_sigma;
      size_t m_smoothSize;
      
    };


    inline bool
    operator<(const SegmenterFelzenszwalb::Edge& arg0,
              const SegmenterFelzenszwalb::Edge& arg1) {
      return arg0.weight < arg1.weight;
    }
    
  } // namespace computerVision
  
} // namespace dlr


/* ============ Definitions of inline & template functions ============ */


#include <cmath>

namespace dlr {

  namespace computerVision {

    template <ImageFormat FORMAT>
    void
    SegmenterFelzenszwalb::
    segment(const Image<FORMAT>& inputImage)
    {
      m_imageSize.setValue(inputImage.rows(), inputImage.columns());
      
      // Smooth the image slightly to reduce artifacts.
      Image<GRAY_FLOAT32> smoothedImage;
      if(m_sigma == 0.0) {
        smoothedImage = convertColorspace<GRAY_FLOAT32>(inputImage);
      } else {
        Kernel<double> gaussian =
          getGaussianKernel<double>(m_smoothSize, m_smoothSize,
                                    m_sigma, m_sigma);
        smoothedImage = 
          filter2D<GRAY_FLOAT32, GRAY_FLOAT32>(
            gaussian, inputImage, Float32(0));
      }

      // Get a vector of the edges in the image, sorted in ascending
      // order.
      typedef std::vector<Edge>::iterator EdgeIter;
      std::vector<Edge> edges = this->getEdges(smoothedImage);
      std::sort(edges.begin(), edges.end());

      // Start with segmentation S^0, where every vertex is its own component.
      typedef numeric::Array1D<Segment>::iterator SegmentIter;
      m_segmentation.reinit(smoothedImage.size());
      for(SegmentIter segmentIter = m_segmentation.begin();
          segmentIter != m_segmentation.end(); ++segmentIter) {
        segmentIter->setPayload(m_k);        
      }
      
      // Iteratively merge segments, as described in the paper.
      EdgeIter edgeIter = edges.begin();
      while(edgeIter != edges.end()) {
        Segment& C_i = m_segmentation[edgeIter->end0].find();
        Segment& C_j = m_segmentation[edgeIter->end1].find();
        if(&C_i != &C_j) {
          float threshold = this->getCost(C_i, C_j);
          if(edgeIter->weight <= threshold) {
            C_i.merge(C_j);
            this->updateCost(C_i, edgeIter->weight);
          }
        }
        ++edgeIter;
      }

      // Merge any undersize segments, merging weak edges first.
      edgeIter = edges.begin();
      while(edgeIter != edges.end()) {
        Segment& C_i = m_segmentation[edgeIter->end0].find();
        Segment& C_j = m_segmentation[edgeIter->end1].find();
        if(C_i.getSize() < m_minimumSegmentSize
           || C_i.getSize() < m_minimumSegmentSize) {
          C_i.merge(C_j);
        }
        ++edgeIter;
      }
    
    }

    
    inline float
    SegmenterFelzenszwalb::
    getCost(const Segment& C_i, const Segment& C_j)
    {
      return std::min(C_i.getPayload(), C_j.getPayload());
    }


    inline void
    SegmenterFelzenszwalb::
    setEdge(Edge& edge, size_t index0, size_t index1,
            Image<GRAY_FLOAT32> inImage)
    {
      edge.end0 = index0;
      edge.end1 = index1;
      edge.weight = inImage[index1] - inImage[index0];
      if(edge.weight < 0.0) {
        edge.weight = -edge.weight;
      }
    }

  
    inline void
    SegmenterFelzenszwalb::
    updateCost(Segment& C_i, float weight)
    {
      Segment& head = C_i.find();
      head.setPayload(weight + m_k / head.getSize());
    }
  
  } // namespace computerVision

} // namespace dlr

#endif /* #ifndef DLR_COMPUTERVISION_SEGMENTERFELZENSZWALB_H */
