/**
***************************************************************************
* @file dlrComputerVision/segmenterFelzenszwalb.cpp
*
* Source file defining an image segmentation class.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#include <dlrComputerVision/segmenterFelzenszwalb.h>

namespace dlr {

  namespace computerVision {

    SegmenterFelzenszwalb::
    SegmenterFelzenszwalb(float k, float sigma, size_t minSegmentSize)
      : m_imageSize(0, 0),
        m_k(k),
        m_minimumSegmentSize(minSegmentSize),
        m_segmentation(),
        m_sigma(sigma),
        m_smoothSize(static_cast<size_t>(std::fabs(6 * sigma + 1)))
    {
      // Empty.
    }

    
    Array2D<UnsignedInt32>
    SegmenterFelzenszwalb::
    getLabelArray()
    {
      Array2D<UnsignedInt32> labelArray(
        m_imageSize.getRow(), m_imageSize.getColumn());

      numeric::Array1D<Segment>::iterator setIter = m_segmentation.begin();
      Array2D<UnsignedInt32>::iterator labelIter = labelArray.begin();
      while(setIter != m_segmentation.end()) {
        Segment& head = setIter->find();

        // Warning(xxx): Assuming we know something about how both
        // vectors and DisjointSets are implemented.
        UnsignedInt32 labelIndex = static_cast<UnsignedInt32>(
          &head - &(m_segmentation[0]));
        *labelIter = labelIndex;

        ++labelIter;
        ++setIter;
      }

      return labelArray;
    }

      
    Array2D<UnsignedInt32>
    SegmenterFelzenszwalb::
    getLabelArray(UnsignedInt32& numberOfSegments,
                  std::vector<size_t>& segmentSizes)
    {
      Array2D<UnsignedInt32> labelArray(
        m_imageSize.getRow(), m_imageSize.getColumn());
      std::vector<UnsignedInt32> labelMap(
        m_segmentation.size(), std::numeric_limits<UnsignedInt32>::max());
      UnsignedInt32 currentLabel = 0;
      segmentSizes.clear();
      
      // Iterate over each pixel.
      numeric::Array1D<Segment>::iterator setIter = m_segmentation.begin();
      Array2D<UnsignedInt32>::iterator labelIter = labelArray.begin();
      while(setIter != m_segmentation.end()) {

        // Figure out to which segment the current pixel belongs.
        // 
        // Warning(xxx): Assuming we know something about how both
        // vectors and DisjointSets are implemented.
        Segment& head = setIter->find();
        UnsignedInt32 labelIndex = static_cast<UnsignedInt32>(
          &head - &(m_segmentation[0]));

        // Have we labeled this segment yet?
        if(labelMap[labelIndex] > currentLabel) {
          // No.  Label it now and remember how big the segment is.
          labelMap[labelIndex] = currentLabel;
          segmentSizes.push_back(head.getSize());
          ++currentLabel;
        }
        // Record the label in our output label image.
        *labelIter = labelMap[labelIndex];

        // Move on to next pixel.
        ++labelIter;
        ++setIter;
      }

      numberOfSegments = currentLabel;
      return labelArray;
    }


    std::vector<SegmenterFelzenszwalb::Edge>
    SegmenterFelzenszwalb::
    getEdges(const Image<GRAY_FLOAT32>& inImage)
    {
      // 8-connected means 4 undirected edges per pixel.
      size_t numEdges = 4 * inImage.size();

      // Except that some point off the side of the image.
      numEdges -= 3 * inImage.rows();

      // And some point off the bottom/top of the image.
      numEdges -= 3 * inImage.columns();

      // Oops! We counted the bottom-right and bottom-left corners twice.
      numEdges += 2;

      std::vector<Edge> edges(numEdges);
      size_t interiorRows = inImage.rows() - 1;
      size_t interiorColumns = inImage.columns() - 1;
      size_t edgeNumber = 0;
      size_t pixelIndex0 = 0;
      size_t pixelIndex1;
      for(size_t row = 0; row < interiorRows; ++row) {
        // Get the first pixel of the row;
        pixelIndex1 = pixelIndex0 + 1;
        this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);
      
        pixelIndex1 += inImage.columns();
        this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);

        pixelIndex1 -= 1;
        this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);
        ++pixelIndex0;
      
        // Get the interior pixels of the row.
        for(size_t column = 1; column < interiorColumns; ++column) {
          pixelIndex1 = pixelIndex0 + 1;
          this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);

          pixelIndex1 += inImage.columns();
          this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);

          pixelIndex1 -= 1;
          this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);

          pixelIndex1 -= 1;
          this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);
          ++pixelIndex0;
        }

        // Get the last pixel of the row.
        pixelIndex1 = pixelIndex0 + inImage.columns();
        this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);

        pixelIndex1 -= 1;
        this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);
        ++pixelIndex0;
      }

      // Sanity check.
      if(pixelIndex0 != (inImage.rows() - 1) * inImage.columns()) {
        DLR_THROW(LogicException, "SegmenterFelzenszwalb::getEdges()",
                  "Indexing error.");
      }
    
      // Get the last row of pixels;
      for(size_t column = 0; column < interiorColumns; ++column) {
        pixelIndex1 = pixelIndex0 + 1;
        this->setEdge(edges[edgeNumber++], pixelIndex0, pixelIndex1, inImage);
        ++pixelIndex0;
      }

      // Sanity check.
      if(edgeNumber != numEdges) {
        DLR_THROW(LogicException, "SegmenterFelzenszwalb::getEdges()",
                  "Edge counting error.");
      }
      return edges;
    }

  } // namespace computerVision
  
} // namespace dlr
