/**
***************************************************************************
* @file dlrComputerVision/disjointSet.h
*
* Header file declaring a class to be used in implementing algorithms
* that need a "forest of disjoint sets" data structure.
*
* Copyright (C) 2008 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef DLR_COMPUTERVISION_DISJOINTSET_H
#define DLR_COMPUTERVISION_DISJOINTSET_H


// xxx
#include <iostream>

namespace dlr {

  namespace computerVision {

    namespace privateCode {
      
      /**
       ** This class implements one tree in the "forest of disjoint
       ** sets" data structure first described by Bernard Galler and
       ** Michael Fischer[1].  This implementation includes union by
       ** rank and path compression, yielding "nearly" constant time
       ** amortized complexity for find/merge operations.
       **
       ** Our implementation includes a size member that allows the
       ** calling context to easily see how many members belong to the
       ** set.  This adds a small constant to the execution time of the
       ** merge() operation.
       **
       ** To make a forest of disjoint sets, you'll need to put a
       ** bunch of DisjointSet instances in some kind of container,
       ** such as an Array1D instance
       ** (dlr::numeric::Array1D<DisjointSet>).  Unfortunately, most
       ** of the C++ standard library con, such as std::vector, use
       ** the copy constructor to fill in the array, which breaks the
       ** parent pointers.  So don't use a std::vector (or std::list,
       ** or...).
       **
       ** An Improved Equivalence Algorithm.  Galler, Bernard, and
       ** Fischer, Michael. Communnications of the ACM, Volume 7, pages
       ** 301-303, 1964.
       **/
      template <class Type>
      class DisjointSet {
      public:

        /** 
         * The default creates a leaf in a disjoint set tree.
         */
        DisjointSet()
          : m_parentPtr(this), m_rank(0), m_size(1), m_payload() {
          if(m_parentPtr != this) {
            std::cout << "Ack!!" << std::endl;
          }
        }


        /** 
         * This constructor allows you to explicitly set the payload
         * values.
         * 
         * @param payload This argument will be copied into the
         * payload, and then ignored until you call
         * this->getPayload().
         */
        DisjointSet(const Type& payload)
          : m_parentPtr(this), m_rank(0), m_size(1), m_payload(payload) {};


        /**
         * Destructor.
         */
        virtual
        ~DisjointSet() {}

#if 0

        /** 
         * This member function returns a reference to the head of the
         * set to which *this belongs.  Although which member of a set
         * gets to be "head" is a little arbitrary, all members of the
         * set will report the same head (until the set is merged with
         * another set, when the head may change).  This implementation
         * is taken directly from the Wikipedia page on disjoint sets,
         * and uses recursion, which may incur a performance penalty.
         * 
         * @return The return value is the head of the set.
         */
        DisjointSet&
        find() {
          if(m_parentPtr == this) {
            return *this;
          }
          m_parentPtr = &(m_parentPtr->find());
          return *m_parentPtr;
        }

#elif 1
      
        /** 
         * This member function returns a reference to the head of the
         * set to which *this belongs.  Although which member of a set
         * gets to be "head" is a little arbitrary, all members of the
         * set will report the same head (until the set is merged with
         * another set, when the head may change).  This implementation
         * does not use recursion, and may be faster...
         * 
         * @return The return value is the head of the set.
         */
        DisjointSet&
        find() {
          // Follow parent pointers to find the head of the tree.
          DisjointSet* headPtr = this;
          while(headPtr->m_parentPtr != headPtr) {
            headPtr = headPtr->m_parentPtr;
          }

          // Now do the same thing again, but this time update all of
          // the parent pointers to point directly at the head.
          DisjointSet* updatePtr = this;
          while(updatePtr->m_parentPtr != updatePtr) {
            DisjointSet* tmpPtr = updatePtr->m_parentPtr;
            updatePtr->m_parentPtr = headPtr;
            updatePtr = tmpPtr;
          }
          return *m_parentPtr;
        }

#else
      
        /** 
         * This member function implements a crippled version of find
         * that doesn't fully update parent pointers.  I saw in Pedro
         * Felzenszwalb's segementation example code[2].  It turns out
         * to be a little faster for that particular application,
         * although I haven't figured out why yet.
         *
         * [2] Efficient Graph-Based Image Segmentation.  Pedro
         * F. Felzenszwalb and Daniel P. Huttenlocher. International
         * Journal of Computer Vision, Volume 59, Number 2, September
         * 2004.
         *
         * @return The return value is a reference to the head of the
         * tree to which *this belongs.
         */
        DisjointSet&
        find() {
          DisjointSet* headPtr = this;
          while(headPtr->m_parentPtr != headPtr) {
            headPtr = headPtr->m_parentPtr;
          }
          m_parentPtr = headPtr;
          return *m_parentPtr;
        }

#endif
      
        /** 
         * This member function merges two sets.  It doesn't matter
         * whether you call x.merge(y), or y.merge(x); both will get the
         * job done.  After merging, all members of both sets will
         * report the same head, which will be the head reported one of
         * the two sets before merging.  The head reported by the other
         * of the two sets before merging will stop being a head.
         * 
         * @param other This argument is the set with which to merge.
         */
        void
        merge(DisjointSet& other) {
          DisjointSet& myParent = this->find();
          DisjointSet& otherParent = other.find();
          if(&myParent == &otherParent) {
            return;
          }
          if(myParent.m_rank > otherParent.m_rank) {
            otherParent.m_parentPtr = &myParent;
            myParent.m_size += otherParent.m_size;
          } else if(myParent.m_rank < otherParent.m_rank) {
            myParent.m_parentPtr = &otherParent;
            otherParent.m_size += myParent.m_size;
          } else {
            otherParent.m_parentPtr = &myParent;
            myParent.m_size += otherParent.m_size;
            ++(myParent.m_rank);
          }
        }
      

        const Type&
        getPayload() const {return m_payload;}
        

        /** 
         * This member function returns the number of member in the set
         * of which *this is a member.
         * 
         * @return The return value is a count of the number of members
         * in the set.
         */
        size_t
        getSize() {return (this->find()).m_size;}


        void
        setPayload(const Type& payload) {m_payload = payload;}
        
      protected:

        DisjointSet* m_parentPtr;
        size_t m_rank;
        size_t m_size;

        Type m_payload;
      };
      
    } // namespace privateCode
    
  } // namespace computerVision
  
} // namespace dlr


/* ============ Definitions of inline & template functions ============ */


namespace dlr {

  namespace computerVision {


  } // namespace computerVision
  
} // namespace dlr

#endif /* #ifndef DLR_COMPUTERVISION_DISJOINTSET_H */
