/**
***************************************************************************
* @file filterTest.cpp
*
* Source file defining tests for ColorspaceConverter classes.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 915 $
* $Date: 2007-05-18 01:27:47 -0400 (Fri, 18 May 2007) $
***************************************************************************
**/

#include <dlrComputerVision/image.h>
#include <dlrComputerVision/colorspaceConverter.h>
#include <dlrComputerVision/pixelRGB.h>
#include <dlrTest/testFixture.h>

namespace dlr {

  namespace computerVision {

    class ColorspaceConverterTest
      : public TestFixture<ColorspaceConverterTest> {

    public:

      ColorspaceConverterTest();
      ~ColorspaceConverterTest() {}

      void setUp(const std::string& testName) {}
      void tearDown(const std::string& testName) {}

      // Tests.
      void testRGB8ToGRAY8();
      void testRGB8ToBGRA8();
      void testRGB8ToRGBA8();
      void testBGRA8ToRGB8();
      void testRGBA8ToRGB8();

    private:

    }; // class ColorspaceConverterTest


    /* ============== Member Function Definititions ============== */

    ColorspaceConverterTest::
    ColorspaceConverterTest()
      : TestFixture<ColorspaceConverterTest>("ColorspaceConverterTest")
    {
      DLR_TEST_REGISTER_MEMBER(testRGB8ToGRAY8);
      DLR_TEST_REGISTER_MEMBER(testRGB8ToBGRA8);
      DLR_TEST_REGISTER_MEMBER(testRGB8ToRGBA8);
      DLR_TEST_REGISTER_MEMBER(testBGRA8ToRGB8);
      DLR_TEST_REGISTER_MEMBER(testRGBA8ToRGB8);
    }


    void
    ColorspaceConverterTest::
    testRGB8ToGRAY8()
    {
      ColorspaceConverter<RGB8, GRAY8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            double redDbl = redValue;
            double greenDbl = greenValue;
            double blueDbl = blueValue;
            double grayDbl = std::sqrt(
              (redDbl * redDbl + greenDbl * greenDbl + blueDbl * blueDbl) / 3.0);
            UnsignedInt8 grayValue = static_cast<UnsignedInt8>(grayDbl + 0.5);

            PixelRGB8 inputPixel(static_cast<UnsignedInt8>(redValue),
								 static_cast<UnsignedInt8>(greenValue), 
								 static_cast<UnsignedInt8>(blueValue));
            DLR_TEST_ASSERT(converter(inputPixel) == grayValue);
          }
        }
      }
      // Special case tests follow.
      PixelRGB8 inputPixel(255, 255, 255);
      DLR_TEST_ASSERT(converter(inputPixel) == 255);
    }


    void
    ColorspaceConverterTest::
    testRGB8ToBGRA8()
    {
      ColorspaceConverter<RGB8, BGRA8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            PixelRGB8 inputPixel(
              static_cast<UnsignedInt8>(redValue),
              static_cast<UnsignedInt8>(greenValue),
              static_cast<UnsignedInt8>(blueValue));
            PixelBGRA8 referencePixel(
              static_cast<UnsignedInt8>(blueValue),
              static_cast<UnsignedInt8>(greenValue),
              static_cast<UnsignedInt8>(redValue),
              255);
            DLR_TEST_ASSERT(converter(inputPixel) == referencePixel);
          }
        }
      }
    }

 
    void
    ColorspaceConverterTest::
    testRGB8ToRGBA8()
    {
      ColorspaceConverter<RGB8, RGBA8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            PixelRGB8 inputPixel(
              static_cast<UnsignedInt8>(redValue),
              static_cast<UnsignedInt8>(greenValue),
              static_cast<UnsignedInt8>(blueValue));
            PixelRGBA8 referencePixel(
              static_cast<UnsignedInt8>(redValue),
              static_cast<UnsignedInt8>(greenValue),
              static_cast<UnsignedInt8>(blueValue),
              255);
            DLR_TEST_ASSERT(converter(inputPixel) == referencePixel);
          }
        }
      }
    }

 
    void
    ColorspaceConverterTest::
    testBGRA8ToRGB8()
    {
      ColorspaceConverter<BGRA8, RGB8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            for(UnsignedInt16 alphaValue = 0; alphaValue < 256;
                alphaValue += 7) {
              PixelBGRA8 inputPixel(
                static_cast<UnsignedInt8>(blueValue),
                static_cast<UnsignedInt8>(greenValue),
                static_cast<UnsignedInt8>(redValue),
                static_cast<UnsignedInt8>(alphaValue));
              PixelRGB8 referencePixel(
                static_cast<UnsignedInt8>(redValue),
                static_cast<UnsignedInt8>(greenValue),
                static_cast<UnsignedInt8>(blueValue));
              DLR_TEST_ASSERT(converter(inputPixel) == referencePixel);
            }
          }
        }
      }
    }
  
    void
    ColorspaceConverterTest::
    testRGBA8ToRGB8()
    {
      ColorspaceConverter<RGBA8, RGB8> converter;
      for(UnsignedInt16 redValue = 0; redValue < 256; redValue += 7) {
        for(UnsignedInt16 greenValue = 0; greenValue < 256; greenValue += 7) {
          for(UnsignedInt16 blueValue = 0; blueValue < 256; blueValue += 7) {
            for(UnsignedInt16 alphaValue = 0; alphaValue < 256;
                alphaValue += 7) {
              PixelRGBA8 inputPixel(
                static_cast<UnsignedInt8>(redValue),
                static_cast<UnsignedInt8>(greenValue),
                static_cast<UnsignedInt8>(blueValue),
                static_cast<UnsignedInt8>(alphaValue));
              PixelRGB8 referencePixel(
                static_cast<UnsignedInt8>(redValue),
                static_cast<UnsignedInt8>(greenValue),
                static_cast<UnsignedInt8>(blueValue));
              DLR_TEST_ASSERT(converter(inputPixel) == referencePixel);
            }
          }
        }
      }
    }
  
  } // namespace computerVision

} // namespace dlr


#if 0

int main(int argc, char** argv)
{
  dlr::computerVision::ColorspaceConverterTest currentTest;
  bool result = currentTest.run();
  return (result ? 0 : 1);
}

#else

namespace {

  dlr::computerVision::ColorspaceConverterTest currentTest;

}

#endif
