/**
***************************************************************************
* @file dlrComputerVision/sobel.h
*
* Header file declaring some routines to compute sobel edge images.
*
* Copyright (C) 2006 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: $
* $Date: $
***************************************************************************
*/

#ifndef _DLRCOMPUTERVISION_SOBEL_H_
#define _DLRCOMPUTERVISION_SOBEL_H_

#include <dlrComputerVision/image.h>

namespace dlr {

  namespace computerVision {
    

    /** 
     * This function applies the sobel edge operator in the X
     * direction.  Specifically, convolves the image with the 3x3 kernel
     *
     * @code
     *   [[-1, 0, 1],
     *    [-2, 0, 2],
     *    [-1, 0, 1]]
     * @endcode
     *
     * and then optionally rescales the result to be proportional to
     * the image gradient with scale factor 1.0.
     *
     * @param inputImage This argument is the image to be convolved.
     * 
     * @param normalizeResult This argument specifies whether or not
     * to divide the resulting pixel values by 8.  Setting this
     * argument to true will result in lost precision on integer
     * types.  This feature is currently not implemented, so please
     * leave normalizeResult at its default value of false.
     * 
     * @return The return value is the result of the convolution.
     */
    template <ImageFormat FORMAT>
    Image<FORMAT>
    applySobelX(const Image<FORMAT>& inputImage, bool normalizeResult=false);


    /** 
     * This function applies the sobel edge operator in the Y
     * direction.  Specifically, convolves the image with the 3x3 kernel
     *
     * @code
     *   [[-1, -2, -1],
     *    [ 0,  0,  0],
     *    [ 1,  2,  1]]
     * @endcode
     *
     * and then optionally rescales the result to be proportional to
     * the image gradient with scale factor 1.0.
     *
     * @param inputImage This argument is the image to be convolved.
     * 
     * @param normalizeResult This argument specifies whether or not
     * to divide the resulting pixel values by 8.  Setting this
     * argument to true will result in lost precision on integer
     * types.  This feature is currently not implemented, so please
     * leave normalizeResult at its default value of false.
     * 
     * @return The return value is the result of the convolution.
     */
    template <ImageFormat FORMAT>
    Image<FORMAT>
    applySobelY(const Image<FORMAT>& inputImage, bool normalizeResult=false);

  } // namespace computerVision

} // namespace dlr


/* =============== Implementation follows =============== */


namespace dlr {

  namespace computerVision {

    // This function applies the sobel edge operator in the X
    // direction.
    template <ImageFormat FORMAT>
    Image<FORMAT>
    applySobelX(const Image<FORMAT>& inputImage, bool normalizeResult)
    {
      // Argument checking.
      if(normalizeResult == true) {
        DLR_THROW(NotImplementedException, "applySobelX()",
                  "Argument normalizeResult must be false for now.");
      }
      if(inputImage.rows() < 2 || inputImage.columns() < 2) {
        DLR_THROW(ValueException, "applySobelX()",
                  "Argument inputImage must be 2x2 or larger.");
      }

      // Prepare a space for the result.
      Image<FORMAT> gradientImage(inputImage.rows(), inputImage.columns());

      // We'll keep a running index to avoid double-indexing the images.
      size_t index0 = 0;
      const size_t rows = inputImage.rows();
      const size_t cols = inputImage.columns();
      const size_t colsMinusOne = cols - 1;

      // NOTE: In all of the code below, We assume that the image
      // continues with constant first derivative into the (fictional)
      // out-of-bounds rows and columns.  This has the somewhat
      // surprising effect of making the first row of convolution
      // results independent of the second image row.
      
      // Use a very reduced kernel for the upper-left corner.
      gradientImage[index0] = (
        8 * (inputImage[index0 + 1] - inputImage[index0]));
      ++index0;
      
      // Use a reduced kernel for the first row.
      for(size_t column = 1; column < colsMinusOne; ++column) {
        gradientImage[index0] = (
          4 * (inputImage[index0 + 1] - inputImage[index0 - 1]));
        ++index0;
      }

      // Use a very reduced kernel for the upper-right corner.
      gradientImage[index0] = (
        8 * (inputImage[index0] - inputImage[index0 - 1]));
      ++index0;
      
      // Convolve the bulk of the image.
      for(size_t row = 1; row < rows - 1; ++row) {

        // Use a reduced kernel for first pixel in the row.
        gradientImage[index0] = (
          2 * (inputImage[index0 - cols + 1] - inputImage[index0 - cols])
          + 4 * (inputImage[index0 + 1] - inputImage[index0])
          + 2 * (inputImage[index0 + cols + 1] - inputImage[index0 + cols]));
        ++index0;
        
        // Convolve the bulk of the row.
        for(size_t column = 1; column < colsMinusOne; ++column) {
          gradientImage[index0] = (
            (inputImage[index0 - cols + 1] - inputImage[index0 - cols - 1])
            + 2 * (inputImage[index0 + 1] - inputImage[index0 - 1])
            + (inputImage[index0 + cols + 1] - inputImage[index0 + cols - 1]));
          ++index0;
        }

        // Use a reduced kernel for last pixel in the row.
        gradientImage[index0] = (
          2 * (inputImage[index0 - cols] - inputImage[index0 - cols - 1])
          + 4 * (inputImage[index0] - inputImage[index0 - 1])
          + 2 * (inputImage[index0 + cols] - inputImage[index0 + cols - 1]));
        ++index0;
      }

      // Use a very reduced kernel for the lower-left corner.
      gradientImage[index0] = (
        8 * (inputImage[index0 + 1] - inputImage[index0]));
      ++index0;
      
      // Use a reduced kernel for the last row.
      for(size_t column = 1; column < colsMinusOne; ++column) {
        gradientImage[index0] = (
          4 * (inputImage[index0 + 1] - inputImage[index0 - 1]));
        ++index0;
      }

      // Use a very reduced kernel for the lower-right corner.
      gradientImage[index0] = (
        8 * (inputImage[index0] - inputImage[index0 - 1]));
      ++index0;

      return gradientImage;
    }


    // This function applies the sobel edge operator in the Y
    // direction.
    template <ImageFormat FORMAT>
    Image<FORMAT>
    applySobelY(const Image<FORMAT>& inputImage, bool normalizeResult)
    {
      // Argument checking.
      if(normalizeResult == true) {
        DLR_THROW(NotImplementedException, "applySobelY()",
                  "Argument normalizeResult must be false for now.");
      }
      if(inputImage.rows() < 2 || inputImage.columns() < 2) {
        DLR_THROW(ValueException, "applySobelY()",
                  "Argument inputImage must be 2x2 or larger.");
      }

      // Prepare a space for the result.
      Image<FORMAT> gradientImage(inputImage.rows(), inputImage.columns());

      // We'll keep a running index to avoid double-indexing the images.
      size_t index0 = 0;
      const size_t rows = inputImage.rows();
      const size_t cols = inputImage.columns();
      const size_t colsMinusOne = cols - 1;

      // NOTE: In all of the code below, We assume that the image
      // continues with constant first derivative into the (fictional)
      // out-of-bounds rows and columns.  This has the somewhat
      // surprising effect of making the first row of convolution
      // results independent of the second image row.
      
      // Use a very reduced kernel for the upper-left corner.
      gradientImage[index0] = (
        8 * (inputImage[index0 + cols] - inputImage[index0]));
      ++index0;
      
      // Use a reduced kernel for the first row.
      for(size_t column = 1; column < colsMinusOne; ++column) {
        gradientImage[index0] = (
          2 * (inputImage[index0 + cols - 1] - inputImage[index0 - 1])
          + 4 * (inputImage[index0 + cols] - inputImage[index0])
          + 2 * (inputImage[index0 + cols + 1] - inputImage[index0 + 1]));
        ++index0;
      }

      // Use a very reduced kernel for the upper-right corner.
      gradientImage[index0] = (
        8 * (inputImage[index0 + cols] - inputImage[index0]));
      ++index0;
      
      // Convolve the bulk of the image.
      for(size_t row = 1; row < rows - 1; ++row) {

        // Use a reduced kernel for first pixel in the row.
        gradientImage[index0] = (
          4 * (inputImage[index0 + cols] - inputImage[index0 - cols]));
        ++index0;
        
        // Convolve the bulk of the row.
        for(size_t column = 1; column < colsMinusOne; ++column) {
          gradientImage[index0] = (
            (inputImage[index0 + cols - 1] - inputImage[index0 - cols - 1])
            + 2 * (inputImage[index0 + cols] - inputImage[index0 - cols])
            + (inputImage[index0 + cols + 1] - inputImage[index0 - cols + 1]));
          ++index0;
        }

        // Use a reduced kernel for last pixel in the row.
        gradientImage[index0] = (
          4 * (inputImage[index0 + cols] - inputImage[index0 - cols]));
        ++index0;
      }

      // Use a very reduced kernel for the lower-left corner.
      gradientImage[index0] = (
        8 * (inputImage[index0] - inputImage[index0 - cols]));
      ++index0;
      
      // Use a reduced kernel for the last row.
      for(size_t column = 1; column < colsMinusOne; ++column) {
        gradientImage[index0] = (
          2 * (inputImage[index0 - 1] - inputImage[index0 - cols - 1])
          + 4 * (inputImage[index0] - inputImage[index0 - cols])
          + 2 * (inputImage[index0 + 1] - inputImage[index0 - cols + 1]));
        ++index0;
      }

      // Use a very reduced kernel for the lower-right corner.
      gradientImage[index0] = (
        8 * (inputImage[index0] - inputImage[index0 - cols]));
      ++index0;

      return gradientImage;
    }


  } // namespace computerVision
  
} // namespace dlr

#endif /* #ifndef _DLRCOMPUTERVISION_KERNEL_H_ */
