/**
***************************************************************************
* @file dlrComputerVision/imageIO.cpp
*
* Source file defining functions for reading and writing images.
*
* Copyright (C) 2005 David LaRose, dlr@cs.cmu.edu
* See accompanying file, LICENSE.TXT, for details.
*
* $Revision: 878 $
* $Date: 2007-05-04 00:17:41 -0400 (Fri, 04 May 2007) $
***************************************************************************
*/

#include <fstream>
#include <dlrCommon/byteOrder.h>
#include <dlrComputerVision/imageIO.h>

namespace {

  void
  discardComments(std::istream& inputStream,
                  std::istream::char_type commentCharacter)
  {
    std::istream::char_type inputCharacter;
    while(1) {
      inputStream >> inputCharacter;
      if(inputCharacter == commentCharacter) {
        std::string dummyString;
        getline(inputStream, dummyString);
      } else {
        inputStream.putback(inputCharacter);
        break;
      }
    }
  }


  std::string
  readComments(std::istream& inputStream,
               std::istream::char_type commentCharacter)
  {
    std::istream::char_type inputCharacter;
    std::ostringstream commentStream;
    while(1) {
      inputStream >> inputCharacter;
      if(inputCharacter == commentCharacter) {
        std::string dummyString;
        getline(inputStream, dummyString);
        commentStream << dummyString;
      } else {
        inputStream.putback(inputCharacter);
        break;
      }
    }
    return commentStream.str();
  }

} // Anonymous namespace


namespace dlr {

  namespace computerVision {
    
    Image<GRAY8>
    readPGM8(const std::string& fileName)
    {
      std::string commentString;
      return readPGM8(fileName, commentString);
    }

  
    Image<GRAY8>
    readPGM8(const std::string& fileName, std::string& commentString)
    {
      std::ifstream inputStream(fileName.c_str(), std::ios::binary);
      if(!inputStream) {
        std::ostringstream message;
        message << "Couldn't open input file: " << fileName;
        DLR_THROW(IOException, "readPGM8()", message.str().c_str());
      }

      // Read the header.
      commentString.clear();
      std::string magic;
      size_t columns;
      size_t rows;
      long long int imageMax;
      inputStream >> magic;
      commentString += readComments(inputStream, '#');
      inputStream >> columns >> rows;
      commentString += readComments(inputStream, '#');
      inputStream >> imageMax;

      // Image data starts after the next newline.
      std::string dummy;
      std::getline(inputStream, dummy);

      // Check for errors.
      if(!inputStream) {
        std::ostringstream message;
        message << "Couldn't read image header from file: " << fileName;
        DLR_THROW(IOException, "readPGM8()", message.str().c_str());
      }

      // Check that this image will fit in an 8-bit pixel array.
      if(imageMax > 255LL) {
        std::ostringstream message;
        message << "File " << fileName << " has max value of " << imageMax
                << ", which is too big for an 8-bit image.";
        DLR_THROW(IOException, "readPGM8()", message.str().c_str());
      }

      // Allocate storage space.
      Image<GRAY8> newImage(rows, columns);

      // And read the pixels.
      if(magic == "P5") {
        // Looks like a raw image.
        inputStream.read(reinterpret_cast<char*>(newImage.data()),
                         newImage.size());
      } else if(magic == "P2") {
        // Looks like a plain image.
        unsigned short interpreter;
        for(size_t pixelIndex = 0; pixelIndex < newImage.size(); ++pixelIndex) {
          // We can't simply read into the image, since the compiler
          // will assume we want the ascii values of the characters in
          // the file.
          inputStream >> interpreter;
          newImage(pixelIndex) =
            static_cast<Image<GRAY8>::PixelType>(interpreter);
        }
      } else {
        std::ostringstream message;
        message << "Incorrect magic, " << magic << " in file " << fileName
                << ".";
        DLR_THROW(IOException, "readPGM8()", message.str().c_str());
      }

      // Check for errors.
      if(!inputStream) {
        std::ostringstream message;
        message << "Error reading image data from input file: " << fileName;
        DLR_THROW(IOException, "readPGM8()", message.str().c_str());
      }
    
      // All done!
      inputStream.close();
      return newImage;
    }


    Image<GRAY16>
    readPGM16(const std::string& fileName)
    {
      std::ifstream inputStream(fileName.c_str(), std::ios::binary);
      if(!inputStream) {
        std::ostringstream message;
        message << "Couldn't open input file: " << fileName;
        DLR_THROW(IOException, "readPGM16()", message.str().c_str());
      }

      // Read the header.
      std::string magic;
      size_t columns;
      size_t rows;
      long long int imageMax;
      inputStream >> magic;
      discardComments(inputStream, '#');
      inputStream >> columns >> rows;
      discardComments(inputStream, '#');
      inputStream >> imageMax;

      // Image data starts after the next newline.
      std::string dummy;
      std::getline(inputStream, dummy);

      // Check for errors.
      if(!inputStream) {
        std::ostringstream message;
        message << "Couldn't read image header from file: " << fileName;
        DLR_THROW(IOException, "readPGM16()", message.str().c_str());
      }

      // Check that this image will fit in an 16-bit pixel array.
      if(imageMax > 65535LL) {
        std::ostringstream message;
        message << "File " << fileName << " has max value of " << imageMax
                << ", which is too big for an 16-bit image.";
        DLR_THROW(IOException, "readPGM16()", message.str().c_str());
      }

      // Allocate storage space.
      Image<GRAY16> newImage(rows, columns);

      // And read the pixels.
      if(magic == "P5") {
        // Looks like a raw image.
        if(imageMax <= 255LL) {
          // Looks like an 8 bit image.
          DLR_THROW(NotImplementedException, "readPGM16()",
                    "This routine currently cannot read 8-bit PGMs");
        } else {
          size_t numberOfBytes = 2 * newImage.size();
          inputStream.read(
            reinterpret_cast<char*>(newImage.data()), numberOfBytes);
          switchByteOrder(newImage.data(), newImage.size(),
                          DLR_BIG_ENDIAN, getByteOrder());
        }
      } else if(magic == "P2") {
        // Looks like a plain image.
        unsigned short interpreter;
        for(size_t pixelIndex = 0; pixelIndex < newImage.size(); ++pixelIndex) {
          // We can't simply read into the image, since the compiler
          // will assume we want the ascii values of the characters in
          // the file.
          inputStream >> interpreter;
          newImage(pixelIndex) =
            static_cast<Image<GRAY16>::PixelType>(interpreter);
        }
      } else {
        std::ostringstream message;
        message << "Incorrect magic, " << magic << " in file " << fileName
                << ".";
        DLR_THROW(IOException, "readPGM16()", message.str().c_str());
      }

      // Check for errors.
      if(!inputStream) {
        std::ostringstream message;
        message << "Error reading image data from input file: " << fileName;
        DLR_THROW(IOException, "readPGM16()", message.str().c_str());
      }
    
      // All done!
      inputStream.close();
      return newImage;
    }


    Image<RGB8>
    readPPM8(const std::string& fileName)
    {
      std::ifstream inputStream(fileName.c_str(), std::ios::binary);
      if(!inputStream) {
        std::ostringstream message;
        message << "Couldn't open input file: " << fileName;
        DLR_THROW(IOException, "readPPM8()", message.str().c_str());
      }

      // Read the header.
      std::string magic;
      size_t columns;
      size_t rows;
      long long int imageMax;
      inputStream >> magic;
      discardComments(inputStream, '#');
      inputStream >> columns >> rows;
      discardComments(inputStream, '#');
      inputStream >> imageMax;

      // Image data starts after the next newline.
      std::string dummy;
      std::getline(inputStream, dummy);

      // Check for errors.
      if(!inputStream) {
        std::ostringstream message;
        message << "Couldn't read image header from file: " << fileName;
        DLR_THROW(IOException, "readPPM8()", message.str().c_str());
      }

      // Check that this image will fit in an 8-bit pixel array.
      if(imageMax > 255LL) {
        std::ostringstream message;
        message << "File " << fileName << " has max value of " << imageMax
                << ", which is too big for an 8-bit image.";
        DLR_THROW(IOException, "readPPM8()", message.str().c_str());
      }

      // Allocate storage space.
      Image<RGB8> newImage(rows, columns);

      // And read the pixels.
      if(magic == "P6") {
        // Looks like a raw image.
        inputStream.read(reinterpret_cast<char*>(newImage.data()),
                         newImage.size() * 3);
      } else if(magic == "P3") {
        // Looks like a plain image.
        unsigned short interpreterRed;
        unsigned short interpreterGreen;
        unsigned short interpreterBlue;
        for(size_t pixelIndex = 0; pixelIndex < newImage.size(); ++pixelIndex) {
          // We can't simply read into the image, since the compiler
          // will assume we want the ascii values of the characters in
          // the file.
          inputStream >> interpreterRed >> interpreterGreen >> interpreterBlue;
          newImage(pixelIndex).red =
            static_cast<UnsignedInt8>(interpreterRed);
          newImage(pixelIndex).green =
            static_cast<UnsignedInt8>(interpreterGreen);
          newImage(pixelIndex).blue =
            static_cast<UnsignedInt8>(interpreterBlue);
        }
      } else {
        std::ostringstream message;
        message << "Incorrect magic, " << magic << " in file " << fileName
                << ".";
        DLR_THROW(IOException, "readPPM8()", message.str().c_str());
      }

      // Check for errors.
      if(!inputStream) {
        std::ostringstream message;
        message << "Error reading image data from input file: " << fileName;
        DLR_THROW(IOException, "readPPM8()", message.str().c_str());
      }
    
      // All done!
      inputStream.close();
      return newImage;
    }


    void
    writePGM8(const std::string& fileName,
              const Image<GRAY8>& outputImage,
              const std::string& comment)
    {
      std::ofstream outputStream(fileName.c_str(), std::ios::binary);
      if(!outputStream) {
        std::ostringstream message;
        message << "Couldn't open output file: " << fileName;
        DLR_THROW(IOException, "writePGM8()", message.str().c_str());
      }

      // Write the header.
      outputStream << "P5\n";
      if(comment != "") {
        outputStream << "# " << comment << "\n";
      }
      outputStream << outputImage.columns() << " " << outputImage.rows() << "\n"
                   << "255\n";

      // Check for errors.
      if(!outputStream) {
        std::ostringstream message;
        message << "Couldn't write image header to file: " << fileName;
        DLR_THROW(IOException, "writePGM8()", message.str().c_str());
      }

      // And write the pixels.
      outputStream.write(reinterpret_cast<const char*>(outputImage.data()),
                         outputImage.size());

      // Check for errors.
      if(!outputStream) {
        std::ostringstream message;
        message << "Error writeing image data to output file: " << fileName;
        DLR_THROW(IOException, "writePGM8()", message.str().c_str());
      }
    
      // All done!
      outputStream.close();
    }
  

    void
    writePGM16(const std::string& fileName,
               const Image<GRAY16>& outputImage,
               const std::string& comment)
    {
      std::ofstream outputStream(fileName.c_str(), std::ios::binary);
      if(!outputStream) {
        std::ostringstream message;
        message << "Couldn't open output file: " << fileName;
        DLR_THROW(IOException, "writePGM16()", message.str().c_str());
      }

      // Write the header.
      outputStream << "P5\n";
      if(comment != "") {
        outputStream << "# " << comment << "\n";
      }
      outputStream << outputImage.columns() << " " << outputImage.rows() << "\n"
                   << "65535\n";

      // Check for errors.
      if(!outputStream) {
        std::ostringstream message;
        message << "Couldn't write image header to file: " << fileName;
        DLR_THROW(IOException, "writePGM16()", message.str().c_str());
      }

      // And write the pixels.
      size_t numberOfElements = outputImage.size();
      size_t numberOfBytes =
        numberOfElements * sizeof(Image<GRAY16>::value_type);
      if(getByteOrder() == DLR_BIG_ENDIAN) {
        outputStream.write(
          reinterpret_cast<const char*>(outputImage.data()), numberOfBytes);
      } else {
        char* swabbedData = new char[numberOfBytes];
        switchByteOrder(
          outputImage.data(), numberOfElements,
          reinterpret_cast<Image<GRAY16>::value_type*>(swabbedData),
          getByteOrder(), DLR_BIG_ENDIAN);
        outputStream.write(swabbedData, numberOfBytes);
        delete[] swabbedData;
      }

      // Check for errors.
      if(!outputStream) {
        outputStream.close();
        std::ostringstream message;
        message << "Error writeing image data to output file: " << fileName;
        DLR_THROW(IOException, "writePGM16()", message.str().c_str());
      }
    
      // All done!
      outputStream.close();
    }
  

    void
    writePPM8(const std::string& fileName,
              const Image<RGB8>& outputImage,
              const std::string& comment)
    {
      std::ofstream outputStream(fileName.c_str(), std::ios::binary);
      if(!outputStream) {
        std::ostringstream message;
        message << "Couldn't open output file: " << fileName;
        DLR_THROW(IOException, "writePPM8()", message.str().c_str());
      }

      // Write the header.
      outputStream << "P6\n";
      if(comment != "") {
        outputStream << "# " << comment << "\n";
      }
      outputStream << outputImage.columns() << " " << outputImage.rows() << "\n"
                   << "255\n";

      // Check for errors.
      if(!outputStream) {
        std::ostringstream message;
        message << "Couldn't write image header to file: " << fileName;
        DLR_THROW(IOException, "writePPM8()", message.str().c_str());
      }

      // And write the pixels.
      outputStream.write(reinterpret_cast<const char*>(outputImage.data()),
                         3 * outputImage.size());

      // Check for errors.
      if(!outputStream) {
        std::ostringstream message;
        message << "Error writeing image data to output file: " << fileName;
        DLR_THROW(IOException, "writePPM8()", message.str().c_str());
      }
    
      // All done!
      outputStream.close();
    }
  
  } // namespace computerVision    

} // namespace dlr
